﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "TexturePool.h"

#include <nn/nn_Log.h>
#include <nn/gfx.h>
#include <nn/util/util_ScopeExit.h>
#include "Framework.h"
#include "../Config.h"

namespace framework{

    void TexturePool::Initialize(const TexturePoolInfo& info) NN_NOEXCEPT
    {
        m_PoolTextureWidth  = info.textureWidth;
        m_PoolTextureHeight = info.textureHeight;
        m_PoolTextureFormat = info.textureFormat;
        m_PoolTextureCount  = info.textureCount;
        m_EntryList.resize(info.textureCount);
        for(auto& e : m_EntryList)
        {
            e.isUsed = false;
        }

        InitializePoolTexture();
    }

    void TexturePool::Finalize() NN_NOEXCEPT
    {
        FinalizePoolTexture();
    }

    void TexturePool::InitializePoolTexture() NN_NOEXCEPT
    {
        auto pDevice = Framework::GetDevice();

        int width  = m_PoolTextureWidth;
        int height = m_PoolTextureHeight * m_PoolTextureCount;
        auto format = m_PoolTextureFormat;

        nn::gfx::TextureInfo texInfo;
        texInfo.SetDefault();
        texInfo.SetWidth(width);
        texInfo.SetHeight(height);
        texInfo.SetImageFormat(format);
        texInfo.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
        texInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture);
        size_t size = nn::gfx::Texture::CalculateMipDataSize(pDevice, texInfo);
        size_t alignment = nn::gfx::Texture::CalculateMipDataAlignment(pDevice, texInfo);
        ptrdiff_t offset = Framework::AllocateTexturePoolMemory(size, alignment);

        m_PoolTexture.Initialize(pDevice, texInfo, Framework::GetTextureMemoryPool(), offset, size);
        m_PoolTextureOffset = offset;

        nn::gfx::TextureViewInfo viewInfo;
        viewInfo.SetDefault();
        viewInfo.SetImageFormat(format);
        viewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
        viewInfo.SetTexturePtr(&m_PoolTexture);
        m_PoolTextureView.Initialize(pDevice, viewInfo);

        m_PoolTextureDescriptorSlotIndex = Framework::AllocateTextureDescriptorSlot();
        Framework::GetTextureDescriptorPool()->SetTextureView(m_PoolTextureDescriptorSlotIndex, &m_PoolTextureView);
        Framework::GetTextureDescriptorPool()->GetDescriptorSlot(&m_PoolTextureDescriptorSlot, m_PoolTextureDescriptorSlotIndex);
    }

    void TexturePool::FinalizePoolTexture() NN_NOEXCEPT
    {
        m_PoolTextureView.Finalize(Framework::GetDevice());
        m_PoolTexture.Finalize(Framework::GetDevice());
        Framework::FreeTextureDescriptorSlot(m_PoolTextureDescriptorSlotIndex);
        Framework::FreeTexturePoolMemory(m_PoolTextureOffset);
    }

    std::shared_ptr<PoolTexture> TexturePool::Acquire(int width, int height, nn::gfx::ImageFormat format) NN_NOEXCEPT
    {
        if(width > m_PoolTextureWidth || height > m_PoolTextureHeight)
        {
            NN_DEVOVL_FW_LOG("[TexturePool] Acquire texture failed (invalid size)\n");
            return nullptr;
        }
        if(format != m_PoolTextureFormat)
        {
            NN_DEVOVL_FW_LOG("[TexturePool] Acquire texture failed (invalid format)\n");
            return nullptr;
        }

        int index = -1;
        for(int i = 0; i < m_PoolTextureCount; i++)
        {
            if(!m_EntryList[i].isUsed)
            {
                index = i;
                break;
            }
        }
        if(index < 0)
        {
            NN_DEVOVL_FW_LOG("[TexturePool] Acquire texture failed (no texture)\n");
            return nullptr;
        }

        m_EntryList[index].isUsed = true;
        return std::make_shared<PoolTexture>(this, index, width, height);
    }

    void TexturePool::ReleasePoolTexture(int index) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(index, 0, m_PoolTextureCount);
        NN_SDK_REQUIRES(m_EntryList[index].isUsed);

        m_EntryList[index].isUsed = false;
    }

    //------------------------------------

    PoolTexture::PoolTexture(TexturePool* pPool, int index, int width, int height) NN_NOEXCEPT
        : m_pPool(pPool)
        , m_Index(index)
        , m_Width(width)
        , m_Height(height)
    {
    }

    PoolTexture::~PoolTexture() NN_NOEXCEPT
    {
        m_pPool->ReleasePoolTexture(m_Index);
    }

    nn::gfx::Texture* PoolTexture::Get() NN_NOEXCEPT
    {
        return &m_pPool->m_PoolTexture;
    }

    int PoolTexture::GetOriginX() const NN_NOEXCEPT
    {
        return 0;
    }

    int PoolTexture::GetOriginY() const NN_NOEXCEPT
    {
        return m_pPool->m_PoolTextureHeight * m_Index;
    }

    int PoolTexture::GetWidth() const NN_NOEXCEPT
    {
        return m_Width;
    }

    int PoolTexture::GetHeight() const NN_NOEXCEPT
    {
        return m_Height;
    }

    int PoolTexture::GetPoolWidth() const NN_NOEXCEPT
    {
        return m_pPool->m_PoolTextureWidth;
    }

    int PoolTexture::GetPoolHeight() const NN_NOEXCEPT
    {
        return m_pPool->m_PoolTextureHeight * m_pPool->m_PoolTextureCount;
    }

    nn::gfx::ImageFormat PoolTexture::GetFormat() const NN_NOEXCEPT
    {
        return m_pPool->m_PoolTextureFormat;
    }

    const nn::gfx::DescriptorSlot& PoolTexture::GetDescriptorSlot() const NN_NOEXCEPT
    {
        return m_pPool->m_PoolTextureDescriptorSlot;
    }
}
