﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ThreadMessageChannel.h"

#include <nn/util/util_ScopeExit.h>
#include "Config.h"

//#define NN_DEVOVL_LOG_CHANNEL(...) NN_DEVOVL_LOG("[channel]" __VA_ARGS__)

#ifndef NN_DEVOVL_LOG_CHANNEL
#define NN_DEVOVL_LOG_CHANNEL(...)
#endif

ThreadMessageChannel::ThreadMessageChannel(int messageCountMax) NN_NOEXCEPT
{
    m_MessageCountMax = messageCountMax;
    nn::os::InitializeMutex(&m_Mutex, false, 0);
    nn::os::InitializeConditionVariable(&m_Condition);
    nn::os::InitializeEvent(&m_ReceivableEvent, false, nn::os::EventClearMode_ManualClear);
}

ThreadMessageChannel::~ThreadMessageChannel() NN_NOEXCEPT
{
    nn::os::FinalizeEvent(&m_ReceivableEvent);
    nn::os::FinalizeConditionVariable(&m_Condition);
    nn::os::FinalizeMutex(&m_Mutex);
}

// メッセージを送信する。送信するまでブロックする。
std::shared_ptr<ThreadMessageResultReceiver> ThreadMessageChannel::SendMessage(ThreadMessageType message, const std::shared_ptr<void>& pUserData) NN_NOEXCEPT
{
    NN_DEVOVL_LOG_CHANNEL("SendMessage\n");
    std::shared_ptr<ThreadMessageResultReceiver> pReceiver = nullptr;
    {
        nn::os::LockMutex(&m_Mutex);
        NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

        // 前のメッセージが消化されるまで待つ
        while(IsMessageQueueFull())
        {
            nn::os::WaitConditionVariable(&m_Condition, &m_Mutex);
        }

        pReceiver = std::make_shared<ThreadMessageResultReceiver>(pUserData);
        auto pReporter = std::make_shared<ThreadMessageResultReporter>(pReceiver);

        // 新しいメッセージを送る
        MessageEntry e;
        e.message = message;
        e.pReporter = pReporter;

        m_MessageQueue.push(e);
        nn::os::SignalEvent(&m_ReceivableEvent);
    }

    // メッセージが送信されたことを通知
    nn::os::SignalConditionVariable(&m_Condition);

    NN_DEVOVL_LOG_CHANNEL("SendMessage done\n");
    return pReceiver;
}

// メッセージを受信する。受信するまでブロックする。
void ThreadMessageChannel::ReceiveMessage(ThreadMessageType* pOutMessage, std::shared_ptr<ThreadMessageResultReporter>* pOutReporter) NN_NOEXCEPT
{
    NN_DEVOVL_LOG_CHANNEL("ReceiveMessage\n");
    NN_SDK_REQUIRES_NOT_NULL(pOutMessage);
    NN_SDK_REQUIRES_NOT_NULL(pOutReporter);
    {
        nn::os::LockMutex(&m_Mutex);
        NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

        while(IsMessageQueueEmpty())
        {
            nn::os::WaitConditionVariable(&m_Condition, &m_Mutex);
        }

        auto e = m_MessageQueue.front();
        m_MessageQueue.pop();
        if(m_MessageQueue.size() == 0)
        {
            nn::os::ClearEvent(&m_ReceivableEvent);
        }

        *pOutMessage = e.message;
        *pOutReporter = std::move(e.pReporter);
    }

    // メッセージが受信されたことを通知
    nn::os::SignalConditionVariable(&m_Condition);
    NN_DEVOVL_LOG_CHANNEL("ReceiveMessage done\n");
}

// メッセージを受信する。なにもなければ false が返る
bool ThreadMessageChannel::TryReceiveMessage(ThreadMessageType* pOutMessage, std::shared_ptr<ThreadMessageResultReporter>* pOutReporter) NN_NOEXCEPT
{
    NN_DEVOVL_LOG_CHANNEL("TryReceiveMessage\n");
    NN_SDK_REQUIRES_NOT_NULL(pOutMessage);
    NN_SDK_REQUIRES_NOT_NULL(pOutReporter);
    {
        nn::os::LockMutex(&m_Mutex);
        NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

        if(IsMessageQueueEmpty())
        {
            NN_DEVOVL_LOG_CHANNEL("TryReceiveMessage no-message\n");
            return false;
        }

        auto e = m_MessageQueue.front();
        m_MessageQueue.pop();
        if(m_MessageQueue.size() == 0)
        {
            nn::os::ClearEvent(&m_ReceivableEvent);
        }

        *pOutMessage = e.message;
        *pOutReporter = std::move(e.pReporter);
    }

    // メッセージが受信されたことを通知
    nn::os::SignalConditionVariable(&m_Condition);
    NN_DEVOVL_LOG_CHANNEL("TryReceiveMessage done\n");
    return true;
}

void ThreadMessageChannel::IntializeReceivableEventHolder(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
{
    nn::os::InitializeMultiWaitHolder(pHolder, &m_ReceivableEvent);
}


bool ThreadMessageChannel::IsMessageQueueEmpty() const NN_NOEXCEPT
{
    return m_MessageQueue.empty();
}

bool ThreadMessageChannel::IsMessageQueueFull() const NN_NOEXCEPT
{
    return m_MessageQueue.size() == static_cast<size_t>(m_MessageCountMax);
}


//---------------------

ThreadMessageResultReceiver::ThreadMessageResultReceiver(const std::shared_ptr<void>& pUserData) NN_NOEXCEPT
{
    NN_DEVOVL_LOG_CHANNEL("receiver ctor\n");
    m_IsComplete = false;
    nn::os::InitializeEvent(&m_CompleteEvent, false, nn::os::EventClearMode_ManualClear);
    m_pUserData = pUserData;
}

ThreadMessageResultReceiver::~ThreadMessageResultReceiver() NN_NOEXCEPT
{
    NN_DEVOVL_LOG_CHANNEL("receiver dtor\n");
    nn::os::FinalizeEvent(&m_CompleteEvent);
}

bool ThreadMessageResultReceiver::TryWaitComplete() NN_NOEXCEPT
{
    if(m_IsComplete)
    {
        return true;
    }

    m_IsComplete = nn::os::TryWaitEvent(&m_CompleteEvent);
    return m_IsComplete;
}

void ThreadMessageResultReceiver::WaitCompletion() NN_NOEXCEPT
{
    if(m_IsComplete)
    {
        return;
    }
    nn::os::WaitEvent(&m_CompleteEvent);
    m_IsComplete = true;
}

void ThreadMessageResultReceiver::InitializeCompletionMultiWaitHolder(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
{
    nn::os::InitializeMultiWaitHolder(pHolder, &m_CompleteEvent);
}

ThreadMessageResult ThreadMessageResultReceiver::GetResult() const NN_NOEXCEPT
{
    return m_Result;
}

std::shared_ptr<void> ThreadMessageResultReceiver::GetUserData() const NN_NOEXCEPT
{
    return m_pUserData;
}


void ThreadMessageResultReceiver::NotifyCompletion(ThreadMessageResult result) NN_NOEXCEPT
{
    NN_DEVOVL_LOG_CHANNEL("receiver notified\n");
    m_Result = result;
    nn::os::SignalEvent(&m_CompleteEvent);
}

//---------------------

ThreadMessageResultReporter::ThreadMessageResultReporter(const std::shared_ptr<ThreadMessageResultReceiver>& pReceiver) NN_NOEXCEPT
{
    NN_DEVOVL_LOG_CHANNEL("reporter ctor\n");
    m_IsResultNotified = false;
    m_pReceiver = pReceiver;
}

ThreadMessageResultReporter::~ThreadMessageResultReporter() NN_NOEXCEPT
{
    NN_DEVOVL_LOG_CHANNEL("reporter dtor\n");
    if(!m_IsResultNotified)
    {
        m_pReceiver->NotifyCompletion(ThreadMessageResult_Aborted);
        m_IsResultNotified = true;
    }
}

std::shared_ptr<void> ThreadMessageResultReporter::GetUserData() const NN_NOEXCEPT
{
    return m_pReceiver->GetUserData();
}

void ThreadMessageResultReporter::NotifyCompletion(ThreadMessageResult result) NN_NOEXCEPT
{
    m_pReceiver->NotifyCompletion(result);
    m_IsResultNotified = true;
}
