﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <unordered_map>

#include <nn/bcat/bcat_ApiAdmin.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/time.h>
#include <nn/util/util_StringUtil.h>

#include "DevMenu_SaveData.h"
#include "DevMenu_SaveDataConfig.h"
#include "../Accounts/DevMenu_AccountsSdkHelper.h"
#include "../DevMenu_Config.h"

#if defined ( NN_CUSTOMERSUPPORTTOOL )
    #include "DevMenu_SaveDataCustomerSupportTool.h"
#endif

#include "../../BackupSaveData/BackupSaveData.h"

namespace {

#if defined ( NN_CUSTOMERSUPPORTTOOL )
    const int SaveDataUserIdFileNameLength = ( 8 * 4 ) + 3 + 1;  //!< UserId文字列長 XXXXXXXX_XXXXXXXX_XXXXXXXX_XXXXXXXXと終端
#endif

} // end of anonymous namespace

namespace devmenu { namespace savedata {

/**
 * @brief セーブデータ読み込みの結果を示します。
 */
enum DevMenuSaveData
{
    DevMenuSaveData_Exists,      //!< DevMenu 以外のセーブデータです。
    DevMenuSaveData_None,        //!< セーブデータが存在しません。
    DevMenuSaveData_Failure,     //!< セーブデータの読み込みに失敗しました。
};

/*********************************
 * struct SaveDataProperty
 *********************************/

/**
 * @brief セーブデータプロパティ型です。
 */
template< size_t N >
class SaveDataProperty
{
public: // メンバ関数

    /**
     * @brief セーブデータが有効か返します。
     */
    bool IsValid() const NN_NOEXCEPT;

    /**
     * @brief アプリケーション ID を返します。
     */
    const nn::ncm::ApplicationId GetApplicationId() const NN_NOEXCEPT;

    /**
     * @brief セーブデータリスト用プロパティデータを構築します。
     * @details イテレータからセーブデータをフェッチします。
     * @details イテレータのカレントから 1 要素を取得します。@n
     * イテレータは取得時に一つ進みます。
     * @return DevMenuSaveData_Exists(存在する), DevMenuSaveData_None(存在しない), DevMenuSaveData_Failure(取得失敗) のいずれかが返ります。
     */
    DevMenuSaveData Fetch( nn::fs::SaveDataIterator& iterator ) NN_NOEXCEPT;

    /**
     * @brief プロパティが示すセーブデータを SD カードにエクスポートします。
     */
    const nn::Result ExportSaveData() const NN_NOEXCEPT;

    const nn::Result ExportSaveDataImpl( const nn::time::CalendarTime& cTime ) const NN_NOEXCEPT;

    /**
     * @brief セーブデータが有効か判定します。
     */
    const nn::Result Verify( nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataId saveDataId ) NN_NOEXCEPT;

    /**
     * @brief プロパティが示すセーブデータを削除します。
     */
    const nn::Result Delete() const NN_NOEXCEPT;

    /**
     * @brief プロパティが示すセーブデータを破壊します。
     */
    const nn::Result Corrupt() const NN_NOEXCEPT;

    /**
     * @brief プロパティが示すセーブデータを RAW データも含め SD カードにエクスポートします。
     */
    const nn::Result ExportRawSaveData() const NN_NOEXCEPT;

    const nn::Result ExportRawSaveDataImpl( const nn::time::CalendarTime& cTime ) const NN_NOEXCEPT;

    /**
     * @brief Get the current SaveDataInfo - primarily used to initialize the Details scene
     */
    const nn::fs::SaveDataInfo& GetSaveDataInfo() const NN_NOEXCEPT;

    bool operator<( const SaveDataProperty& right ) const NN_NOEXCEPT;


public: // メンバ変数

    char                    saveDataIdStr[ 32 ]; // 参照せずに saveDataIdWideStr を変換すれば削除可能

    // リストビュー
    glv::WideCharacterType  saveDataIdWideStr[ N ];
    glv::WideCharacterType  saveDataTypeWideStr[ N ];
    glv::WideCharacterType  applicationIdWideStr[ N ];

    // 下部の詳細情報
    char                    systemSaveDataIdStr[ 32 ];
    char                    saveDataUserIdStr[ 64 ];
    char                    availableSizeStr[ 32 ];
    char                    journalSizeStr[ 32 ];
    char                    flagStr[ 32 ];

private: // メンバ定数

    static const char* DataIdStatementFormat;
    static const char* SizeFormat;

private: // メンバ関数

#if !defined( DEBUG_DATA )

    /**
     * @brief セーブデータの種類を取得します。アカウントセーブデータの場合はニックネームも合わせて取得します。
     */
    void GetTypeAndNickname() NN_NOEXCEPT;

#endif

    static const std::string GetUint32String( uint32_t value ) NN_NOEXCEPT;

    static const std::string GetUint64String( uint64_t value ) NN_NOEXCEPT;

    static const std::string GetInt64String( int64_t value ) NN_NOEXCEPT;

    /**
     * @brief アプリケーション ID を文字列に変換して取得します。
     */
    static const std::string GetApplicationIdString( uint64_t applicationIdValue ) NN_NOEXCEPT;

    static const std::string GetSaveDataCountString( uint8_t count ) NN_NOEXCEPT;

    /**
     * @brief 指定したパスにディレクトリが存在しなければ作成します。
     */
    static const nn::Result CreateDirectoryIfNotExists( const std::string& path ) NN_NOEXCEPT;

    static uint8_t GetTitleSaveDataCount( uint64_t applicationIdValue ) NN_NOEXCEPT;

    /**
     * @brief interger 型の数値を指定した長さの文字列に変化して返します。
     */
    template< typename T >
    static const std::string DateIntValueToString( T data, std::streamsize width ) NN_NOEXCEPT;

private: // メンバ変数

    nn::fs::SaveDataInfo    m_SaveDataInfo;
    bool                    m_IsValid = true;
};

/**
 * @brief セーブデータプロパティ( 48 文字上限 )型です。
 */
typedef SaveDataProperty< 48 > SaveDataPropertyType;

/*********************************
 * class SaveDataProperty
 *********************************/

#if !defined( DEBUG_DATA )

/**
 * @brief セーブデータが有効か返します。
 */
template< size_t N > bool SaveDataProperty< N >::IsValid() const NN_NOEXCEPT
{
    return m_IsValid;
}

/**
 * @brief アプリケーション ID を返します。
 */
template< size_t N > const nn::ncm::ApplicationId SaveDataProperty< N >::GetApplicationId() const NN_NOEXCEPT
{
    return m_SaveDataInfo.applicationId;
}

/**
 * @brief セーブデータリスト用プロパティデータを構築します。
 * @details イテレータからセーブデータをフェッチします。
 * @details イテレータのカレントから 1 要素を取得します。@n
 * イテレータは取得時に一つ進みます。
 * @return true の場合取得成功。
 */
template< size_t N > DevMenuSaveData SaveDataProperty< N >::Fetch( nn::fs::SaveDataIterator& iterator ) NN_NOEXCEPT
{
    int64_t fetchCount;
    auto result = iterator.ReadSaveDataInfo( &fetchCount, &m_SaveDataInfo, 1 );

    if ( result.IsFailure() )
    {
        DEVMENU_LOG( "Failed to read save data info as 0x%08x\n", result.GetInnerValueForDebug() );
        return DevMenuSaveData_Failure;
    }
    else if ( 0 >= fetchCount )
    {
        return DevMenuSaveData_None;
    }

    BuildUtf16< N >( saveDataIdWideStr, SaveDataProperty< N >::DataIdStatementFormat, m_SaveDataInfo.saveDataId );

    // applicationId
    if ( nn::fs::InvalidProgramId != m_SaveDataInfo.applicationId )
    {
        BuildUtf16< N >( applicationIdWideStr, SaveDataProperty< N >::DataIdStatementFormat, m_SaveDataInfo.applicationId.value );
    }
    else
    {
        BuildUtf16< N >( applicationIdWideStr, "%s", "N/A" );
    }

    // systemSaveDataId
    if ( m_SaveDataInfo.systemSaveDataId != nn::fs::InvalidSystemSaveDataId )
    {
        nn::util::SNPrintf( systemSaveDataIdStr, sizeof( systemSaveDataIdStr ), SaveDataProperty< N >::DataIdStatementFormat, m_SaveDataInfo.systemSaveDataId );
    }
    else
    {
        nn::util::SNPrintf( systemSaveDataIdStr, sizeof( systemSaveDataIdStr ), "N/A" );
    }

    // saveDataUserId, Type(Nickname)
    if ( nn::fs::InvalidUserId != m_SaveDataInfo.saveDataUserId )
    {
        GetTypeAndNickname();
    }
    else
    {
        nn::util::SNPrintf( saveDataUserIdStr, sizeof( saveDataUserIdStr ), "N/A" );
        BuildUtf16< N >( saveDataTypeWideStr, "%s", GetTypeString( m_SaveDataInfo.saveDataType, m_SaveDataInfo.saveDataSpaceId, m_SaveDataInfo.index ) );
    }

    nn::util::SNPrintf( saveDataIdStr, sizeof( saveDataIdStr ), DataIdStatementFormat, m_SaveDataInfo.saveDataId );

    {
        // 一時的に GET API が Abort しないようにする
        nn::fs::SetEnabledAutoAbort( false );

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::SetEnabledAutoAbort( true );
        };

        int64_t availableSize = 0;
        result = nn::fs::GetSaveDataAvailableSize( &availableSize, m_SaveDataInfo.saveDataSpaceId, m_SaveDataInfo.saveDataId );
        if ( result.IsSuccess() )
        {
            ConvertBytesToLargestType( availableSizeStr, sizeof( availableSizeStr ), availableSize );
        }
        else
        {
            DEVMENU_LOG( "Failed to get save data[%s] available size as 0x%08x\n", saveDataIdWideStr, result.GetInnerValueForDebug() );
            nn::util::SNPrintf( availableSizeStr, sizeof( availableSizeStr ), "Unknown" );
        }

        int64_t journalSize = 0;
        result = nn::fs::GetSaveDataJournalSize( &journalSize, m_SaveDataInfo.saveDataSpaceId, m_SaveDataInfo.saveDataId );
        if ( result.IsSuccess() )
        {
            ConvertBytesToLargestType( journalSizeStr, sizeof( journalSizeStr ), journalSize );
        }
        else
        {
            DEVMENU_LOG( "Failed to get save data[%s] journal size as 0x%08x\n", saveDataIdWideStr, result.GetInnerValueForDebug() );
            nn::util::SNPrintf( journalSizeStr, sizeof( journalSizeStr ), "Unknown" );
        }

        uint32_t flag = 0;
        result = nn::fs::GetSaveDataFlags( &flag, m_SaveDataInfo.saveDataSpaceId, m_SaveDataInfo.saveDataId );
        if ( result.IsSuccess() )
        {
            nn::util::SNPrintf( flagStr, sizeof( flagStr ), "0x%08X", flag );
        }
        else
        {
            DEVMENU_LOG( "Failed to get save data[%s] flags as 0x%08x\n", saveDataIdWideStr, result.GetInnerValueForDebug() );
            nn::util::SNPrintf( flagStr, sizeof( flagStr ), "Unknown" );
        }
    }

    return DevMenuSaveData_Exists;
}

template< size_t N > void SaveDataProperty< N >::GetTypeAndNickname() NN_NOEXCEPT
{
    nn::account::Uid uid;

    uid._data[ 0 ] = m_SaveDataInfo.saveDataUserId._data[ 0 ];
    uid._data[ 1 ] = m_SaveDataInfo.saveDataUserId._data[ 1 ];

    accounts::UidToString( saveDataUserIdStr, sizeof( saveDataUserIdStr ), uid );

    char buffer[ 64 ];

    if ( nn::fs::SaveDataType::Account == m_SaveDataInfo.saveDataType )
    {
        nn::account::Nickname name;
        auto result = nn::account::GetNickname( &name, uid );

        if ( result.IsSuccess() )
        {
            nn::util::SNPrintf( buffer, sizeof( buffer ), "%s (%s)", GetTypeString( m_SaveDataInfo.saveDataType ), name.name );
        }
        else
        {
            nn::util::SNPrintf( buffer, sizeof( buffer ), "%s (%s)", GetTypeString( m_SaveDataInfo.saveDataType ), "N/A" );
        }
    }
    else
    {
        nn::util::Strlcpy( buffer, GetTypeString( m_SaveDataInfo.saveDataType ), sizeof( buffer ) );
    }

    BuildUtf16< N >( saveDataTypeWideStr, "%s", buffer );

    const size_t maxLength = 20;
    // 表示可能領域の都合により、一定文字数以上の場合は確実に表示できるであろう長さまでカットする
    if ( glv::WideString( saveDataTypeWideStr ).size() > maxLength )
    {
        saveDataTypeWideStr[ maxLength - 1 ] = GLV_TEXT_API_WIDE_STRING( ')' );
        saveDataTypeWideStr[ maxLength ] = GLV_TEXT_API_WIDE_STRING( '\0' );
    }
}

/**
 * @brief プロパティが示すセーブデータを SD カードにエクスポートします。
 */
template< size_t N > const nn::Result SaveDataProperty< N >::ExportSaveData() const NN_NOEXCEPT
{
#if !defined ( NN_BUILD_CONFIG_OS_WIN )
    // SD カードをマウント
    NN_RESULT_DO( nn::fs::MountSdCardForDebug( "sd" ) );
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount( "sd" );
    };

    nn::time::PosixTime posixTime;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetCurrentTime( &posixTime ) );

    nn::time::CalendarTime cTime;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::ToCalendarTime( &cTime, nullptr, posixTime ) );

    NN_RESULT_DO( ExportSaveDataImpl( cTime ) );
#endif // !defined ( NN_BUILD_CONFIG_OS_WIN )

    NN_RESULT_SUCCESS;
}

template< size_t N > const nn::Result SaveDataProperty< N >::ExportSaveDataImpl( const nn::time::CalendarTime& cTime ) const NN_NOEXCEPT
{
    nn::Bit64 ownerId;
    uint32_t saveDataFlags;
    int64_t saveDataAvailableSize;
    int64_t saveDataJournalSize;
    NN_RESULT_DO( nn::fs::GetSaveDataOwnerId( &ownerId, m_SaveDataInfo.saveDataSpaceId, m_SaveDataInfo.saveDataId ) );
    NN_RESULT_DO( nn::fs::GetSaveDataFlags( &saveDataFlags, m_SaveDataInfo.saveDataSpaceId, m_SaveDataInfo.saveDataId ) );
    NN_RESULT_DO( nn::fs::GetSaveDataAvailableSize( &saveDataAvailableSize, m_SaveDataInfo.saveDataSpaceId, m_SaveDataInfo.saveDataId ) );
    NN_RESULT_DO( nn::fs::GetSaveDataJournalSize( &saveDataJournalSize, m_SaveDataInfo.saveDataSpaceId, m_SaveDataInfo.saveDataId ) );

    nn::ncm::ApplicationId applicationId;
    applicationId.value = m_SaveDataInfo.applicationId.value;
    NN_RESULT_DO( MountSaveData( m_SaveDataInfo.saveDataType, "save", applicationId, m_SaveDataInfo.saveDataUserId, m_SaveDataInfo.index ) );
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount( "save" );
    };
    // sd:/NintendoSDK/BackupSaveData ディレクトリの作成
    NN_RESULT_DO( CreateDirectoryIfNotExists( "sd:/NintendoSDK" ) );
    NN_RESULT_DO( CreateDirectoryIfNotExists( DataMoveInfo::ExportPathRoot ) );

    // sd:/NintendoSDK/BackupSaveData/<date_applicationId> ディレクトリの作成
    std::string exportPathSubRoot = DataMoveInfo::ExportPathRoot + '/'
                            + DateIntValueToString( cTime.year  , 4 ) + DateIntValueToString( cTime.month,  2 )
                            + DateIntValueToString( cTime.day   , 2 ) + DateIntValueToString( cTime.hour ,  2 )
                            + DateIntValueToString( cTime.minute, 2 ) + DateIntValueToString( cTime.second, 2 )
                            + '_' + GetApplicationIdString( m_SaveDataInfo.applicationId.value );

#if !defined( NN_CUSTOMERSUPPORTTOOL )
    NN_RESULT_DO( CreateDirectoryIfNotExists( exportPathSubRoot ) );

    // config.xml の作成
    if ( !ExistsFile( exportPathSubRoot + '/' + DataMoveInfo::ConfigFileName ) ) // Note: このチェックは不要かも
    {
        // Partial な config.xml を作成する
        NN_RESULT_DO( CreateConfigFile( exportPathSubRoot, true ) );
    }
#else
    // config.xml のデータをバッファに保存する
    const int MaxConfigXmlBufferSize = 1024;
    std::unique_ptr< char[] > configXmlBuffer( new char[ MaxConfigXmlBufferSize ] );
    int configXmlBufferSize = 0;

    // Partial な config.xml を作成する
    NN_RESULT_DO( CreateConfigFileToMemory( configXmlBuffer.get(), true, &configXmlBufferSize ) );
#endif

    // sd:/NintendoSDK/BackupSaveData/<date>/<saveDataType> ディレクトリの作成
    std::string exportPath = exportPathSubRoot;

    for ( const auto& iter : DataMoveInfo::TypeDirectoryInfo )
    {
        if ( iter.type == m_SaveDataInfo.saveDataType )
        {
            exportPath += iter.name;
            break;
        }
    }

#if !defined( NN_CUSTOMERSUPPORTTOOL )
    NN_RESULT_DO( CreateDirectoryIfNotExists( exportPath ) );
#endif

    // sd:/NintendoSDK/BackupSaveData/<date>/user/<applicationId> ディレクトリの作成
    exportPath += GetApplicationIdString( m_SaveDataInfo.applicationId.value );

#if !defined( NN_CUSTOMERSUPPORTTOOL )
    NN_RESULT_DO( CreateDirectoryIfNotExists( exportPath ) );
#endif

    // sd:/NintendoSDK/BackupSaveData/<date>/<saveDataType>/<applicationId>/<xxxx> ディレクトリの作成
    const uint8_t titleSaveDataCount = GetTitleSaveDataCount( m_SaveDataInfo.applicationId.value );
    exportPath += "/" + GetSaveDataCountString( titleSaveDataCount );

#if !defined( NN_CUSTOMERSUPPORTTOOL )
    NN_RESULT_DO( CreateDirectoryIfNotExists( exportPath ) );
#endif

#if 0
    DEVMENU_LOG( "data info : %016lx\n"
                    "            %016lx %016lx\n"
                    "            %016lx\n"
                    "            %016lx\n"
                    "            %016lx\n"
                    "            %016lx\n"
                    "            %08x\n"
                    "            %08x\n"
                    "            %08x\n",
                    m_SaveDataInfo.saveDataType, m_SaveDataInfo.applicationId, m_SaveDataInfo.saveDataUserId, ownerId, m_SaveDataInfo.saveDataSize, saveDataAvailableSize, saveDataJournalSize, saveDataFlags, m_SaveDataInfo.saveDataSpaceId, m_SaveDataInfo.index );
#endif

#if !defined( NN_CUSTOMERSUPPORTTOOL )

    // Note: セーブデータの書き出しに失敗した場合を考慮すると、書き出し後に info ファイルを作成する方がベター
    NN_RESULT_DO( CreateSaveDataInfoFile( exportPath,
                                            std::string( GetTypeString( m_SaveDataInfo.saveDataType, true ) ),
                                            GetUint64String( m_SaveDataInfo.saveDataUserId._data[ 0 ] ),
                                            GetUint64String( m_SaveDataInfo.saveDataUserId._data[ 1 ] ),
                                            GetUint64String( m_SaveDataInfo.applicationId.value ),
                                            GetUint64String( m_SaveDataInfo.saveDataSize ),
                                            GetUint64String( ownerId ),
                                            GetUint32String( saveDataFlags ),
                                            GetInt64String( saveDataAvailableSize ),
                                            GetInt64String( saveDataJournalSize ),
                                            GetUint32String( static_cast<uint32_t>(m_SaveDataInfo.saveDataSpaceId) ),
                                            GetUint32String( m_SaveDataInfo.index ) ) );

    // セーブデータから SD カードへの書き出し
    NN_RESULT_DO( CopySaveData( exportPath, "save:/" ) );

#else

    const int MaxbackupXmlBufferSize = 1024;
    std::unique_ptr< char[] > backupXmlBuffer( new char[ MaxbackupXmlBufferSize ] );
    int backupXmlBufferSize = 0;
    NN_RESULT_DO( CreateSaveDataInfoFileToMemory( backupXmlBuffer.get(),
                                            std::string( GetTypeString( m_SaveDataInfo.saveDataType, true ) ),
                                            GetUint64String( m_SaveDataInfo.saveDataUserId._data[0] ),
                                            GetUint64String( m_SaveDataInfo.saveDataUserId._data[1] ),
                                            GetUint64String( m_SaveDataInfo.applicationId.value ),
                                            GetUint64String( m_SaveDataInfo.saveDataSize ),
                                            GetUint64String( ownerId ),
                                            GetUint32String( saveDataFlags ),
                                            GetInt64String( saveDataAvailableSize ),
                                            GetInt64String( saveDataJournalSize ),
                                            &backupXmlBufferSize,
                                            GetUint32String( static_cast<uint32_t>(m_SaveDataInfo.saveDataSpaceId) ),
                                            GetUint32String( m_SaveDataInfo.index ) ) );

    char fileNameSaveDataUserIdStr[ SaveDataUserIdFileNameLength ];
    nn::account::Uid uid;
    uid._data[ 0 ] = m_SaveDataInfo.saveDataUserId._data[ 0 ];
    uid._data[ 1 ] = m_SaveDataInfo.saveDataUserId._data[ 1 ];
    accounts::UidToString( fileNameSaveDataUserIdStr, sizeof( fileNameSaveDataUserIdStr ), uid );
    // メモリ経由でのSDカードへの書き出し
    const std::string exportFile = exportPathSubRoot + "_" + fileNameSaveDataUserIdStr + DataMoveInfo::NsaveFilenameExtension.c_str();
    NN_RESULT_DO( CreatePartitionFsArchive( exportFile, exportPath, "save:/", backupXmlBuffer.get(), backupXmlBufferSize, configXmlBuffer.get(), configXmlBufferSize ) );

#endif

    NN_RESULT_SUCCESS;
}

/**
 * @brief プロパティが示すセーブデータを検証します。
 */
template< size_t N > const nn::Result SaveDataProperty< N >::Verify( nn::fs::SaveDataSpaceId saveDataSpaceId, nn::fs::SaveDataId saveDataId ) NN_NOEXCEPT
{
    const size_t verifyBufferSize = 64 * 1024;

    static char s_WorkBuffer[ verifyBufferSize ]; // 動的に確保するとパフォーマンスがかなり落ちるので static に

    NN_RESULT_TRY( nn::fs::VerifySaveData( &m_IsValid, saveDataSpaceId, saveDataId, s_WorkBuffer, sizeof( s_WorkBuffer ) ) )
        NN_RESULT_CATCH( nn::fs::ResultTargetLocked )
        {
            // ロックされているファイルでは VerifySaveData はできないので、正常と返します。
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

/**
 * @brief プロパティが示すセーブデータを削除します。
 */
template< size_t N > const nn::Result SaveDataProperty< N >::Delete() const NN_NOEXCEPT
{
    // ロットチェック向けの DevMenu は権限を絞りたいため、そちらは今までの削除 API を呼び出すようにする
#if !defined( NN_DEVMENULOTCHECK )
    if ( m_SaveDataInfo.saveDataType == nn::fs::SaveDataType::Bcat )
    {
        nn::ApplicationId applicationId = { m_SaveDataInfo.applicationId.value };
        NN_RESULT_DO( nn::bcat::DeleteDeliveryCacheStorage( applicationId ) );
        NN_RESULT_SUCCESS;
    }
#endif
    NN_RESULT_DO( nn::fs::DeleteSaveData( m_SaveDataInfo.saveDataSpaceId, m_SaveDataInfo.saveDataId ) );
    NN_RESULT_SUCCESS;
}

/**
 * @brief プロパティが示すセーブデータを破壊します。
 */
template< size_t N > const nn::Result SaveDataProperty< N >::Corrupt() const NN_NOEXCEPT
{
    NN_RESULT_DO( nn::fs::CorruptSaveDataForDebug( m_SaveDataInfo.saveDataSpaceId, m_SaveDataInfo.saveDataId ) );
    NN_RESULT_SUCCESS;
}

/**
 * @brief プロパティが示すセーブデータを RAW データを含め SD カードにエクスポートします。
 */
template< size_t N > const nn::Result SaveDataProperty< N >::ExportRawSaveData() const NN_NOEXCEPT
{
#if !defined ( NN_BUILD_CONFIG_OS_WIN )
    // SD カードをマウント
    NN_RESULT_DO( nn::fs::MountSdCardForDebug( "sd" ) );
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount( "sd" );
    };

    nn::time::PosixTime posixTime;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetCurrentTime( &posixTime ) );

    nn::time::CalendarTime cTime;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::ToCalendarTime( &cTime, nullptr, posixTime ) );

    NN_RESULT_DO( ExportSaveDataImpl( cTime ) );
    NN_RESULT_DO( ExportRawSaveDataImpl( cTime ) );
#endif // !defined ( NN_BUILD_CONFIG_OS_WIN )

    NN_RESULT_SUCCESS;
}

template< size_t N > const nn::Result SaveDataProperty< N >::ExportRawSaveDataImpl( const nn::time::CalendarTime& cTime ) const NN_NOEXCEPT
{
    /* SaveDataFs のキャッシュを追い出す（適当なセーブ(SaveDataIndexer)をマウント＆アンマウント） */
    NN_RESULT_DO( nn::fs::MountSystemSaveData( "save", 0x8000000000000000 ) );
    nn::fs::Unmount("save");

    std::string mountName = nn::fs::GetBisMountName(nn::fs::BisPartitionId::User);
    NN_RESULT_DO( nn::fs::MountBis( nn::fs::BisPartitionId::User, nullptr ) );
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount( mountName.c_str() );
    };

    // sd:/NintendoSDK/BackupSaveData ディレクトリの作成
    NN_RESULT_DO( CreateDirectoryIfNotExists( "sd:/NintendoSDK" ) );
    NN_RESULT_DO( CreateDirectoryIfNotExists( DataMoveInfo::ExportPathRoot ) );

    // sd:/NintendoSDK/BackupSaveData/<date_applicationId>_raw ディレクトリの作成
    std::string exportPathSubRoot = DataMoveInfo::ExportPathRoot + '/'
                            + DateIntValueToString( cTime.year  , 4 ) + DateIntValueToString( cTime.month,  2 )
                            + DateIntValueToString( cTime.day   , 2 ) + DateIntValueToString( cTime.hour ,  2 )
                            + DateIntValueToString( cTime.minute, 2 ) + DateIntValueToString( cTime.second, 2 )
                            + '_' + GetApplicationIdString( m_SaveDataInfo.applicationId.value ) + "_raw";
    NN_RESULT_DO( CreateDirectoryIfNotExists( exportPathSubRoot ) );

    // セーブデータ情報
    nn::fs::RawSaveDataInfo saveDataInfo;
    std::memset( &saveDataInfo, 0, sizeof(saveDataInfo) );

    NN_RESULT_DO( nn::fs::GetSaveDataOwnerId( &saveDataInfo.ownerId, m_SaveDataInfo.saveDataId ) );
    NN_RESULT_DO( nn::fs::GetSaveDataFlags( &saveDataInfo.flags, m_SaveDataInfo.saveDataId ) );
    NN_RESULT_DO( nn::fs::GetSaveDataAvailableSize( &saveDataInfo.availableSize, m_SaveDataInfo.saveDataId ) );
    NN_RESULT_DO( nn::fs::GetSaveDataJournalSize( &saveDataInfo.journalSize, m_SaveDataInfo.saveDataId ) );
    saveDataInfo.type = m_SaveDataInfo.saveDataType;
    saveDataInfo.applicationId.value = m_SaveDataInfo.applicationId.value;
    saveDataInfo.userId._data[0] = m_SaveDataInfo.saveDataUserId._data[0];
    saveDataInfo.userId._data[1] = m_SaveDataInfo.saveDataUserId._data[1];
    saveDataInfo.id = m_SaveDataInfo.saveDataId;

    // sd:/NintendoSDK/BackupSaveData/<date_applicationId>_raw/__BackupSaveDataInfo ファイルの作成
    std::string saveDataInfoFile = exportPathSubRoot + '/' + nn::fs::RawSaveDataInfoFileName;
    nn::fs::FileHandle fileHandle;
    NN_RESULT_DO( nn::fs::CreateFile( saveDataInfoFile.c_str(), sizeof(saveDataInfo) ) );
    NN_RESULT_DO( nn::fs::OpenFile( &fileHandle, saveDataInfoFile.c_str(), nn::fs::OpenMode_Write) );
    NN_RESULT_DO( nn::fs::WriteFile( fileHandle, 0, &saveDataInfo, sizeof(saveDataInfo), nn::fs::WriteOption() ) );
    NN_RESULT_DO( nn::fs::FlushFile( fileHandle ) );
    nn::fs::CloseFile( fileHandle );

    std::string srcPath = mountName + ":/save/" + GetUint64String( m_SaveDataInfo.saveDataId );
    // sd:/NintendoSDK/BackupSaveData/<date_applicationId>_raw/save ディレクトリの作成
    std::string exportPath = exportPathSubRoot + "/save";
    NN_RESULT_DO( CreateDirectoryIfNotExists( exportPath ) );

    exportPath = exportPath + '/' + GetUint64String( m_SaveDataInfo.saveDataId );
    // セーブデータの RAW データを SD カードへ書き出し
    NN_RESULT_DO( CopySaveData( exportPath, srcPath ) );

    srcPath = mountName + ":/saveMeta/" + GetUint64String( m_SaveDataInfo.saveDataId );
    nn::fs::DirectoryEntryType outValue;
    NN_RESULT_TRY( nn::fs::GetEntryType( &outValue, srcPath.c_str() ) );
        NN_RESULT_CATCH( nn::fs::ResultPathNotFound )
        {
            // bis:/saveMeta/<saveDataId> ディレクトリが存在しない場合はメタ情報のコピーをスキップ
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;

    // sd:/NintendoSDK/BackupSaveData/<date_applicationId>_raw/saveMeta ディレクトリの作成
    exportPath = exportPathSubRoot + "/saveMeta";
    NN_RESULT_DO( CreateDirectoryIfNotExists( exportPath ) );
    // sd:/NintendoSDK/BackupSaveData/<date_applicationId>_raw/saveMeta/<saveDataId> ディレクトリの作成
    exportPath = exportPath + '/' + GetUint64String( m_SaveDataInfo.saveDataId );
    NN_RESULT_DO( CreateDirectoryIfNotExists( exportPath ) );

    // セーブデータの メタ情報 RAW データを SD カードへ書き出し
    NN_RESULT_DO( CopySaveData( exportPath, srcPath ) );

    NN_RESULT_SUCCESS;
}

#else

static int g_DebugDataCountMax = 10;
static int g_DebugDataCount = 0;

/**
* @brief Define the IsValid function for Debug Data
*/
template< size_t N > bool SaveDataProperty< N >::IsValid() const NN_NOEXCEPT
{
    return true;
}

template< size_t N > const DevMenuSaveData SaveDataProperty< N >::Fetch( nn::fs::SaveDataIterator& iterator ) NN_NOEXCEPT
{
    int index;
    if ( g_DebugDataCountMax > ( index = g_DebugDataCount ) )
    {
        g_DebugDataCount = index + 1;
        m_SaveDataInfo.saveDataId = static_cast< nn::fs::SaveDataId >( index );
        m_SaveDataInfo.applicationId.value = static_cast< nn::Bit64 >( index );
        BuildUtf16< N >( saveDataIdWideStr, DataIdStatementFormat, static_cast< nn::fs::SaveDataId >( index ) );
        BuildUtf16< N >( applicationIdWideStr, DataIdStatementFormat, static_cast< nn::fs::SaveDataId >( index ) );
        BuildUtf16< N >( saveDataTypeWideStr, "%s", GetTypeString( m_SaveDataInfo.saveDataType ) );

        return DevMenuSaveData_Exists;
    }
    return DevMenuSaveData_None;
}

template< size_t N > nn::Result SaveDataProperty< N >::ExportUserSaveData() const NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

template< size_t N > nn::Result SaveDataProperty< N >::ExportUserSaveDataImpl() const NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

template< size_t N > nn::Result SaveDataProperty< N >::Delete() const NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_ADDRESS_SUPPORTS_64 )
    DEVMENU_LOG( "Request deleting[ 0x%016llX ]\n", m_SaveDataInfo.saveDataId );
#else
    DEVMENU_LOG( "Request deleting[ 0x%016lX ]\n", m_SaveDataInfo.saveDataId );
#endif

    if ( g_DebugDataCountMax > 0 )
    {
        --g_DebugDataCountMax;
    }
    NN_RESULT_SUCCESS;
}

template< size_t N > nn::Result SaveDataProperty< N >::Corrupt() const NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}
#endif


/**
 * @brief SaveDataInfo を取得します。
 */
template< size_t N > const nn::fs::SaveDataInfo& SaveDataProperty< N >::GetSaveDataInfo() const NN_NOEXCEPT
{
    return m_SaveDataInfo;
}

template< size_t N > bool SaveDataProperty< N >::operator<( const SaveDataProperty& right ) const NN_NOEXCEPT
{
    return applicationIdWideStr < right.applicationIdWideStr;
}

#if defined( NN_BUILD_CONFIG_ADDRESS_SUPPORTS_64 )
template< size_t N > const char* SaveDataProperty< N >::DataIdStatementFormat = "0x%016llX";
#else
template< size_t N > const char* SaveDataProperty< N >::DataIdStatementFormat = "0x%016lX";
#endif

#if defined( NN_BUILD_CONFIG_ADDRESS_SUPPORTS_64 )
template< size_t N > const char* SaveDataProperty< N >::SizeFormat = "%016lld";
#else
template< size_t N > const char* SaveDataProperty< N >::SizeFormat = "%016ld";
#endif


template< size_t N > const std::string SaveDataProperty< N >::GetUint32String( uint32_t value ) NN_NOEXCEPT
{
    char outString[ 16 ];
    memset( outString, 0, sizeof( outString ) );
    sprintf( outString, "%08x", value );
    return outString;
}

template< size_t N > const std::string SaveDataProperty< N >::GetUint64String( uint64_t value ) NN_NOEXCEPT
{
    char outString[ 32 ];
    memset( outString, 0, sizeof( outString ) );
    // TORIAEZU: uint64_t は unsigned long long のはずだが、unsigned long と言われるので明示的にキャスト
    nn::util::SNPrintf( outString, sizeof( outString ), "%016llx", static_cast< unsigned long long >( value ) );
    return outString;
}

template< size_t N > const std::string SaveDataProperty< N >::GetInt64String( int64_t value ) NN_NOEXCEPT
{
    return GetUint64String( value );
}

/**
 * @brief アプリケーション ID を文字列に変換して取得します。
 */
template< size_t N > const std::string SaveDataProperty< N >::GetApplicationIdString( uint64_t applicationIdValue ) NN_NOEXCEPT
{
    return GetUint64String( applicationIdValue );
}

template< size_t N > const std::string SaveDataProperty< N >::GetSaveDataCountString( uint8_t count ) NN_NOEXCEPT
{
    char countString[ 32 ];
    memset( countString, 0, sizeof( countString ) );
    sprintf( countString, "%04x", count );
    return countString;
}

/**
 * @brief 指定したパスにディレクトリが存在しなければ作成します。
 */
template< size_t N > const nn::Result SaveDataProperty< N >::CreateDirectoryIfNotExists( const std::string& path ) NN_NOEXCEPT
{
    NN_RESULT_TRY( nn::fs::CreateDirectory( path.c_str() ) );
        NN_RESULT_CATCH( nn::fs::ResultPathAlreadyExists )
        {
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_CATCH_ALL
        {
            DEVMENU_LOG( "error: failed to create directory %s (0x%08x)\n", path.c_str(), NN_RESULT_CURRENT_RESULT.GetInnerValueForDebug() );
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;

    DEVMENU_LOG_DEBUG( "Created %s\n", path.c_str() );
    NN_RESULT_SUCCESS;
}

template< size_t N > uint8_t SaveDataProperty< N >::GetTitleSaveDataCount( uint64_t applicationIdValue ) NN_NOEXCEPT
{
#if 1

    // 現状は 1 つの xxxx ディレクトリしか applicationId ディレクトリ下に作成しない
    return 1;

#else

    // BackupSaveData 同様に複数のセーブデータを applicationId ディレクトリ下に保存する場合
    static std::unordered_map< uint64_t, uint8_t > s_TitleSaveDataCountMap;

    if ( s_TitleSaveDataCountMap.count( applicationIdValue ) == 0 )
    {
        return s_TitleSaveDataCountMap[ applicationIdValue ] = 1;
    }
    return ++s_TitleSaveDataCountMap[ applicationIdValue ];

#endif
}

/**
 * @brief interger 型の数値を指定した長さの文字列に変化して返します。
 */
template< size_t N > template< typename T >
const std::string SaveDataProperty< N >::DateIntValueToString( T data, std::streamsize width ) NN_NOEXCEPT
{
    std::ostringstream out;
    out << std::setw( width ) << std::setfill( '0' ) << static_cast< int >( data );
    return out.str();
}

}} // ~namespace devmenu::savedata, ~namespace devmenu
