﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_FileSystem.h>

#include "../Common/DevMenu_CommonScene.h"
#include "../DevMenu_Common.h"
#include "../DevMenu_RootSurface.h"

namespace devmenu { namespace savedata {

enum SceneType
{
    SceneType_ApplicationSaveDataList,
#if defined ( NN_DEVMENUSYSTEM )
    SceneType_SystemSaveDataList,
#endif
    SceneType_ExportedSaveDataList,
    SceneType_SaveDataTypeList,
    SceneType_SaveDataDetail,
};

/**
 * @brief  SD カードをマウントします。
 */
const nn::Result MountSdCard( const char* name ) NN_NOEXCEPT;

/**
 * @brief 指定したパスにファイルが存在するかをチェックします。
 */
bool ExistsFile( const std::string& path ) NN_NOEXCEPT;

/**
 * @brief 指定したファイルがセーブデータの info ファイルと名前が一致するかを判別します。
 */
bool IsSaveDataInfoFile( const std::string& fileName ) NN_NOEXCEPT;

/**
 * @brief セーブデータをマウントします。
 */
const nn::Result MountSaveData( nn::fs::SaveDataType saveDataType, const std::string& mountName, nn::ncm::ApplicationId applicationId, nn::fs::UserId userId, int cacheIndex ) NN_NOEXCEPT;

/**
 * @brief 指定したディレクトリ以下のエントリを取得します。
 */
const nn::Result GetFsEntries( nn::fs::DirectoryHandle* pOutDirecotyHandle, std::vector< std::unique_ptr< nn::fs::DirectoryEntry> >* pOutDirectoryEntryAreaList, int64_t* readCount,
                         const std::string& directoryPath, nn::fs::OpenDirectoryMode mode ) NN_NOEXCEPT;

/**
 * @brief 指定したパスからマウントしたストレージ名を取得します。
 */
const std::string GetStorageName( const std::string& path ) NN_NOEXCEPT;

/**
 * @brief srcPath で指定したディレクトリ以下のセーブデータを destPath にコピーします。
 */
const nn::Result CopySaveData( const std::string& destPath, const std::string& srcPath, bool isJournalingEnabled = false, int64_t journalSize = 0  ) NN_NOEXCEPT;

/**
 * @brief SaveDataType の文字列を取得します。
 */
const char* GetTypeString( nn::fs::SaveDataType saveDataType, bool isEnumName = false ) NN_NOEXCEPT;

/**
 * @brief SaveDataType の文字列を取得します。
 */
const char* GetTypeString( nn::fs::SaveDataType saveDataType, nn::fs::SaveDataSpaceId spaceId, int cacheIndex ) NN_NOEXCEPT;

/*********************************
 * class SaveDataSceneBase
 *********************************/
class SaveDataSceneBase : public ProgressModalSceneBase
{
protected:
    SaveDataSceneBase( Page* pParentPage, const glv::Rect& rect, bool isDefaultbackButtonCallbackUsed = false, const std::function< void() >& backButtonCallback = nullptr ) NN_NOEXCEPT
        : ProgressModalSceneBase( pParentPage, rect, isDefaultbackButtonCallbackUsed, backButtonCallback )
    {
    }
};

}} // ~namespace devmenu::savedata, ~namespace devmenu

