﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*
 * インクルード
 */

#include <cstdlib>
#include <cstring>

#include "QREL.h"

/*
 * マクロ定義
 */
#define MALLOC_MOD

/*
 * 定数定義
 */
enum ConstantValue_SetMask
{
    CellOutArea = 0x80,       /* 除外エリア */
    QrHorizon = 0,                       /* 行 */
    QrVertucal = 1,                       /* 列 */
    MaskPatternMaxNum = 8,

/* 減点スコア */
    MaskDecreasePointN1 = 3,
    MaskDecreasePointN2 = 3,
    MaskDecreasePointN3 = 40,
    MaskDecreasePointN4 = 10,
};

/*
 * 外部変数定義
 */

/*
 * 内部関数定義
 */
static int32_t GetMaskPattern(int32_t i, int32_t j, int8_t mp);
static bool SetFunctionPattern(int32_t version, uint8_t **matrix);
#ifndef MALLOC_MOD
static int32_t GetMaskPatternScore(int32_t ncells, uint8_t **matrix);
#else
static int32_t GetMaskPatternScore(int32_t ncells, uint8_t **matrix, int32_t **, int32_t **);
#endif
static bool SetPDPPattern(uint8_t **matrix, int32_t x, int32_t y);
static bool SetAllPOPPattern(int32_t version, uint8_t **matrix);
static bool SetPOPPattern(uint8_t **matrix, int32_t x, int32_t y);
static int32_t GetDemeritScore_1(int32_t ncells, int32_t **runlength_hor,
                              int32_t **runlength_ver);
static int32_t GetDemeritScore_2(int32_t ncells, uint8_t **matrix);
static int32_t GetDemeritScore_3(int32_t ncells, int32_t **runlength_hor,
                              int32_t **runlength_ver);
static int32_t GetDemeritScore_4(int32_t ncells, uint8_t **matrix);
static bool SetRunLengthData(int32_t **runlength, int32_t ncells,
                              uint8_t **matrix,
                              int32_t dir);

/****************************************************************
 * 外部関数
 ****************************************************************/
/*
 * 関数名 QREL_SetMask
 * 機能   最終データワード列をマトリックスに配置する
 * 引数   versioninfo(IN)
 *        mp(IN/OUT)
 *          ecclevel(IN) 2003.02.13 Add A.Kurosawa マスクパターンの評価ロジック修正
 *          ncodes(IN)
 *          matrix(IN/OUT)
 * 戻り値 正常終了時0以外を異常終了時0を返す
 * 説明
 *        引数に指定が無ければ、マスクパターン決定後、
 *        マスクを全体にかける。最終的に正式な機能パターンを配置する。
 *
 */
bool QREL_SetMask(VERSIONINFO versioninfo, int8_t *mp, ECCLEVEL ecclevel,
                   uint8_t **matrix)
{
    FORMATINFO formatinfo;
    /* 2003.02.13 Add A.Kurosawa マスクパターンの評価ロジック修正  */
    int8_t i = 0, j = 0, k = 0;
    /* ループ用変数            */
    int32_t ncells = 0;
    /* 一辺のセル数            */
    int32_t score[MaskPatternMaxNum] = { 0, 0, 0, 0, 0, 0, 0, 0 };
    /* マスク評価スコア        */
    int32_t minscore = 0;
    /* スコア最小値記憶変数 */
    // bool    boFlg = false;
    int8_t buff[256];
    memset(buff, 0x00, 256);

    QREL_SetLastError(ERR_NOERROR, true);
    /* 一辺あたりのセル数取得 */
    if (!VER_GetnCellNum(versioninfo.version, &ncells)) {
        QREL_SetLastError(ERR_VERSIONINFOERROR, true);
        return false;
    }
    /* マスクパターンを決定する */
    auto mask_any = MaskAny;
    if (*mp == (int8_t)(mask_any)) {
        // boFlg = true;       // 自動フラグ設定
        /* 最適マスク判定処理
         * 仮のマトリックスを作成
         */
#ifdef MALLOC_MOD
        int32_t **runlength_hor = (int32_t **)QRE_Malloc(ncells * sizeof(int32_t *));
        int32_t **runlength_ver = (int32_t **)QRE_Malloc(ncells * sizeof(int32_t *));
#endif
        uint8_t **tempmatrix
            = (uint8_t **)QRE_Malloc(ncells * sizeof(uint8_t *));
        for (i = 0; i < ncells; i++) {
            tempmatrix[i] = (uint8_t *)QRE_Malloc(ncells);
            memset(tempmatrix[i], 0x00, ncells);
#ifdef MALLOC_MOD
            /* ＋１は長さ分 */
            runlength_hor[i] = (int32_t *)QRE_Malloc(sizeof(int32_t) * (ncells + 1));
            runlength_ver[i] = (int32_t *)QRE_Malloc(sizeof(int32_t) * (ncells + 1));
            memset(runlength_hor[i], 0x00, sizeof(int32_t) * (ncells + 1));
            memset(runlength_ver[i], 0x00, sizeof(int32_t) * (ncells + 1));
#endif
        }
        /* ・ｽマスクパターンを全タイプかけたデータをそれぞれ作成 */
        for (k = 0; k < MaskPatternMaxNum; k++) {
            *mp = k;
            /* マトリックス初期化 */
            for (i = 0; i < ncells; i++)
                for (j = 0; j < ncells; j++)
                    tempmatrix[i][j] = matrix[i][j];
                 /* マスクパターンを全体にかける */
            for (i = 0; i < ncells; i++) {
                for (j = 0; j < ncells; j++) {
                    if (!GetMaskPattern(i, j, *mp)) {
                        /* 念のため、除外エリアにはかけない */
                        if (tempmatrix[i][j] != CellOutArea)
                            tempmatrix[i][j] = tempmatrix[i][j] ^ 0xFF;
                    }
                }
            }
            /* 正式な機能パターンを配置する */
            if (SetFunctionPattern(versioninfo.version, tempmatrix) == false)
                return false;
            /* 除外エリアを空(明モジュール)にする */
            for (i = 0; i < ncells; i++) {
                for (j = 0; j < ncells; j++)
                    if (tempmatrix[i][j] == CellOutArea)
                        tempmatrix[i][j] = 0x00;
            }
            /* 2003.02.13 Add A.Kurosawa マスクパターンの評価ロジック修正
             * マスク評価用シンボルに形式情報・型番情報を付加する
             */
            formatinfo.ecclevel = ecclevel;
            formatinfo.mp = k;
            QREL_GenerateQRCode(versioninfo, formatinfo, tempmatrix);
            /* ・ｽ作成した各仮シンボルを評価する */
#ifndef MALLOC_MOD
            score[k] = GetMaskPatternScore(ncells, tempmatrix);
#else
            score[k] = GetMaskPatternScore(ncells, tempmatrix, runlength_hor, runlength_ver);
#endif
        }
        /* ・ｽ評価結果のマスクを設定する */
        for (i = 0; i < MaskPatternMaxNum; i++) {
            if (i == 0) {
                minscore = score[i];
                *mp = 0;
            }
            else {  /* 同点の場合はMPの若い番号 */
                if (minscore > score[i]) {
                    minscore = score[i];
                    *mp = i;
                }
            }
        }
        /* TEMP領域の開放 */
        for (i = 0; i < ncells; i++) {
#ifdef MALLOC_MOD
            QRE_Free(runlength_ver[i]);
            QRE_Free(runlength_hor[i]);
#endif
            QRE_Free(tempmatrix[i]);
        }
#ifdef MALLOC_MOD
        QRE_Free(runlength_ver);
        QRE_Free(runlength_hor);
#endif
        QRE_Free(tempmatrix);
    }
    /* マスクパターンを全体にかける */
    for (i = 0; i < ncells; i++) {
        for (j = 0; j < ncells; j++) {
            if (!GetMaskPattern(i, j, *mp)) {
                /* 念のため、除外エリアにはかけない */
                if (matrix[i][j] != CellOutArea)
                    matrix[i][j] = matrix[i][j] ^ 0xFF;
            }
        }
    }
    /* 正式な機能パターンを配置する */
    if (SetFunctionPattern(versioninfo.version, matrix) == false)
        return false;
    // if (boFlg == true) {
    // }
    return true;
}

/****************************************************************
 * 内部関数
 ****************************************************************/
/*
 * 関数名 GetMaskPattern
 * 機能   マスクパターンを生成
 * 引数   i(IN)        y座標
 *        j(IN)        x座標
 *          mp(IN)    マスクパターン
 * 戻り値 正常終了時マスクパターンを異常終了時-1を返す
 * 説明
 *
 */
int32_t GetMaskPattern(int32_t i, int32_t j, int8_t mp)
{
    switch (mp) {
    case Mask000:
        return (i + j) % 2;

    case Mask001:
        return i % 2;

    case Mask010:
        return j % 3;

    case Mask011:
        return (i + j) % 3;

    case Mask100:
        return ((i / 2) + (j / 3)) % 2;

    case Mask101:
        return ((i * j) % 2) + ((i * j) % 3);

    case Maslk110:
        return (((i * j) % 2) + ((i * j) % 3)) % 2;

    case Mask111:
        return (((i * j) % 3) + ((i + j) % 2)) % 2;

    default:
        return -1;      /* 失敗 */
    }
}

/*
 * 関数名 SetFunctionPattern
 * 機能   機能パターンを生成
 * 引数   version(IN)        型番
 *        matrix(IN/OUT)     マトリックス
 * 戻り値 正常終了時0以外を異常終了時0を返す
 * 説明
 *
 */
bool SetFunctionPattern(int32_t version, uint8_t **matrix)
{
    int32_t i = 0;                    /* ループ用変数 */
    int32_t ncells = 0;
    /* 一辺あたりのセル数取得 */
    if (!VER_GetnCellNum(version, &ncells)) {
        QREL_SetLastError(ERR_VERSIONINFOERROR, true);
        return false;
    }
    /* タイミングパターン：２箇所 */
    for (i = 0; i < ncells; i++) {
        if (i % 2)
            matrix[i][6] = 0x00;
        else
            matrix[i][6] = 0xFF;
    }
    for (i = 0; i < ncells; i++) {
        if (i % 2)
            matrix[6][i] = 0x00;
        else
            matrix[6][i] = 0xFF;
    }
    /* 位置検出パターン領域：左上・左下・右上（7×7） */
    SetPDPPattern(matrix, 0, 0);
    SetPDPPattern(matrix, ncells - 7, 0);
    SetPDPPattern(matrix, 0, ncells - 7);
    /* 位置検出パターン分離領域：左上・左下・右上（8+8） */
    for (i = 0; i < 8; i++)
        matrix[7][i] = 0x00;
    for (i = 0; i < 8; i++)
        matrix[i][7] = 0x00;
    for (i = 0; i < 8; i++)
        matrix[ncells - 1 - 7][i] = 0x00;
    for (i = ncells - 1; i > ncells - 1 - 8; i--)
        matrix[i][7] = 0x00;
    for (i = ncells - 1; i > ncells - 1 - 8; i--)
        matrix[7][i] = 0x00;
    for (i = 0; i < 8; i++)
        matrix[i][ncells - 1 - 7] = 0x00;
    /* 位置合わせパターン領域 */
    if (SetAllPOPPattern(version, matrix) == false)
        return false;
    return true;
}

/*
 * 関数名 SetPDPPattern
 * 機能   位置検出パターン領域を書き込み
 * 引数   matrix(IN/OUT)    マトリックス
 *          x                    スタート位置
 *          y                    スタート位置
 * 戻り値 正常終了時0以外を異常終了時0を返す
 * 説明
 *
 */
bool SetPDPPattern(uint8_t **matrix, int32_t x, int32_t y)
{
    int32_t i = 0, j = 0;
    /* 上辺の黒 */
    for (i = 0; i < 7; i++)
        matrix[y][x + i] = (uint8_t)0xFF; /* 下辺の黒 */
    for (i = 0; i < 7; i++)
        matrix[y + 6][x + i] = (uint8_t)0xFF; /* 左辺の黒 */
    for (i = 0; i < 7; i++)
        matrix[y + i][x] = (uint8_t)0xFF; /* 右辺の黒 */
    for (i = 0; i < 7; i++)
        matrix[y + i][x + 6] = (uint8_t)0xFF;
    /* 中の白上 */
    for (i = 0; i < 5; i++)
        matrix[y + 1][x + 1 + i] = (uint8_t)0x00; /* 中の白下 */
    for (i = 0; i < 5; i++)
        matrix[y + 5][x + 1 + i] = (uint8_t)0x00; /* 中の白左 */
    for (i = 0; i < 5; i++)
        matrix[y + 1 + i][x + 1] = (uint8_t)0x00; /* 中の白右 */
    for (i = 0; i < 5; i++)
        matrix[y + 1 + i][x + 5] = (uint8_t)0x00;
    /* 中心の黒 */
    for (i = 0; i < 3; i++)
        for (j = 0; j < 3; j++)
            matrix[y + 2 + i][x + 2 + j] = (uint8_t)0xFF;
    return true;
}

/*
 * 関数名 SetAllPOPPattern
 * 機能   位置合わせパターン領域を書き込み
 * 引数   version(IN)        型番
 *        matrix(IN/OUT)     マトリックス
 * 戻り値 正常終了時0以外を異常終了時0を返す
 * 説明
 *
 */
bool SetAllPOPPattern(int32_t version, uint8_t **matrix)
{
    int32_t x[PopMaxNum], y[PopMaxNum];  /* 位置合わせパターンの中心座標 */
    int32_t popnum = 0;                   /* 位置合わせパターン数 */
    int32_t i = 0;
    /* 処理部 */
    if (VER_GetPOPCoord(version, &popnum, x, y) == false) {
        QREL_SetLastError(ERR_VERSIONINFOERROR, true);
        return false;
    }
    else
        for (i = 0; i < popnum; i++)
            SetPOPPattern(matrix, x[i], y[i]);
    return true;
}

/*
 * 関数名 SetPOPPattern
 * 機能   位置検出パターン領域を書き込み
 * 引数   matrix(IN/OUT)    マトリックス
 *          x               スタート位置
 *          y               スタート位置
 * 戻り値 正常終了時0以外を異常終了時0を返す
 * 説明
 *
 */
bool SetPOPPattern(uint8_t **matrix, int32_t x, int32_t y)
{
    /* 変数宣言 */
    int32_t i = 0, j = 0;   /* ループ用変数
                             * 処理部
                             */
    for (i = y - 2; i < y - 2 + 5; i++)
        for (j = x - 2; j < x - 2 + 5; j++)
            matrix[i][j] = (uint8_t)0xFF;
    for (i = y - 1; i < y - 2 + 4; i++)
        for (j = x - 1; j < x - 2 + 4; j++)
            matrix[i][j] = (uint8_t)0x00;
    matrix[y][x] = (uint8_t)0xFF;

    return true;
}

/*
 * 関数名 GetMaskPatternScore
 * 機能   位置検出パターン領域を書き込み
 * 引数   ncells(IN)    一辺あたりのセル数
 *          matrix(IN)    マトリックス
 * 戻り値 評価結果のスコアを返す
 * 説明
 *
 */
#ifndef MALLOC_MOD
int32_t GetMaskPatternScore(int32_t ncells, uint8_t **matrix)
#else
int32_t GetMaskPatternScore(int32_t ncells, uint8_t **matrix, int32_t **runlength_hor, int32_t **runlength_ver)
#endif
{
    /* 変数宣言 */
    int32_t i = 0;  /* ループ用変数 */
    int32_t score1 = 0, score2 = 0, score3 = 0, score4 = 0; /* 減点スコア変数 */
#ifndef MALLOC_MOD
    int32_t **runlength_hor = NULL;     /* ランレングス（縦）*/
    int32_t **runlength_ver = NULL;     /* ランレングス（横）*/

    /* 処理部
     * ランレングスを作成する
     */
    runlength_hor = (int32_t **)QRE_Malloc(ncells * sizeof(int32_t *));
    runlength_ver = (int32_t **)QRE_Malloc(ncells * sizeof(int32_t *));
#endif
    for (i = 0; i < ncells; i++) {
#ifndef MALLOC_MOD
        runlength_hor[i] = (int32_t *)QRE_Malloc(sizeof(int32_t) * (ncells + 1));  /* ＋１は長さ分 */
        runlength_ver[i] = (int32_t *)QRE_Malloc(sizeof(int32_t) * (ncells + 1));
#endif
        memset(runlength_hor[i], 0x00, sizeof(int32_t) * (ncells + 1));
        memset(runlength_ver[i], 0x00, sizeof(int32_t) * (ncells + 1));
    }
    SetRunLengthData(runlength_hor, ncells, matrix, QrHorizon);
    SetRunLengthData(runlength_ver, ncells, matrix, QrVertucal);

    /* 減点処理１ */
    score1 = GetDemeritScore_1(ncells, runlength_hor, runlength_ver);

    /* 減点処理２ */
    score2 = GetDemeritScore_2(ncells, matrix);

    /* 減点処理３ */
    score3 = GetDemeritScore_3(ncells, runlength_hor, runlength_ver);

    /* 減点処理４ */
    score4 = GetDemeritScore_4(ncells, matrix);
    /* ランレングスメモリの開放 */
#ifndef MALLOC_MOD
    for (i = 0; i < ncells; i++) {
        QRE_Free(runlength_hor[i]);
        QRE_Free(runlength_ver[i]);
    }
    QRE_Free(runlength_hor);
    QRE_Free(runlength_ver);
#endif

    return score1 + score2 + score3 + score4;     /* スコアの合計を返す */
}

/*
 * 関数名 GetDemeritScore_1
 * 機能   減点評価１
 * 引数   ncells(IN)          一辺あたりのセル数
 *          matrix(IN)        マトリックス
 *          runlength_hor(IN) ランレングス（横方向）
 *          runlength_ver(IN) ランレングス（縦方向）
 * 戻り値 評価結果のスコアを返す
 * 説明
 *        同色モジュールが連続５個を超えた場合減算。
 *          行／列の両方からチェックする。
 */
int32_t GetDemeritScore_1(int32_t ncells, int32_t **runlength_hor, int32_t **runlength_ver)
{
    /* 変数宣言 */
    int32_t i = 0, j = 0;                          /* ループ用変数   */
    int32_t demeritscore = 0;                      /* 減点スコア変数 */
    for (i = 0; i < ncells; i++) {                 /* 行をチェック
                                                    * 行のチェック結果をスコアに代入
                                                    */
        for (j = 1; j <= runlength_hor[i][0]; j++)
            if (abs(runlength_hor[i][j]) >= 5)
                demeritscore += MaskDecreasePointN1
                                + (abs(runlength_hor[i][j]) - 5);
    }
    for (i = 0; i < ncells; i++) {                 /* 列をチェック
                                                    * 列のチェック結果をスコアに代入
                                                    */
        for (j = 1; j <= runlength_ver[i][0]; j++)
            if (abs(runlength_ver[i][j]) >= 5)
                demeritscore += MaskDecreasePointN1
                                + (abs(runlength_ver[i][j]) - 5);
    }
    return demeritscore;
}

/*
 * 関数名 GetDemeritScore_2
 * 機能   減点評価２
 * 引数   ncells(IN)      一辺あたりのセル数
 *          matrix(IN)    マトリックス
 * 戻り値 評価結果のスコアを返す
 * 説明
 *          同色のモジュールブロック（２×２）があった場合、
 *        減算。
 */
int32_t GetDemeritScore_2(int32_t ncells, uint8_t **matrix)
{
    /* 変数宣言 */
    int32_t i = 0, j = 0;                           /* ループ用変数   */
    int32_t demeritscore = 0;                       /* 減点スコア変数 */
    for (i = 0; i < ncells; i++) {
        for (j = 0; j < ncells; j++) {
            if (i <= ncells - 2 && j <= ncells - 2) {
                if (matrix[i][j] == matrix[i + 1][j]
                    && matrix[i][j] == matrix[i][j + 1]
                    && matrix[i][j] == matrix[i + 1][j + 1])
                    demeritscore += MaskDecreasePointN2;
            }
        }
    }
    return demeritscore;
}

/*
 * 関数名 GetDemeritScore_3
 * 機能   減点評価２
 * 引数   ncells(IN)      一辺あたりのセル数
 *          matrix(IN)    マトリックス
 *          runlength_hor(IN)    ランレングス（横方向）
 *          runlength_ver(IN)    ランレングス（縦方向）
 * 戻り値 評価結果のスコアを返す
 * 説明
 *        モジュールが1:1:3:1:1（暗：明：暗：明：暗）のパターン
 *          で存在した場合減算。行／列の両方からチェックする。
 */
int32_t GetDemeritScore_3(int32_t ncells, int32_t **runlength_hor, int32_t **runlength_ver)
{
    /* 変数宣言 */
    int32_t i = 0, j = 0;                                      /* ループ用変数 */
    int32_t demeritscore = 0;                                  /* 減点スコア変数 */

    for (i = 0; i < ncells; i++) {                             /* 行をチェック
                                                                * 行のチェック結果をスコアに代入
                                                                */
        for (j = 1; j <= (runlength_hor[i][0] - 4); j++) {       /* 1:1:3:1:1の最小5件のデータと１オリジンのため-4*/
            if (runlength_hor[i][j] < 0) {
                if ((runlength_hor[i][j] == runlength_hor[i][j + 4])
                    && (runlength_hor[i][j] == (runlength_hor[i][j + 1] * (-1)))
                    && (runlength_hor[i][j] == (runlength_hor[i][j + 3] * (-1))))
                {
                    if (runlength_hor[i][j] * (3) == runlength_hor[i][j + 2]) {
                        if ((j + 5) > runlength_hor[i][0])  /* 境界チェック */
                            demeritscore += MaskDecreasePointN3;
                        else if (runlength_hor[i][j + 4] !=
                                 runlength_hor[i][j + 5])
                            demeritscore += MaskDecreasePointN3;
                    }
                }
            }
        }
    }
    for (i = 0; i < ncells; i++) {                             /* 列をチェック
                                                                * 列のチェック結果をスコアに代入
                                                                */
        for (j = 1; j <= (runlength_ver[i][0] - 4); j++) {     /* 1:1:3:1:1の最小5件のデータと１オリジンのため-4*/
            if (runlength_ver[i][j] < 0) {
                if ((runlength_ver[i][j] == runlength_ver[i][j + 4])
                    && (runlength_ver[i][j] == (runlength_ver[i][j + 1] * (-1)))
                    && (runlength_ver[i][j] == (runlength_ver[i][j + 3] * (-1))))
                {
                    if (runlength_ver[i][j] * (3) == runlength_ver[i][j + 2]) {
                        if ((j + 5) > runlength_ver[i][0])  /* 境界チェック */
                            demeritscore += MaskDecreasePointN3;
                        else if (runlength_ver[i][j + 4] !=
                                 runlength_ver[i][j + 5])
                            demeritscore += MaskDecreasePointN3;
                    }
                }
            }
        }
    }
    return demeritscore;
}

/*
 * 関数名 GetDemeritScore_4
 * 機能   減点評価４
 * 引数   ncells(IN)    一辺あたりのセル数
 *          matrix(IN)    マトリックス
 * 戻り値 評価結果のスコアを返す
 * 説明
 *          全体に対する暗モジュール(セル)の占める割合により
 *        減算（50%からはなれるほど減算）。
 *          50±(5*k)%～50±(5*(k+1)%  →N4*k
 */
int32_t GetDemeritScore_4(int32_t ncells, uint8_t **matrix)
{
    /* 変数宣言 */
    int32_t i = 0, j = 0;                       /* ループ用変数 */
    int32_t DarkCell = 0;                       /* 暗モジュールの個数 */
    double Percentage = 0;                      /* 暗モジュールの割合 */
    int32_t k = 0;                              /* 減算用定数 */
    for (i = 0; i < ncells; i++) {
        for (j = 0; j < ncells; j++)
            if (matrix[i][j] == (uint8_t)0xFF)
                DarkCell++;
    }
    Percentage = ((double)DarkCell / (ncells * ncells) * 100.0);
    if (Percentage > 45.0 && Percentage < 55.0)
        k = 0;
    else if ((Percentage >= 55.0 && Percentage < 60.0)
             || (Percentage <= 45.0 && Percentage > 40.0))
        k = 1;
    else if ((Percentage >= 60.0 && Percentage < 65.0)
             || (Percentage <= 40.0 && Percentage > 35.0))
        k = 2;
    else if ((Percentage >= 65.0 && Percentage < 70.0)
             || (Percentage <= 35.0 && Percentage > 30.0))
        k = 3;
    else if ((Percentage >= 70.0 && Percentage < 75.0)
             || (Percentage <= 30.0 && Percentage > 25.0))
        k = 4;
    else if ((Percentage >= 75.0 && Percentage < 80.0)
             || (Percentage <= 25.0 && Percentage > 20.0))
        k = 5;
    else if ((Percentage >= 80.0 && Percentage < 85.0)
             || (Percentage <= 20.0 && Percentage > 15.0))
        k = 6;
    else if ((Percentage >= 85.0 && Percentage < 90.0)
             || (Percentage <= 15.0 && Percentage > 10.0))
        k = 7;
    else if ((Percentage >= 90.0 && Percentage < 95.0)
             || (Percentage <= 10.0 && Percentage > 5.0))
        k = 8;
    else if ((Percentage >= 95.0 && Percentage < 100.0)
             || (Percentage <= 5.0 && Percentage > 0.0))
        k = 9;
    else    /* 0% or 100% */
        k = 10;
    return MaskDecreasePointN4 * k;
}

/*
 * 関数名 SetRunLengthData
 * 機能   ランレングスの作成
 * 引数   runlength(OUT)    ランレングス
 *        ncells(IN)        一辺あたりのセル数
 *          matrix(IN)      マトリックス
 *          dir(IN)         方向
 * 戻り値 正常終了時0以外を異常終了時0を返す
 * 説明
 *        マトリックスから行列のランレングスを作成する
 *        最初の要素（runlength[0]）は長さとする。
 */
bool SetRunLengthData(int32_t **runlength, int32_t ncells, uint8_t **matrix,
                       int32_t dir)
{
    /* 変数宣言 */
    int32_t i = 0, j = 0, k = 0, l = 0;     /* ループ用変数 */
    uint8_t precell = 0;                    /* 前回データ */
    int8_t length = 0;                      /* 連続数
                                             * 処理部
                                             */
    for (i = 0; i < ncells; i++) {
        for (j = 0, precell = 0, length = 0; j < ncells; j++) {
            if (dir == QrHorizon) {
                k = i;
                l = j;
            }
            else {
                k = j;
                l = i;
            }
            if (j == 0) {
                if (matrix[k][l] == (uint8_t)0xFF)
                    length--;
                else
                    length++;
                runlength[i][0] = 1;
            }
            else if (j == ncells - 1) {
                if (precell == matrix[k][l]) {
                    if (matrix[k][l] == (uint8_t)0xFF)
                        runlength[i][runlength[i][0]] = --length;
                    else
                        runlength[i][runlength[i][0]] = ++length;
                }
                else {
                    runlength[i][runlength[i][0]] = length;
                    (runlength[i][0])++;
                    if (matrix[k][l] == (uint8_t)0xFF)
                        runlength[i][runlength[i][0]] = -1;
                    else
                        runlength[i][runlength[i][0]] = 1;
                }
            }
            else {
                if (precell == matrix[k][l]) {
                    if (matrix[k][l] == (uint8_t)0xFF)
                        length--;
                    else
                        length++;
                }
                else {
                    runlength[i][runlength[i][0]] = length;
                    runlength[i][0]++;
                    if (matrix[k][l] == (uint8_t)0xFF)
                        length = -1;
                    else
                        length = 1;
                }
            }
            precell = matrix[k][l];
        }
    }
    return true;
}
