﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/nn_Common.h>
#include <nn/err/err_Types.h>
#include <nn/erpt/erpt_Types.h>
#include <nn/settings/system/settings_ErrorReport.h>
#include <nn/time/time_PosixTime.h>
#include "../DevMenu_Common.h"
#include "../Common/DevMenu_CommonScene.h"

#include <glv_CustomVerticalListView.h>

namespace devmenu {
class Page;
}

namespace devmenu { namespace error {

struct ErrorProperty
{
    enum class Type : nn::Bit8
    {
        ErrorViewer,
        ApplicationCrash,
        Fatal,
        Other,
    };

    void Prepare(int inputNumber, const nn::msgpack::MpWalker& erptMpWalker, nn::erpt::ReportFlagSet flagSet) NN_NOEXCEPT;

    Type type;      //!< エラーの種類。
    bool isUploaded;//!< 送信済みかどうか。
    nn::settings::system::ErrorReportSharePermission sharePermission; //!< 共有許可状態。
    char number[4]; //!< リスト表示上の通し番号。
    char pid[32];   //!< プログラムID。
    char code[64];  //!< エラーコード文字列（表示用）。"2xxx-xxxx (result=0xXXXXXXXXX)" のように result を併記する場合有り
    char time[64];  //!< エラー発生時刻文字列（表示用）。

    static const size_t CrashReportHashSize = 32;
    std::unique_ptr<nn::Bit8>  crashReportHash; //!< クラッシュレポートハッシュ。

    static const int MaxDescriptionLength = 2048;
    std::unique_ptr<char> description; //!< 自由記述のエラー表示を行った場合に保存されるメッセージの内容。

    char rawErrorCode[nn::err::ErrorCode::StringLengthMax]; //!< 生のエラーコード文字列。2xxx-xxxx。
    nn::time::PosixTime posixTime; //!< エラー発生時刻。
};

/**
* @brief リストビュー
*/
class PropertyListView : public glv::CustomVerticalListView< ErrorProperty >
{
public:
    typedef CustomVerticalListView< ErrorProperty > ParentType;

    static const glv::space_t LeftMargin;       //!< リスト要素の左マージン( ピクセル単位 )
    static const glv::space_t NoWidth;          //!< リスト要素中の[No]表示横幅( ピクセル単位 )
    static const glv::space_t TypeWidth;        //!< リスト要素中の[Type]表示横幅( ピクセル単位 )
    static const glv::space_t ErrorCodeWidth;   //!< リスト要素中の[ErrorCode]表示横幅( ピクセル単位 )

    explicit PropertyListView(const glv::Rect& parentClipRegion) NN_NOEXCEPT;

protected:
    virtual void OnQueryBounds(const ItemType& item, glv::space_t& outWidth, glv::space_t& outHeight) NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnDrawItem(const ItemType& item, const IndexType index, const glv::Rect& contentRegion) NN_NOEXCEPT NN_OVERRIDE;
};

class ErrorReportScene : public devmenu::Scene
{
public:
    ErrorReportScene(devmenu::Page* pParent, glv::Rect rect) NN_NOEXCEPT;

    void Activate() NN_NOEXCEPT;
    void Deactivate() NN_NOEXCEPT;
    void UpdateCrashReportStatus(bool updateDetails) NN_NOEXCEPT;

    glv::View* GetFocusableChild() NN_NOEXCEPT;
    virtual void Refresh() NN_NOEXCEPT final NN_OVERRIDE;
private:
    void OnExecuteErrorDetail() NN_NOEXCEPT;
    static void OnPropertyListUpdateNotification(const glv::Notification& notification) NN_NOEXCEPT;
    void EntryProperties(const PropertyListView::CollectionType* pItems) NN_NOEXCEPT;
    void FinalizeProperties() NN_NOEXCEPT;

    nn::Result MakeErrorReportList() NN_NOEXCEPT;
    void UpdateButtonStatus(bool hasErrorReport) NN_NOEXCEPT;

    PropertyListView::CollectionType*   m_pItems;
    PropertyListView*                   m_pListView;
    glv::Label*                         m_pLabelNoItem;
    glv::Label                          m_CrashReportStatusLabel;
    devmenu::Button                     m_ExportButton;
#if defined( NN_DEVMENUSYSTEM )
    devmenu::Button                     m_DumpButton;
#endif
};

}} // ~namespace devmenu::error, ~namespace devmenu
