﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>
#include <nn/time.h>

#include "../Common/DevMenu_CommonCheckBox.h"
#include "../Common/DevMenu_CommonDropDown.h"
#include "../Common/DevMenu_CommonScrollBox.h"
#include "DevMenu_DeviceSettingsCommon.h"

namespace devmenu { namespace devicesettings {

/**************************************
 * class RebootRequiredSetting
 **************************************/
class RebootRequiredSetting : public glv::Table
{
public:
    typedef std::pair< std::string, std::string > valuePair;

public:
    explicit RebootRequiredSetting( const char* labelString, glv::space_t itemWidth, glv::space_t dropDownWidth, Page* pParentPage,
        const std::function< void() >& adhocScrollCallback,
        const std::function< const std::vector< RebootRequiredSetting::valuePair >&() >& getDropDownItemsCallback,
        const std::function< const std::string() >& getCurrentSettingValueCallback,
        const std::function< void( const std::string& value ) >& setSettingValueCallback ) NN_NOEXCEPT;

    void DisplayRebootDialog( const std::string& selected, const std::string& current ) NN_NOEXCEPT;

    void ForceMoveFocusToRebootDialog() NN_NOEXCEPT;

    void SetFocusTransitionPath( FocusManager* pFocusManager, glv::View* pPreviousView, glv::View* pNextView ) NN_NOEXCEPT;

    glv::View* GetFirstFocusTargetView() NN_NOEXCEPT;

    glv::View* GetLastFocusTargetView() NN_NOEXCEPT;

private:
    class RebootRequiredDropDown : public DropDownStringKeyValueBase
    {
    public:
        RebootRequiredDropDown( const glv::Rect& rect, float textSize,
            const std::function< void( const std::string&, const std::string& ) >& onClickCallback,
            const std::function< void() >& adhocScrollCallback,
            const std::function< const std::vector< RebootRequiredSetting::valuePair >&() >& getDropDownItemsCallback,
            const std::function< const std::string() >& getCurrentSettingValueCallback ) NN_NOEXCEPT;

    private:
        virtual std::string GetSettingsValue() NN_NOEXCEPT NN_OVERRIDE;

        virtual void RegisterDropDownItems() NN_NOEXCEPT NN_OVERRIDE;

        virtual void UpdateSettingsValue() NN_NOEXCEPT NN_OVERRIDE;

        virtual bool onEvent( glv::Event::t event, glv::GLV& glvRoot ) NN_NOEXCEPT NN_OVERRIDE;

    private:
        std::function< void() > m_AdhocScrollCallback;  // Callback for adjusting scroll position
        std::function< const std::vector< RebootRequiredSetting::valuePair >&() > m_GetDropDownItemsCallback;  // Callback for get drop down items
        std::function< const std::string() > m_GetCurrentSettingValueCallback;  // Callback for get current setting
    };

private:
    std::string              m_LabelString;
    RebootRequiredDropDown*  m_pDropDown;
    Page*                    m_pParentPage;
    MessageView*             m_pRebootDialog;
    bool                     m_IsRebootDialogUnvisible;
    std::function< void( const std::string& selected ) > m_SetSettingValueCallback;  // Callback for set setting value
};

/**************************************
 * class TimeZoneSetting
 **************************************/
class TimeZoneSetting : public glv::Table
{
public:

    TimeZoneSetting( glv::space_t width, Page* pParentPage ) NN_NOEXCEPT;

    void RefreshCurrentTimeZoneLabel() NN_NOEXCEPT;

    void SetFocusTransitionPath( FocusManager* pFocusManager, glv::View* pPreviousView, glv::View* pNextView ) NN_NOEXCEPT;

    glv::View* GetFirstFocusTargetView() NN_NOEXCEPT;

    glv::View* GetLastFocusTargetView() NN_NOEXCEPT;

private:
    Page*         m_pParentPage;
    glv::Group*   m_pItemGroup;
    Button*       m_pEditButton;
    glv::Label*   m_pTimeZoneLabel;
};

/**************************************
 * class DateTimeSetting
 **************************************/
class DateTimeSetting : public glv::Table
{
public:
    struct CalendarTime
    {
        uint16_t year;
        uint8_t month;
        uint8_t day;
        uint8_t hour;
        uint8_t minute;
        uint8_t second;
    };

    DateTimeSetting(glv::space_t width, Page* pPage) NN_NOEXCEPT;

    void SetFocusTransitionPath( FocusManager* pFocusManager, glv::View* pPreviousView, glv::View* pNextView ) NN_NOEXCEPT;

    glv::View* GetFirstFocusTargetView() NN_NOEXCEPT;

    glv::View* GetLastFocusTargetView() NN_NOEXCEPT;

    void UpdateTime( int yearDelta = 0, int monthDelta = 0, int dayDelta = 0, int hourDelta = 0, int minuteDelta = 0 ) NN_NOEXCEPT;

    void UpdateTimeOnLoop() NN_NOEXCEPT;

    void Refresh() NN_NOEXCEPT;

private:
    void RegisterUpdateTimeFunction() NN_NOEXCEPT;

    void SetTime( const nn::time::CalendarTime& carendarTime, bool isSetToDeviceRequired ) NN_NOEXCEPT;

    void UpdateDateUpDownButtonsDisplay( bool isEnabled ) NN_NOEXCEPT;

    void OnCheckBoxPressed( const glv::Notification& notificaiton ) NN_NOEXCEPT;

    void SetAutomaticCorrectionEnabledAndUpdateCheckBoxValue( bool isEnabled ) NN_NOEXCEPT;

    void ScheduleRefreshDateTime( int64_t nowSecond ) NN_NOEXCEPT;

private:
    Page* m_pParentPage;

    glv::Label* m_pYearLabel;
    glv::Label* m_pMonthLabel;
    glv::Label* m_pDayLabel;
    glv::Label* m_pHourLabel;
    glv::Label* m_pMinuteLabel;

    glv::View* m_pYearUp;
    glv::View* m_pMonthUp;
    glv::View* m_pDayUp;
    glv::View* m_pHourUp;
    glv::View* m_pMinuteUp;

    glv::View* m_pYearDown;
    glv::View* m_pMonthDown;
    glv::View* m_pDayDown;
    glv::View* m_pHourDown;
    glv::View* m_pMinuteDown;

    CheckBoxButton* m_pAutomaticCorrectionCheckBox;
    int64_t         m_DisplayedDateClock;
};

/**************************************
 * class SystemSettings
 **************************************/
class SystemSettings : public SubsectionWithFocusUtility
{
public:
    SystemSettings( Page* pPage, glv::space_t width, ScrollableBoxView* pContainer ) NN_NOEXCEPT;

    void ForceMoveFocusToRebootDialog() NN_NOEXCEPT;

    virtual void SetFocusTransitionPath( FocusManager* pFocusManager, glv::View* pPreviousFocusItem, glv::View* pNextFocusItem ) const NN_NOEXCEPT final NN_OVERRIDE;
    virtual glv::View* GetFirstFocusTargetView() const NN_NOEXCEPT final NN_OVERRIDE;
    virtual glv::View* GetLastFocusTargetView() const NN_NOEXCEPT final NN_OVERRIDE;

    virtual void OnLoopBeforeSceneRenderer() NN_NOEXCEPT final NN_OVERRIDE;
    virtual void OnLoopAfterSceneRenderer() NN_NOEXCEPT final NN_OVERRIDE;
    virtual void Refresh() NN_NOEXCEPT final NN_OVERRIDE;

private:
    RebootRequiredSetting* m_pLanguage;
    RebootRequiredSetting* m_pRegion;
    TimeZoneSetting*       m_pTimeZone;
    DateTimeSetting*       m_pDateTime;
};

}} // ~namespace devmenu::devicesettings, ~namespace devmenu
