﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <vector>
#include <nn/nn_Abort.h>
#include <nn/audioctrl.h>
#include <nn/audio.h>

#include "../Common/DevMenu_CommonCheckBox.h"
#include "../Common/DevMenu_CommonDropDown.h"
#include "../DevMenu_Sound.h"
#include "../DevMenu_SoundUtil.h"
#include "DevMenu_DeviceSettingsSound.h"

namespace devmenu { namespace devicesettings {

namespace {
    const char* GetOutputModeLabelString( nn::audioctrl::AudioOutputMode mode ) NN_NOEXCEPT
    {
        switch ( mode )
        {
        case nn::audioctrl::AudioOutputMode_Pcm1ch:
            return "Mono";
        case nn::audioctrl::AudioOutputMode_Pcm2ch:
            return "Stereo";
        case nn::audioctrl::AudioOutputMode_Pcm6ch:
            return "Surround Sound";
        case nn::audioctrl::AudioOutputMode_PcmAuto:
            return "Auto";
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

TvOutputModeDropDown::TvOutputModeDropDown( const glv::Rect& rect, float textSize ) NN_NOEXCEPT : DropDownBase( rect, textSize )
{
    for ( const auto& value : TvOutputModeTable )
    {
        addItem( GetOutputModeLabelString( value ) );
    }

    const auto currentMode = nn::audioctrl::GetOutputModeSetting( nn::audioctrl::AudioTarget_Tv );
    setValue( GetOutputModeLabelString( currentMode ) );

    attach([]( const glv::Notification& notification )->void { notification.receiver< TvOutputModeDropDown >()->UpdateOutputMode(); }, glv::Update::Action, this );
}

void TvOutputModeDropDown::UpdateOutputMode() NN_NOEXCEPT
{
    const auto outputMode = TvOutputModeTable[mSelectedItem];
    nn::audioctrl::SetOutputModeSetting( nn::audioctrl::AudioTarget_Tv, outputMode );
    nn::audioctrl::SetAudioOutputMode( nn::audioctrl::AudioTarget_Tv, outputMode );
}


SystemOutputModeDropDown::SystemOutputModeDropDown( const glv::Rect& rect, float textSize ) NN_NOEXCEPT
    : DropDownBase( rect, textSize )
{
    for ( const auto& value : SystemOutputModeTable )
    {
         addItem(GetOutputModeLabelString( value ) );
    }

    const auto currentMode = nn::audioctrl::GetOutputModeSetting( nn::audioctrl::AudioTarget_Speaker );
    setValue( GetOutputModeLabelString( currentMode ) );

    attach([]( const glv::Notification& notification )->void { notification.receiver< SystemOutputModeDropDown >()->UpdateOutputMode(); }, glv::Update::Action, this );
}

void SystemOutputModeDropDown::UpdateOutputMode() NN_NOEXCEPT
{
    const auto outputMode = SystemOutputModeTable[mSelectedItem];
    nn::audioctrl::SetOutputModeSetting( nn::audioctrl::AudioTarget_Speaker, outputMode );
    nn::audioctrl::SetAudioOutputMode( nn::audioctrl::AudioTarget_Speaker, outputMode );

     // UsbOutputDevice's output mode should be same as Speaker's.
    nn::audioctrl::SetOutputModeSetting( nn::audioctrl::AudioTarget_UsbOutputDevice, outputMode );
    nn::audioctrl::SetAudioOutputMode( nn::audioctrl::AudioTarget_UsbOutputDevice, outputMode );
}

TvOutputModeSetting::TvOutputModeSetting( glv::space_t width ) NN_NOEXCEPT
{
    auto pLabel = new glv::Label( "TV Sound", glv::Label::Spec( glv::Place::TL, 5.0f, 0.0f, CommonValue::InitialFontSize ) );
    auto pDropDown = new TvOutputModeDropDown( glv::Rect( 240.0f, 35.0f ), CommonValue::InitialFontSize );
    pDropDown->anchor( glv::Place::TR ).pos( -pDropDown->width(), 0.0f );
    pDropDown->enable( glv::Property::KeepWithinParent );

    auto pGroup = new glv::Group( glv::Rect( width, pDropDown->h ) );
    *pGroup << pLabel << pDropDown;

    *this << pGroup;
    arrange();
    fit( false );

    m_pDropDown = pDropDown;
}

void TvOutputModeSetting::SetFocusTransitionPath( FocusManager* pFocusManager, glv::View* pPreviousView, glv::View* pNextView ) NN_NOEXCEPT
{
    pFocusManager->AddFocusSwitch< FocusButtonUp >( m_pDropDown, pPreviousView );
    pFocusManager->AddFocusSwitch< FocusButtonDown >( m_pDropDown, pNextView );
}

glv::View* TvOutputModeSetting::GetFirstFocusTargetView() NN_NOEXCEPT
{
    return m_pDropDown;
}

glv::View* TvOutputModeSetting::GetLastFocusTargetView() NN_NOEXCEPT
{
    return m_pDropDown;
}

SystemOutputModeSetting::SystemOutputModeSetting(glv::space_t width) NN_NOEXCEPT
{
    auto pLabel = new glv::Label( "System Sound (Speaker & HP)", glv::Label::Spec(glv::Place::TL, 5.0f, 0.0f, CommonValue::InitialFontSize ) );
    auto pDropDown = new SystemOutputModeDropDown( glv::Rect( 240.0f, 35.0f ), CommonValue::InitialFontSize );
    pDropDown->anchor( glv::Place::TR ).pos(-pDropDown->width(), 0.0f );
    pDropDown->enable( glv::Property::KeepWithinParent );

    auto pGroup = new glv::Group( glv::Rect(width, pDropDown->h ) );
    *pGroup << pLabel << pDropDown;

    *this << pGroup;
    arrange();
    fit( false );

    m_pDropDown = pDropDown;
}

void SystemOutputModeSetting::SetFocusTransitionPath( FocusManager* pFocusManager, glv::View* pPreviousView, glv::View* pNextView ) NN_NOEXCEPT
{
    pFocusManager->AddFocusSwitch< FocusButtonUp >( m_pDropDown, pPreviousView );
    pFocusManager->AddFocusSwitch< FocusButtonDown >( m_pDropDown, pNextView );
}

glv::View* SystemOutputModeSetting::GetFirstFocusTargetView() NN_NOEXCEPT
{
    return m_pDropDown;
}

glv::View* SystemOutputModeSetting::GetLastFocusTargetView() NN_NOEXCEPT
{
    return m_pDropDown;
}

ForceMuteOnHeadphoneRemovedSetting::ForceMuteOnHeadphoneRemovedSetting( glv::space_t width ) NN_NOEXCEPT
{
    auto pCheckbox = new devmenu::CheckBoxButton( "Mute when Headphones are Disconnected", 0.0f, "< < >", true );
    pCheckbox->SetWidth( width );

    *this << pCheckbox;
    arrange().fit( false );

    // Default padding for tables and groups, so that it lines up with other Sound options.
    pCheckbox->resizeLeftTo( 6.0f );

    const auto currentForceMutePolicy = nn::audioctrl::GetForceMutePolicy();
    pCheckbox->SetValue( currentForceMutePolicy == nn::audioctrl::ForceMutePolicy_SpeakerMuteOnHeadphoneUnplugged );
    pCheckbox->SetCallback([]( const glv::Notification& notification )->void { notification.receiver< ForceMuteOnHeadphoneRemovedSetting >()->UpdateSetting(); }, this );

    m_pCheckbox = pCheckbox;
}

void ForceMuteOnHeadphoneRemovedSetting::UpdateSetting() NN_NOEXCEPT
{
    if ( m_pCheckbox->GetValue() )
    {
        nn::audioctrl::SetForceMutePolicy( nn::audioctrl::ForceMutePolicy_SpeakerMuteOnHeadphoneUnplugged );
    }
    else
    {
        nn::audioctrl::SetForceMutePolicy( nn::audioctrl::ForceMutePolicy_Disable );
    }
}

void ForceMuteOnHeadphoneRemovedSetting::SetFocusTransitionPath( FocusManager* pFocusManager, glv::View* pPreviousView, glv::View* pNextView ) NN_NOEXCEPT
{
    pFocusManager->AddFocusSwitch< FocusButtonUp >( m_pCheckbox->GetButtonFocus(), pPreviousView );
    pFocusManager->AddFocusSwitch< FocusButtonDown >( m_pCheckbox->GetButtonFocus(), pNextView );
}

glv::View* ForceMuteOnHeadphoneRemovedSetting::GetFirstFocusTargetView() NN_NOEXCEPT
{
    return m_pCheckbox->GetButtonFocus();
}

glv::View* ForceMuteOnHeadphoneRemovedSetting::GetLastFocusTargetView() NN_NOEXCEPT
{
    return m_pCheckbox->GetButtonFocus();
}

HeadphoneOutputLeveLimitSetting::HeadphoneOutputLeveLimitSetting( glv::space_t width ) NN_NOEXCEPT
{
    auto pCheckbox = new CheckBoxButton( "Limit Headphone Output Level", 0.0f, "< < >", true );
    pCheckbox->SetWidth( width );
    *this << pCheckbox;
    arrange().fit( false );

    // Default padding for tables and groups, so that it lines up with other Sound options.
    pCheckbox->resizeLeftTo( 6.0f );

    const auto isLimited = ( nn::audioctrl::GetHeadphoneOutputLevelMode() == nn::audioctrl::HeadphoneOutputLevelMode_Normal );
    pCheckbox->SetValue( isLimited );
    pCheckbox->SetCallback([]( const glv::Notification& notification )->void { notification.receiver< HeadphoneOutputLeveLimitSetting >()->UpdateSetting(); }, this );
    m_pCheckbox = pCheckbox;
}

void HeadphoneOutputLeveLimitSetting::UpdateSetting() NN_NOEXCEPT
{
    if ( m_pCheckbox->GetValue () )
    {
        nn::audioctrl::SetHeadphoneOutputLevelMode( nn::audioctrl::HeadphoneOutputLevelMode_Normal );
    }
    else
    {
        nn::audioctrl::SetHeadphoneOutputLevelMode( nn::audioctrl::HeadphoneOutputLevelMode_HighPower );
    }
}

void HeadphoneOutputLeveLimitSetting::SetFocusTransitionPath( FocusManager* pFocusManager, glv::View* pPreviousView, glv::View* pNextView ) NN_NOEXCEPT
{
    pFocusManager->AddFocusSwitch< FocusButtonUp >( m_pCheckbox->GetButtonFocus(), pPreviousView );
    pFocusManager->AddFocusSwitch< FocusButtonDown >( m_pCheckbox->GetButtonFocus(), pNextView );
}

glv::View* HeadphoneOutputLeveLimitSetting::GetFirstFocusTargetView() NN_NOEXCEPT
{
    return m_pCheckbox->GetButtonFocus();
}

glv::View* HeadphoneOutputLeveLimitSetting::GetLastFocusTargetView() NN_NOEXCEPT
{
    return m_pCheckbox->GetButtonFocus();
}

PlaybackButtonPanel::PlaybackButtonPanel( glv::space_t width ) NN_NOEXCEPT
{
    auto pLabel = new glv::Label( "Sound Test", glv::Label::Spec( glv::Place::TL, 5.0f, 0.0f, CommonValue::InitialFontSize ) );

    Button* pPlaybackButton = new Button( "Playback", [&] { Playback(); } );
    pPlaybackButton->anchor( glv::Place::TR ).pos( -( pPlaybackButton->width() ), 0.0f );
    pPlaybackButton->enable( glv::Property::KeepWithinParent );

    auto pGroup = new glv::Group( glv::Rect( width, pPlaybackButton->height() ) );
    *pGroup << pLabel << pPlaybackButton;

    *this << pGroup;
    arrange().fit( false );

    m_pPlaybackButton = pPlaybackButton;
}

void PlaybackButtonPanel::PlaybackButtonPanel::Playback() NN_NOEXCEPT
{
    int channelCount = 0;

    const auto currentTarget = nn::audioctrl::GetDefaultTarget();
    const auto currentMode = nn::audioctrl::GetOutputModeSetting( currentTarget );

    if ( currentMode == nn::audioctrl::AudioOutputMode_Pcm1ch )
    {
        channelCount = 1;
    }
    else if ( currentMode == nn::audioctrl::AudioOutputMode_Pcm2ch )
    {
        channelCount = 2;
    }
    else if ( currentMode == nn::audioctrl::AudioOutputMode_Pcm6ch || currentMode == nn::audioctrl::AudioOutputMode_PcmAuto )
    {
        channelCount = 6;
    }
    else
    {
        NN_ASSERT( false );
    }

    PlaybackTestSound( channelCount );
}

void PlaybackButtonPanel::SetFocusTransitionPath( FocusManager* pFocusManager, glv::View* pPreviousView, glv::View* pNextView ) NN_NOEXCEPT
{
    pFocusManager->AddFocusSwitch< FocusButtonUp >( m_pPlaybackButton, pPreviousView );
    pFocusManager->AddFocusSwitch< FocusButtonDown >( m_pPlaybackButton, pNextView );
}

glv::View* PlaybackButtonPanel::GetFirstFocusTargetView() NN_NOEXCEPT
{
    return m_pPlaybackButton;
}

glv::View* PlaybackButtonPanel::GetLastFocusTargetView() NN_NOEXCEPT
{
    return m_pPlaybackButton;
}

OutputDeviceInformation::OutputDeviceInformation( glv::space_t w ) NN_NOEXCEPT
{
    m_pNameLabel = new glv::Label( "Output Device", glv::Label::Spec( glv::Place::CC, 0.0f, 0.0f, CommonValue::InitialFontSize ) );
    m_pSpacer = new Spacer( 0.0f, 0.0f );
    m_pValueLabel = new glv::Label( "", glv::Label::Spec( glv::Place::CC, 0.0f, 0.0f, CommonValue::InitialFontSize ) );
    m_pTable = new glv::Table( "<x>" );

    set( glv::Rect( w, m_pNameLabel->h ) );

    *m_pTable << m_pNameLabel << m_pSpacer << m_pValueLabel;
    m_pTable->arrange().fit();
    *this << m_pTable;
}

void OutputDeviceInformation::UpdateInformation( const char* name, int channelCount ) NN_NOEXCEPT
{
    m_Name = name;
    m_ChannelCount = channelCount;
    const auto combinedStr = m_Name + " (" + std::to_string(m_ChannelCount) + ")";
    m_pValueLabel->setValue( combinedStr );
    const auto spacerWidth = w - m_pNameLabel->w - m_pValueLabel->w;
    m_pSpacer->set( glv::Rect( spacerWidth, 0.0f ) );
    m_pSpacer->remove();
    m_pValueLabel->remove();
    *m_pTable << m_pSpacer << m_pValueLabel;
    m_pTable->arrange().fit( false );
}

void OutputDeviceInformation::SetFocusTransitionPath( FocusManager* pFocusManager, glv::View* pPreviousView, glv::View* pNextView ) NN_NOEXCEPT
{
    pFocusManager->AddFocusSwitch< FocusButtonUp >( m_pValueLabel, pPreviousView );
    pFocusManager->AddFocusSwitch< FocusButtonDown >( m_pValueLabel, pNextView );
}

glv::View* OutputDeviceInformation::GetFirstFocusTargetView() NN_NOEXCEPT
{
    return m_pValueLabel;
}

glv::View* OutputDeviceInformation::GetLastFocusTargetView() NN_NOEXCEPT
{
    return m_pValueLabel;
}

/**************************************
 * class SoundSettings
 **************************************/

SoundSettings::SoundSettings( Page* pPage, glv::space_t width ) NN_NOEXCEPT
    : SubsectionWithFocusUtility( pPage, "Sound Settings", width )
{
    m_pTvOutputMode = new TvOutputModeSetting( this->GetItemWidth() );
    m_pSystemOutputMode = new SystemOutputModeSetting( this->GetItemWidth() );
    m_pForceMuteOnHeadphoneRemoved = new ForceMuteOnHeadphoneRemovedSetting( this->GetItemWidth() );
    m_pHeadphoneOutputLimit = new HeadphoneOutputLeveLimitSetting( this->GetItemWidth() );
    m_pPlaybackButtonPanel = new PlaybackButtonPanel( this->GetItemWidth() );
    m_pOutputDeviceInformation = new OutputDeviceInformation( this->GetItemWidth() );

    *this << m_pTvOutputMode << m_pSystemOutputMode << m_pForceMuteOnHeadphoneRemoved << m_pHeadphoneOutputLimit << m_pPlaybackButtonPanel << m_pOutputDeviceInformation;
    arrange().fit( false );

    UpdateOutputDeviceInformation();

#if defined( NN_BUILD_CONFIG_SPEC_NX )
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::audio::AcquireAudioDeviceNotificationForOutput( &m_DeviceNotificationEventForOutput ) );
#endif // defined( NN_BUILD_CONFIG_SPEC_NX )
}

void SoundSettings::SetFocusTransitionPath( FocusManager* pFocusManager, glv::View* pPreviousFocusItem, glv::View* pNextFocusItem ) const NN_NOEXCEPT
{
    m_pTvOutputMode->SetFocusTransitionPath( pFocusManager, pPreviousFocusItem, m_pSystemOutputMode->GetFirstFocusTargetView() );
    m_pSystemOutputMode->SetFocusTransitionPath( pFocusManager, m_pTvOutputMode->GetLastFocusTargetView(), m_pForceMuteOnHeadphoneRemoved->GetFirstFocusTargetView() );
    m_pForceMuteOnHeadphoneRemoved->SetFocusTransitionPath( pFocusManager, m_pSystemOutputMode->GetLastFocusTargetView(), m_pHeadphoneOutputLimit->GetFirstFocusTargetView() );
    m_pHeadphoneOutputLimit->SetFocusTransitionPath( pFocusManager, m_pForceMuteOnHeadphoneRemoved->GetLastFocusTargetView(), m_pPlaybackButtonPanel->GetFirstFocusTargetView() );
    m_pPlaybackButtonPanel->SetFocusTransitionPath( pFocusManager, m_pHeadphoneOutputLimit->GetLastFocusTargetView(), pNextFocusItem );
}

void SoundSettings::UpdateOutputDeviceInformation() NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_SPEC_NX )
    nn::audio::AudioDeviceName deviceName = { { 0 } };
    nn::audio::GetActiveAudioDeviceName( &deviceName );
    const auto channelCount = nn::audio::GetActiveAudioDeviceChannelCountForOutput();
    m_pOutputDeviceInformation->UpdateInformation( deviceName.name, channelCount );
#endif // defined( NN_BUILD_CONFIG_SPEC_NX )
}

glv::View* SoundSettings::GetFirstFocusTargetView() const NN_NOEXCEPT
{
    return m_pTvOutputMode->GetFirstFocusTargetView();
}

glv::View* SoundSettings::GetLastFocusTargetView() const NN_NOEXCEPT
{
    return m_pPlaybackButtonPanel->GetLastFocusTargetView();
}

void SoundSettings::OnLoopBeforeSceneRenderer() NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_SPEC_NX )
    if ( m_DeviceNotificationEventForOutput.TryWait() )
    {
        UpdateOutputDeviceInformation();
    }
#endif // defined( NN_BUILD_CONFIG_SPEC_NX )
}

void SoundSettings::OnLoopAfterSceneRenderer() NN_NOEXCEPT
{
    // Do Nothing
}

void SoundSettings::Refresh() NN_NOEXCEPT
{
    // Do Nothing
}

}} // ~namespace devmenu::devicesettings, ~namespace devmenu
