﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "DevMenu_DeviceSettingsItemListScene.h"
#include "DevMenu_DeviceSettingsDisplay.h"
#include "DevMenu_DeviceSettingsSleep.h"
#include "DevMenu_DeviceSettingsSound.h"
#include "DevMenu_DeviceSettingsSystem.h"
#include "../Common/DevMenu_CommonSettingsApi.h"

namespace devmenu { namespace devicesettings {

ItemListScene::ItemListScene( Page* pPage, const glv::Rect& rect) NN_NOEXCEPT
    : Scene( pPage, rect, true )
    , m_ScrollableContainer( "x", glv::Rect( rect.width(), rect.height() - CommonValue::InitialFontSize ) )
{
    const auto subsectionWidth = this->width() * 0.98f;

    // Allocate Subsections
    m_SubsectionList.push_back( new SystemSettings( pPage, subsectionWidth, &m_ScrollableContainer ) );
    m_SubsectionList.push_back( new DisplaySettings( pPage, subsectionWidth ) );
    m_SubsectionList.push_back( new SoundSettings( pPage, subsectionWidth ) );

    if ( IsDebugModeEnabled() )
    {
        // 現状 Sleep Settings の全項目が Prod Mode で無効なので、Sleep Settings 全体を提供対象外にする
        m_SubsectionList.push_back( new SleepSettings( pPage, subsectionWidth ) );
    }
}

void ItemListScene::PrepareSubsections() NN_NOEXCEPT
{
    // Add Subsections and Set Focus Transition Path
    for ( size_t index = 0; index < m_SubsectionList.size(); index++ )
    {
        auto& pSubsection = m_SubsectionList[ index ];
        m_ScrollableContainer << pSubsection;

        if ( nullptr == this->GetFirstFocusTargetView() )
        {
            this->SetFirstFocusTargetView( pSubsection->GetFirstFocusTargetView() );
        }

        glv::View* pLastFocusTargetViewOfPreviousSubsection = nullptr;
        glv::View* pFirstFocusTargetViewOfNextSubsection = nullptr;

        if ( 0 == index ) // Top
        {
            pLastFocusTargetViewOfPreviousSubsection = this->GetRootSurfaceContext()->GetRebootButtonView();
            pFirstFocusTargetViewOfNextSubsection =
                ( m_SubsectionList.size() - 1 > index ) ? m_SubsectionList[ index + 1 ]->GetFirstFocusTargetView() : nullptr ;
        }
        else if ( m_SubsectionList.size() - 1 == index ) // Bottom
        {
            pLastFocusTargetViewOfPreviousSubsection = m_SubsectionList[ index - 1 ]->GetLastFocusTargetView();
            pFirstFocusTargetViewOfNextSubsection = nullptr;
        }
        else
        {
            pLastFocusTargetViewOfPreviousSubsection = m_SubsectionList[ index - 1 ]->GetLastFocusTargetView();
            pFirstFocusTargetViewOfNextSubsection = m_SubsectionList[ index + 1 ]->GetFirstFocusTargetView();
        }

        pSubsection->SetFocusTransitionPath(
            &m_FocusManager, pLastFocusTargetViewOfPreviousSubsection, pFirstFocusTargetViewOfNextSubsection );
    }

    *this << m_ScrollableContainer;

    this->Refresh();
}

void ItemListScene::Refresh() NN_NOEXCEPT
{
    for ( auto& iter : m_SubsectionList )
    {
        iter->Refresh();
    }
}

void ItemListScene::OnChangeIntoForeground() NN_NOEXCEPT
{
    this->Refresh();
}

void ItemListScene::OnActivatePage() NN_NOEXCEPT
{
    this->Refresh();
}

void ItemListScene::OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT
{
    NN_UNUSED( context );
    NN_UNUSED( events );

    for ( auto& iter : m_SubsectionList )
    {
        iter->OnLoopBeforeSceneRenderer();
    }
}

}} // ~namespace devmenu::devicesettings, ~namespace devmenu
