﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <glv_CustomVerticalListView.h>
#include <glv_ScissorBoxView.h>
#include <nn/es.h>
#include <nn/util/util_FormatString.h>
#include <glv/utils/glv_MessageThread.h>

#include "DevMenu_Config.h"
#include "DevMenu_RootSurface.h"

namespace devmenu { namespace ticket {

namespace {

/**
* @brief チケット読み込みの結果を示します。
*/
enum DevMenuTicket
{
    DevMenuTicket_Exists,        //!< チケットが存在します。
    DevMenuTicket_NotExists,     //!< チケットが存在しません。
};

void ConvertToCharString(char* outChar, const uint8_t* byte, int size) NN_NOEXCEPT
{
    for (int i = 0; i < size; i++)
    {
        nn::util::SNPrintf(outChar + 2 * i, 16, "%02hhx", *(byte + i));
    }

    outChar[size * 2] = '\0';
}

} // ~namespace devmenu::ticket::<anonymous>

/**
* @brief チケットデータを取得し、保持する構造体です。
*/
struct TicketList
{
    struct TicketInfo
    {
        enum TicketType
        {
            TicketType_Common,
            TicketType_Personalized,
            TicketType_PrepurchaseRecord,
        } type;

        nn::es::RightsIdIncludingKeyId rightsId;

        static const char* ToString(TicketType type)
        {
            switch (type)
            {
            case TicketList::TicketInfo::TicketType_Common:
                return "Common";
            case TicketList::TicketInfo::TicketType_Personalized:
                return "Personalized";
            case TicketList::TicketInfo::TicketType_PrepurchaseRecord:
                return "Prepurchase";
            default:
                NN_ABORT("Unexpected TicketType\n");
            }
        }
    };

    int ticketCount;
    std::unique_ptr<TicketInfo[]> ticketInfoList;

    int CountTicket() NN_NOEXCEPT
    {
        return nn::es::CountCommonTicket() + nn::es::CountPersonalizedTicket() + nn::es::CountPrepurchaseRecord();
    }

    int UpdateTicketInfoList() NN_NOEXCEPT
    {
        int commonTicketCount = nn::es::CountCommonTicket();
        int personalizedTicketCount = nn::es::CountPersonalizedTicket();
        int prepurchaseRecordCount = nn::es::CountPrepurchaseRecord();
        ticketCount = commonTicketCount + personalizedTicketCount + prepurchaseRecordCount;

        std::unique_ptr<nn::es::RightsIdIncludingKeyId[]> commonRightsIdList(new nn::es::RightsIdIncludingKeyId[commonTicketCount]);
        std::unique_ptr<nn::es::RightsIdIncludingKeyId[]> personalizedRightsIdList(new nn::es::RightsIdIncludingKeyId[personalizedTicketCount]);
        std::unique_ptr<nn::es::RightsIdIncludingKeyId[]> prepurchaseRecordRightsIdList(new nn::es::RightsIdIncludingKeyId[prepurchaseRecordCount]);
        int commonTicketRightsIdCount = nn::es::ListCommonTicketRightsIds(commonRightsIdList.get(), commonTicketCount);
        int personalizedTicketRightsIdCount = nn::es::ListPersonalizedTicketRightsIds(personalizedRightsIdList.get(), personalizedTicketCount);
        int prepurchaseRecordRightsIdCount = nn::es::ListPrepurchaseRecordRightsIds(prepurchaseRecordRightsIdList.get(), prepurchaseRecordCount);

        int offset = 0;
        ticketInfoList.reset(new TicketInfo[ticketCount]);

        std::unique_ptr<nn::es::LightTicketInfo[]> lightTicketInfoList(new nn::es::LightTicketInfo[commonTicketCount + personalizedTicketCount]);
        for (int i = 0; i < commonTicketRightsIdCount; i++)
        {
            int count = nn::es::ListLightTicketInfo(lightTicketInfoList.get() + offset, ticketCount - offset, commonRightsIdList[i]);
            for (int j = 0; j < count; j++)
            {
                ticketInfoList[offset + j].type = TicketInfo::TicketType_Common;
                ticketInfoList[offset + j].rightsId = (lightTicketInfoList.get() + offset + j)->rightsId;
            }
            offset += count;
        }
        for (int i = 0; i < personalizedTicketRightsIdCount; i++)
        {
            int count = nn::es::ListLightTicketInfo(lightTicketInfoList.get() + offset, ticketCount - offset, personalizedRightsIdList[i]);
            for (int j = 0; j < count; j++)
            {
                ticketInfoList[offset + j].type = TicketInfo::TicketType_Personalized;
                ticketInfoList[offset + j].rightsId = (lightTicketInfoList.get() + offset + j)->rightsId;
            }
            offset += count;
        }
        NN_SDK_ASSERT_EQUAL(offset, commonTicketCount + personalizedTicketCount);

        std::unique_ptr<nn::es::PrepurchaseRecord[]> prepurchaseRecordInfoList(new nn::es::PrepurchaseRecord[prepurchaseRecordCount]);
        for (int i = 0; i < prepurchaseRecordRightsIdCount; i++)
        {
            int count = nn::es::ListPrepurchaseRecordInfo(prepurchaseRecordInfoList.get() + offset - (commonTicketCount + personalizedTicketCount), ticketCount - offset, prepurchaseRecordRightsIdList[i]);
            for (int j = 0; j < count; j++)
            {
                ticketInfoList[offset + j].type = TicketInfo::TicketType_PrepurchaseRecord;
                ticketInfoList[offset + j].rightsId = (prepurchaseRecordInfoList.get() + offset - (commonTicketCount + personalizedTicketCount) + j)->rightsId;
            }
            offset += count;
        }
        NN_SDK_ASSERT_EQUAL(offset, ticketCount);

        return ticketCount;
    }

    TicketInfo GetTicketInfo(int index) NN_NOEXCEPT
    {
        return ticketInfoList[index];
    }

    void DeleteTicket(int index) NN_NOEXCEPT
    {
        nn::es::DeleteTicket(&ticketInfoList[index].rightsId, 1);
    }
};

TicketList g_TicketList;

/**
* @brief チケットプロパティ型です。
*/
template<size_t N>
struct TicketProperty
{
    static const char* QueryingConditionMessage;

    /**
    * @brief チケットリスト用プロパティデータを構築します。
    * @details TicketList からチケットをフェッチします。
    * @pre TicketList::UpdateTicketInfoList() で最新のチケットを取得している。
    */
    const DevMenuTicket Fetch(int index) NN_NOEXCEPT
    {
        ticketIndex = index;

        if (ticketIndex < g_TicketList.ticketCount)
        {
            char rightsIdCharString[sizeof(nn::es::RightsIdIncludingKeyId) * 2 + 1];
            ConvertToCharString(rightsIdCharString, g_TicketList.GetTicketInfo(ticketIndex).rightsId._data, sizeof(nn::es::RightsIdIncludingKeyId));

            nn::util::SNPrintf(labelOfRightsId, sizeof(labelOfRightsId), "0x%32s", rightsIdCharString);
            nn::util::SNPrintf(labelOfTicketType, sizeof(labelOfTicketType), "%s", TicketList::TicketInfo::ToString(g_TicketList.GetTicketInfo(ticketIndex).type));

            return DevMenuTicket_Exists;
        }
        else
        {
            return DevMenuTicket_NotExists;
        }
    }

    /**
    * @brief プロパティが示すチケットを削除します。
    */
    const bool Delete() const NN_NOEXCEPT
    {
        g_TicketList.DeleteTicket(ticketIndex);
        return true;
    }

    char       labelOfRightsId[N];
    char       labelOfTicketType[N];
    int        ticketIndex;
    bool       isQueried;
};

template< size_t N > const char* TicketProperty< N >::QueryingConditionMessage = "Loading ...";

/**
* @brief チケットデータプロパティ( 35 文字上限 )型です。
*/
typedef TicketProperty<sizeof(nn::es::RightsIdIncludingKeyId) * 2 + 3> TicketPropertyType;

/**
* @brief リストビュー
*/
class PropertyListView : public glv::CustomVerticalListView< TicketPropertyType >
{
public:
    typedef glv::CustomVerticalListView< TicketPropertyType > ParentType;

    static const glv::space_t HorizontalMargin_RightsId;       //!< リスト要素中の[Rights ID]表示左マージン( ピクセル単位 )
    static const glv::space_t HorizontalLength_RightsId;       //!< リスト要素中の[Rights ID]表示横幅( ピクセル単位 )

    /**
    * @brief コンストラクタです。
    */
    explicit PropertyListView(const glv::Rect& parentClipRegion) NN_NOEXCEPT
        : ParentType(parentClipRegion)
    {
        glv::Style* pStyle = new glv::Style();
        pStyle->color = glv::Style::standard().color;
        pStyle->color.selection.set(0.1f, 0.85f, 0.2f);
        style(pStyle);
        font().size(25.f);

        // 親クラスで OKボタン( A )に対してUpdate::Actionイベントが発生するので Hack.
        changePadClickDetectableButtons(glv::BasicPadEventType::Button::X::Mask);
        changePadClickDetectableButtons(glv::DebugPadEventType::Button::X::Mask);
    }

protected:
    /**
    * @copydoc CustomVerticalListView<>::OnQueryBounds( const CustomVerticalListView<>::ItemType&, glv::space_t&, glv::space_t& )
    */
    virtual void OnQueryBounds(const ItemType& item, glv::space_t& outWidth, glv::space_t& outHeight) NN_NOEXCEPT NN_OVERRIDE
    {
        const char* const pLabel = (false == item.isQueried) ? TicketPropertyType::QueryingConditionMessage : item.labelOfRightsId;
        this->font().getBounds(outWidth, outHeight, pLabel);
        outWidth = this->width();
    }

    /**
    * @copydoc CustomVerticalListView<>::OnDrawItem( const CustomVerticalListView<>::ItemType&, const CustomVerticalListView<>::IndexType, const glv::Rect& )
    */
    virtual void OnDrawItem(const ItemType& item, const IndexType index, const glv::Rect& contentRegion) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(index);
        glv::Font& font = this->font();

        if (item.isQueried == false)
        {
            glv::space_t posId = contentRegion.left() + HorizontalMargin_RightsId;
            font.render(TicketPropertyType::QueryingConditionMessage, posId, contentRegion.top());
        }
        else
        {
            glv::space_t posId = contentRegion.left() + HorizontalMargin_RightsId;
            font.render(item.labelOfRightsId, posId, contentRegion.top());

            glv::space_t outWidth, outHeight;
            font.getBounds(outWidth, outHeight, item.labelOfTicketType);
            const glv::space_t ownerExpect = contentRegion.right() - (outWidth + (paddingX() * 2) + 4.f);
            const glv::space_t ownerLimit = posId + HorizontalMargin_RightsId + HorizontalLength_RightsId + 12.f;
            font.render(item.labelOfTicketType, (ownerExpect < ownerLimit) ? ownerLimit : ownerExpect, contentRegion.top());
        }
    }
};

const glv::space_t PropertyListView::HorizontalMargin_RightsId = 4.f;
const glv::space_t PropertyListView::HorizontalLength_RightsId = 332.f;

/**
* @brief クエリ結果通知用メッセージ ( 192 byte )
*/
class QueryReply : public glv::utils::ObjectRecyclerBase< QueryReply >::BasedNode
{
public:
    static const unsigned MessageQueueCapacity = 8;
    typedef glv::utils::StaticObjectRecycler< QueryReply, MessageQueueCapacity > RecyclerType;
    typedef glv::utils::MessageReceiver< MessageQueueCapacity > ReplyReceiver;

    //!--------------------------------------------------------------------------------------
    //! @brief
    //!--------------------------------------------------------------------------------------
    explicit QueryReply(RecyclerType* pOwnerRecycler) NN_NOEXCEPT
        : RecyclerType::BasedNode(pOwnerRecycler) {}

    //!--------------------------------------------------------------------------------------
    //! @brief
    //!--------------------------------------------------------------------------------------
    void Initialize(const TicketPropertyType& inPropoerty, const int index) NN_NOEXCEPT
    {
        m_Index = index;
        m_Property = inPropoerty;
    }

    //!--------------------------------------------------------------------------------------
    //! @brief
    //!--------------------------------------------------------------------------------------
    const TicketPropertyType& GetProperty() const NN_NOEXCEPT
    {
        return m_Property;
    }

    //!--------------------------------------------------------------------------------------
    //! @brief
    //!--------------------------------------------------------------------------------------
    const int GetIndex() const NN_NOEXCEPT
    {
        return m_Index;
    }

private:
    TicketPropertyType m_Property;
    int                m_Index;
};

/**
* @brief クエリ要求用メッセージ ( 80 byte )
*/
class QueryRequest : public glv::utils::ObjectRecyclerBase< QueryRequest >::BasedNode, public glv::utils::ThreadMessage
{
public:
    static const unsigned MessageQueueCapacity = 1;
    static const size_t QueryThreadStackSize = nn::os::ThreadStackAlignment * 3;    // 12KB( 4096 * 3 )
    typedef glv::utils::StaticObjectRecycler< QueryRequest, MessageQueueCapacity > RecyclerType;
    typedef glv::utils::MessageThread< QueryThreadStackSize, MessageQueueCapacity > QueryThreadBaseType;

    //!--------------------------------------------------------------------------------------
    //! @brief
    //!--------------------------------------------------------------------------------------
    explicit QueryRequest(RecyclerType* pOwnerRecycler) NN_NOEXCEPT
        : RecyclerType::BasedNode(pOwnerRecycler) {}
};

/**
* @brief クエリスレッド
: @details 12KBスタック( from Heap ), 1キュー
*/
class QueryThread : public QueryRequest::QueryThreadBaseType
{
public:
    typedef QueryRequest::QueryThreadBaseType ParentType;

    //!--------------------------------------------------------------------------------------
    //! @brief コンストラクタ
    //!--------------------------------------------------------------------------------------
    explicit QueryThread() NN_NOEXCEPT : m_pQueryRequest(nullptr)
    {
    }

    //!--------------------------------------------------------------------------------------
    //! @brief 初期化
    //!--------------------------------------------------------------------------------------
    void Initialize() NN_NOEXCEPT
    {
        m_QueryRecycler.Initialize();
        m_ReplyRecycler.Initialize();
        m_ReplyReceiver.Initialize();
        ParentType::InitializeWithRelativePriority(0);
    }

    //!--------------------------------------------------------------------------------------
    //! @brief 初期化
    //!--------------------------------------------------------------------------------------
    virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE
    {
        ClearReceivedReplyMessage();
        ParentType::Finalize();
        m_ReplyReceiver.Finalize();
        m_ReplyRecycler.Finalize();
        m_QueryRecycler.Finalize();
    }

    //!--------------------------------------------------------------------------------------
    //! @brief クエリ開始
    //!--------------------------------------------------------------------------------------
    void Query() NN_NOEXCEPT
    {
        auto& query = m_QueryRecycler.Obtain();
        m_pQueryRequest = &query;
        query.Initialize(nullptr, &query);
        Send(query);
    }

    //!--------------------------------------------------------------------------------------
    //! @brief クエリキャンセル
    //!--------------------------------------------------------------------------------------
    void CancelQuery() NN_NOEXCEPT
    {
        QueryRequest* pQuery;
        if (nullptr != (pQuery = m_pQueryRequest))
        {
            pQuery->Cancel();
        }
        ClearReceivedReplyMessage();        // Replyキューをクリーン
        m_QueryRecycler.WaitCanObtain();    // Queryコンテナ取得可能まで待機
        ClearReceivedReplyMessage();        // Query可能になったら、再度 Replyキューをクリーン
    }

    //!--------------------------------------------------------------------------------------
    //! @brief メッセージ受信通知
    //!--------------------------------------------------------------------------------------
    virtual void OnMessageReceived(glv::utils::ThreadMessage* pMessage) NN_NOEXCEPT NN_OVERRIDE
    {
        if (nullptr == pMessage)
        {
            return; // 終了要求
        }

        QueryRequest* pQuery;
        if (nullptr != (pQuery = static_cast< QueryRequest* >(pMessage)))
        {
            DEVMENU_LOG_DEBUG("Start of ticket query.\n");

            // チケットデータ更新
            const int count = g_TicketList.UpdateTicketInfoList();

            for (int i = 0; i < count; ++i)
            {
                if (false == pQuery->IsCancelled())
                {
                    TicketPropertyType prop;
                    prop.Fetch(i);

                    prop.isQueried = true;
                    QueryReply& reply = m_ReplyRecycler.Obtain();
                    reply.Initialize(prop, i);
                    m_ReplyReceiver.Send(reinterpret_cast< const uintptr_t >(&reply));
                }
                else
                {
                    // キャンセルにより中断。
                    DEVMENU_LOG_DEBUG("Cancel of ticket query.\n");
                    break;
                }
            }
            m_ReplyReceiver.Send(0U); // 処理終端メッセージ送信
            pQuery->Recycle();          // クエリ要求ハンドルの返却

            DEVMENU_LOG_DEBUG("End of ticket query.\n");
        }
    }

    //!--------------------------------------------------------------------------------------
    //! @brief 返答メッセージ受信
    //!--------------------------------------------------------------------------------------
    void ObserveReceiveProperty(PropertyListView::CollectionType* const pItems) NN_NOEXCEPT
    {
        DoActionReceivedReplyMessage([pItems](QueryReply* pReply)
        {
            if (nullptr != pItems)
            {
                pItems->at(pReply->GetIndex()) = pReply->GetProperty();
            }
        });
    }

private:
    //!--------------------------------------------------------------------------------------
    //! @brief リプライメッセージの共通処理
    //!--------------------------------------------------------------------------------------
    void DoActionReceivedReplyMessage(std::function< void(QueryReply*) > doRunnable) NN_NOEXCEPT
    {
        uintptr_t receive;
        while (true == m_ReplyReceiver.TryReceive(receive))
        {
            if (0U == receive)
            {
                m_pQueryRequest = nullptr;
            }
            else
            {
                QueryReply* pReply = reinterpret_cast< QueryReply* >(receive);
                if (nullptr != doRunnable)
                {
                    doRunnable(pReply);
                }
                pReply->Recycle();
            }
        }
    }

    //!--------------------------------------------------------------------------------------
    //! @brief キュークリア
    //!--------------------------------------------------------------------------------------
    void ClearReceivedReplyMessage() NN_NOEXCEPT
    {
        DoActionReceivedReplyMessage(nullptr);
    }

    QueryRequest* volatile      m_pQueryRequest;    //!< アクティブなクエリ要求ハンドル
    QueryRequest::RecyclerType  m_QueryRecycler;    //!< クエリ要求ハンドルリサイクラ
    QueryReply::ReplyReceiver   m_ReplyReceiver;    //!< クエリ結果受信機
    QueryReply::RecyclerType    m_ReplyRecycler;    //!< クエリ結果受信ハンドルリサイクラ
};

/**
* @brief チケットデータ一覧のページです。
*/
class TicketCatalogPage : public devmenu::Page
{
public:
    /**
    * @brief コンストラクタです。
    */
    TicketCatalogPage(int pageId, const glv::WideCharacterType* pageCaption, glv::Rect rect) NN_NOEXCEPT
        : devmenu::Page(pageId, pageCaption, rect), m_pItems(nullptr), m_pListView(nullptr), m_pLabelNoItem(nullptr)
    {
    }

    /**
    * @brief ページがコンテナに追加された後に呼び出されます。
    */
    virtual void OnAttachedPage() NN_NOEXCEPT NN_OVERRIDE
    {
        // シザーボックスの領域設定
        static const glv::space_t HeaderRegion = 48.f;
        static const glv::space_t FooterRegion = 48.f;
        glv::ScissorBoxView* pListContainer = new glv::ScissorBoxView(0, HeaderRegion, width(), height() - (HeaderRegion + FooterRegion));

        // リスト要素のパディング ( ボーダーを表示する場合は 2 以上必要 )
        static const glv::space_t ListMarginL = 16;
        static const glv::space_t ListMarginR = 16;
        static const glv::space_t ListMarginY = 2;

        // ヘッダ
        *this << new glv::Label(GLV_TEXT_API_WIDE_STRING("Rights ID"), glv::Label::Spec(glv::Place::TL, ListMarginL + 8, 8, CommonValue::InitialFontSize));
        *this << new glv::Label(GLV_TEXT_API_WIDE_STRING("Ticket Type"), glv::Label::Spec(glv::Place::TR, -(ListMarginR + 16), 8, CommonValue::InitialFontSize));

        // フッタ
        *this << new glv::Label(GLV_TEXT_API_WIDE_STRING("↑↓: Select ticket  X: Delete the selected ticket"),
            glv::Label::Spec(glv::Place::BR, -(ListMarginL + 16), -8, CommonValue::InitialFontSize));

        // リストビュー
        glv::Rect clipRegion(ListMarginL, ListMarginY, pListContainer->width() - (ListMarginL + ListMarginR), pListContainer->height() - (ListMarginY * 2));
        PropertyListView* pListView = new PropertyListView(clipRegion);
        pListView->attach(OnPropertyListUpdateNotification, glv::Update::Action, this);
        *pListContainer << pListView;
        m_pListView = pListView;

        // アイテムなしメッセージ
        glv::Label* pLabelNoItem = new glv::Label(GLV_TEXT_API_WIDE_STRING("No ticket exists."), glv::Label::Spec(glv::Place::CC, 0, 0, 32.f));
        *pListContainer << pLabelNoItem;
        m_pLabelNoItem = pLabelNoItem;

        *this << pListContainer;

        this->attach( this->FocusMenuTabOnBbuttonPress, glv::Update::Clicked, this );

        m_QueryThread.Initialize();
    }

    /**
    * @brief ページがコンテナから削除された前に呼び出されます。
    */
    virtual void OnDetachedPage() NN_NOEXCEPT NN_OVERRIDE
    {
        FinalizeProperties();
        m_QueryThread.Finalize();
    }

    /**
    * @brief ページがアクティブ( 選択により表示開始 )になる際に呼び出されます。
    */
    virtual void OnActivatePage() NN_NOEXCEPT NN_OVERRIDE
    {
        QueryProperties();
    }

    /**
    * @brief ページがディアクティブ( 選択により非表示開始 )になる際に呼び出されます。
    */
    virtual void OnDeactivatePage() NN_NOEXCEPT NN_OVERRIDE
    {
        m_pLabelNoItem->disable(glv::Property::t::Visible);
        FinalizeProperties();
    }

    /**
    * @brief バックグラウンド遷移処理です。
    */
    virtual void OnChangeIntoBackground() NN_NOEXCEPT NN_OVERRIDE
    {
        m_pLabelNoItem->disable(glv::Property::t::Visible);
        FinalizeProperties();
    }

    /**
    * @brief フォアグラウンド遷移処理です。
    */
    virtual void OnChangeIntoForeground() NN_NOEXCEPT NN_OVERRIDE
    {
        QueryProperties();
    }

    /**
    * @brief アプリケーションメインループからのコールバックです。
    *
    * @details glvシーンレンダラへ hid系イベントが通知される前に呼び出されます。@n
    * この時点ではまだ glvコンテキストのレンダリングは開始していません。
    */
    virtual void OnLoopBeforeSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(context);
        NN_UNUSED(events);

        // ページが変わると呼ばれなくなる。
        m_QueryThread.ObserveReceiveProperty(m_pItems);
    }

    /**
    * @brief アプリケーションメインループからのコールバックです。
    *
    * @details glvシーンレンダラのレンダリングが終わった後に呼び出されます。
    */
    virtual void OnLoopAfterSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(context);
        NN_UNUSED(events);
    }

    /**
    * @brief ページにフォーカスが与えられた際に、フォーカスを横取る子供を指定します。
    */
    virtual View* GetFocusableChild() NN_NOEXCEPT NN_OVERRIDE
    {
        PropertyListView::CollectionType* pItems;
        return (nullptr == (pItems = m_pItems) || pItems->empty())
            ? static_cast< View* >(m_pLabelNoItem)
            : static_cast< View* >(m_pListView);
    }

private:
    /**
    * @brief This sets up the Modal dialog to ask the user for confirmation before deleting the Ticket
    */
    void OnDeleteTicket() NN_NOEXCEPT
    {
        auto pView = new devmenu::MessageView();

        // Set up the Title and Message
        pView->AddMessage("Delete Ticket");
        pView->AddMessage("Are you sure you want to delete this Ticket?");

        pView->AddButton( GLV_TEXT_API_WIDE_STRING( "Cancel" ) );

        // Add the button with the Lambda that will delete the Ticket
        pView->AddButton(
            GLV_TEXT_API_WIDE_STRING("Delete"),
            [](void* pParam, nn::TimeSpan& timespan)
        {
            TicketCatalogPage *_this = static_cast<TicketCatalogPage *>(pParam);
            NN_ABORT_UNLESS_NOT_NULL(_this);

            const TicketPropertyType *currentProperty = _this->m_pListView->GetSelectedValue();
            if (currentProperty == nullptr) {
                DEVMENU_LOG("Could not delete Ticket - Selected Value was null.\n");
            }
            else
            {
                currentProperty->Delete();
                _this->QueryProperties();
            }
        },
            this,
            MessageView::ButtonTextColor::Red
        );

        GetRootSurfaceContext()->StartModal(pView, true);
    }

    static void OnPropertyListUpdateNotification(const glv::Notification& n) NN_NOEXCEPT
    {
        TicketCatalogPage* pPage;
        const TicketPropertyType* pProperty;
        if (nullptr != (pPage = n.receiver< TicketCatalogPage >()) && nullptr != (pProperty = n.data< PropertyListView::ItemType >()))
        {
            pPage->OnDeleteTicket();
        }
    }

    /**
    * @brief チケットデータの問い合わせ。
    */
    void QueryProperties() NN_NOEXCEPT
    {
        FinalizeProperties();
        PropertyListView::CollectionType* pItems;
        m_pItems = pItems = new PropertyListView::CollectionType();

        int ticketCount = g_TicketList.CountTicket();
        pItems->resize(ticketCount);

        for (int i = 0; i < ticketCount; i++)
        {
            pItems->at(i).isQueried = false;
        }

        // プロパティリスト登録
        EntryProperties(pItems);
        // スレッド経由で問い合わせ開始
        m_QueryThread.Query();
    }

    /**
    * @brief プロパティコレクションを登録します。
    * @details 登録内容に応じてリスト or アイテムなしメッセージの切り替えを行います。
    */
    void EntryProperties(const PropertyListView::CollectionType* pItems) NN_NOEXCEPT
    {
        const glv::Property::t focusableProperty = glv::Property::t::Controllable | glv::Property::t::HitTest;
        PropertyListView* pListView;
        if (nullptr != (pListView = m_pListView))
        {
            pListView->EntryCollection(*pItems);
            if (true == pItems->empty())
            {
                pListView->disable(focusableProperty);
                auto* const pLabelNoItem = m_pLabelNoItem;
                pLabelNoItem->enable(glv::Property::t::Visible);
                pLabelNoItem->bringToFront();
                // Focus出来るViewが無くなるのでメインメニューにフォーカスを戻す。
                GetRootSurfaceContext()->MoveFocusToMenuTabs();
            }
            else
            {
                pListView->enable(focusableProperty);
                m_pLabelNoItem->disable(glv::Property::t::Visible);
            }
        }
    }

    /**
    * @brief 問い合わせ済プロパティデータの解放。
    */
    void FinalizeProperties() NN_NOEXCEPT
    {
        m_QueryThread.CancelQuery();

        PropertyListView::CollectionType* pItems;
        if (nullptr != (pItems = m_pItems))
        {
            m_pItems = nullptr;
            delete pItems;
        }
    }

    PropertyListView::CollectionType*   m_pItems;
    PropertyListView*                   m_pListView;
    glv::Label*                         m_pLabelNoItem;
    QueryThread                         m_QueryThread;
};

/**
* @brief ページ生成 ( 専用クリエイター )
*/
template< size_t ID >
class TicketCatalogPageCreator : devmenu::PageCreatorBase
{
public:
    /**
    * @brief コンストラクタです。
    */
    explicit TicketCatalogPageCreator(const char* pPageName) NN_NOEXCEPT
        : devmenu::PageCreatorBase(ID, pPageName) {}

protected:
    /**
    * @brief ページインスタンスを生成します。
    */
    virtual glv::PageBase* newInstance() NN_NOEXCEPT NN_OVERRIDE
    {
        int resolution[2];
        const auto& display = glv::ApplicationFrameworkGetRuntimeContext().GetDisplay();
        display.GetResolution(resolution[0], resolution[1]);
        const auto width = static_cast< glv::space_t >(resolution[0]);
        const auto height = static_cast< glv::space_t >(resolution[1]);
        //const glv::Rect pageBounds( width - ( ( 12.0f ) * 2.0f ), height - 118.0f );    // 横は 8 + 4 マージン
        const glv::Rect pageBounds(width - 218.f, height - 118.0f);
        return new TicketCatalogPage(ID, GLV_TEXT_API_WIDE_STRING("Ticket"), pageBounds);
    }
};

/**
* @brief Declearation for the statical instance of page creator.
*/
#define LOCAL_PAGE_CREATOR( _id, _name ) TicketCatalogPageCreator< _id > g_TicketCatalogPageCreator##_id( _name );
LOCAL_PAGE_CREATOR(DevMenuPageId_Ticket, "TicketCatalog");

}} // ~namespace devmenu::ticket, ~namespace devmenu
