﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/socket/socket_Types.h>
#include <glv_CustomVerticalListView.h>
#include <glv_ScissorBoxView.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_Optional.h>
#include <algorithm>
#include <memory>
#include <functional>
#include <sstream>
#include <iomanip>
#include <new>

#include <nn/applet/applet_LibraryApplet.h>
#include <nn/netdiag/netdiag_GlobalIpAddressApi.h>
#include <nn/netdiag/netdiag_NatApi.h>
#include <nn/netdiag/netdiag_Result.h>
#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nifm/nifm_ApiNetworkProfile.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_ApiIpAddress.h>
#include <nn/nifm/nifm_ApiNetworkConnection.h>
#include <nn/nifm/nifm_ResultMenu.h>

#include "Common/DevMenu_CommonCheckBox.h"
#include "Common/DevMenu_CommonSettingsApi.h"
#include "DevMenu_Config.h"
#include "Launcher/DevMenu_LauncherLibraryAppletApis.h"
#include "DevMenu_RootSurface.h"

//#define DEBUG_DATA

namespace devmenu { namespace networksettings {

const float InformationFontSize = 20.0f;

namespace {

    nn::util::TypedStorage<nn::nifm::NetworkConnection, sizeof(nn::nifm::NetworkConnection), NN_ALIGNOF(nn::nifm::NetworkConnection)> g_NetworkConnectionStorage;
    nn::nifm::NetworkConnection* g_pNetworkConnection = nullptr;

    nn::nifm::NetworkConnection* GetNetworkConnectionPointer() NN_NOEXCEPT
    {
        if (g_pNetworkConnection == nullptr)
        {
            g_pNetworkConnection = new(&g_NetworkConnectionStorage)nn::nifm::NetworkConnection();

#if defined(NN_DEVMENU_ENABLE_SYSTEM_APPLET)
            // プリセット
            nn::nifm::SetRequestRequirementPreset(
                g_pNetworkConnection->GetRequestHandle(),
                nn::nifm::RequirementPreset_InternetForDevMenu
            );
#else
            // 疎通確認を強制化
            nn::nifm::SetRequestConnectionConfirmationOption(
                g_pNetworkConnection->GetRequestHandle(),
                nn::nifm::ConnectionConfirmationOption_Forced
            );
#endif
        }

        return g_pNetworkConnection;
    }
}

class Scene
    : public glv::Group
{
protected:
    // シザーボックスの領域設定
    static const int HeaderRegion = 32;
    static const int FooterRegion = 32;

    explicit Scene(glv::Rect rect) NN_NOEXCEPT
        : glv::Group(rect)
    {
    }
};

// ネットワーク接続テスト、Result 表示
class NetworkConnectTestTable : public glv::Table
{
private:
    static const int ButtonHeight = 33;

    glv::Label  m_LabelResultCode;

    glv::Style  m_StyleGreen;
    glv::Style  m_StyleGray;
    glv::Style  m_StyleRed;

    devmenu::Button m_CancelButton;
    devmenu::Button m_DoTestButton;

    nn::nifm::NetworkConnection* m_pNetworkConnection;

    bool m_IsNetworkRequestOnHold;
    bool m_IsNetworkAvailable;
    int m_NetworkConnectionPolingCounter;
    nn::Result m_PrevResult;
    nn::Result m_Result;

public:
    NetworkConnectTestTable() NN_NOEXCEPT
        : glv::Table("ppp", 3.0f, 3.0f),
          m_LabelResultCode("", glv::Label::Spec(glv::Place::TL, 0.0f, 8.0f, CommonValue::InitialFontSize)),
          m_CancelButton("Cancel network request", [&] {OnClickCancelEventButton_(); }, glv::Rect(320.0f, ButtonHeight), glv::Place::TL),
          m_DoTestButton("Submit network request", [&] {OnClickDoEventButton_(); }, glv::Rect(320.0f, ButtonHeight), glv::Place::TL),
          m_pNetworkConnection(GetNetworkConnectionPointer()),
          m_NetworkConnectionPolingCounter(0),
          m_PrevResult(nn::ResultSuccess()),
          m_Result(nn::ResultSuccess())
    {
        this->disable(glv::Property::DrawBorder | glv::Property::Focused | glv::Property::HitTest | glv::Property::Controllable);

        m_StyleGreen.color.set(glv::StyleColor::BlackOnWhite);
        m_StyleGreen.color.fore.set(0.8, 0.8, 0.8);
        m_StyleGreen.color.back.set(0.0, 0.6, 0.0);

        m_StyleGray.color.set(glv::StyleColor::BlackOnWhite);
        m_StyleGray.color.fore.set(0.8, 0.8, 0.8);
        m_StyleGray.color.back.set(0.3, 0.3, 0.3);

        m_StyleRed.color.set(glv::StyleColor::BlackOnWhite);
        m_StyleRed.color.fore.set(0.8, 0.8, 0.8);
        m_StyleRed.color.back.set(0.6, 0.0, 0.0);

        m_LabelResultCode.enable(glv::Property::DrawBack);

        m_DoTestButton.enable(glv::Property::Visible);

        NN_ASSERT_NOT_NULL(m_pNetworkConnection);
        m_IsNetworkRequestOnHold = m_pNetworkConnection->IsRequestOnHold();
        m_IsNetworkAvailable = m_pNetworkConnection->IsAvailable();
        m_Result = m_pNetworkConnection->GetResult();

        auto buttonArea = new glv::ScissorBoxView(0, 0, 320, ButtonHeight);
        buttonArea->enable(glv::Property::DrawBorder);
        *buttonArea << m_DoTestButton << m_CancelButton;

        auto spacer = new glv::Label(GLV_TEXT_API_WIDE_STRING("　　"));

        *this << buttonArea << spacer << m_LabelResultCode;
        this->enable( glv::Property::KeepWithinParent );

        arrange();
    }

private:

    //----------------------------------------------------------

    void Clear() NN_NOEXCEPT
    {
    }

    //----------------------------------------------------------
    void UpdateLabelesPerFrame_()
    {
        // 進捗ラべル
        {
            const int speed = 10;

            const char* table[] =
            {
                " Connecting ",
                " Connecting. ",
                " Connecting.. ",
                " Connecting... ",
            };

            const int patternCount = sizeof(table) / sizeof(table[0]);

            m_LabelResultCode.setValue(table[(m_NetworkConnectionPolingCounter / speed) % patternCount]);
        }
    }

    //----------------------------------------------------------

    void UpdateLabeles_()
    {
        // テスト実行ボタンの表示・非表示
        if (m_IsNetworkRequestOnHold || m_IsNetworkAvailable)
        {
            m_DoTestButton.disable(glv::Property::Visible);
        }
        else {
            m_DoTestButton.enable(glv::Property::Visible);
        }

        // キャンセルボタンの表示・非表示
        if (m_IsNetworkRequestOnHold || m_IsNetworkAvailable)
        {
            m_CancelButton.enable(glv::Property::Visible);
        }
        else {
            m_CancelButton.disable(glv::Property::Visible);
        }

        // リザルトラベルのスタイル
        if (m_IsNetworkRequestOnHold)
        {
            m_LabelResultCode.style(&m_StyleGray);
        }
        else {
            if (m_IsNetworkAvailable)
            {
                m_LabelResultCode.style(&m_StyleGreen);
            }
            else {
                m_LabelResultCode.style(&m_StyleRed);
            }
        }

        if (m_IsNetworkAvailable)
        {
            m_LabelResultCode.setValue(" Available ");
        }
        else if (m_IsNetworkRequestOnHold)
        {
            UpdateLabelesPerFrame_();
        }
        else
        {
            std::stringstream ss;

            ss << " Not available (";
            {
                std::stringstream ss1;
                ss1 << "0x" << std::setfill('0') << std::setw(8) << std::hex << m_Result.GetInnerValueForDebug();
                ss << ss1.str();
            }
            ss << ", " << (m_Result.GetModule() + 2000) << "-";
            {
                std::stringstream ss1;
                ss1 << std::setfill('0') << std::setw(4) << m_Result.GetDescription();
                ss << ss1.str();
            }
            ss << ") ";
            m_LabelResultCode.setValue(ss.str().c_str());

            // 切断系の失敗 Result が変化した場合に LA を起動
            if (m_Result.IsFailure() &&
                m_Result.GetInnerValueForDebug() != m_PrevResult.GetInnerValueForDebug() &&
                m_Result <= nn::nifm::ResultRequestRejected())
            {
                launcher::HandleNetworkRequestResult( m_pNetworkConnection );
            }
        }
        m_PrevResult = m_Result;
    }

    //----------------------------------------------------------

    void DoUpdate()
    {
        m_IsNetworkRequestOnHold = m_pNetworkConnection->IsRequestOnHold();
        m_IsNetworkAvailable = m_pNetworkConnection->IsAvailable();
        m_Result = m_pNetworkConnection->GetResult();

        if (m_IsNetworkRequestOnHold)
        {
            m_NetworkConnectionPolingCounter++;
        }
    }

    //----------------------------------------------------------

    void OnClickDoEventButton_() NN_NOEXCEPT
    {
        m_pNetworkConnection->SubmitRequest();

        m_NetworkConnectionPolingCounter = 0;
        m_PrevResult = nn::ResultSuccess();

        auto& g = reinterpret_cast<glv::GLV&>(this->root());
        NN_ASSERT(glv::GLV::valid(&g));
        g.setFocus(&m_CancelButton);

        OnLoopBeforeSceneRenderer();
    }

    //----------------------------------------------------------

    void OnClickCancelEventButton_() NN_NOEXCEPT
    {
        m_pNetworkConnection->CancelRequest();

        auto& g = reinterpret_cast<glv::GLV&>(this->root());
        NN_ASSERT(glv::GLV::valid(&g));
        g.setFocus(&m_DoTestButton);

        OnLoopBeforeSceneRenderer();
    }

public:

    //----------------------------------------------------------
    void OnLoopBeforeSceneRenderer() NN_NOEXCEPT // 親ページから呼んでもらう
    {
        NN_FUNCTION_LOCAL_STATIC( bool, s_IsNetworkRequestOnHoldPrevious, = false );
        NN_FUNCTION_LOCAL_STATIC( bool, s_IsNetworkAvailablePrevious, = false );
        NN_FUNCTION_LOCAL_STATIC( nn::Result, s_ResultPrevious, = nn::ResultSuccess() );

        DoUpdate();

        if (m_IsNetworkRequestOnHold ||
            m_IsNetworkRequestOnHold != s_IsNetworkRequestOnHoldPrevious ||
            m_IsNetworkAvailable != s_IsNetworkAvailablePrevious ||
            m_Result.GetInnerValueForDebug() != s_ResultPrevious.GetInnerValueForDebug())
        {
            RefreshUi();

            s_IsNetworkRequestOnHoldPrevious = m_IsNetworkRequestOnHold;
            s_IsNetworkAvailablePrevious = m_IsNetworkAvailable;
            s_ResultPrevious = m_Result;
        }
    }

    //----------------------------------------------------------
    void RefreshUi() NN_NOEXCEPT
    {
        UpdateLabeles_();

        Clear();
    }
};

// 通信有効・無効の切り替え機能の許可設定
class CommunicationControlSetting : public glv::Table
{
public:
    glv::Label  m_Label;

    CommunicationControlSetting() NN_NOEXCEPT
        : glv::Table("x", 3.0f, 3.0f),
          m_Label("Testing mode enabled", glv::Label::Spec(glv::Place::TL, 5.0f, 0.0f, CommonValue::InitialFontSize)),
          m_pCheckBox( new CheckBoxButton( "Testing mode enabled" ) )
    {
        this->disable(glv::Property::DrawBorder | glv::Property::Focused | glv::Property::HitTest | glv::Property::Controllable);

        m_IsCommunicationControlEnabledCache = this->IsCommunicationControlEnabled();
        m_pCheckBox->SetValue(m_IsCommunicationControlEnabledCache);
        m_pCheckBox->SetCallback( [](const glv::Notification& n)->void { n.receiver<CommunicationControlSetting>()->UpdateCommunicationControlEnabled( n ); }, this );

        *this << m_pCheckBox;

        this->enable( glv::Property::KeepWithinParent );

        arrange();
    }

    void OnLoopBeforeSceneRenderer() NN_NOEXCEPT // 親ページから呼んでもらう
    {
        this->RefreshUi();
    }

    void RefreshUi() NN_NOEXCEPT // 親ページから呼んでもらう
    {
        bool currentValue = false;
        GetFixedSizeFirmwareDebugSettingsItemValue( &currentValue, "nifm", "is_communication_control_enabled_for_test" );

        if ( m_IsCommunicationControlEnabledCache != currentValue )
        {
            m_pCheckBox->SetValue( currentValue );
            m_IsCommunicationControlEnabledCache = currentValue;
        }
    }

private:
    CheckBoxButton* m_pCheckBox;
    bool            m_IsCommunicationControlEnabledCache; // RefreshUi() を高頻度で呼ばれても大丈夫なようにキャッシュ

    bool IsCommunicationControlEnabled() NN_NOEXCEPT
    {
        bool isEnabled = false;
        GetFixedSizeFirmwareDebugSettingsItemValue( &isEnabled, "nifm", "is_communication_control_enabled_for_test" );
        return isEnabled;
    }

    void SetCommunicationControlEnabled(bool isEnabled) NN_NOEXCEPT
    {
        SetFixedSizeFirmwareDebugSettingsItemValue( "nifm", "is_communication_control_enabled_for_test", isEnabled );
    }

    void UpdateCommunicationControlEnabled( const glv::Notification& notification ) NN_NOEXCEPT
    {
        NN_UNUSED( notification );
        this->SetCommunicationControlEnabled( m_pCheckBox->GetValue() );
        this->m_IsCommunicationControlEnabledCache = m_pCheckBox->GetValue();
    }
};

class NetworkConnectionView : public glv::View
{
private:
    NetworkConnectTestTable m_NetworkConnectionTestTable;
    CommunicationControlSetting m_CommunicationControlSettingTable;

public:
    NetworkConnectionView()
        : glv::View(glv::Rect(1020.0f, 90.0f))
    {
        this->disable(glv::Property::DrawBorder | glv::Property::Focused | glv::Property::HitTest | glv::Property::Controllable);
        this->enable(glv::Property::CropSelf);

        auto m_Layout = new glv::Table("x");

        *m_Layout << m_NetworkConnectionTestTable << m_CommunicationControlSettingTable;
        m_Layout->arrange();
        *this << m_Layout;
    }

    void OnLoopBeforeSceneRenderer() NN_NOEXCEPT
    {
        m_NetworkConnectionTestTable.OnLoopBeforeSceneRenderer();
        m_CommunicationControlSettingTable.OnLoopBeforeSceneRenderer();
    }

    void RefreshUi() NN_NOEXCEPT
    {
        m_NetworkConnectionTestTable.RefreshUi();
        m_CommunicationControlSettingTable.RefreshUi();
    }
};

class NetworkProfileView : public glv::View
{
private:
    devmenu::Button m_RefreshButton;
    glv::View m_Spacer;
    glv::Table m_Table;

    nn::nifm::NetworkProfileData m_NetworkProfileData = {};
    in_addr m_IpAddress;
    in_addr m_SubnetMask;
    in_addr m_DefaultGateway;
    in_addr m_PreferredDns;
    in_addr m_AlternateDns;

    glv::Label m_NameLabel;
    glv::Label m_AutoConnectLabel;
    glv::Label m_NicTypeLabel;
    glv::Label m_ProfileTypeLabel;
    glv::Label m_IpSettingAutoLabel;
    glv::Label m_IpAddressLabel;
    glv::Label m_SubnetMaskLabel;
    glv::Label m_DefaultGatewayLabel;
    glv::Label m_DnsSettingAutoLabel;
    glv::Label m_PreferredDnsLabel;
    glv::Label m_AlternateDnsLabel;

private:
    void RefreshUi() NN_NOEXCEPT
    {
        m_NameLabel.setValue("-");
        m_AutoConnectLabel.setValue("-");
        m_NicTypeLabel.setValue("-");
        m_ProfileTypeLabel.setValue("-");
        m_IpSettingAutoLabel.setValue("-");
        m_IpAddressLabel.setValue("-");
        m_SubnetMaskLabel.setValue("-");
        m_DefaultGatewayLabel.setValue("-");
        m_DnsSettingAutoLabel.setValue("-");
        m_PreferredDnsLabel.setValue("-");
        m_AlternateDnsLabel.setValue("-");

        nn::Result result = nn::nifm::GetCurrentNetworkProfile(&m_NetworkProfileData);
        if (result.IsSuccess())
        {
            {
                m_NameLabel.setValue(static_cast<const char*>(m_NetworkProfileData.name));
                m_AutoConnectLabel.setValue(m_NetworkProfileData.isAutoConnect ? "true" : "false");
                switch (m_NetworkProfileData.networkInterfaceType)
                {
                case nn::nifm::NetworkInterfaceType_Ethernet:
                    m_NicTypeLabel.setValue("Ethernet");
                    break;
                case nn::nifm::NetworkInterfaceType_Ieee80211:
                    m_NicTypeLabel.setValue("Ieee802.11");
                    break;
                default: NN_UNEXPECTED_DEFAULT;
                }
                switch (m_NetworkProfileData.networkProfileType)
                {
                case nn::nifm::NetworkProfileType_User:
                    m_ProfileTypeLabel.setValue("User");
                    break;
                case nn::nifm::NetworkProfileType_SsidList:
                    m_ProfileTypeLabel.setValue("SSID List");
                    break;
                case nn::nifm::NetworkProfileType_Temporary:
                    m_ProfileTypeLabel.setValue("Temporary");
                    break;
                default: NN_UNEXPECTED_DEFAULT;
                }

                // ip
                m_IpSettingAutoLabel.setValue(m_NetworkProfileData.ipSetting.ip.isAuto ? "true" : "false");
                m_DnsSettingAutoLabel.setValue(m_NetworkProfileData.ipSetting.dns.isAuto ? "true" : "false");
                result = nn::nifm::GetCurrentIpConfigInfo(&m_IpAddress, &m_SubnetMask, &m_DefaultGateway, &m_PreferredDns, &m_AlternateDns);
                if (result.IsSuccess())
                {
                    char buffer[sizeof("000.000.000.000")];
                    unsigned long addr;
                    addr = m_IpAddress.s_addr;
                    nn::util::SNPrintf(buffer, sizeof(buffer), "%u.%u.%u.%u", (addr >> 0) & 0xff, (addr >> 8) & 0xff, (addr >> 16) & 0xff, (addr >> 24) & 0xff);
                    m_IpAddressLabel.setValue(static_cast<const char*>(buffer));
                    addr = m_SubnetMask.s_addr;
                    nn::util::SNPrintf(buffer, sizeof(buffer), "%u.%u.%u.%u", (addr >> 0) & 0xff, (addr >> 8) & 0xff, (addr >> 16) & 0xff, (addr >> 24) & 0xff);
                    m_SubnetMaskLabel.setValue(static_cast<const char*>(buffer));
                    addr = m_DefaultGateway.s_addr;
                    nn::util::SNPrintf(buffer, sizeof(buffer), "%u.%u.%u.%u", (addr >> 0) & 0xff, (addr >> 8) & 0xff, (addr >> 16) & 0xff, (addr >> 24) & 0xff);
                    m_DefaultGatewayLabel.setValue(static_cast<const char*>(buffer));
                    addr = m_PreferredDns.s_addr;
                    nn::util::SNPrintf(buffer, sizeof(buffer), "%u.%u.%u.%u", (addr >> 0) & 0xff, (addr >> 8) & 0xff, (addr >> 16) & 0xff, (addr >> 24) & 0xff);
                    m_PreferredDnsLabel.setValue(static_cast<const char*>(buffer));
                    addr = m_AlternateDns.s_addr;
                    nn::util::SNPrintf(buffer, sizeof(buffer), "%u.%u.%u.%u", (addr >> 0) & 0xff, (addr >> 8) & 0xff, (addr >> 16) & 0xff, (addr >> 24) & 0xff);
                    m_AlternateDnsLabel.setValue(static_cast<const char*>(buffer));
                }
            }
        }
    }

public:
    NetworkProfileView() NN_NOEXCEPT
      : glv::View(glv::Rect(500.0f, 360.0f), glv::Place::CC),
        m_RefreshButton("Refresh", [&] { RefreshUi(); }, glv::Rect(100.0f, 33.0f)),
        m_Spacer(glv::Rect(480.0f, 1.0f)),
        m_Table("<>,..,x-,<<,<<,<<,<<,<<,<<,<<,<<,<<,<<,<<", 8.0f),
        m_NameLabel("-                                             ", glv::Label::Spec(glv::Place::TL, 5.0f, 0.0f, InformationFontSize)),
        m_AutoConnectLabel("-", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize)),
        m_NicTypeLabel("-", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize)),
        m_ProfileTypeLabel("-", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize)),
        m_IpSettingAutoLabel("-", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize)),
        m_IpAddressLabel("-", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize)),
        m_SubnetMaskLabel("-", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize)),
        m_DefaultGatewayLabel("-", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize)),
        m_DnsSettingAutoLabel("-", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize)),
        m_PreferredDnsLabel("-", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize)),
        m_AlternateDnsLabel("-", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize))
    {
        this->disable(glv::Property::DrawBorder | glv::Property::Focused | glv::Property::HitTest | glv::Property::Controllable);
        this->enable(glv::Property::CropChildren);
        m_Spacer.disable(glv::Property::DrawBack | glv::Property::Focused | glv::Property::HitTest | glv::Property::Controllable);

        *this << (m_Table
            << new glv::Label("NetworkProfile", glv::Label::Spec(glv::Place::TL, 5, 0, CommonValue::InitialFontSize))
            << m_RefreshButton
            << m_Spacer
            << new glv::Label("Name : ", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize))
            << m_NameLabel
            << new glv::Label("AutoConnect : ", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize))
            << m_AutoConnectLabel
            << new glv::Label("NicType : ", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize))
            << m_NicTypeLabel
            << new glv::Label("ProfileType : ", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize))
            << m_ProfileTypeLabel
            << new glv::Label("AutoIpAddress : ", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize))
            << m_IpSettingAutoLabel
            << new glv::Label("IpAddress : ", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize))
            << m_IpAddressLabel
            << new glv::Label("SubnetMask : ", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize))
            << m_SubnetMaskLabel
            << new glv::Label("DefaultGateway : ", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize))
            << m_DefaultGatewayLabel
            << new glv::Label("AutoDns : ", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize))
            << m_DnsSettingAutoLabel
            << new glv::Label("PreferredDns : ", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize))
            << m_PreferredDnsLabel
            << new glv::Label("AlternateDns : ", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize))
            << m_AlternateDnsLabel
            );

        m_Table.arrange();

        RefreshUi();
    }
};

//
class AccessPointView : public glv::View
{
private:
    nn::nifm::AccessPointData m_AccessPointData;

    devmenu::Button m_RefreshButton;
    glv::View  m_Spacer;
    glv::Table m_Table;
    glv::Label m_SsidLabel;
    glv::Label m_BssidLabel;
    glv::Label m_SecurityLabel;
    glv::Label m_LinkLevelLabel;
    glv::Label m_ChannelLabel;

private:
    void RefreshUi() NN_NOEXCEPT
    {
        m_SsidLabel.setValue("-");
        m_BssidLabel.setValue("-");
        m_SecurityLabel.setValue("-");
        m_LinkLevelLabel.setValue("-");
        m_ChannelLabel.setValue("-");

        nn::Result result = nn::nifm::GetCurrentAccessPoint(&m_AccessPointData);
        if (result.IsSuccess())
        {
            const nn::nifm::Ssid& ssid = m_AccessPointData.ssid;
            char buffer[nn::nifm::Ssid::HexSize * 2 + 1] = {};

            bool hasNonAscii = false;
            for (int i = 0; i < ssid.length; ++i)
            {
                if (ssid.hex[i] < 0x20 || 0x7e < ssid.hex[i])
                {
                    hasNonAscii = true;
                    break;
                }
            }

            if (hasNonAscii)
            {
                for (int i = 0; i < ssid.length; ++i)
                {
                    nn::util::SNPrintf(&buffer[i * 2], sizeof(char) * 3, "%02x", ssid.hex[i]);
                }
            }
            else
            {
                for (int i = 0; i < ssid.length; ++i)
                {
                    nn::util::SNPrintf(&buffer[i], sizeof(char) * 2, "%c", ssid.hex[i]);
                }
            }
            m_SsidLabel.setValue(static_cast<const char*>(buffer));
            std::memset(buffer, 0, sizeof(buffer));

            auto m = m_AccessPointData.bssid.data;
            nn::util::SNPrintf(buffer, sizeof(buffer), "%02x:%02x:%02x:%02x:%02x:%02x", m[0], m[1], m[2], m[3], m[4], m[5]);
            m_BssidLabel.setValue(static_cast<const char*>(buffer));
            std::memset(buffer, 0, sizeof(buffer));

            switch (m_AccessPointData.authentication)
            {
            case nn::nifm::Authentication_Open:
                nn::util::SNPrintf(buffer, sizeof(buffer), "Open");
                break;
            case nn::nifm::Authentication_Shared:
                nn::util::SNPrintf(buffer, sizeof(buffer), "Shared");
                break;
            case nn::nifm::Authentication_WpaPsk:
                nn::util::SNPrintf(buffer, sizeof(buffer), "WPA-PSK");
                break;
            case nn::nifm::Authentication_Wpa2Psk:
                nn::util::SNPrintf(buffer, sizeof(buffer), "WPA2-PSK");
                break;
            default: NN_UNEXPECTED_DEFAULT;
            }
            auto length = nn::util::Strnlen(buffer, sizeof(buffer));
            switch (m_AccessPointData.encryption)
            {
            case nn::nifm::Encryption_None:
                nn::util::SNPrintf(&buffer[length], sizeof(buffer) - length, "(None)");
                break;
            case nn::nifm::Encryption_Wep:
                nn::util::SNPrintf(&buffer[length], sizeof(buffer) - length, "(WEP)");
                break;
            case nn::nifm::Encryption_Tkip:
                nn::util::SNPrintf(&buffer[length], sizeof(buffer) - length, "TKIP");
                break;
            case nn::nifm::Encryption_Aes:
                nn::util::SNPrintf(&buffer[length], sizeof(buffer) - length, "(AES)");
                break;
            default: NN_UNEXPECTED_DEFAULT;
            }
            m_SecurityLabel.setValue(static_cast<const char*>(buffer));
            std::memset(buffer, 0, sizeof(buffer));

            nn::util::SNPrintf(buffer, sizeof(buffer), "%d", m_AccessPointData.channel);
            m_ChannelLabel.setValue(static_cast<const char*>(buffer));

            nn::util::SNPrintf(buffer, sizeof(buffer), "%d", m_AccessPointData.linkLevel);
            m_LinkLevelLabel.setValue(static_cast<const char*>(buffer));
        }
    }

public:
    AccessPointView() NN_NOEXCEPT
      : glv::View(glv::Rect(500.0f, 218.0f), glv::Place::CC),
        m_RefreshButton("Refresh", [&] { RefreshUi(); }, glv::Rect(100.0f, 33.0f)),
        m_Spacer(glv::Rect(480.0f, 1.0f)),
        m_Table("<>,..,x-,<<,<<,<<,<<,<<", 8.0f),
        m_SsidLabel("-                                                  ", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize)),
        m_BssidLabel("-", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize)),
        m_SecurityLabel("-", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize)),
        m_LinkLevelLabel("-", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize)),
        m_ChannelLabel("-", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize))

    {
        this->disable(glv::Property::DrawBorder | glv::Property::Focused | glv::Property::HitTest | glv::Property::Controllable);
        this->enable(glv::Property::CropChildren);
        m_Spacer.disable(glv::Property::DrawBack | glv::Property::Focused | glv::Property::HitTest | glv::Property::Controllable);

        *this << (m_Table
            << new glv::Label("AccessPoint", glv::Label::Spec(glv::Place::TL, 5, 0, CommonValue::InitialFontSize))
            << m_RefreshButton
            << m_Spacer
            << new glv::Label("Ssid : ", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize))
            << m_SsidLabel
            << new glv::Label("Bssid : ", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize))
            << m_BssidLabel
            << new glv::Label("Security : ", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize))
            << m_SecurityLabel
            << new glv::Label("Channel : ", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize))
            << m_ChannelLabel
            << new glv::Label("LinkLevel : ", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize))
            << m_LinkLevelLabel
            );

        m_Table.arrange();

        RefreshUi();
    }
};

class InterfaceView : public glv::View
{
private:
    nn::nifm::NetworkInterfaceInfo m_NetworkInterfaceInfo[2];

    devmenu::Button m_RefreshButton;
    glv::View m_Spacer;
    glv::Table m_Table;
    glv::Label m_WirelessInterfaceMacAddressLabel;
    glv::Label m_EthernetInterfaceMacAddressLabel;

private:
    void RefreshUi()
    {
        m_WirelessInterfaceMacAddressLabel.setValue("-");
        m_EthernetInterfaceMacAddressLabel.setValue("-");

        for (auto& info : m_NetworkInterfaceInfo)
        {
            info.isAvailable = false;
        }

        int outCount = 0;
        nn::Result result = nn::nifm::EnumerateNetworkInterfaces(m_NetworkInterfaceInfo, &outCount, 2,
            nn::nifm::EnumerateNetworkInterfacesFilter_Ieee80211 | nn::nifm::EnumerateNetworkInterfacesFilter_Ethernet);

        if (result.IsSuccess())
        {
            for (auto&& info : m_NetworkInterfaceInfo)
            {
                if (!info.isAvailable)
                {
                    continue;
                }
                const auto& m = info.macAddress.data;
                char buffer[sizeof("00:00:00:00:00:00")];
                nn::util::SNPrintf(buffer, sizeof(buffer), "%02x:%02x:%02x:%02x:%02x:%02x", m[0], m[1], m[2], m[3], m[4], m[5]);
                if (info.type == nn::nifm::NetworkInterfaceType_Ieee80211)
                {
                    m_WirelessInterfaceMacAddressLabel.setValue(static_cast<const char*>(buffer));
                }
                else if (info.type == nn::nifm::NetworkInterfaceType_Ethernet)
                {
                    m_EthernetInterfaceMacAddressLabel.setValue(static_cast<const char*>(buffer));
                }
            }
        }
    }

public:
    InterfaceView()
        : glv::View(glv::Rect(500.0f, 128.0f), glv::Place::CC),
          m_RefreshButton("Refresh", [&] { RefreshUi(); }, glv::Rect(100.0f, 33.0f)),
          m_Spacer(glv::Rect(480.0f, 1.0f)),
          m_Table("<>,..,x-,<<,<<", 8.0f),
          m_WirelessInterfaceMacAddressLabel("-                                              ", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize)),
          m_EthernetInterfaceMacAddressLabel("-", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize))
    {
        this->disable(glv::Property::DrawBorder | glv::Property::Focused | glv::Property::HitTest | glv::Property::Controllable);
        this->enable(glv::Property::CropChildren);
        m_Spacer.disable(glv::Property::DrawBack | glv::Property::Focused | glv::Property::HitTest | glv::Property::Controllable);

        *this << (m_Table
            << new glv::Label("Interface", glv::Label::Spec(glv::Place::TL, 5, 0, CommonValue::InitialFontSize))
            << m_RefreshButton
            << m_Spacer
            << new glv::Label("Wireless : ", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize))
            << m_WirelessInterfaceMacAddressLabel
            << new glv::Label("Ethernet : ", glv::Label::Spec(glv::Place::TL, 5, 0, InformationFontSize))
            << m_EthernetInterfaceMacAddressLabel
            );

        m_Table.arrange();

        RefreshUi();
    }
};

// グローバル IP アドレス取得・表示
class GlobalIpAddrTable : public glv::Table
{
private:
    enum class ExecutionState
    {
        Idle,
        PrepareToRun,
        Running,
        Finished,
        Cancelling,
    };
    ExecutionState m_ExecutionState;
    devmenu::Button m_ExecuteButton;
    glv::Label m_ResultLabel;

    nn::Result m_LastResult;
    nn::netdiag::GlobalIpAddress m_GlobalIpAddress;
    std::atomic<bool> m_IsFinished;
    int m_WaitingCount;

    static const size_t ThreadStackSize = 32 * 1024;
    nn::os::ThreadType m_Thread;
    void* m_pStack;

    static void ThreadFunc( void* arg )
    {
        // グローバルアドレス取得
        GlobalIpAddrTable* pThis = reinterpret_cast<GlobalIpAddrTable*>(arg);
        NN_ASSERT_NOT_NULL( pThis );
        pThis->m_LastResult = nn::netdiag::GetGlobalIpAddress(&pThis->m_GlobalIpAddress);
        pThis->m_IsFinished = true;
    }

    void LaunchExecutionThread() NN_NOEXCEPT
    {
        // スタックを取りスレッドを開始
        m_pStack = MemoryAllocator::AllocAlignedMemory( nn::os::ThreadStackAlignment, ThreadStackSize );
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::os::CreateThread( &m_Thread, ThreadFunc, this, m_pStack, ThreadStackSize, nn::os::DefaultThreadPriority ) );
        nn::os::SetThreadNamePointer( &m_Thread, "DevMenuNetworkConnectionGetGlobalIpAddress" );
        nn::os::StartThread( &m_Thread );
    }
    void DestroyExecutionThread() NN_NOEXCEPT
    {
        // スレッドやスタックを破棄
        nn::os::WaitThread( &m_Thread );
        nn::os::DestroyThread( &m_Thread );
        MemoryAllocator::FreeAlignedMemory(m_pStack);
    }

    void OnClickExecuteButton() NN_NOEXCEPT
    {
        if (!GetNetworkConnectionPointer()->IsAvailable())
        {
            m_ResultLabel.setValue("In advance, submit network request!");
            return;
        }

        if (m_ExecutionState == ExecutionState::Idle)
        {
            m_ExecutionState = ExecutionState::PrepareToRun;
        }

        // 取得中の場合はキャンセル
        if ( m_ExecutionState == ExecutionState::Running )
        {
            m_ExecutionState = ExecutionState::Cancelling;
            nn::netdiag::InterruptGetGlobalIpAddress();
        }
    }

public:
    GlobalIpAddrTable() NN_NOEXCEPT
        : glv::Table("xxx", 3.0f, 3.0f),
          m_ExecutionState(ExecutionState::Idle),
          m_ExecuteButton("Get Global IP Address", [&] {OnClickExecuteButton(); }, glv::Rect(320.0f, 33.0f), glv::Place::TL),
          m_ResultLabel("", glv::Label::Spec(glv::Place::TL, 5, 0, CommonValue::InitialFontSize))
    {
        this->disable(glv::Property::DrawBorder | glv::Property::Focused | glv::Property::HitTest | glv::Property::Controllable);

        *this
            << m_ExecuteButton
            << new glv::Label(GLV_TEXT_API_WIDE_STRING("　"))
            << m_ResultLabel;
        this->arrange();
    }

    void OnLoopBeforeSceneRenderer() NN_NOEXCEPT // 親ページから呼んでもらう
    {
        this->RefreshUi();
    }

    void RefreshUi() NN_NOEXCEPT // 親ページから呼んでもらう
    {
        switch( m_ExecutionState )
        {
            case ExecutionState::Idle:
                // do nothing
                break;
            case ExecutionState::PrepareToRun:
                m_IsFinished = false;
                m_ExecutionState = ExecutionState::Running;
                m_WaitingCount = 0;
                LaunchExecutionThread();
                break;
            case ExecutionState::Running:
            case ExecutionState::Cancelling:
                {
                    const int speed = 10;
                    const char gettingStr[] = "Getting....";
                    const char cancellingStr[] = "Cancelling....";
                    auto isGetting = (m_ExecutionState == ExecutionState::Running);
                    char outputStr[ 32 ];
                    int length = (isGetting? sizeof(gettingStr): sizeof(cancellingStr)) - sizeof("....") + ((m_WaitingCount++ / speed) % 4);

                    sprintf( outputStr, "%.*s", length, (isGetting? gettingStr: cancellingStr) );
                    m_ResultLabel.setValue( static_cast<const char*>(outputStr) );

                    if ( m_IsFinished.load() )
                    {
                        DestroyExecutionThread();
                        m_ExecutionState = ExecutionState::Finished;
                    }
                }
                break;
            case ExecutionState::Finished:
                if (m_LastResult.IsSuccess())
                {
                    m_ResultLabel.setValue(static_cast<const char*>(m_GlobalIpAddress.value));
                }
                else
                {
                    if ( nn::netdiag::ResultInterrupted().Includes( m_LastResult ) )
                    {
                        m_ResultLabel.setValue( "Cancelled." );
                    }
                    else
                    {
                        std::stringstream ss;
                        ss
                            << "Failed. result("
                            << "0x" << std::setfill('0') << std::setw(8) << std::hex << m_LastResult.GetInnerValueForDebug()
                            << ")";
                        m_ResultLabel.setValue(ss.str().c_str());
                    }
                }

                m_ExecutionState = ExecutionState::Idle;
                break;
            default: NN_UNEXPECTED_DEFAULT;
        }
    }
};

// NAT タイプ判定・表示
class NatTypeTable : public glv::Table
{
private:
    enum class ExecutionState
    {
        Idle,
        PrepareToRun,
        Running,
        Finished,
        Cancelling,
    };
    ExecutionState m_ExecutionState;
    devmenu::Button m_ExecuteButton;
    glv::Label m_ResultLabel;

    nn::Result m_LastResult;
    nn::netdiag::NatType m_NatType;
    std::atomic<bool> m_IsFinished;
    int m_WaitingCount;

    static const size_t ThreadStackSize = 32 * 1024;
    nn::os::ThreadType m_Thread;
    void* m_pStack;

    static void ThreadFunc( void* arg )
    {
        // NAT タイプ
        NatTypeTable* pThis = reinterpret_cast<NatTypeTable*>(arg);
        NN_ASSERT_NOT_NULL( pThis );
        pThis->m_LastResult = nn::netdiag::DetectNatType(&pThis->m_NatType);
        pThis->m_IsFinished = true;
    }

    void LaunchExecutionThread() NN_NOEXCEPT
    {
        // スタックを取りスレッドを開始
        m_pStack = MemoryAllocator::AllocAlignedMemory( nn::os::ThreadStackAlignment, ThreadStackSize );
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::os::CreateThread( &m_Thread, ThreadFunc, this, m_pStack, ThreadStackSize, nn::os::DefaultThreadPriority ) );
        nn::os::SetThreadNamePointer( &m_Thread, "DevMenuNetworkConnectionGetNatType" );
        nn::os::StartThread( &m_Thread );
    }
    void DestroyExecutionThread() NN_NOEXCEPT
    {
        // スレッドやスタックを破棄
        nn::os::WaitThread( &m_Thread );
        nn::os::DestroyThread( &m_Thread );
        MemoryAllocator::FreeAlignedMemory(m_pStack);
    }

    void OnClickExecuteButton() NN_NOEXCEPT
    {
        if (!GetNetworkConnectionPointer()->IsAvailable())
        {
            m_ResultLabel.setValue("In advance, submit network request!");
            return;
        }

        if (m_ExecutionState == ExecutionState::Idle)
        {
            m_ExecutionState = ExecutionState::PrepareToRun;
        }

        // 取得中の場合はキャンセル
        if ( m_ExecutionState == ExecutionState::Running )
        {
            m_ExecutionState = ExecutionState::Cancelling;
            nn::netdiag::InterruptDetectNatType();
        }
    }

public:
    NatTypeTable()
        : glv::Table("xxx", 3.0f, 3.0f),
          m_ExecutionState(ExecutionState::Idle),
          m_ExecuteButton("Get NAT Type", [&] {OnClickExecuteButton(); }, glv::Rect(320.0f, 33.0f), glv::Place::TL),
          m_ResultLabel("", glv::Label::Spec(glv::Place::TL, 5, 0, CommonValue::InitialFontSize))
    {
        this->disable(glv::Property::DrawBorder | glv::Property::Focused | glv::Property::HitTest | glv::Property::Controllable);

        *this
            << m_ExecuteButton
            << new glv::Label(GLV_TEXT_API_WIDE_STRING("　"))
            << m_ResultLabel;
        this->arrange();
    }

    void OnLoopBeforeSceneRenderer() NN_NOEXCEPT // 親ページから呼んでもらう
    {
        this->RefreshUi();
    }

    void RefreshUi() NN_NOEXCEPT // 親ページから呼んでもらう
    {
        switch( m_ExecutionState )
        {
            case ExecutionState::Idle:
                // do nothing;
                break;
            case ExecutionState::PrepareToRun:
                m_IsFinished = false;
                m_ExecutionState = ExecutionState::Running;
                m_WaitingCount = 0;
                LaunchExecutionThread();
                break;
            case ExecutionState::Running:
            case ExecutionState::Cancelling:
                {
                    const int speed = 10;
                    const char gettingStr[] = "Getting....";
                    const char cancellingStr[] = "Cancelling....";
                    auto isGetting = (m_ExecutionState == ExecutionState::Running);
                    char outputStr[ 32 ];
                    int length = (isGetting? sizeof(gettingStr): sizeof(cancellingStr)) - sizeof("....") + ((m_WaitingCount++ / speed) % 4);

                    sprintf( outputStr, "%.*s", length, (isGetting? gettingStr: cancellingStr) );
                    m_ResultLabel.setValue( static_cast<const char*>(outputStr) );

                    if ( m_IsFinished.load() )
                    {
                        DestroyExecutionThread();
                        m_ExecutionState = ExecutionState::Finished;
                    }
                }
                break;
            case ExecutionState::Finished:
                if (m_LastResult.IsSuccess())
                {
                    switch (m_NatType)
                    {
                        case nn::netdiag::NatType_A:
                            {
                                m_ResultLabel.setValue("NAT type A");
                            }
                            break;
                        case nn::netdiag::NatType_B:
                            {
                                m_ResultLabel.setValue("NAT type B");
                            }
                            break;
                        case nn::netdiag::NatType_C:
                            {
                                m_ResultLabel.setValue("NAT type C");
                            }
                            break;
                        case nn::netdiag::NatType_D:
                            {
                                m_ResultLabel.setValue("NAT type D");
                            }
                            break;
                        case nn::netdiag::NatType_Z:
                            {
                                // サポート上 Z は F と表示
                                // (http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=165372838)
                                m_ResultLabel.setValue("NAT type F");
                            }
                            break;
                        default: NN_UNEXPECTED_DEFAULT;
                    }
                }
                else
                {
                    if ( nn::netdiag::ResultInterrupted().Includes( m_LastResult ) )
                    {
                        m_ResultLabel.setValue( "Cancelled." );
                    }
                    else
                    {
                        std::stringstream ss;
                        ss
                            << "Failed. result("
                            << "0x" << std::setfill('0') << std::setw(8) << std::hex << m_LastResult.GetInnerValueForDebug()
                            << ")";
                        m_ResultLabel.setValue(ss.str().c_str());
                    }
                }
                m_ExecutionState = ExecutionState::Idle;
                break;
            default: NN_UNEXPECTED_DEFAULT;
        }
    }
};

class NetdiagView : public glv::View
{
private:
    GlobalIpAddrTable m_GlobalIpAddrTable;
    NatTypeTable m_NatTypeTable;

public:
    NetdiagView()
        : glv::View(glv::Rect(1020.0f, 90.0f))
    {
        this->disable(glv::Property::DrawBorder | glv::Property::Focused | glv::Property::HitTest | glv::Property::Controllable);

        auto m_Layout = new glv::Table("x");

        *m_Layout << m_GlobalIpAddrTable << m_NatTypeTable;
        m_Layout->arrange();
        *this << m_Layout;
    }

    void OnLoopBeforeSceneRenderer() NN_NOEXCEPT
    {
        m_GlobalIpAddrTable.OnLoopBeforeSceneRenderer();
        m_NatTypeTable.OnLoopBeforeSceneRenderer();
    }

    void RefreshUi() NN_NOEXCEPT
    {
        m_GlobalIpAddrTable.RefreshUi();
        m_NatTypeTable.RefreshUi();
    }
};


/**
 * @brief ネットワーク接続機能のページです。
 */
class NetworkConnectionPage : public Page
{
private:
    NetworkConnectionView* m_pNetworkConnectionView;
    NetdiagView* m_pNetdiagView;
    NetworkProfileView* m_pNetworkProfileView;
    AccessPointView* m_pAccessPointView;
    InterfaceView* m_pInterfaceView;

public:
    /**
     * @brief コンストラクタです。
     */
    NetworkConnectionPage(int pageId, const glv::WideCharacterType* pageCaption, glv::Rect rect) NN_NOEXCEPT
        : Page(pageId, pageCaption, rect)
    {
    }

    virtual void OnChangeIntoForeground() NN_NOEXCEPT NN_OVERRIDE
    {
        RefreshUi();
    }

    /**
     * @brief ページがコンテナに追加された後に呼び出されます。
     */
    virtual void OnAttachedPage() NN_NOEXCEPT NN_OVERRIDE
    {
        m_pNetworkConnectionView = new NetworkConnectionView();
        m_pNetdiagView = new NetdiagView();

        m_pNetworkProfileView = new NetworkProfileView();
        m_pAccessPointView = new AccessPointView();
        m_pInterfaceView = new InterfaceView();

        auto pTable = new glv::Table("x-,"
                                     "..,"
                                     "x-,"
                                     "..,"
                                     "<>,"
                                     "|>,", 15, 15);

         *pTable
            << m_pNetworkConnectionView
            << m_pNetdiagView
            << m_pNetworkProfileView
            << m_pAccessPointView
            << m_pInterfaceView;

        pTable->arrange();

        // Set the Table to be able to detect clicks
        pTable->enable( glv::Property::HitTest | glv::Property::GestureDetectability );
        // Attach a Click handler to detect if the B button was pressed
        pTable->attach( this->FocusMenuTabOnBbuttonPress, glv::Update::Clicked, this );

        *this << pTable;
    }

    /**
     * @brief ページがコンテナから削除された前に呼び出されます。
     */
    virtual void OnDetachedPage() NN_NOEXCEPT NN_OVERRIDE
    {
        FinalizeProperties();
    }

    /**
     * @brief ページがアクティブ( 選択により表示開始 )になる際に呼び出されます。
     */
    virtual void OnActivatePage() NN_NOEXCEPT NN_OVERRIDE
    {
        RefreshUi();
    }

    /**
     * @brief ページがディアクティブ( 選択により非表示開始 )になる際に呼び出されます。
     */
    virtual void OnDeactivatePage() NN_NOEXCEPT NN_OVERRIDE
    {
        FinalizeProperties();
    }

    /**
     * @brief アプリケーションメインループからのコールバックです。
     *
     * @details glvシーンレンダラへ hid系イベントが通知される前に呼び出されます。@n
     * この時点ではまだ glvコンテキストのレンダリングは開始していません。
     */
    virtual void OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED( context );
        NN_UNUSED( events );

        m_pNetworkConnectionView->OnLoopBeforeSceneRenderer();
        m_pNetdiagView->OnLoopBeforeSceneRenderer();
    }

    /**
     * @brief アプリケーションメインループからのコールバックです。
     *
     * @details glvシーンレンダラのレンダリングが終わった後に呼び出されます。
     */
    virtual void OnLoopAfterSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED( context );
        NN_UNUSED( events );
    }

    /**
     * @brief ページにフォーカスが与えられた際に、フォーカスを横取る子供を指定します。
     */
    virtual View* GetFocusableChild() NN_NOEXCEPT NN_OVERRIDE
    {
        return this;
    }

private:
    void RefreshUi() NN_NOEXCEPT
    {
        m_pNetworkConnectionView->RefreshUi();
        m_pNetdiagView->RefreshUi();
    }

    /**
     * @brief 問い合わせ済プロパティデータの解放。
     */
    void FinalizeProperties() NN_NOEXCEPT
    {
    }
};

/**
 * @brief ページ生成 ( 専用クリエイター )
 */
template< size_t ID >
class NetworkConnectionPageCreator : PageCreatorBase
{
public:
    /**
     * @brief コンストラクタです。
     */
    explicit NetworkConnectionPageCreator( const char* pageName ) NN_NOEXCEPT
        : PageCreatorBase( ID, pageName ) {}

protected:

    /**
     * @brief ページインスタンスを生成します。
     */
    virtual glv::PageBase* newInstance() NN_NOEXCEPT NN_OVERRIDE
    {
        int resolution[ 2 ];
        const glv::DisplayMetrics& display = glv::ApplicationFrameworkGetRuntimeContext().GetDisplay();
        display.GetResolution( resolution[ 0 ], resolution[ 1 ] );
        const glv::space_t width = static_cast< glv::space_t >( resolution[ 0 ] );
        const glv::space_t height = static_cast< glv::space_t >( resolution[ 1 ] );
        const glv::Rect pageBounds( width - 218.f, height - 118.0f );
        return new NetworkConnectionPage( ID,  GLV_TEXT_API_WIDE_STRING( "Network" ), pageBounds );
    }
};

/**
 * @brief Declearation for the statical instance of page creator.
 */
#define LOCAL_PAGE_CREATOR( _id, _name ) NetworkConnectionPageCreator< _id > g_NetworkConnectionPageCreator##_id( _name );
LOCAL_PAGE_CREATOR(DevMenuPageId_NetworkConnection, "NetworkConnectionPage" );

}} // ~namespace devmenu::networksettings, ~namespace devmenu

