﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdlib>
#include <cstring>
#include <glv_viewutility.h>
#include <nn/nn_Assert.h>

#include "DevMenu_Config.h"
#include "DevMenu_ModalView.h"
#include "DevMenu_RootSurface.h"
#include "DevMenu_Image.h"

namespace devmenu {

/**
 * @brief       モーダルビューの定義です。
 */
namespace {

    // 実行情報の初期数
    static const nn::TimeSpan ZeroSpan;

} // end of unnamed namespace

ModalView::ModalView( const glv::Rect& rect ) NN_NOEXCEPT
    : glv::ViewContainer(rect)
    , m_Timeout()
    , m_StartTick(0)
    , m_IsRunning(false)
    , m_IsExitRequested(false)
{
    enable(glv::Visible | glv::DrawBack | glv::DrawBorder);
    anchor(glv::Place::CC);
}

ModalView& ModalView::SetTimeout(nn::TimeSpan timeout) NN_NOEXCEPT
{
    m_Timeout = timeout;
    return *this;
}

ModalView& ModalView::ExtendTimeout(nn::TimeSpan timeout) NN_NOEXCEPT
{
    m_Timeout = timeout;
    m_StartTick = nn::os::GetSystemTick();
    return *this;
}

bool ModalView::IsExpired() const NN_NOEXCEPT
{
    bool isExpired = false;
    if ( m_IsRunning && (m_Timeout > ZeroSpan) )
    {
        nn::os::Tick current = nn::os::GetSystemTick();
        nn::TimeSpan elapsed = nn::os::ConvertToTimeSpan(current - m_StartTick);
        isExpired = (elapsed > m_Timeout);
    }
    return isExpired;
}

bool ModalView::IsTimeoutValid() const NN_NOEXCEPT
{
    return m_Timeout > ZeroSpan;
}

void ModalView::StartModal() NN_NOEXCEPT
{
    NN_ASSERT( !m_IsRunning );

    if ( !m_IsRunning )
    {
        m_IsRunning = true;
        m_IsExitRequested = false;
        m_StartTick = nn::os::GetSystemTick();
        enable(glv::Visible);
        OnActivate();
    }
}

void ModalView::EndModal() NN_NOEXCEPT
{
    NN_ASSERT( m_IsRunning );

    if ( m_IsRunning )
    {
        m_IsRunning = false;
        m_IsExitRequested = false;
        OnDeactivate();
    }
}


/**
 * @brief       スプラッシュビューの定義です。
 */
SplashView::SplashView() NN_NOEXCEPT
    :   ModalView(glv::Rect(1028, 512))
{
    enable(glv::DrawBorder);
    SetTimeout(nn::TimeSpan::FromSeconds(2));
}

SplashView& SplashView::Initialize(const char* pJpegPath) NN_NOEXCEPT
{
    ImageData image;
    image.LoadJpegImage(pJpegPath);
    *this << (new TextureView())->CreateTexture(image, true).pos(glv::Place::CC).anchor(glv::Place::CC);
    return *this;
}

void SplashView::OnLoopAfterSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT
{
    if( IsExpired() != false )
    {
        ExitModal();
    }
}


/**
 * @brief       メッセージビューの定義です。
 */
namespace {

    // 実行情報の初期数
    static const int InitialActionCapacity = 4;

    // パッディング
    const glv::space_t MessageViewPaddingX = 32.f;
    const glv::space_t MessageViewPaddingY = 32.f;
    const glv::space_t MessageButtonPaddingX = 32.f;
    const glv::space_t MessageButtonPaddingY = 16.f;

} // end of unnamed namespace

MessageView::MessageView() NN_NOEXCEPT
    :   ModalView(glv::Rect(0)),
        m_MessagesTable("<", MessageViewPaddingX, MessageViewPaddingY, glv::Rect(0)),
        m_ButtonsTable("", MessageViewPaddingX, MessageViewPaddingY, glv::Rect(0)),
        m_ImageTable("x", MessageViewPaddingX, MessageViewPaddingY, glv::Rect(0)),
        m_ViewTable("<,x,x", 0, 0, glv::Rect(0)),
        m_IsClosable(true)
{
    Initialize();
}

MessageView::MessageView(bool isClosable) NN_NOEXCEPT
    :   ModalView(glv::Rect(0)),
        m_MessagesTable("<", MessageViewPaddingX, MessageViewPaddingY, glv::Rect(0)),
        m_ButtonsTable("", MessageViewPaddingX, MessageViewPaddingY, glv::Rect(0)),
        m_ImageTable("x", MessageViewPaddingX, MessageViewPaddingY, glv::Rect(0)),
        m_ViewTable("<,x,x", 0, 0, glv::Rect(0)),
        m_IsClosable(isClosable)
{
    Initialize();
}

glv::Style MessageView::m_StyleWhite = glv::Style();
glv::Style MessageView::m_StyleGreen = glv::Style();
glv::Style MessageView::m_StyleRed = glv::Style();

void MessageView::Initialize() NN_NOEXCEPT
{
    enable(glv::Property::DrawBorder | glv::Property::DrawBack);
    disable(glv::Property::FocusHighlight);
    m_Actions.reserve(InitialActionCapacity);

    *this << (m_ViewTable << m_MessagesTable << m_ImageTable << m_ButtonsTable);

}

MessageView::~MessageView() NN_NOEXCEPT
{
    m_Actions.clear();
}

void MessageView::Remove() NN_NOEXCEPT
{
    // 現状は m_ImageTable だけ remove すればいいが、子孫を全て remove する実装にしておく
    glv::View* pView = child;

    while ( nullptr != pView )
    {
        while ( nullptr != pView->child )
        {
            pView = pView->child;
        }

        glv::View* pRemovedView = pView;

        // 次に削除する要素に移動する
        if ( nullptr != pView->sibling )
        {
            pView = pView->sibling;
        }
        else if ( this != pView->parent )
        {
            pView = pView->parent;
        }
        else
        {
            pView = nullptr;
        }
        pRemovedView->remove();
    }
}

MessageView& MessageView::AddMessage(const std::string& message) NN_NOEXCEPT
{
    AddMessageT<std::string>(message);
    return *this;
}

// TORIAEZU: 名前は暫定, 不要なら後で削除する
glv::Label* MessageView::AddMessageReturnLabel(const std::string& message) NN_NOEXCEPT
{
    return AddMessageT<std::string>(message);
}

MessageView& MessageView::AddMessage(const glv::WideString& message) NN_NOEXCEPT
{
    AddMessageT<glv::WideString>(message);
    return *this;
}

template<typename StringType>
glv::Label* MessageView::AddMessageT(const StringType& message) NN_NOEXCEPT
{
    // ラベル作成
    auto pLabel = new glv::Label(message);
    pLabel->size(CommonValue::InitialFontSize);

    // レイアウト調整
    m_MessagesTable << pLabel;
    m_MessagesTable.arrange();
    m_ViewTable.arrange();
    extent(m_ViewTable.width(), m_ViewTable.height());

    return pLabel;
}

MessageView& MessageView::AddButton( const std::string& caption, ActionFunc pFunc, void* pParam, ButtonTextColor textColor ) NN_NOEXCEPT
{
    AddButtonT<std::string>( caption, pFunc, pParam, textColor );
    return *this;
}

MessageView& MessageView::AddButton( const glv::WideString& caption, ActionFunc pFunc, void* pParam, ButtonTextColor textColor ) NN_NOEXCEPT
{
    AddButtonT<glv::WideString>( caption, pFunc, pParam, textColor );
    return *this;
}

template<typename StringType>
glv::Button* MessageView::AddButtonT( const StringType& caption, ActionFunc pFunc, void* pParam, ButtonTextColor textColor ) NN_NOEXCEPT
{
    // ラベル作成
    auto pLabel = new glv::Label( caption );

    switch ( textColor )
    {
    case ButtonTextColor::White:
        m_StyleWhite.color.set( glv::StyleColor::BlackOnWhite );
        m_StyleWhite.color.text.set( 0.8f, 0.8f, 0.8f );
        pLabel->style( &m_StyleWhite );
        break;
    case ButtonTextColor::Green:
        m_StyleGreen.color.set( glv::StyleColor::BlackOnWhite );
        m_StyleGreen.color.text.set( 0.0f, 0.6f, 0.0f );
        pLabel->style( &m_StyleGreen );
        break;
    case ButtonTextColor::Red:
        m_StyleRed.color.set( glv::StyleColor::BlackOnWhite );
        m_StyleRed.color.text.set( 0.6f, 0.0f, 0.0f );
        pLabel->style( &m_StyleRed );
        break;
    default:
        break;
    }

    pLabel->size(CommonValue::InitialFontSize).pos(glv::Place::CC).anchor(glv::Place::CC);

    // ボタン作成
    glv::Rect rect(pLabel->width() + MessageButtonPaddingX * 2, pLabel->height() + MessageButtonPaddingY * 2);
    auto pButton = new glv::Button(rect, true);
    *pButton << pLabel;
    pButton ->attach(ReceiveUpdateNotification, glv::Update::Clicked, this);

    // 配置方法指定
    const std::string align = m_ButtonsTable.arrangement() + "^";
    m_ButtonsTable.arrangement(align.c_str());

    // レイアウト調整
    m_ButtonsTable << pButton;
    m_ButtonsTable.arrange();
    m_ViewTable.arrange();
    extent(m_ViewTable.width(), m_ViewTable.height());

    // 実行情報を登録
    Action action;
    action.pButton = pButton;
    action.pFunc = pFunc;
    action.pParam = pParam;
    m_Actions.push_back(action);

    return pButton;
}

Icon* MessageView::AddImage( void* imageData, uint32_t imageDataSize, uint16_t imageWidth, uint16_t imageHeight ) NN_NOEXCEPT
{
    const size_t imagePadding = 8;
    auto pImageIcon = new Icon( imageWidth, imageHeight, imagePadding, imagePadding );
    if ( imageData != nullptr && imageDataSize > 0 )
    {
        pImageIcon->SetRawTexture( reinterpret_cast< unsigned char* >( imageData ), imageDataSize );
    }
    else
    {
        pImageIcon->ClearTexture();
    }
    // 配置方法指定
    const std::string align = m_ImageTable.arrangement() + "^";
    m_ImageTable.arrangement( align.c_str() );

    // レイアウト調整
    m_ImageTable << pImageIcon;
    m_ImageTable.arrange();
    m_ViewTable.arrange();
    extent( m_ViewTable.width(), m_ViewTable.height() );

    return pImageIcon;
}


glv::View* MessageView::focusableView() NN_NOEXCEPT
{
    if( m_Actions.size() > 0 )
    {
        auto it = m_Actions.end();
        return (--it)->pButton;
    }
    return nullptr;
}

void MessageView::OnLoopAfterSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT
{
    bool isExit = IsExpired();
    if( IsTimeoutValid() != false )
    {
        isExit = IsExpired();
    }
    else if( m_IsClosable != false )
    {
        {
            const auto& dpad = events.GetDebugPad();
            isExit = dpad.IsButtonDown<glv::DebugPadEventType::Button::B>();
        }
        if( isExit == false )
        {
            const auto& bpad = events.GetBasicPad(0);
            isExit = bpad.IsButtonDown<glv::BasicPadEventType::Button::B>();
        }
    }
    if( isExit != false )
    {
        ExitModal();
    }
}

void MessageView::ReceiveUpdateNotification(const glv::Notification& notification) NN_NOEXCEPT
{
    auto pView = notification.receiver<MessageView>();
    const void* pSender = notification.sender();

    auto& actions = pView->m_Actions;
    for ( auto it = actions.begin(); it != actions.end(); ++it )
    {
        if( pSender == reinterpret_cast<void*>(it->pButton) )
        {
            nn::TimeSpan timespan = ZeroSpan;
            ActionFunc pFunc = it->pFunc;
            if( pFunc != nullptr )
            {
                pFunc(it->pParam, timespan);
            }
            if( timespan > ZeroSpan )
            {
                pView->ExtendTimeout(timespan);
                pView->disable(glv::Visible);
            }
            else
            {
                pView->ExitModal();
            }
            break;
        }
    }
}


/**
 * @brief       プログレスバー用のメッセージビューの定義です。
 */
ProgressModalView::ProgressModalView( bool isClosedAutomatically ) NN_NOEXCEPT
    : MessageView( false )
    , m_pMessageLabel( nullptr )
    , m_pCloseButton( nullptr )
{
    // MessageView で構築されたビューツリーをクリア
    Remove();

    // Table の登録
    m_ViewTable.arrangement( "<, x" );
    *this << ( m_ViewTable << m_MessagesTable << m_ButtonsTable );

    // Message の登録
    m_pMessageLabel = new glv::Label( "", glv::Label::Spec( glv::Place::TL, 0.f, 0.f, CommonValue::InitialFontSize ) );
    AddLabel( m_pMessageLabel );
    m_pMessageLabel->disable( glv::Visible );

    // Button の登録
    const glv::space_t messageButtonPaddingX = 32.f;
    const glv::space_t messageButtonPaddingY = 16.f;

    if ( !isClosedAutomatically )
    {
        auto pButtonLabel = new glv::Label( "Close" );
        pButtonLabel->size( CommonValue::InitialFontSize ).pos( glv::Place::CC ).anchor( glv::Place::CC );
        const glv::Rect rect( pButtonLabel->width() + messageButtonPaddingX * 2, pButtonLabel->height() + messageButtonPaddingY * 2 );
        m_pCloseButton = new glv::Button( rect, true );
        *m_pCloseButton << pButtonLabel;
        AddButton( m_pCloseButton );
        m_pCloseButton->disable( glv::Visible );
    }
}

ProgressModalView::~ProgressModalView() NN_NOEXCEPT
{
}

void ProgressModalView::onResize( glv::space_t dx, glv::space_t dy ) NN_NOEXCEPT
{
    // 画面サイズが変更されても、位置の再調整が行われない為、変更前後の差分から手動でセンターへ調整する。
    glv::space_t addX = dx / 2;
    glv::space_t addY = dy / 2;
    posAdd( -addX, -addY );
}

ProgressModalView& ProgressModalView::AddLabel( glv::Label* pLabel ) NN_NOEXCEPT
{
    // レイアウト調整
    m_MessagesTable << pLabel;
    m_MessagesTable.arrange();
    m_ViewTable.arrange();

    return *this;
}

ProgressModalView& ProgressModalView::AddButton( glv::Button* pButton ) NN_NOEXCEPT
{
    pButton->attach( ReceiveUpdateNotification, glv::Update::Clicked, this );

    // 配置方法指定
    const std::string align = m_ButtonsTable.arrangement() + "x";
    m_ButtonsTable.arrangement( align.c_str() );

    // レイアウト調整
    m_ButtonsTable << pButton;
    m_ButtonsTable.arrange();
    m_ViewTable.arrange();
    extent( m_ViewTable.width(), m_ViewTable.height() );

    // 実行情報を登録
    Action action;
    action.pButton = pButton;
    action.pFunc = nullptr;
    action.pParam = nullptr;
    m_Actions.push_back( action );

    return *this;
}

void ProgressModalView::SetMessage( const std::string& message ) NN_NOEXCEPT
{
    m_pMessageLabel->setValue( message );
}

void ProgressModalView::DisplayMessage() NN_NOEXCEPT
{
    m_pMessageLabel->enable( glv::Visible );
}

void ProgressModalView::ArrangeViewTable() NN_NOEXCEPT
{
    m_MessagesTable.arrange();
    m_ButtonsTable.arrange();
    m_ViewTable.arrange();

    if ( m_ButtonsTable.enabled( glv::Property::Visible ) )
    {
        extent( m_ViewTable.width(), m_ViewTable.height() );
    }
    else
    {
        extent( m_ViewTable.width(), m_MessagesTable.height() );
    }
}

void ProgressModalView::DisplayCloseButton( RootSurfaceContext* pRootSurface ) NN_NOEXCEPT
{
    if ( m_pCloseButton )
    {
        m_pCloseButton->enable( glv::Visible );
        pRootSurface->setFocus( m_pCloseButton );
    }
}

void ProgressModalView::HideCloseButton( RootSurfaceContext* pRootSurface ) NN_NOEXCEPT
{
    if ( m_pCloseButton )
    {
        m_pCloseButton->disable( glv::Visible );
    }
}


/**
 * @brief       モーダルビューの親ビューの定義です。
 */
InvisibleWall::InvisibleWall() NN_NOEXCEPT
    :   glv::View(glv::Rect(0), glv::Place::TL),
        m_pRootContext(nullptr),
        m_pModalView(nullptr),
        m_IsDisposable(false),
        m_IsAutomaticExitAtBackgroundEnabled(false),
        m_IsFloatingApplicationLaunchBlocked(false),
        m_KeepsVisibleAfterExit(false)
{
    disable(glv::Visible | glv::FocusHighlight | glv::DrawBorder);
    enable(glv::Controllable);
    stretch(1, 1);
    SetOpaqueStyle();
}

InvisibleWall::~InvisibleWall() NN_NOEXCEPT
{
    EndModal();
}

void InvisibleWall::OnAttached(RootSurfaceContext* pContext) NN_NOEXCEPT
{
    m_pRootContext = pContext;
}

void InvisibleWall::OnDetached(RootSurfaceContext* pContext) NN_NOEXCEPT
{
    NN_UNUSED( pContext );

    EndModal();
    m_pRootContext = nullptr;
}

void InvisibleWall::SetTransparentStyle() NN_NOEXCEPT
{
    m_WallColorType = WallColorType_Transparent;
    m_WallStyle.color.back.set(0, 0, 0, 0.0f);
    style(&m_WallStyle);
}

void InvisibleWall::SetOpaqueStyle() NN_NOEXCEPT
{
    m_WallColorType = WallColorType_Opaque;
    m_WallStyle.color.back.set(0, 0, 0, 0.9f);
    style(&m_WallStyle);
}

void InvisibleWall::DisplayModal(ModalView* pModalView) NN_NOEXCEPT
{
    NN_ASSERT( nullptr != pModalView );

    bringToFront();

    // 登録
    pModalView->pos(glv::Place::CC).anchor(glv::Place::CC);
    *this << pModalView;

    // 可視化
    enable(glv::Visible);
    pModalView->StartModal();

    // フォーカス
    auto pFocusView = pModalView->focusableView();
    if( pFocusView != nullptr )
    {
        m_pRootContext->setFocus(pFocusView);
    }
    m_pModalView = pModalView;
}

void InvisibleWall::StartModal(
    ModalView* pModalView,
    bool isDisposable,
    bool isDelayExecution,
    bool isAutomaticExitAtBackgroundEnabled,
    bool isFloatingApplicationLaunchBlocked ) NN_NOEXCEPT
{
    if( isDelayExecution == false )
    {
        NN_ASSERT( m_pModalView == nullptr );
    }

    if( pModalView != nullptr )
    {
        m_IsDisposable = isDisposable;
        m_IsDelayExecution = isDelayExecution;
        m_IsAutomaticExitAtBackgroundEnabled = isAutomaticExitAtBackgroundEnabled;
        m_IsFloatingApplicationLaunchBlocked = isFloatingApplicationLaunchBlocked;

        if (isDelayExecution == false)
        {
            DisplayModal(pModalView);
        }
        else
        {
            m_pNextModalView = pModalView;
        }
    }
}

void InvisibleWall::EndModal() NN_NOEXCEPT
{
    EndModal( m_pModalView );
}

void InvisibleWall::EndModal( bool keepsVisibleAfterExit ) NN_NOEXCEPT
{
    m_KeepsVisibleAfterExit = keepsVisibleAfterExit;
    EndModal( m_pModalView );
}

void InvisibleWall::EndModal( ModalView* pModalView, bool isForced ) NN_NOEXCEPT
{
    if ( pModalView  )
    {
        pModalView->EndModal();

        if ( ( !m_IsDelayExecution && !m_KeepsVisibleAfterExit ) || isForced )
        {
            // 不可視化
            disable(glv::Visible);

            // フォーカス調整
            m_pRootContext->MoveFocusToPageView();

            // ページ外にフォーカスがあった場合にフォーカスが維持されるようにする
            m_pRootContext->MoveFocusToPreviousFocusedView();
        }

        // 破棄
        pModalView->remove();
        if ( m_IsDisposable && pModalView->dynamicAlloc() )
        {
            delete pModalView;
        }
        m_pModalView = nullptr;
        m_IsDisposable = false;
        m_IsAutomaticExitAtBackgroundEnabled = false;
        m_IsFloatingApplicationLaunchBlocked = false;
        m_KeepsVisibleAfterExit = false;
    }
}

void InvisibleWall::ForceGainFocus(ModalView* pModalView) NN_NOEXCEPT
{
    // 元々何故か pModalView を null 判定にしか使用していなかったので、m_pModalView のnull判定条件追加。
    // 多分引数いらない？ もしくは、引数を利用するのが正しい？
    if( pModalView != nullptr && nullptr != ( pModalView = m_pModalView ) )
    {
        // フォーカス
        auto pFocusView = pModalView->focusableView();
        if( pFocusView != nullptr )
        {
            m_pRootContext->setFocus(pFocusView);
        }
    }
}

void InvisibleWall::OnLoopBeforeSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT
{
    auto pModalView = m_pModalView;
    if( nullptr != pModalView )
    {
        pModalView->OnLoopBeforeSceneRenderer(context, events);
    }

    // EndModal()→StartModal() の繋ぎのため、InvisibleWall が Visible なときは必ず前面に表示する
    if ( enabled( glv::Property::Visible ) )
    {
        bringToFront();
    }
}

void InvisibleWall::OnLoopAfterSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT
{
    auto pModalView = m_pModalView;
    if( pModalView != nullptr )
    {
        pModalView->OnLoopAfterSceneRenderer(context, events);
        if( pModalView->KeepsRunning() == false )
        {
            EndModal( pModalView );
            if ( m_IsDelayExecution  )
            {
                ModalView* pNextView;
                m_IsDelayExecution = false;
                if ( nullptr != ( pNextView = m_pNextModalView ) )
                {
                    m_pNextModalView = nullptr;
                    DisplayModal( pNextView );
                }
            }
        }
    }
}

bool InvisibleWall::onEvent(glv::Event::t events, glv::GLV& context) NN_NOEXCEPT
{
    // フォーカス移動の操作
    int flags = glv::toMoveFlags(events, context);
    if( (flags & glv::MoveArrowMask) != 0 )
    {
        View* prev = context.focusedView();
        View* next = nullptr;
        if( (glv::isVisible(prev) == false) || (glv::isFocusable(prev) == false) )
        {
            prev = nullptr;
        }

        // 現在フォーカス表示されていない
        if( prev == nullptr )
        {
            auto pModalView = m_pModalView;
            if( pModalView != nullptr )
            {
                next = pModalView->focusableView();
            }
            if( next != nullptr )
            {
                context.setFocus(next);
            }
        }
    }
    // InvisibleWall で処理された際は親にイベントを移譲しない
    return false;
}

bool InvisibleWall::IsModalViewRunning() NN_NOEXCEPT
{
    return nullptr != m_pModalView;
}

bool InvisibleWall::CanLaunchFloatingApplication() NN_NOEXCEPT
{
    return false == m_IsFloatingApplicationLaunchBlocked;
}

void InvisibleWall::OnChangeIntoBackground() NN_NOEXCEPT
{
    if ( m_IsAutomaticExitAtBackgroundEnabled && nullptr != m_pModalView )
    {
        EndModal( m_pModalView, true );
    }
}

void InvisibleWall::OnChangeIntoForeground() NN_NOEXCEPT
{
}

} // ~namespace devmenu
