﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <vector>

#include <nn/nn_Abort.h>
#include <nn/fs.h>
#include <nn/time.h>
#include <nn/mii.h>
#include <nn/mii/detail/mii_DatabaseFile.h>
#include <nn/result/result_HandlingUtility.h>

#include <glv_ScissorBoxView.h>

#include "Common/DevMenu_CommonDropDown.h"
#include "Common/DevMenu_CommonSettingsApi.h"
#include "DevMenu_DebugSettingsConfig.h"
#include "DevMenu_MiiSetting.h"

namespace devmenu { namespace debugsettings {

namespace {

    const char* FolderPath = "sd:/mii/";
    const char* SdDriveLetter = "sd";
    const char* SettingName = "mii";
    const char* KeyName = "is_db_test_mode_enabled";
    const char* KeyImgName = "is_img_db_test_mode_enabled";

    void ConvertWideStringToString( char* pOutStr, const glv::WideCharacterType* fileName ) NN_NOEXCEPT
    {
        int fileListLength = 0;
        const uint16_t* pFileList = reinterpret_cast< const uint16_t* >( fileName );
        nn::util::GetLengthOfConvertedStringUtf16NativeToUtf8( &fileListLength, pFileList, MiiImportFilePropertyType::MaxDirectoryNameLength );
        nn::util::ConvertStringUtf16NativeToUtf8( pOutStr, fileListLength, pFileList );
    }

    bool IsMiiDatabaseTestEnabled() NN_NOEXCEPT
    {
        bool isEnabled = true;
        GetFixedSizeFirmwareDebugSettingsItemValue( &isEnabled, SettingName, KeyName );
        return isEnabled;
    }

    bool IsMiiImgDatabaseTestEnabled() NN_NOEXCEPT
    {
        bool isEnabled = true;
        GetFixedSizeFirmwareDebugSettingsItemValue( &isEnabled, SettingName, KeyImgName );
        return isEnabled;
    }

    #if !defined ( NN_BUILD_CONFIG_OS_WIN )
    static nn::Result CreateDirectory(const char* pFolderPath) NN_NOEXCEPT
    {
        NN_RESULT_TRY(nn::fs::CreateDirectory(pFolderPath))
            NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists){}
        NN_RESULT_END_TRY
        NN_RESULT_SUCCESS;
    }
    #endif

    static nn::Result OutputFile(const char* pPath,const void* pSrc,size_t size) NN_NOEXCEPT
    {
        /// 既にファイルがあれば削除して再生成
        NN_RESULT_TRY(nn::fs::DeleteFile(pPath))
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound){}
        NN_RESULT_END_TRY
        NN_RESULT_DO(nn::fs::CreateFile( pPath, 0 ));

        nn::fs::FileHandle handle;
        NN_RESULT_DO(nn::fs::OpenFile(&handle, pPath, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));
        NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(handle); };

        NN_RESULT_DO(nn::fs::WriteFile(handle, 0, pSrc, size,nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));

        NN_RESULT_SUCCESS;
    }
    template<typename T>
    static nn::Result OutputFile(const char* pPath,const T& obj) NN_NOEXCEPT
    {
        return OutputFile(pPath,&obj,sizeof(obj));
    }

    nn::Result GetCurrentDateString(char* pOutString) NN_NOEXCEPT
    {
        /// パス生成用に現在の日時を取得
        nn::time::PosixTime posixTime;
        NN_RESULT_DO(nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime));
        nn::time::CalendarTime calenderTime;
        NN_RESULT_DO(nn::time::ToCalendarTime(&calenderTime, nullptr, posixTime));

        /// 日時を使って出力パスを決める
        sprintf(pOutString,"%04d%02d%02d%02d%02d%02d"
            ,calenderTime.year
            ,calenderTime.month
            ,calenderTime.day
            ,calenderTime.hour
            ,calenderTime.minute
            ,calenderTime.second
            );
        NN_RESULT_SUCCESS;
    }

    const char* GetSuccessMessage(MiiSetting::ExecuteType type) NN_NOEXCEPT
    {
        switch(type)
        {
        case MiiSetting::ExecuteType_ImportMiiDatabase:
            return "Import Completed";
        case MiiSetting::ExecuteType_ExportMiiDatabase:
            return "Export Completed";
        case MiiSetting::ExecuteType_FormatMiiDatabase:
            return "Format Completed";
        case MiiSetting::ExecuteType_ExportCharInfo:
            return "Export Completed";
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
    const char* GetResultMessage(const nn::Result& result) NN_NOEXCEPT
    {
        /// SDアクセスエラー
        if(nn::fs::ResultSdCardAccessFailed::Includes(result))
        {
            return "Failed to mount the SD card.";
        }
        /// パスが見つからない
        if(nn::fs::ResultPathNotFound::Includes(result))
        {
            return "Failed to open the Mii directory.";
        }
        /// ディレクトリが見つからない
        if(nn::fs::ResultTargetLocked::Includes(result))
        {
            return "Target Locked";
        }
        /// Miiが1件もいない
        if(nn::mii::ResultNotFound::Includes(result))
        {
            return "The Mii database is empty.";
        }
        DEVMENU_LOG("Result %u,%u\n",result.GetModule(),result.GetDescription());
        return "Unknown Error";
    }

    class LogFile {
    public:
        NN_IMPLICIT LogFile(size_t bufferSize = 100 * 1024)
        : m_Buffer(new char[bufferSize])
        , m_BufferSize(bufferSize)
        , m_Index(0)
        {
        }
        ~LogFile()
        {
        }

        void Append(const char* pMessage) NN_NOEXCEPT
        {
            auto length = strlen(pMessage);
            NN_ABORT_UNLESS_LESS(m_Index + length, m_BufferSize);
            strncpy(&m_Buffer[m_Index],pMessage, m_BufferSize - m_Index);
            m_Index += length;
        }

        void AppendFormat(const char* pFormat,...) NN_NOEXCEPT
        {
            auto* pBuffer = &m_Buffer[m_Index];

            std::va_list list;
            va_start(list, pFormat);
            m_Index += nn::util::VSNPrintf(pBuffer, m_BufferSize - m_Index, pFormat, list);
            va_end(list);
        }

        /// リセットする
        void Reset()
        {
            m_Index = 0;
        }

        /// ファイルに書き込む
        nn::Result Write(const char* pPath) NN_NOEXCEPT
        {
            return OutputFile(pPath,m_Buffer.get(),m_Index);
        }
    private:
        std::unique_ptr<char[]> m_Buffer;
        size_t m_BufferSize;
        size_t m_Index;
    };

    void GetSettingsValue( void* pOutValue ) NN_NOEXCEPT
    {
        auto* pIsEnabled = reinterpret_cast< bool* >( pOutValue );
        *pIsEnabled = IsMiiDatabaseTestEnabled();
    }

    void GetImgSettingsValue( void* pOutValue ) NN_NOEXCEPT
    {
        auto* pIsEnabled = reinterpret_cast< bool* >( pOutValue );
        *pIsEnabled = IsMiiImgDatabaseTestEnabled();
    }

    typedef std::pair< const bool, const char* > valuePair;

    void RegisterDropDownItems( void* pOutValues, devmenu::DropDownBase* pDropDown ) NN_NOEXCEPT
    {
        const valuePair valueList[] = {
            valuePair( false , "Disabled" ),
            valuePair( true ,  "Enabled" ),
        };

        auto pVectorValues = reinterpret_cast< std::vector< valuePair >* >( pOutValues );

        for ( const auto& iter: valueList )
        {
            pVectorValues->push_back( iter );
            pDropDown->addItem( iter.second );
        }
    }

    void SetSettingsValue( const bool selected, bool current ) NN_NOEXCEPT
    {
        SetFixedSizeFirmwareDebugSettingsItemValue( SettingName, KeyName, selected );
    }

    void SetImgSettingsValue( const bool selected, bool current ) NN_NOEXCEPT
    {
        SetFixedSizeFirmwareDebugSettingsItemValue( SettingName, KeyImgName, selected );
    }

} // end of unnamed namespace

const glv::space_t MiiImportFileListView::HorizontalMargin_Name = 4.f;
const glv::space_t MiiImportFileListView::HorizontalLength_Name = 768.f;

MiiImportFileListView::MiiImportFileListView( const glv::Rect& parentClipRegion ) NN_NOEXCEPT
        : CustomVerticalListView( parentClipRegion )
{
    SetTouchAndGo( true );
    glv::Style* pStyle = new glv::Style();
    pStyle->color = glv::Style::standard().color;
    pStyle->color.selection.set( 0.1f, 0.85f, 0.2f );
    style( pStyle );
    font().size( 20.f );
}

void MiiImportFileListView::OnQueryBounds( const ItemType& item, glv::space_t& outWidth, glv::space_t& outHeight ) NN_NOEXCEPT
{
    this->font().getBounds( outWidth, outHeight, item.name );
    outWidth = this->width();
}

void MiiImportFileListView::OnDrawItem( const ItemType& item, const IndexType index, const glv::Rect& contentRegion ) NN_NOEXCEPT
{
    NN_UNUSED( index );
    glv::Font& font = this->font();

    glv::space_t outWidth, outHeight;

    font.getBounds( outWidth, outHeight, item.size );
    glv::space_t posName = contentRegion.left() + HorizontalMargin_Name;
    font.render( item.name, posName, contentRegion.top() );

    font.getBounds( outWidth, outHeight, item.size );
    const glv::space_t sizeExpect = contentRegion.right() - ( outWidth + ( paddingX() * 2 ) + 4.f );
    const glv::space_t sizeLimit = posName + HorizontalLength_Name + 12.f;
    font.render( item.size, ( sizeExpect < sizeLimit ) ? sizeLimit : sizeExpect, contentRegion.top() );
}

void MiiImportFilePropertyType::Prepare( const nn::fs::DirectoryEntry& dirEntry, const char* pFilePath ) NN_NOEXCEPT
{
    BuildUtf16< MiiImportFilePropertyType::MaxDirectoryNameLength >( name, "%s%s", pFilePath, dirEntry.name );
    auto delimitedSizeStr = devmenu::GetDelimitedNumberString( dirEntry.fileSize );
    BuildUtf16< 32 >( size, "%s", delimitedSizeStr.c_str() );
}

void MiiImportFileScene::Open(devmenu::Page* pParentPage) NN_NOEXCEPT
{
#if !defined ( NN_BUILD_CONFIG_OS_WIN )
    /// SDカードをマウント
    auto result = nn::fs::MountSdCardForDebug( SdDriveLetter );
    if(result.IsFailure())
    {
        auto pView = new MessageView( false );

        // 表示メッセージ追加
        pView->AddMessage( "Failed to mount the SD card." );
        pView->AddButton( "Close" );

        pParentPage->GetRootSurfaceContext()->StartModal(pView, true);
        return;
    }

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount( SdDriveLetter );
    };

    /// リストを生成
    result = MakeImportFileList();
    if(result.IsFailure())
    {
        auto pView = new MessageView( false );

        // 表示メッセージ追加
        pView->AddMessage( "Failed to open the Mii directory." );
        pView->AddButton( "Close" );

        pParentPage->GetRootSurfaceContext()->StartModal(pView, true);
        return;
    }
    ///
    if(GetListCount() == 0)
    {
        auto pView = new MessageView();

        // 表示メッセージ追加
        pView->AddMessage( "The Mii directory is empty." );
        pView->AddButton( "Close" );

        pParentPage->GetRootSurfaceContext()->StartModal(pView, true);
        return;
    }

    this->m_OpenCallback();
#endif
}

void MiiImportFileScene::Close() NN_NOEXCEPT
{
    this->m_CloseCallback();
}

glv::View* MiiImportFileScene::GetFirstFocusTargetView() NN_NOEXCEPT
{
    return &m_BackButton;
}

MiiImportFileScene::MiiImportFileScene(
    const glv::Rect& rect,
    std::function<void()> openCallback,
    std::function<void()> closeCallback,
    std::function<void(const char* pPath)> importCallback) NN_NOEXCEPT
    : glv::Group(rect)
    , m_BackButton("Back"
    , [&] {
        this->Close();
    }
    , glv::Rect(this->width() / 2, 35.0f)
    , glv::Place::TL)
    , m_OpenCallback(openCallback)
    , m_CloseCallback(closeCallback)
    , m_ImportCallback(importCallback)
    , m_pImportFileView(nullptr)
{
    auto baseTable = new glv::Table("<", 3, 10);

    /// シザーボックス作成
    const glv::space_t headerRegion = 48.0f * 2.0f;
    const glv::space_t listMarginL = 16.0f;
    const glv::space_t listMarginR = 16.0f;
    const glv::space_t listMarginY = 2.0f;

    auto pListContainer = new glv::ScissorBoxView( 0, headerRegion, width(), height() - ( headerRegion ) );
    {
        glv::Rect clipRegion( listMarginL, listMarginY, pListContainer->width() - ( listMarginL + listMarginR ), pListContainer->height() - ( listMarginY * 2 ) );

        auto pImportFileView = new MiiImportFileListView(clipRegion);

        /// 要素選択時のデリゲータを作成
        pImportFileView->attach([](const glv::Notification& notification)->void {
            auto pScene = notification.receiver< MiiImportFileScene >();
            const MiiImportFilePropertyType* pProperty = notification.data< MiiImportFileListView::ItemType >();
            if ( nullptr != pScene && nullptr != pProperty )
            {
                pScene->ImportMiiDatabase( *pProperty );
            }
        },glv::Update::Action,this);

        *pListContainer << pImportFileView;


        m_pImportFileView = pImportFileView;
    }

    // フッタ
    auto pFooterExplaination = new glv::Label( "A: Import the selected file", glv::Label::Spec( glv::Place::BR, -( listMarginL + 16.0f ), -8.0f, CommonValue::InitialFontSize ) );

    *baseTable << m_BackButton << pListContainer << pFooterExplaination;
    baseTable->arrange().fit(false);

    *this << baseTable;

    this->disable(glv::Property::Visible); // 非表示が初期状態
}

nn::Result MiiImportFileScene::MakeImportFileList() NN_NOEXCEPT
{
    /// リストをクリア
    m_FileList.clear();

    /// 指定フォルダを開く
    nn::fs::DirectoryHandle directoryHandle;
    const char* pDirectoryPath = FolderPath;
    NN_RESULT_DO(nn::fs::OpenDirectory( &directoryHandle, pDirectoryPath, nn::fs::OpenDirectoryMode_File ));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseDirectory( directoryHandle );
    };

    /// .datを列挙
    nn::fs::DirectoryEntry entry;
    int64_t entryCount = 0;
    while(NN_STATIC_CONDITION(true))
    {
        NN_RESULT_DO( nn::fs::ReadDirectory( &entryCount, &entry, directoryHandle, 1 ) );

        if(0 >= entryCount)
        {
            break;
        }

        if ( entry.directoryEntryType == nn::fs::DirectoryEntryType_File )
        {
            auto length = strlen( entry.name );
            if (length > 4 && strcmp( entry.name + length - 4, ".dat" ) == 0)
            {
                MiiImportFilePropertyType fileProp;
                fileProp.Prepare( entry, pDirectoryPath );
                m_FileList.push_back( fileProp );
            }
        }
    }

    /// コレクションを生成
    auto pDatum = new MiiImportFileListView::CollectionType();
    pDatum->resize( m_FileList.size() );
    for(size_t i = 0; i < m_FileList.size();++i)
    {
        pDatum->at( i ) = m_FileList[ i ];
    }

    {
    m_pImportFileView->EntryCollection( *pDatum );
    const glv::Property::t focusableProperty = glv::Property::t::Controllable | glv::Property::t::HitTest;
    m_pImportFileView->enable( focusableProperty );
    }

    ///
    NN_RESULT_SUCCESS;
}

int MiiImportFileScene::GetListCount() const NN_NOEXCEPT
{
    return m_FileList.size();

}

void MiiImportFileScene::ImportMiiDatabase(const MiiImportFilePropertyType& property) NN_NOEXCEPT
{
    /// コールバック実行
    char filePath[ MiiImportFilePropertyType::MaxDirectoryNameLength ];
    ConvertWideStringToString( filePath, property.name );
    m_ImportCallback(filePath);
}

MiiSetting::MiiSetting (glv::space_t width,devmenu::RootSurfaceContext* pRootSurface, std::function< void() > openImportScene) NN_NOEXCEPT
    :m_pRootSurface(pRootSurface)
    ,m_OpenImportCallback(openImportScene)
    ,m_pImportDatabaseButton(nullptr)
    ,m_pExportDatabaseButton(nullptr)
    ,m_pFormatDatabaseButton(nullptr)
    ,m_pExportCharInfoButton(nullptr)
{
    // Testing Mode
    {
        auto pLabel = new glv::Label( "Mii Testing Mode", ConstantValue::DefaultLabelSpec );
        auto pDropDown = new DropDownComparableKeyValue< bool >(
            ConstantValue::DefaultDropDownRect, CommonValue::InitialFontSize,
            [&]( void* outValues, devmenu::DropDownBase* pDropDown ) { RegisterDropDownItems( outValues, pDropDown ); },
            [&]( void* value ) { GetSettingsValue( value ); },
            [&]( const bool& selected, bool current ) { SetSettingsValue( selected, current ); UpdateButtonFocusAndColor( IsMiiDatabaseTestEnabled() ); }
        );
        pDropDown->enable( glv::Property::KeepWithinParent );

        auto pGroup = new Group( glv::Rect( width, pDropDown->h ) );
        *pGroup << pLabel << pDropDown;

        *this << pGroup;
    }

    const glv::space_t tableHeight = 40.0f;
    const glv::space_t ButtonRightMargin = 24.0f;

    auto pTable = new glv::Table( "xxxx", 3.0f, 3.0f, glv::Rect( width, tableHeight ) );

    // Import Mii Database
    m_pImportDatabaseButton = new devmenu::Button("Import Database"
            ,[&]{ Execute(ExecuteType_ImportMiiDatabase); }
            );

    // Export Mii Database
    m_pExportDatabaseButton = new devmenu::Button("Export Database"
        ,[&]{ Execute(ExecuteType_ExportMiiDatabase); }
        );

    // Format Mii Database
    m_pFormatDatabaseButton = new devmenu::Button("Format Database"
        ,[&]{ Execute(ExecuteType_FormatMiiDatabase); }
        );

    // Export Mii CharInfo
    m_pExportCharInfoButton = new devmenu::Button("Export CharInfo"
        ,[&]{ Execute(ExecuteType_ExportCharInfo); }
        );

    // Table に追加するだけだと位置が期待通りにならないので明示的に指定する
    m_pExportCharInfoButton->anchor( glv::Place::TR ).pos( -( m_pExportCharInfoButton->width() + ButtonRightMargin ), 0.0f );
    m_pFormatDatabaseButton->anchor( glv::Place::TR ).pos( m_pExportCharInfoButton->left() - ( m_pFormatDatabaseButton->width() + ButtonRightMargin ), 0.0f );
    m_pExportDatabaseButton->anchor( glv::Place::TR ).pos( m_pFormatDatabaseButton->left() - ( m_pExportDatabaseButton->width() + ButtonRightMargin ), 0.0f );
    m_pImportDatabaseButton->anchor( glv::Place::TR ).pos( m_pExportDatabaseButton->left() - ( m_pImportDatabaseButton->width() + ButtonRightMargin ), 0.0f );

    *this << ( *pTable << m_pImportDatabaseButton << m_pExportDatabaseButton << m_pFormatDatabaseButton << m_pExportCharInfoButton );
    UpdateButtonFocusAndColor( IsMiiDatabaseTestEnabled() );

    arrange();
    fit(false);
}

void MiiSetting::Execute(ExecuteType type) NN_NOEXCEPT
{
    /// 実行できる環境にあるかをチェック
    if(!IsMiiDatabaseTestEnabled())
    {
        auto pView = new MessageView(false);
        pView->AddMessage("Mii Testing Mode is Disabled");
        pView->AddButton("OK");
        m_pRootSurface->StartModal(pView, true);
        return;
    }

    /// フォーマットはリスクが高いので確認メッセージを表示する
    if(type == ExecuteType_FormatMiiDatabase) {
        auto pView = new MessageView();
        pView->AddMessage("Are you sure you want to format Mii Database?");

        pView->AddButton( "Cancel" );
        pView->AddButton("Execute",
                [this,type]( void* pParam, nn::TimeSpan& timespan )
                {
                    this->ExecuteImpl(type);
                    timespan = nn::TimeSpan::FromMilliSeconds(600);
                },
                nullptr,
                MessageView::ButtonTextColor::Red
            );
        m_pRootSurface->StartModal(pView, true);
        return;
    }
    ExecuteImpl(type);
}

void MiiSetting::ExecuteImpl(ExecuteType type) NN_NOEXCEPT
{
    NN_UNUSED(type);
    nn::Result result;
    switch(type)
    {
    case ExecuteType_ImportMiiDatabase:
        /// シーン起動
        m_OpenImportCallback();
        return;
    case ExecuteType_FormatMiiDatabase:
        /// フォーマットは事前確認をしているので完了通知を行わない
        FormatMiiDatabase();
        return;
    case ExecuteType_ExportMiiDatabase:
        result = ExportMiiDatabase();
        break;
    case ExecuteType_ExportCharInfo:
        result = ExportCharInfo();
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    ViewMessage(result,type);
}

nn::Result MiiSetting::ImportDatabaseImpl(const char* pFilePath) NN_NOEXCEPT
{
#if !defined ( NN_BUILD_CONFIG_OS_WIN )
    /// SDカードをマウント
    NN_RESULT_DO(nn::fs::MountSdCardForDebug( SdDriveLetter ));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount( SdDriveLetter );
    };

    /// ファイルを読み込み
    nn::fs::FileHandle file;
    NN_RESULT_DO(nn::fs::OpenFile(&file, pFilePath, nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(file); };
    std::unique_ptr<nn::mii::detail::ImportFile> importFile(new nn::mii::detail::ImportFile);
    NN_RESULT_DO(nn::fs::ReadFile(file, 0, importFile.get(), sizeof(*importFile)));

    /// 読み込んだファイルをデータベースにインポート
    nn::mii::Database database;
    NN_UTIL_SCOPE_EXIT{ database.Finalize(); };
    NN_RESULT_DO(database.Initialize());
    NN_RESULT_DO(database.Import(importFile.get(),sizeof(*importFile)));
    NN_RESULT_SUCCESS;
#endif
    NN_RESULT_SUCCESS;
}
nn::Result MiiSetting::ExportMiiDatabase() NN_NOEXCEPT
{
#if !defined ( NN_BUILD_CONFIG_OS_WIN )
    /// SDカードをマウント
    NN_RESULT_DO(nn::fs::MountSdCardForDebug( SdDriveLetter ));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount( SdDriveLetter );
    };

    /// mii ディレクトリを作成
    NN_RESULT_DO(CreateDirectory(FolderPath));

    /// パス生成用に現在の日時を取得
    char dateString[MiiImportFilePropertyType::MaxDirectoryNameLength];
    NN_RESULT_DO(GetCurrentDateString(dateString));

    /// 日時を使って出力パスを決める
    char path[MiiImportFilePropertyType::MaxDirectoryNameLength];
    sprintf(path,"%s%s.dat"
        ,FolderPath
        ,dateString
        );

    /// データベースエクスポート
    std::unique_ptr<nn::mii::detail::ImportFile> importFile(new nn::mii::detail::ImportFile);
    nn::mii::Database database;
    NN_RESULT_DO(database.Initialize());
    NN_UTIL_SCOPE_EXIT{ database.Finalize(); };
    database.Export(importFile.get(),sizeof(*importFile));

    /// ファイルとして出力
    NN_RESULT_DO(OutputFile(path,*importFile));
#endif
    NN_RESULT_SUCCESS;
}

nn::Result MiiSetting::FormatMiiDatabase() NN_NOEXCEPT
{
    nn::mii::Database database;
    NN_RESULT_DO(database.Initialize());
    NN_UTIL_SCOPE_EXIT{ database.Finalize(); };

    database.Format();

    NN_RESULT_SUCCESS;
}

nn::Result MiiSetting::ExportCharInfo() NN_NOEXCEPT
{
#if !defined ( NN_BUILD_CONFIG_OS_WIN )
    /// SDカードをマウント
    NN_RESULT_DO(nn::fs::MountSdCardForDebug( SdDriveLetter ));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount( SdDriveLetter );
    };

    /// mii ディレクトリを作成
    NN_RESULT_DO(CreateDirectory(FolderPath));

    /// パス生成用に現在の日時を取得
    char dateString[MiiImportFilePropertyType::MaxDirectoryNameLength];
    NN_RESULT_DO(GetCurrentDateString(dateString));

    const char* pFolderPath = FolderPath;

    /// 日時を使って出力パスを決める
    char path[MiiImportFilePropertyType::MaxDirectoryNameLength];
    sprintf(path,"%s%s",pFolderPath,dateString);

    /// CharInfoを取り出す
    int miiCount;
    std::unique_ptr<nn::mii::CharInfo[]> charInfoArray(new nn::mii::CharInfo[nn::mii::DatabaseMiiCount]);
    nn::mii::Database database;
    NN_UTIL_SCOPE_EXIT{ database.Finalize(); };
    NN_RESULT_DO(database.Initialize());
    NN_RESULT_DO(database.Get(&miiCount,charInfoArray.get(),nn::mii::DatabaseMiiCount,nn::mii::SourceFlag_Database));

    /// Miiが1体もいない
    if(miiCount <= 0)
    {
        NN_RESULT_THROW(nn::mii::ResultNotFound());
    }

    /// 出力先ディレクトリを作成
    NN_RESULT_DO(CreateDirectory(path));

    LogFile exportLog;
    for(int i = 0; i < miiCount;++i)
    {
        nn::mii::CharInfoAccessor accessor(charInfoArray[i]);

        /// ニックネームを取得
        nn::mii::Nickname miiNameUtf16;
        accessor.GetNickname(&miiNameUtf16, nn::mii::FontRegionFlag_All);

        /// UTF-8に変換
        char miiNameUtf8[MiiImportFilePropertyType::MaxDirectoryNameLength];
        auto convResult = nn::util::ConvertStringUtf16NativeToUtf8(miiNameUtf8, sizeof(miiNameUtf8), miiNameUtf16.name);
        NN_ABORT_UNLESS(convResult == nn::util::CharacterEncodingResult_Success);

        sprintf(path,"%s%s/%02d.dat",pFolderPath,dateString,i);

        /// ファイルを出力
        NN_RESULT_DO(OutputFile(path,charInfoArray[i]));

        /// ニックネームを出力
        exportLog.AppendFormat("%02d.dat : %s\n",i,miiNameUtf8);
    }

    sprintf(path,"%s%s/nicknames.txt",pFolderPath,dateString);
    NN_RESULT_DO(exportLog.Write(path));
#endif
    NN_RESULT_SUCCESS;
}

void MiiSetting::ViewMessage(nn::Result result,ExecuteType type) NN_NOEXCEPT
{
    auto pView = new MessageView(false);

    if(result.IsSuccess())
    {
        /// 成功時のメッセージ
        pView->AddMessage(GetSuccessMessage(type));
    }
    else
    {
        /// 失敗時のメッセージ
        pView->AddMessage(GetResultMessage(result));
    }
    pView->AddButton("Close");
    m_pRootSurface->StartModal(pView, true);
    return;
}

void MiiSetting::UpdateButtonFocusAndColor( bool isValid ) NN_NOEXCEPT
{
    m_pImportDatabaseButton->UpdateFocusAndColor( isValid, true );
    m_pExportDatabaseButton->UpdateFocusAndColor( isValid, true );
    m_pFormatDatabaseButton->UpdateFocusAndColor( isValid, true );
    m_pExportCharInfoButton->UpdateFocusAndColor( isValid, true );
}

MiiImgSetting::MiiImgSetting  (glv::space_t width) NN_NOEXCEPT
{
    auto pLabel = new glv::Label( "Mii Image Testing Mode", ConstantValue::DefaultLabelSpec );
    auto pDropDown = new DropDownComparableKeyValue< bool >(
        ConstantValue::DefaultDropDownRect, CommonValue::InitialFontSize,
        [&]( void* outValues, devmenu::DropDownBase* pDropDown ) { RegisterDropDownItems( outValues, pDropDown ); },
        [&]( void* value ) { GetImgSettingsValue( value ); },
        [&]( const bool& selected, bool current ) { SetImgSettingsValue( selected, current ); }
    );
    pDropDown->enable( glv::Property::KeepWithinParent );

    auto pGroup = new Group( glv::Rect( width, pDropDown->h ) );
    *pGroup << pLabel << pDropDown;

    *this << pGroup;

    this->arrange().fit(false);
}

}} // ~namespace devmenu::debugsettings, ~namespace devmenu

