﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <glv_CustomVerticalListView.h>
#include <glv_ScissorBoxView.h>
#include <nn/fs/fs_SaveDataManagement.h>

#include <nn/fs/fs_FileSystemPrivate.h>

#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_Optional.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <algorithm>
#include <memory>
#include <functional>
#include <sstream>
#include <iomanip>
#include <new>

#include <nn/fs/fs_Bis.h>

#include "DevMenu_Config.h"
#include "DevMenu_RootSurface.h"

//#define DEBUG_DATA

namespace devmenu { namespace system { namespace filesystem {

//----------------------------------------------------------

namespace
{
    const int64_t MaxDirectoryNameLength = 300;

    template< size_t TemporarySize, size_t OutBufferSize  >
    glv::WideCharacterType* BuildUtf16(glv::WideCharacterType(&pOutBuffer)[OutBufferSize], const char* pFormat, ...) NN_NOEXCEPT
    {
        va_list vaList;
        va_start(vaList, pFormat);

        char pTemp[TemporarySize];
        std::vsprintf(pTemp, pFormat, vaList);
        nn::util::ConvertStringUtf8ToUtf16Native(reinterpret_cast<uint16_t*>(pOutBuffer), OutBufferSize, pTemp);

        va_end(vaList);
        return pOutBuffer;
    }

}

//----------------------------------------------------------
/**
* @brief アプリケーションプロパティ型です。
*/
struct FilePropertyType
{
    //----------------------------------------------------------
    /**
    * @brief アプリケーションリスト用プロパティデータを構築します。
    */
    void Prepare(const nn::fs::DirectoryEntry& dirEntry, const char* pFilePath) NN_NOEXCEPT
    {
#if defined( NN_BUILD_CONFIG_ADDRESS_SUPPORTS_64 )
        const char* pLogMessageFormat = "%s%s   %12ld\n";
#else
        const char* pLogMessageFormat = "%s%s   %12lld\n";
#endif
        sprintf(logMessage, pLogMessageFormat, pFilePath, dirEntry.name, dirEntry.fileSize);
        BuildUtf16<300>(name, "%s%s", pFilePath, dirEntry.name);
        BuildUtf16<32>(size, "%ld",    dirEntry.fileSize);
    }

    //----------------------------------------------------------

    char   logMessage[332];

    glv::WideCharacterType name[300];
    glv::WideCharacterType size[32];
};

//----------------------------------------------------------
/**
* @brief リストビュー
*/
class PropertyListView : public glv::CustomVerticalListView< FilePropertyType >
{
public:

    //----------------------------------------------------------

    typedef glv::CustomVerticalListView< FilePropertyType > ParentType;

    static const glv::space_t HorizontalMargin_Name;  //!< リスト要素中の[プログラムID]表示左マージン( ピクセル単位 )
    static const glv::space_t HorizontalLength_Name;  //!< リスト要素中の[プログラムID]表示横幅( ピクセル単位 )
    static const glv::space_t HorizontalMargin_Size;  //!< リスト要素中の[プログラムVersion]表示左マージン( ピクセル単位 )
    static const glv::space_t HorizontalLength_Size;  //!< リスト要素中の[プログラムVersion]表示横幅( ピクセル単位 )

    //----------------------------------------------------------
    /**
    * @brief コンストラクタです。
    */
    explicit PropertyListView(const glv::Rect& parentClipRegion) NN_NOEXCEPT
        : CustomVerticalListView(parentClipRegion)
    {
        SetTouchAndGo(false);
        glv::Style* pStyle = new glv::Style();
        pStyle->color = glv::Style::standard().color;
        pStyle->color.selection.set(0.1f, 0.85f, 0.2f);
        style(pStyle);
        font().size(15.f);

        // 親クラスで OKボタン( A )に対してUpdate::Actionイベントが発生するので Hack.
        changePadClickDetectableButtons(glv::BasicPadEventType::Button::X::Mask);
        changePadClickDetectableButtons(glv::DebugPadEventType::Button::X::Mask);
    }

protected:
    //----------------------------------------------------------
    /**
    * @copydoc CustomVerticalListView<>::OnQueryBounds( const CustomVerticalListView<>::ItemType&, glv::space_t&, glv::space_t& )
    */
    virtual void OnQueryBounds(const ItemType& item, glv::space_t& outWidth, glv::space_t& outHeight) NN_NOEXCEPT NN_OVERRIDE
    {
        this->font().getBounds(outWidth, outHeight, item.name);
        outWidth = this->width();
    }

    //----------------------------------------------------------
    /**
    * @copydoc CustomVerticalListView<>::OnDrawItem( const CustomVerticalListView<>::ItemType&, const CustomVerticalListView<>::IndexType, const glv::Rect& )
    */
    virtual void OnDrawItem(const ItemType& item, const IndexType index, const glv::Rect& contentRegion) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(index);
        glv::Font& font = this->font();

        glv::space_t outWidth, outHeight;

        font.getBounds(outWidth, outHeight, item.size);
        glv::space_t posName = contentRegion.left() + HorizontalMargin_Name;
        font.render(item.name, posName, contentRegion.top());

        font.getBounds(outWidth, outHeight, item.size);
        const glv::space_t nameExpect = contentRegion.right() - (outWidth + (paddingX() * 2) + 4.f);
        const glv::space_t nameLimit = posName + HorizontalLength_Name + HorizontalMargin_Size + HorizontalLength_Size + 12.f;
        font.render(item.size, (nameExpect < nameLimit) ? nameLimit : nameExpect, contentRegion.top());
    }
};

//----------------------------------------------------------

const glv::space_t PropertyListView::HorizontalMargin_Name = 4.f;
const glv::space_t PropertyListView::HorizontalMargin_Size = 4.f;
const glv::space_t PropertyListView::HorizontalLength_Name = 832.f;
const glv::space_t PropertyListView::HorizontalLength_Size = 32.f;

//----------------------------------------------------------

/**
 * @brief ファイルシステムのページです。
 */
class FileSystemPage : public devmenu::Page
{
public:
    //----------------------------------------------------------
    FileSystemPage(int pageId, const glv::WideCharacterType* pageCaption, glv::Rect rect) NN_NOEXCEPT
        : devmenu::Page(pageId, pageCaption, rect), m_pItems(nullptr), m_pListView(nullptr), m_pLabelNoItem(nullptr), m_pLabelSpaceSize(nullptr)
    {
    }

    //----------------------------------------------------------
    virtual void OnAttachedPage() NN_NOEXCEPT NN_OVERRIDE
    {
        // シザーボックスの領域設定
        static const glv::space_t HeaderRegion = 48.f;
        static const glv::space_t FooterRegion = 96.f;
        static const glv::space_t FooterRegion2 = 24.f;
        glv::ScissorBoxView* pListContainer = new glv::ScissorBoxView(0, HeaderRegion, width(), height() - (HeaderRegion + FooterRegion + FooterRegion2));

        // リスト要素のパディング ( ボーダーを表示する場合は 2以上必要 )
        static const glv::space_t ListMarginL = 16;
        static const glv::space_t ListMarginR = 16;
        static const glv::space_t ListMarginY = 2;

        // ヘッダ
        m_pLabelName = new glv::Label("Name", glv::Label::Spec(glv::Place::TL, ListMarginL + 8, 8, CommonValue::InitialFontSize));
        *this << m_pLabelName;
        *this << new glv::Label(GLV_TEXT_API_WIDE_STRING("Size"), glv::Label::Spec(glv::Place::TR, -(ListMarginL + 16), 8, CommonValue::InitialFontSize));

        // リストビュー
        glv::Rect clipRegion(ListMarginL, ListMarginY, pListContainer->width() - (ListMarginL + ListMarginR), pListContainer->height() - (ListMarginY * 2));
        PropertyListView* pListView = new PropertyListView(clipRegion);
        *pListContainer << pListView;
        m_pListView = pListView;

        // アイテムなしメッセージ
        glv::Label* pLabelNoItem = new glv::Label("No files are listed.", glv::Label::Spec(glv::Place::CC, 0, 0, 32.f));
        *pListContainer << pLabelNoItem;
        m_pLabelNoItem = pLabelNoItem;

        *this << pListContainer;

        // Console out
        {
            auto button = new glv::Button(glv::Rect(ListMarginL, height() - (FooterRegion + FooterRegion2), width() - ListMarginL - ListMarginR, 48), true);
            button->attach([](const glv::Notification& n)->void { n.receiver< FileSystemPage >()->OnOutConsole(); }, glv::Update::Clicked, this);
            auto outConsoleLabel = new glv::Label(GLV_TEXT_API_WIDE_STRING("Dump list to console"), glv::Label::Spec(glv::Place::CC, 0, 0, CommonValue::InitialFontSize));
            *button << outConsoleLabel;
            *this << button;
        }

        // ストレージ空きサイズの表示
        {
            m_pLabelSpaceSize = new glv::Label("", glv::Label::Spec(glv::Place::BL, ListMarginL, -FooterRegion2, 20.f));
            *this << m_pLabelSpaceSize;
        }

        this->attach( this->FocusMenuTabOnBbuttonPress, glv::Update::Clicked, this );

    }

    //----------------------------------------------------------
    virtual void OnDetachedPage() NN_NOEXCEPT NN_OVERRIDE
    {
        FinalizeProperties();
    }

    //----------------------------------------------------------
    virtual void OnActivatePage() NN_NOEXCEPT NN_OVERRIDE
    {
        QueryProperties();
    }

    //----------------------------------------------------------
    virtual void OnDeactivatePage() NN_NOEXCEPT NN_OVERRIDE
    {
        m_pLabelNoItem->disable(glv::Property::t::Visible);
        FinalizeProperties();
    }

    //----------------------------------------------------------
    virtual void OnLoopBeforeSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(context);
        NN_UNUSED(events);
    }

    //----------------------------------------------------------
    virtual void OnLoopAfterSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(context);
        NN_UNUSED(events);
    }

    //----------------------------------------------------------
    virtual glv::View* GetFocusableChild() NN_NOEXCEPT NN_OVERRIDE
    {
        PropertyListView::CollectionType* pItems;
        return (nullptr == (pItems = m_pItems) || pItems->empty())
            ? static_cast<View*>(m_pLabelNoItem)
            : static_cast<View*>(m_pListView);
    }

private:

    //----------------------------------------------------------
    void UpdateBuildInSystemSpaceSizeLabel() NN_NOEXCEPT
    {
        //
        int64_t outFreeSize = 0;
        int64_t outTotalSize = 0;
        char spaceSizeStr[256];

        // この関数は、nn::fs::MountBis("system") でマウント中に呼び出す必要があります。
#if !defined( NN_BUILD_CONFIG_OS_WIN )
        // Win では、nn::fs::ResultNotImplemented で失敗するので呼び出しを省略しています。
        nn::Result res1 = nn::fs::GetFreeSpaceSize(&outFreeSize, "system:/");
        NN_ABORT_UNLESS_RESULT_SUCCESS(res1);
        NN_UNUSED(res1);
        nn::Result res2 = nn::fs::GetTotalSpaceSize(&outTotalSize, "system:/");
        NN_ABORT_UNLESS_RESULT_SUCCESS(res2);
        NN_UNUSED(res2);
#endif

        float freeKib = outFreeSize / 1024.0f;
        float totalKib = outTotalSize / 1024.0f;
        sprintf(spaceSizeStr, "BuiltInSystem SpaceSize : %8.1f / %8.1f (free / total KiB)", freeKib, totalKib);

        NN_SDK_ASSERT_NOT_NULL(m_pLabelSpaceSize);
        const char* pStr = &spaceSizeStr[0];
        m_pLabelSpaceSize->setValue(pStr);
    }

    //----------------------------------------------------------
    void OnOutConsole() NN_NOEXCEPT
    {
        NN_LOG("DevMenuSystem - FileSystemPage ----------------------------------------- \n", m_FileList.size());
        NN_LOG("Files count=%zu \n", m_FileList.size());
        for (size_t i = 0; i < m_FileList.size(); i++)
        {
            NN_LOG(m_FileList[i].logMessage);
        }
    }

    //----------------------------------------------------------
    nn::Result EnumerateFiles(std::vector< FilePropertyType >& entries, const char* directoryPath)
    {
        nn::fs::DirectoryHandle directoryHandle;
        NN_RESULT_DO(nn::fs::OpenDirectory(&directoryHandle, directoryPath, nn::fs::OpenDirectoryMode_All));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseDirectory(directoryHandle);
        };

        int64_t entryCount = 1;
        nn::fs::DirectoryEntry entry;

        while (entryCount)
        {
            NN_RESULT_DO(nn::fs::ReadDirectory(&entryCount, &entry, directoryHandle, 1));

            if (entry.directoryEntryType == nn::fs::DirectoryEntryType_File)
            {
                FilePropertyType fileProp;
                fileProp.Prepare(entry, directoryPath);
                entries.push_back(fileProp);
            }

            else if (entry.directoryEntryType == nn::fs::DirectoryEntryType_Directory)
            {
                char childDirectoryPath[MaxDirectoryNameLength];
                sprintf(childDirectoryPath, "%s%s/", directoryPath, entry.name);

                // 子ディレクトリに対して再帰的に関数を呼び出す
                EnumerateFiles(entries, childDirectoryPath);
            }
        }
        NN_RESULT_SUCCESS;
    }

    //----------------------------------------------------------
    nn::Result QueryProperties() NN_NOEXCEPT
    {
        FinalizeProperties();
        PropertyListView::CollectionType* pItems;
        m_pItems = pItems = new PropertyListView::CollectionType();

        m_FileList.clear();

        // system
        {
            NN_RESULT_DO(nn::fs::MountBis("system", nn::fs::BisPartitionId::System));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::Unmount("system");
            };

            char sysDirectory[MaxDirectoryNameLength];
            sprintf(sysDirectory, "system%s", ":/");

            EnumerateFiles(m_FileList, sysDirectory);

            UpdateBuildInSystemSpaceSizeLabel();
        }

        // user
        {
            NN_RESULT_DO(nn::fs::MountBis("user", nn::fs::BisPartitionId::User));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::Unmount("user");
            };

            char usrDirectory[MaxDirectoryNameLength];
            sprintf(usrDirectory, "user%s", ":/");

            EnumerateFiles(m_FileList, usrDirectory);
        }

        // リストに反映
        {
            pItems->resize(m_FileList.size());
            for (size_t i = 0; i < m_FileList.size(); i++)
            {
                pItems->at(i) = m_FileList[i];
            }

            EntryProperties(pItems);
        }

        NN_RESULT_SUCCESS;
    }

    //----------------------------------------------------------
    void EntryProperties(const PropertyListView::CollectionType* pItems) NN_NOEXCEPT
    {
        const glv::Property::t focusableProperty = glv::Property::t::Controllable | glv::Property::t::HitTest;
        PropertyListView* pListView;
        if (nullptr != (pListView = m_pListView))
        {
            pListView->EntryCollection(*pItems);
            if (true == pItems->empty())
            {
                pListView->disable(focusableProperty);
                m_pLabelNoItem->enable(glv::Property::t::Visible);
            }
            else
            {
                pListView->enable(focusableProperty);
                m_pLabelNoItem->disable(glv::Property::t::Visible);
            }
        }
    }

    //----------------------------------------------------------
    void FinalizeProperties() NN_NOEXCEPT
    {
        PropertyListView::CollectionType* pItems;
        if (nullptr != (pItems = m_pItems))
        {
            m_pItems = nullptr;
            delete pItems;
        }
    }

    //----------------------------------------------------------

    std::vector< FilePropertyType >     m_FileList;

    PropertyListView::CollectionType*   m_pItems;
    PropertyListView*                   m_pListView;
    glv::Label*                         m_pLabelNoItem;
    glv::Label*                         m_pLabelName;
    glv::Label*                         m_pLabelSpaceSize;
};

//----------------------------------------------------------

/**
 * @brief ページ生成 ( 専用クリエイター )
 */
template< size_t ID >
class FileSystemPageCreator : devmenu::PageCreatorBase
{
public:
    /**
     * @brief コンストラクタです。
     */
    explicit FileSystemPageCreator( const char* pPageName ) NN_NOEXCEPT
        : devmenu::PageCreatorBase( ID, pPageName ) {}

protected:

    /**
     * @brief ページインスタンスを生成します。
     */
    virtual glv::PageBase* newInstance() NN_NOEXCEPT NN_OVERRIDE
    {
        int resolution[ 2 ];
        const auto& display = glv::ApplicationFrameworkGetRuntimeContext().GetDisplay();
        display.GetResolution( resolution[ 0 ], resolution[ 1 ] );
        const auto width = static_cast< glv::space_t >( resolution[ 0 ] );
        const auto height = static_cast< glv::space_t >( resolution[ 1 ] );
        const glv::Rect pageBounds( width - 218.f, height - 118.0f );
        return new FileSystemPage( ID, GLV_TEXT_API_WIDE_STRING( "FileSystem" ), pageBounds );
    }
};

//----------------------------------------------------------

/**
 * @brief Declearation for the statical instance of page creator.
 */
#define LOCAL_PAGE_CREATOR( _id, _name ) FileSystemPageCreator< _id > g_FileSystemPageCreator##_id( _name );
LOCAL_PAGE_CREATOR(DevMenuPageId_FileSystem, "FileSystemPage" );

}}} // ~namespace devmenu::system::filesystem, ~namespace devmenu::system, ~namespace devmenu
