﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_ContentMetaUtil.h>
#include <nn/ns/ns_ApplicationVersionSystemApi.h>
#include <nn/ns/ns_Result.h>

#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_InstallUtil.h"
#include "DevMenuCommand_HashUtil.h"
#include "DevMenu_ApplicationCatalog.h"
#include "DevMenu_DeleteContents.h"
#include "Applications/DevMenu_ApplicationsCommon.h"
#include "Launcher/DevMenu_Launcher.h"

#if defined( NN_DEVMENULOTCHECK )
    #include "QrCodeEncoder/DevMenu_CreateQrCode.h"
#endif

namespace devmenu { namespace application {

namespace
{

char* GetContentMetaTypeStringInternal(const nn::ncm::ContentMetaKey& key, const nn::ncm::ApplicationId& applicationId) NN_NOEXCEPT
{
    static char s_Buffer[128];
    if (key.type != nn::ncm::ContentMetaType::AddOnContent)
    {
        nn::util::SNPrintf(s_Buffer, sizeof(s_Buffer), "%s", devmenuUtil::GetContentMetaTypeString(key.type));
    }
    else
    {
        auto index = key.id - (applicationId.value + 0x1000);
        nn::util::SNPrintf(s_Buffer, sizeof(s_Buffer), "%s (%llu)", devmenuUtil::GetContentMetaTypeString(key.type), index);
    }
    return s_Buffer;
}

} // ~namespace devmenu::applicaiton::<anonymous>

DeleteContentsScene::CloseButton::CloseButton( DeleteContentsScene* pParent, const glv::Rect& rect, glv::Place::t anchor ) NN_NOEXCEPT
    : Button( "< Back", [&] { m_pParent->OnBackButtonPressed(); }, rect, anchor )
    , m_pParent( pParent )
{
}

void DeleteContentsScene::CloseButton::CloseEventCallback() NN_NOEXCEPT
{
    // アプリケーション本体が存在すれば GetApplicationControlData() に成功するはずなので詳細画面に戻す
    if ( m_pParent->m_ExistsApplication )
    {
        m_pParent->CloseScene( m_pParent->m_pDetailScene );
    }
    else
    {
        m_pParent->CloseScene( m_pParent->m_pCatalogScene );
    }
}

DeleteContentsScene::ResetRequiredVersionButton::ResetRequiredVersionButton( DeleteContentsScene* pParent, const glv::Rect& rect, glv::Place::t anchor ) NN_NOEXCEPT
    : devmenu::Button( "Reset RequiredVersion", [&] { ResetRequiredVersionEventCallback(); }, rect, anchor ),
      m_pParent( pParent )
{
}

void DeleteContentsScene::ResetRequiredVersionButton::ResetRequiredVersionEventCallback() NN_NOEXCEPT
{
    m_pParent->ResetRequiredVersion();
}

DeleteContentsScene::DeleteButton::DeleteButton( DeleteContentsScene* pParent, const glv::Rect& rect, glv::Place::t anchor ) NN_NOEXCEPT
    : devmenu::Button( "A: Delete", [&] { DeleteEventCallback(); }, rect, anchor ),
      m_pParent( pParent )
{
}

void DeleteContentsScene::DeleteButton::DeleteEventCallback() NN_NOEXCEPT
{
    m_pParent->DeleteContent();
}

DeleteContentsScene::DeleteAllButton::DeleteAllButton( DeleteContentsScene* pParent, const glv::Rect& rect, glv::Place::t anchor ) NN_NOEXCEPT
    : devmenu::Button( "-/+: Delete All", [&] { DeleteAllEventCallback(); }, rect, anchor),
      m_pParent( pParent )
{
}

void DeleteContentsScene::DeleteAllButton::DeleteAllEventCallback() NN_NOEXCEPT
{
    m_pParent->DeleteAllContents();
}

DeleteContentsScene::ShowContentsHashButton::ShowContentsHashButton( DeleteContentsScene* pParent, const glv::Rect& rect, glv::Place::t anchor ) NN_NOEXCEPT
    : devmenu::Button( "Y: Show Hash", [&] { ShowContentsHashEventCallback(); }, rect, anchor),
      m_pParent( pParent )
{
}

void DeleteContentsScene::ShowContentsHashButton::ShowContentsHashEventCallback() NN_NOEXCEPT
{
    m_pParent->ShowContentsHash();
}

DeleteContentsScene::DescriptionLabel::DescriptionLabel() NN_NOEXCEPT
: m_Label("<blank>", glv::Label::Spec(glv::Place::TL, 0.0f, 50.0f, CommonValue::InitialFontSize))
{
}

void DeleteContentsScene::DescriptionLabel::SetApplicationId( const nn::ncm::ApplicationId& applicationId ) NN_NOEXCEPT
{
    char buffer[128];
    nn::util::SNPrintf( buffer, sizeof(buffer), "Application ID: 0x%016llx", applicationId.value );

    std::string str(buffer);
    m_Label.setValue(str);
}

DeleteContentsScene::ContentsTileView::ContentsTileView( glv::space_t width, glv::space_t height ) NN_NOEXCEPT
    : glv::View( glv::Rect( width, height ) )
    , m_ContentInformationTable( "< < < < >", 10.0f, 10.0f, glv::Rect( width, height ) )
    , m_pContentTarget( new glv::Label( "N/A", glv::Label::Spec( glv::Place::CL, 0, 0, Default::FontSize ) ) )
    , m_pContentVersion( new glv::Label( "N/A", glv::Label::Spec( glv::Place::CL, 0, 0, Default::FontSize ) ) )
    , m_pContentStorageId( new glv::Label( "N/A", glv::Label::Spec( glv::Place::CR, 0, 0, Default::FontSize ) ) )
{
    auto pSeparator = new glv::Label( "|", glv::Label::Spec( glv::Place::CL, 0, 0, Default::FontSize ) );
    auto pVersionLabel = new glv::Label( "Version", glv::Label::Spec( glv::Place::CL, 0, 0, Default::FontSize ) );

    m_ContentInformationTable << m_pContentTarget << pSeparator << pVersionLabel << m_pContentVersion << m_pContentStorageId;

    m_ContentInformationTable.arrange().fit( false );
    *this << m_ContentInformationTable;

    m_ContentInformationTable.maximize();
    m_ContentInformationTable.disable( glv::Property::HitTest );
    this->disable( glv::Property::DrawBorder | glv::Property::HitTest | glv::Property::Controllable | glv::Property::DrawBack );
}

void DeleteContentsScene::ContentsTileView::UpdateContentTileViewValues(
    const nn::ncm::StorageContentMetaKey& storageContent,
    const nn::ncm::ApplicationId& applicationId ) NN_NOEXCEPT
{
    char buffer[128];
    nn::util::SNPrintf( buffer, sizeof( buffer ), "%u", ( storageContent.key.version >> 16 ) );

    m_pContentTarget->setValue(
        std::string( GetContentMetaTypeStringInternal( storageContent.key, applicationId ) ) );
    m_pContentVersion->setValue( std::string( buffer ) );
    m_pContentStorageId->setValue( GetStorageIdString( storageContent.storageId ) );

    // Reflect the style to all view elements in table
    auto updateStyle = [&]( glv::Style* pStyle )-> void
    {
        auto pView = m_ContentInformationTable.child;
        while ( nullptr != pView )
        {
            pView->style( pStyle );
            pView = pView->sibling;
        }
    };

    NN_FUNCTION_LOCAL_STATIC( nn::util::optional<glv::Style>, s_DisabledStyle );

    // Check if the Application Content can be deleted
    if ( !nn::ncm::IsInstallableStorage( storageContent.storageId ) )
    {
        if ( !s_DisabledStyle )
        {
            s_DisabledStyle = glv::Style::standard();
            s_DisabledStyle->color.text.set( 0.5f, 0.5f, 0.5f );
        }
        updateStyle( &( *s_DisabledStyle ) );
    }
    else
    {
        updateStyle( &glv::Style::standard() );
    }

    // 表示位置を期待通りにするために一度 remove して再度 add する
    m_ContentInformationTable.ReAddOwnselfToParent( this );
}

const char* DeleteContentsScene::ContentsTileView::GetStorageIdString( nn::ncm::StorageId storageId ) NN_NOEXCEPT
{
    switch ( storageId )
    {
    case nn::ncm::StorageId::GameCard: return "Game Card";
    case nn::ncm::StorageId::SdCard: return "microSD Card";
    case nn::ncm::StorageId::BuiltInUser: return "System Memory";
    default: return "Unknown";
    }
}

void DeleteContentsScene::ContentsTileView::onDraw( glv::GLV& glvContext ) NN_NOEXCEPT
{
    // Call the parent view's Draw
    glv::View::onDraw( glvContext );
}

DeleteContentsScene::ContentTileContainer::ContentTileContainer(
        ContentsTileView* pContentTile,
        glv::space_t width, glv::space_t height,
        const nn::ncm::StorageContentMetaKey& content,
        const nn::ncm::ApplicationId& applicationId,
        DeleteContentsScene* pParent ) NN_NOEXCEPT
    : glv::View( glv::Rect( width, height ) )
    , m_StorageContent( content )
    , m_ApplicationId( applicationId )
    , m_pContentTile( pContentTile )
    , m_pParent( pParent )
{
    this->enable( glv::Property::DrawBorder | glv::Property::FocusHighlight | glv::Property::HitTest | glv::Property::Controllable );
}

DeleteContentsScene::ContentTileContainer::~ContentTileContainer() NN_NOEXCEPT
{
    // Detach the Content tile so that it doesn't get destroyed on Scrollable container refresh
    m_pContentTile->remove();
    m_pContentTile = nullptr;
}

void DeleteContentsScene::ContentTileContainer::onDraw( glv::GLV& glvContext ) NN_NOEXCEPT
{
    // Check if the the given tile index is already attached to this container
    if ( m_pContentTile->parent != this )
    {
        // If it is not, update the tile contents by passing this container's item information
        m_pContentTile->UpdateContentTileViewValues( m_StorageContent, m_ApplicationId );

        // Attach the ApplicationTile to this container
        *this << m_pContentTile;
    }

    // Call the parent view's Draw
    glv::View::onDraw( glvContext );
}

bool DeleteContentsScene::ContentTileContainer::onEvent( glv::Event::t event, glv::GLV& glvRoot ) NN_NOEXCEPT
{
    NN_UNUSED( glvRoot );

    switch ( event )
    {
    case glv::Event::FocusGained:
        // Update the selected content
        m_pParent->m_pSelectedContent = &m_StorageContent;
        break;
    default:
        break;
    }

    return true;
}

DeleteContentsScene::DeleteContentsScene( ApplicationCatalogPage* pParentPage, Scene* pDetailScene, Scene* pCatalogScene ) NN_NOEXCEPT
    : Scene( pParentPage, pParentPage->rect() )
    , m_pParentPage( pParentPage )
    , m_pDetailScene( pDetailScene )
    , m_pCatalogScene( pCatalogScene)
    , m_ExistsApplication( false )
{
    auto pHeader = new glv::Table( "xxxxx", 10.f, 10.f );
    {
        m_pCloseButton = new CloseButton( this, glv::Rect( 128.0f, 40.0f ), glv::Place::TL );
        m_pResetRequiredVersionButton = new ResetRequiredVersionButton( this, glv::Rect( 300.0f, 40.f ), glv::Place::TL );
        m_pDeleteButton = new DeleteButton( this, glv::Rect( 156.0f, 40.f ), glv::Place::TL );
        m_pDeleteAllButton = new DeleteAllButton( this, glv::Rect( 216.0f, 40.0f ), glv::Place::TL );
        m_pShowContentsHashButton = new ShowContentsHashButton( this, glv::Rect( 200.0f, 40.f ), glv::Place::TL );
        *pHeader << m_pCloseButton << m_pResetRequiredVersionButton << m_pDeleteButton << m_pDeleteAllButton << m_pShowContentsHashButton;

        pHeader->arrange().fit( false );

        *this << pHeader;
    }

    auto pDescription = new DescriptionLabel();
    {
        // Update the header styles
        glv::Style* pHeaderStyle = new glv::Style();
        pHeaderStyle->color.text.set( 0.6f, 0.6f, 0.6f );
        pDescription->SetStyle( pHeaderStyle );
        m_pDescriptionLabel = pDescription;
    }

    // Adding a scrollable container
    m_pContainer = new glv::View( glv::Rect( 0.0f, pHeader->height() + 20.0f, width(), height() - ( pHeader->height() + 20.0f ) ) );
    m_pContainer->disable( glv::Property::DrawBack | glv::Property::FocusHighlight | glv::Property::DrawBorder );

    *this << m_pContainer;
    m_pScrollContainer = new ScrollableBoxView( "<", glv::Rect( 0.0f, 0.0f, m_pContainer->width(), m_pContainer->height() ), 0.0f, 0.0f );
    const auto innerRect = m_pScrollContainer->GetInnerRect();

    m_pHeaderTable = new glv::Table( "<", 0.0f, 0.0f, glv::Rect( innerRect.width(), innerRect.height() ) );

    *m_pHeaderTable << pDescription->GetLabel();
    // Creates a Divider
    glv::Divider* pHeaderDivider = new glv::Divider( 10.0f, 2.0f );
    *m_pHeaderTable << pHeaderDivider;
    m_pHeaderTable->arrange();

    *m_pScrollContainer << m_pHeaderTable;

    // Container view
    *m_pContainer << *m_pScrollContainer;

    this->SetFirstFocusTargetView( m_pCloseButton );

    for ( int i = 0; i < MaxCountOfVisibleContentsTiles; i++ )
    {
        m_ContentsTilesList.push_back( new ContentsTileView( innerRect.width(), 50.0f ) );
    }
}

void DeleteContentsScene::SetApplicationId( const nn::ncm::ApplicationId& applicationId ) NN_NOEXCEPT
{
    m_ApplicationId = applicationId;
    m_pDescriptionLabel->SetApplicationId( applicationId );
    LoadContents();
}


void DeleteContentsScene::OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT
{
    NN_UNUSED( context );
    NN_UNUSED( events );
}
void DeleteContentsScene::OnButtonClickNotification( const glv::Notification& notification ) NN_NOEXCEPT
{
    const auto pSelf = notification.receiver< DeleteContentsScene >();

    auto pSender = notification.sender< Scene >();
    auto& glvRoot = reinterpret_cast< glv::GLV& >( pSender->root() );

    // Check if the A button was pressed
    if ( glvRoot.getBasicPadEvent().IsButtonUp< glv::BasicPadEventType::Button::A >() ||
         glvRoot.getDebugPadEvent().IsButtonUp< glv::DebugPadEventType::Button::A >() )
    {
        // Call the DeletContent callback
        pSelf->DeleteContent();
    }
    else if ( glvRoot.getBasicPadEvent().IsButtonUp< glv::BasicPadEventType::Button::Start >() ||
              glvRoot.getDebugPadEvent().IsButtonUp< glv::DebugPadEventType::Button::Start >() ||
              glvRoot.getBasicPadEvent().IsButtonUp< glv::BasicPadEventType::Button::Select >() ||
              glvRoot.getDebugPadEvent().IsButtonUp< glv::DebugPadEventType::Button::Select >() )
    {
        // Check if the Plus or Minus buttons were pressed, if so call the DeleteAllContents function
        pSelf->DeleteAllContents();
    }
    else if ( glvRoot.getBasicPadEvent().IsButtonUp< glv::BasicPadEventType::Button::Y >() ||
              glvRoot.getDebugPadEvent().IsButtonUp< glv::DebugPadEventType::Button::Y >() )
    {
        // Check if the Y button was pressed, if so call the ShowContentsHash function
        pSelf->ShowContentsHash();
    }
}

void DeleteContentsScene::LoadContents() NN_NOEXCEPT
{
    m_ContentMetaKey.clear();
    m_ExistsApplication = false;

    // Refresh selected content
    m_pSelectedContent = nullptr;

    // Clear callbacks
    m_Callback = nullptr;

    // Refresh scrollable list
    m_pHeaderTable->remove();
    m_pHeaderTable->parent = nullptr;
    m_pScrollContainer->Refresh();

    // Scroll to the top of the container
    m_pScrollContainer->ScrollToTop();
    *m_pScrollContainer << m_pHeaderTable;

    auto f = [&] ( nn::ncm::StorageId storageId ) -> void
    {
        nn::ncm::ContentMetaDatabase db;
        auto result = nn::ncm::OpenContentMetaDatabase( &db, storageId );
        if ( result.IsFailure() )
        {
            return;
        }
        const int maxKeys = 512;
        std::unique_ptr<nn::ncm::ContentMetaKey[]> keys( new nn::ncm::ContentMetaKey[maxKeys] );
        auto listcount = db.ListContentMeta( keys.get(), maxKeys, nn::ncm::ContentMetaType::Unknown, m_ApplicationId );

        // Get inner rect
        const auto innerRect = m_pScrollContainer->GetInnerRect();

        for ( int i = 0; i < listcount.listed; ++i )
        {
            nn::ncm::StorageContentMetaKey skey = { keys[i], storageId };
            m_ContentMetaKey.push_back( skey );

            if ( nn::ncm::ContentMetaType::Application == keys[ i ].type )
            {
                m_ExistsApplication = true;
            }

            // Create Content Tile
            DeleteContentsScene::ContentTileContainer* pContentTile = new DeleteContentsScene::ContentTileContainer(
                m_ContentsTilesList.at( i % MaxCountOfVisibleContentsTiles ),
                innerRect.width(), 50.0f, skey, m_ApplicationId, this );
            *m_pScrollContainer << pContentTile;
            // Attach click listener to tile
            pContentTile->attach( OnButtonClickNotification, glv::Update::Clicked, this );
        }
    };
    f( nn::ncm::StorageId::Card );
    f( nn::ncm::StorageId::BuiltInUser );
    f( nn::ncm::StorageId::SdCard );
}

void DeleteContentsScene::CloseScene( Scene* pDstScene ) NN_NOEXCEPT
{
    m_pParentPage->SwitchScene( this, pDstScene, true );
}

void DeleteContentsScene::ResetRequiredVersion() NN_NOEXCEPT
{
    auto pView = new MessageView( true );
    {
        char buffer[128];
        nn::util::SNPrintf(buffer, sizeof(buffer), "Target application ID: 0x%016llx", m_ApplicationId);
        pView->AddMessage(buffer);
        pView->AddMessage( "Reset required-version may causes save data compatibility problem." );
        pView->AddMessage( "Are you OK to reset?" );
    }

    pView->AddButton( "Cancel" );

    pView->AddButton(
        "OK",
        [&]( void* pParam, nn::TimeSpan& timespan )
        {
            nn::ns::PushLaunchVersion(m_ApplicationId, 0);
            timespan = nn::TimeSpan::FromMilliSeconds( 100 );
        },
        nullptr,
        MessageView::ButtonTextColor::Red
    );
    this->StartModal( pView, true );
}

void DeleteContentsScene::DeleteContent() NN_NOEXCEPT
{
    if ( m_pSelectedContent != nullptr && nn::ncm::IsInstallableStorage( m_pSelectedContent->storageId ) )
    {
        auto pView = new MessageView( true );
        {
            auto content = m_pSelectedContent;
            char buffer[ 128 ];
            nn::util::SNPrintf( buffer, sizeof( buffer ), "Target: %s, Ver.%u, Storage: %s",
                GetContentMetaTypeStringInternal( content->key, m_ApplicationId ), ( content->key.version >> 16 ), devmenuUtil::GetStorageIdString( content->storageId ) );
            pView->AddMessage( buffer );
            pView->AddMessage( "Are you sure you want to delete this?" );
        }

        pView->AddButton( "Cancel" );

        pView->AddButton(
            "Delete",
            [&]( void* pParam, nn::TimeSpan& timespan )
            {
                // 実行中でなければ削除する
                if ( launcher::GetActiveApplicationId() != m_ApplicationId )
                {
                    const auto content = m_pSelectedContent;
                    const auto applicationId = m_ApplicationId;

                    auto Delete = []( const nn::ncm::StorageContentMetaKey* content, nn::ncm::ApplicationId id ) -> nn::Result
                    {
                        // DevMenuCommand_InstallUtil.h
                        NN_RESULT_DO( devmenuUtil::Delete( content->key, content->storageId ) );
                        NN_RESULT_DO( devmenuUtil::PushApplicationRecordBasedOnDatabase( id ) );
                        NN_RESULT_SUCCESS;
                    };

                    auto result = Delete( content, applicationId );
                    if ( result.IsSuccess() )
                    {
                        LoadContents();
                    }
                    else
                    {
                        auto pView = new MessageView( true );
                        if ( nn::ns::ResultGameCardIsInserted::Includes( result ) )
                        {
                            CreateErrorMessageView(pView, "Failed to delete the content. Remove the Game Card and try again.", result);
                        }
                        else
                        {
                            CreateErrorMessageView(pView, "Failed to delete the content.", result);
                        }
                        this->StartModal( pView, true, true );
                    }
                    timespan = nn::TimeSpan::FromMilliSeconds( 100 );
                }
                else
                {
                    auto pView = new MessageView( true );
                    pView->AddMessage( "Failed to delete the content. Application is running." );
                    pView->AddButton( "Close" );
                    m_pParentPage->GetRootSurfaceContext()->StartModal( pView, true, true );
                }
            },
            nullptr,
            MessageView::ButtonTextColor::Red
        );
        this->StartModal( pView, true );
    }
}

void DeleteContentsScene::DeleteAllContents() NN_NOEXCEPT
{
    auto pView = new MessageView( true );
    {
        char buffer[ 128 ];
        nn::util::SNPrintf( buffer, sizeof( buffer ), "Target Application ID: 0x%016llx", m_ApplicationId );
        pView->AddMessage( buffer );
        pView->AddMessage( "Are you sure you want to delete all?" );
    }

    pView->AddButton( "Cancel" );
    pView->AddButton(
        "Delete All",
        [&]( void* pParam, nn::TimeSpan& timespan )
        {
            timespan = nn::TimeSpan::FromMilliSeconds( 100 );

            // 実行中でなければ削除する
            if ( launcher::GetActiveApplicationId() != m_ApplicationId )
            {
                auto result = nn::ns::DeleteApplicationCompletely( m_ApplicationId );
                if ( result.IsSuccess() )
                {
                    CloseScene( m_pCatalogScene );
                }
                else
                {
                    auto pView = new MessageView( false );
                    if ( nn::ns::ResultGameCardIsInserted::Includes( result ) )
                    {
                        CreateErrorMessageView( pView, "Failed to delete the contents. Remove the Game Card and try again.", result );
                    }
                    else
                    {
                        CreateErrorMessageView( pView, "Failed to delete the contents.", result );
                    }
                    this->StartModal( pView, true, true );
                }
            }
            else
            {
                auto pView = new MessageView( false );
                pView->AddMessage( "Failed to delete the contents. Application is running." );
                pView->AddButton( "Close" );
                this->StartModal( pView, true, true );
            }
        },
        nullptr,
        MessageView::ButtonTextColor::Red
    );
    this->StartModal( pView, true );
}

void DeleteContentsScene::ShowContentsHash() NN_NOEXCEPT
{
    if ( nullptr != m_pSelectedContent )
    {
        auto pView = new MessageView( false );
        AddApplicationRomIdHash( pView );
        pView->AddButton( "Close" );
        this->StartModal( pView, true );
    }
}

nn::Result DeleteContentsScene::GetApplicationRomIdHash( nn::ncm::Hash* pOutHash ) NN_NOEXCEPT
{
    const auto pStorageKey = m_pSelectedContent;

    if ( nullptr != pStorageKey )
    {
        NN_RESULT_DO( devmenuUtil::GetApplicationRomIdHash( pOutHash, pStorageKey->key, pStorageKey->storageId ) );
    }

    NN_RESULT_SUCCESS;
}

void DeleteContentsScene::AddApplicationRomIdHash( MessageView* pView ) NN_NOEXCEPT
{
    nn::ncm::Hash hash;

    auto result = GetApplicationRomIdHash( &hash );
    if ( result.IsFailure() )
    {
        if ( nn::ncm::ResultContentMetaNotFound::Includes( result ) )
        {
            pView->AddMessage( "[Rom ID Hash]\n  Selected application is not stored in game card or has been uninstalled." );
        }
        else
        {
            char errorValueStr[ 64 ];
            nn::util::SNPrintf( errorValueStr, sizeof( errorValueStr ), "(result: 0x%08x)", result.GetInnerValueForDebug() );
            const std::string ErrorMessage = "[Rom ID Hash]\n  Failed to get hash value. " + std::string( errorValueStr );
            DEVMENU_LOG( "%s\n", ErrorMessage.c_str() );
            pView->AddMessage( ErrorMessage );
        }
        return;
    }

    const std::string titleStr( "[Rom ID Hash]\n  " );
    std::string displayHashStr, qrCodeHashStr;
    GetHashString( &displayHashStr, &qrCodeHashStr, &hash, sizeof( hash ) );
    pView->AddMessage( titleStr + displayHashStr );

#if defined( NN_DEVMENULOTCHECK )

    mw::qre::ImageInfo imageInfo = { 0, 0, nullptr, mw::qre::Rect() };
    uint32_t imageSize = 0;
    const auto hashSize = qrCodeHashStr.size();
    std::unique_ptr< uint8_t[] > hashData( new uint8_t[ hashSize ] );
    memcpy( hashData.get(), qrCodeHashStr.c_str(), hashSize );
    CreateQrCode( &imageInfo, &imageSize, hashData.get(), hashSize );
    pView->AddImage( imageInfo.rgbData, imageSize, imageInfo.width, imageInfo.height );
    delete[] imageInfo.rgbData;

#endif // defined( NN_DEVMENULOTCHECK )

}

}} // ~namespace devmenu::application, ~namespace devmenu

