﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/fs/fs_ResultPrivate.h>
#include <nn/ns/ns_Result.h>

#include "DevMenu_ApplicationCatalog.h"
#include "DevMenu_ApplicationWriter.h"
#include "DevMenuCommand_GameCardWriterImpl.h"
#include "DevMenuCommand_Result.h"

namespace devmenu { namespace application { namespace write {

const glv::space_t SdCardListView::HorizontalMargin_Name = 4.0f;
const glv::space_t SdCardListView::HorizontalLength_Name = 768.0f;

namespace {

    const size_t MaxVisibleFileNameLength   = 35;

} // end of anonymous namespace

/**************************************
 * class ConfirmView
 **************************************/

ConfirmView::ConfirmView( bool isRadioButtonRequired ) NN_NOEXCEPT
    : MessageView( false )
    , m_WriteLocationStorageId( nn::ncm::StorageId::Any )
{
    Remove();

    // Table
    m_ViewTable << m_MessagesTable;
    if ( isRadioButtonRequired )
    {
        m_ViewTable.arrangement( "<, <, x" );
    }
    else
    {
        m_ViewTable.arrangement( "<, x" );
    }

    m_ViewTable << m_ButtonsTable;
    m_ViewTable.arrange();
    *this << m_ViewTable;
}

ConfirmView::~ConfirmView() NN_NOEXCEPT
{
}

void ConfirmView::ExtentSize() NN_NOEXCEPT
{
    const auto maxWidth = m_ViewTable.width();
    const auto correctionWidth = ( maxWidth - width() ) / 2.0f;

    extent( maxWidth, m_ViewTable.height() );
    m_ButtonsTable.anchor( glv::Place::BC ).pos( m_ButtonsTable.left() + correctionWidth, m_ButtonsTable.top() );
}

/**************************************
 * class WriteView
 **************************************/

const size_t       WriteView::InitialActionCapacity = 4;
const glv::space_t WriteView::WriteViewWidth      = 800.0f;
const glv::space_t WriteView::ProgressBarMargineX   = 8.0f;
const glv::space_t WriteView::MessageButtonPaddingX = 32.0f;
const glv::space_t WriteView::MessageButtonPaddingY = 16.0f;
const glv::space_t WriteView::MessageViewPaddingX   = 32.0f;
const glv::space_t WriteView::MessageViewPaddingY   = 32.0f;

const glv::Rect        WriteView::ProgressBarRect( WriteView::WriteViewWidth - WriteView::ProgressBarMargineX - WriteView::ProgressBarMargineX, CommonValue::InitialFontSize );
const glv::Label::Spec WriteView::DefaultLabelSpec( glv::Place::TL, 0.0f, 0.0f, CommonValue::InitialFontSize );

WriteView::WriteView( const ButtonCallback& closeCallback ) NN_NOEXCEPT
    : MessageView( false ),
    m_ProgressBarsTable( "", MessageViewPaddingX, MessageViewPaddingY, glv::Rect( 0.0f ) ),
    m_ProgressBar( WriteView::ProgressBarRect, 0.0f ),
    m_ProgressCaption( "[Write Progress]", WriteView::DefaultLabelSpec ),
    m_ProgressFileName( "", WriteView::DefaultLabelSpec ),
    m_ProgressLabel( "Prepare writing...", WriteView::DefaultLabelSpec ),
    m_CloseButton( "Close", closeCallback, CommonValue::InitialFontSize, 24.0f ),
    m_WriteResult( nn::ResultSuccess() )
{
    Initialize();
}

WriteView::~WriteView() NN_NOEXCEPT
{
}

void WriteView::Initialize() NN_NOEXCEPT
{
    enable( glv::Property::DrawBorder | glv::Property::DrawBack );
    disable( glv::Property::FocusHighlight );
    m_Actions.reserve( WriteView::InitialActionCapacity );

    // MessageView で構築されたビューツリーをクリア
    Remove();

    // Table
    m_ViewTable.arrangement( "<,x,x" );
    *this << ( m_ViewTable << m_MessagesTable << m_ProgressBarsTable << m_ButtonsTable );

    // メッセージ
    AddProgressMessage( &m_ProgressCaption, &m_ProgressFileName, &m_ProgressLabel );

    // プログレスバー
    AddProgressBar( &m_ProgressBar );

    const glv::Property::t focusableProperty = glv::Property::t::Controllable | glv::Property::t::HitTest;
    m_ProgressBar.disable( focusableProperty );

    // 下部ボタン
    m_CloseButton.IncreaseLabelSize( WriteView::MessageButtonPaddingX , WriteView::MessageButtonPaddingY );

    // ModalView の表示位置調整のために非表示で追加しておく
    AddButton( &m_CloseButton );
    m_CloseButton.disable( glv::Property::Visible );
}

WriteView& WriteView::AddProgressMessage( glv::Label* pProgressCaption, glv::Label* pProgressFileName, glv::Label* pProgressLabel ) NN_NOEXCEPT
{
    // レイアウト調整
    m_MessagesTable << pProgressCaption;
    m_MessagesTable << pProgressFileName;
    m_MessagesTable << pProgressLabel;
    m_MessagesTable.arrange();
    m_ViewTable.arrange();

    return *this;
}

WriteView& WriteView::AddButton( glv::Button* pButton ) NN_NOEXCEPT
{
    return AddButton( pButton, nullptr, nullptr );
}

WriteView& WriteView::AddButton( glv::Button* pButton, ActionFunc pFunc, void* pParam ) NN_NOEXCEPT
{
    pButton->attach( ReceiveUpdateNotification, glv::Update::Clicked, this );

    // 配置方法指定
    const std::string align = m_ButtonsTable.arrangement() + "x";
    m_ButtonsTable.arrangement(align.c_str());

    // レイアウト調整
    m_ButtonsTable << pButton;
    m_ButtonsTable.arrange();
    m_ViewTable.arrange();
    extent( m_ViewTable.width(), m_ViewTable.height() );

    // 実行情報を登録
    Action action;
    action.pButton = pButton;
    action.pFunc = pFunc;
    action.pParam = pParam;
    m_Actions.push_back( action );

    return *this;
}

WriteView& WriteView::AddProgressBar( glv::Slider* pProgressBar ) NN_NOEXCEPT
{
    // 配置方法指定
    const std::string align = m_ProgressBarsTable.arrangement() + "x";
    m_ProgressBarsTable.arrangement( align.c_str() );

    // レイアウト調整
    m_ProgressBarsTable << pProgressBar;
    m_ProgressBarsTable.arrange();
    m_ViewTable.arrange();
    extent( m_ViewTable.width(), m_ViewTable.height() );

    return *this;
}

void WriteView::SetErrorMessageView() NN_NOEXCEPT
{
    SetProgressCaption( "[Write Failed]" );

    std::string errorMessage;

    if ( nn::fs::ResultPortSdCardNoDevice::Includes( m_WriteResult ) )
    {
        CreateErrorMessage( &errorMessage, "SD card is not inserted.\n", m_WriteResult, true );
    }
    else if ( nn::fs::ResultSdCardAccessFailed::Includes(m_WriteResult) )
    {
        CreateErrorMessage( &errorMessage, "Failed to access to the SD card.\n", m_WriteResult, true );
    }
    else if ( nn::ncm::ResultContentNotFound::Includes(m_WriteResult) )
    {
        CreateErrorMessage( &errorMessage, "Writable content is not found. Confirm the nsp file.", m_WriteResult, true );
    }
    else if ( nn::ns::ResultSdCardNoOwnership::Includes(m_WriteResult) || nn::ns::ResultSdCardDatabaseCorrupted::Includes(m_WriteResult) )
    {
        CreateErrorMessage( &errorMessage,
            "SD card was initialized on another device or database is corrupted.", m_WriteResult, true );
    }
    else if ( nn::ns::ResultSdCardNotMounted::Includes(m_WriteResult) )
    {
        CreateErrorMessage( &errorMessage,
            "SD card was inserted after device boot.\nDevice reboot is required to write contents to the card.", m_WriteResult, true );
    }
    else if ( devmenuUtil::ResultInvalidExternalKey::Includes(m_WriteResult) )
    {
        CreateErrorMessage( &errorMessage, "External key nsp can not be written to the game card.", m_WriteResult, true );
    }
    else
    {
        CreateErrorMessage( &errorMessage, "Failed to write the nsp file. ", m_WriteResult, false );
    }

    SetProgressLabel( errorMessage.c_str() );
    m_ProgressBarsTable.disable( glv::Property::Visible );

    RemoveButtons();

    auto& g = reinterpret_cast< glv::GLV& >( root() );

    // Close ボタン
    AddButton( &m_CloseButton );
    m_CloseButton.enable( glv::Property::Visible );
    g.setFocus( &m_CloseButton );
}

void WriteView::SetCloseButton() NN_NOEXCEPT
{
    RemoveButtons();

    AddButton( &m_CloseButton );
    m_CloseButton.enable( glv::Property::Visible );

    auto& g = reinterpret_cast< glv::GLV& >( root() );
    g.setFocus( &m_CloseButton );
}

void WriteView::RemoveButtons() NN_NOEXCEPT
{
    m_CloseButton.remove();
}

/**************************************
 * class WriteScene
 **************************************/

// シザーボックスの領域設定
const glv::space_t WriteScene::HeaderRegionLength = 80.0f;
const glv::space_t WriteScene::FooterRegionLength = 48.0f;

// リスト要素のパディング ( ボーダーを表示する場合は 2 以上必要 )
const glv::space_t WriteScene::ListMarginLeft = 16.0f;
const glv::space_t WriteScene::ListMarginRight = 16.0f;
const glv::space_t WriteScene::ListMarginVertical = 2.0f;
const glv::space_t WriteScene::NspFileLeft = 318.0f;

const glv::Label::Spec WriteScene::LabelSpecNoNsp( glv::Place::CC, 0.0f, 0.0f, 32.0f );

const std::string WriteScene::NspFileNothing = "<not selected>";
const std::string WriteScene::HeaderName = "Please select the nsp files that you'd like written to the Game Card.";

/**************************************
 * class WriteScene::Header
 **************************************/
WriteScene::Header::Header( glv::Label* pPageTitleLabel, const glv::Rect& rect ) NN_NOEXCEPT
    : glv::Group( rect )
{
    // Update the header styles
    NN_FUNCTION_LOCAL_STATIC( glv::Style, s_HeaderStyle );
    s_HeaderStyle.color.text.set( 0.6f, 0.6f, 0.6f );
    auto pPageTitle = pPageTitleLabel;
    pPageTitle->style( &s_HeaderStyle );

    auto pTitleTable = new glv::Table( "<", 0.0f, 0.0f, glv::Rect( rect.width(), rect.height() ) );
    *pTitleTable << pPageTitle;

    // Creates a Divider
    auto pHeaderDivider = new glv::Divider( 10.0f, 2.0f );
    *pTitleTable << pHeaderDivider;
    pTitleTable->arrange();
    *this << pTitleTable;
}

/**
 * @brief コンストラクタです。
 */
WriteScene::WriteScene( devmenu::application::ApplicationCatalogPage* pParentPage, Scene* pParentScene ) NN_NOEXCEPT
    : Scene( pParentPage->rect() ),
    m_ButtonToCatalog( "< Back", [&]{ this->OnBackButtonPressed(); }, CommonValue::InitialFontSize, 16.0f ),
    m_ButtonWrite( "Write File(s) to Game Card", [&]{ ConfirmSelector(); }, glv::Rect( 450.0f, 40.0f ), glv::Place::TR ),
    m_ListContainer( glv::Rect( 0.0f, WriteScene::HeaderRegionLength, width(), height() - ( WriteScene::HeaderRegionLength + WriteScene::FooterRegionLength ) ) ),
    m_ButtonToApplicationList( "Select nsp File", [&] { ListupCallBack(nn::ncm::ContentMetaType::Application); }, CommonValue::InitialFontSize, 16.0f ),
    m_ButtonToPatchList( "Select nsp File", [&] { ListupCallBack(nn::ncm::ContentMetaType::Patch); }, CommonValue::InitialFontSize, 16.0f ),
    m_pParentScene( pParentScene ),
    m_pParentPage( pParentPage ),
    m_Mutex( true ),
    m_StartTick( 0 ),
    m_pWriteView( nullptr ),
    m_WriteState( WriteState_NotStarted )
{
    // ボタンの登録
    {
        // Catalog に戻るボタン
        m_ButtonToCatalog.anchor( glv::Place::TL ).pos( glv::Place::TL, ListMarginLeft, 4.0f );
        *this << m_ButtonToCatalog;

        m_pFileListSelectorScene = new select::FileListSelectorScene( m_pParentPage, this );
        m_pFileListSelectorScene->SetBackButtonCallback([&] { m_pParentPage->SwitchScene( m_pFileListSelectorScene, this, true); } );
        m_pFileListSelectorScene->anchor( glv::Place::TL).pos(glv::Place::TL, 0.f, 0.f );
        m_pFileListSelectorScene->disable( glv::Property::Visible );
        *m_pParentPage << m_pFileListSelectorScene;
    }

    // ヘッダ
    auto pHeader = new Header( new glv::Label( "Write to Game Card", glv::Label::Spec( glv::Place::TL, ListMarginLeft, m_ButtonToCatalog.bottom() + 8.0f, CommonValue::InitialFontSize ) ),
        glv::Rect( width() - ( ListMarginLeft + ListMarginRight + 30.0f ), WriteScene::HeaderRegionLength ) );
    pHeader->anchor( glv::Place::TL ).pos( ListMarginLeft + 9.0f, 60.0f );
    m_pHeaderName = new glv::Label( HeaderName, glv::Label::Spec( glv::Place::TL, ListMarginLeft + 18.0f, pHeader->bottom(), CommonValue::InitialFontSize ) );
    {
        *this << pHeader;
        *this << m_pHeaderName;
    }

    // NSPからGameCardへの書き込み選択項目
    auto pApplicationName   = new glv::Label( "Application (required)", glv::Label::Spec(glv::Place::TL, ListMarginLeft + 18.0f, m_pHeaderName->bottom() + 60.0f, CommonValue::InitialFontSize) );
    auto pPatchName         = new glv::Label( "Patch (optional)", glv::Label::Spec(glv::Place::TL, ListMarginLeft + 18.0f, pApplicationName->bottom() + 20.0f, CommonValue::InitialFontSize) );
    m_pLabelApplication     = new glv::Label( NspFileNothing, glv::Label::Spec( glv::Place::TL, NspFileLeft, m_pHeaderName->bottom() + 60.0f, CommonValue::InitialFontSize ) );
    m_pLabelPatch           = new glv::Label( NspFileNothing, glv::Label::Spec( glv::Place::TL, NspFileLeft, pApplicationName->bottom() + 20.0f, CommonValue::InitialFontSize ) );
    nn::util::Strlcpy( m_ApplicationFileName, NspFileNothing.c_str(), sizeof(m_ApplicationFileName) );
    nn::util::Strlcpy( m_PatchFileName, NspFileNothing.c_str(), sizeof(m_PatchFileName) );
    {
        *this << pApplicationName;
        *this << pPatchName;
        *this << m_pLabelApplication;
        *this << m_pLabelPatch;
    }

    // ボタン
    m_ButtonToApplicationList.anchor( glv::Place::TR ).pos( glv::Place::TR, -( ListMarginRight + 16.0f ), pApplicationName->top() - 4.0f );
    m_ButtonToPatchList.anchor( glv::Place::TR ).pos( glv::Place::TR, -( ListMarginRight + 16.0f ), pPatchName->top() - 4.0f );
    m_ButtonWrite.anchor( glv::Place::TR ).pos( glv::Place::TR, -( ListMarginRight + 16.0f ), pPatchName->bottom() + 60.0f );
    m_ButtonWrite.UpdateFocusAndColor( false, false );
    {
        *this << m_ButtonToApplicationList;
        *this << m_ButtonToPatchList;
        *this << m_ButtonWrite;
    }

    this->SetFirstFocusTargetView( &m_ButtonToApplicationList );
}

void WriteScene::Refresh() NN_NOEXCEPT
{
    if ( m_pFileListSelectorScene->IsNspFileSelected() == true )
    {
        SetNspFile( m_pFileListSelectorScene->GetSelectedNspFilePath() );

        // ファイル選択情報初期化
        m_pFileListSelectorScene->InitializeFileSelected();
    }
}

/**
 * @brief シーンにフォーカスが与えられた際に、フォーカスを横取る子供を指定します。
 */
glv::View* WriteScene::GetFocusableChild() NN_NOEXCEPT
{
    // デフォルトフォーカスはApplicationボタン
    return &m_ButtonToApplicationList;
}

/**
 * @brief ダイアログを表示して確認を取った後、GameCard書き込みを行います。
 */
void WriteScene::ConfirmSelector( const FilePropertyType* pProperty ) NN_NOEXCEPT
{
    auto applicationFileName = m_pLabelApplication->getValue();
    auto pView = new ConfirmView( false );
    if ( !nn::fs::IsGameCardInserted() )
    {
        pView->AddMessage( "Game Card is not inserted." );
        pView->AddButton( "Close" );
    }
    else
    {
        auto patchFileName = m_pLabelPatch->getValue();

        // 確認表示時のファイル数が1つ
        if ( patchFileName == NspFileNothing )
        {
            pView->AddMessage( "Write the selected file?" );
        }
        // 確認表示時のファイル数が複数
        else
        {
            pView->AddMessage( "Write these selected files?" );
        }
        pView->AddMessage( "Application : " + applicationFileName );
        if ( patchFileName != NspFileNothing )
        {
            pView->AddMessage( "Patch : " + patchFileName );
        }

        pView->AddButton("Cancel");

        // 開始ボタンを追加
        pView->AddButton(
            "Start",
            [ this ]( void* pParam, nn::TimeSpan& timespan )
        {
            WriteNspFile();
        },
        nullptr,
        MessageView::ButtonTextColor::Green
        );
    }

    pView->ExtentSize();
    m_pParentPage->GetRootSurfaceContext()->StartModal( pView, true );
}

/**
 * @brief 選択された nsp ファイルのGameCard書き込みを行います。
 */
void WriteScene::WriteNspFile() NN_NOEXCEPT
{
    // GameCard書き込みスレッドのスタック
    static NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 s_WriteStack[ 32 * 1024 ];

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::CreateThread( &m_WriteTriggerThread, WriteTriggerThreadFunc, this, s_WriteStack, sizeof( s_WriteStack ), nn::os::DefaultThreadPriority )
    );
    nn::os::StartThread( &m_WriteTriggerThread );

    // GameCard書き込み進捗ビューを登録
    m_pWriteView = new WriteView( [&]{ CloseButtonCallback(); } );

    // コンテンツのGameCard書き込み処理中はフローティング起動をブロックする
    m_pParentPage->GetRootSurfaceContext()->StartModal( m_pWriteView, true, true, false, true );
}

/**
 * @brief GameCard書き込み進捗画面の Close ボタン押下時のコールバック関数です。
 */
void WriteScene::CloseButtonCallback() NN_NOEXCEPT
{
    m_WriteState = WriteState_NotStarted;
    DestroyWriteTriggerThread();

    m_pParentPage->GetRootSurfaceContext()->setFocus( &m_ButtonToCatalog);
}

/**
 * @brief GameCard書き込みトリガスレッドのメイン関数です。
 */
void WriteScene::WriteTriggerThreadFunc( void* args ) NN_NOEXCEPT
{
    WriteScene* self = reinterpret_cast< WriteScene* >( args );

    NN_ASSERT_NOT_NULL( self );
    NN_ASSERT_EQUAL( self->m_WriteState, WriteState_NotStarted );

#if !defined ( NN_BUILD_CONFIG_OS_WIN ) // ビルドエラーを回避
    const auto result = nn::fs::MountSdCardForDebug( "sd" );
    if ( result.IsFailure() )
    {
        self->SetWriteResult( result );
        return;
    }
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount( "sd" );
    };
#endif

    const char* pApplicationFileName = self->m_ApplicationFileName;
    const char* pPatchFileName = self->m_PatchFileName;

    self->SetWriteResult( self->ExecuteWrite( pApplicationFileName, pPatchFileName ) );
}

/**
 * @brief GameCard書き込みトリガスレッドを破棄します。
 */
void WriteScene::DestroyWriteTriggerThread() NN_NOEXCEPT
{
    nn::os::WaitThread( &m_WriteTriggerThread );
    nn::os::DestroyThread( &m_WriteTriggerThread);
}

/**
 * @brief GameCard書き込み結果を設定します。
 */
void WriteScene::SetWriteResult( const nn::Result& result ) NN_NOEXCEPT
{
    std::lock_guard< nn::os::Mutex > lock( m_Mutex );
    m_WriteState = result.IsSuccess() ? WriteState_Committed : WriteState_Failed;
    m_pWriteView->SetWriteResult( result );
}

/**
 * @brief nsp ファイルをGameCard書き込みします。
 */
const nn::Result WriteScene::ExecuteWrite( const char* applicationFilePath, const char* patchFilePath ) NN_NOEXCEPT
{
#if !defined ( NN_BUILD_CONFIG_OS_WIN )
    const size_t maxNspPathCount = 2;
    size_t nspCount = 0;

    nn::fs::GameCardSize cardSize = {}; // ゲームカードサイズ。未指定(0)固定
    nn::fs::GameCardClockRate cardClockRate = {}; // ゲームカード動作周波数。未指定(0)固定

    char nspPath[maxNspPathCount][MaxFilePathLength];
    nn::util::Strlcpy( nspPath[0], applicationFilePath, MaxFilePathLength );
    nspCount++;

    if ( patchFilePath != NspFileNothing )
    {
        nn::util::Strlcpy( nspPath[nspCount], patchFilePath, MaxFilePathLength );
        nspCount++;
    }

    // 開発用ゲームカードへの書き込み確認OFF
    nn::fs::SetVerifyWriteEnalbleFlag( false );

    m_WriteState = WriteState_Running;
    DEVMENU_LOG( "Preparing to write %s...\n", applicationFilePath );

    // 開発用ゲームカードへの書き込み
    auto result = WriteFromNsp( nspPath, nspCount, cardSize, cardClockRate, "", false, false, false, 0, false );
    if ( result.IsSuccess() )
    {
        m_WriteState = WriteState_Committed;
    }
    else
    {
        DEVMENU_LOG("Failed to write nsp to Game Card 0x%08x\n", result.GetInnerValueForDebug());
    }

    return result;
#else
    NN_RESULT_SUCCESS;
#endif // !defined ( NN_BUILD_CONFIG_OS_WIN )
}

/**
 * @brief GameCard書き込み状況を更新します。
 */
void WriteScene::UpdateWriteProgress() NN_NOEXCEPT
{
    // Note: 以下の遷移は別スレッドで行われる
    //   - WriteState_NotStarted -> WriteState_Running
    //   - WriteState_NotStarted -> WriteState_Failed
    //   - WriteState_Running    -> WriteState_Committed
    //   - WriteState_Running    -> WriteState_Failed (異常ケースで稀)

    std::lock_guard< nn::os::Mutex > lock( m_Mutex );

    switch ( m_WriteState )
    {
    case WriteState_NotStarted:
        // ApplicationWriteTask が Execute されると WriteState_Running に遷移する
        // ApplicationWriteTask を Execute するまでの処理に Fail すると WriteState_Failed に遷移する
        break;
    case WriteState_Running:
        // ApplicationWriteTask が Commit されると WriteState_Committed に遷移する
        // ApplicationWriteTask の Commit に失敗すると WriteState_Failed に遷移する
        {
            const int64_t pastTimeMilliSeconds = nn::os::ConvertToTimeSpan( nn::os::GetSystemTick() - m_StartTick ).GetMilliSeconds();
            if ( 1000LL < pastTimeMilliSeconds )
            {
                GameCardWriteProgress progress;
                GetGameCardWriteProgress(&progress);
                m_pWriteView->SetLastWrittenNspTotalSize( progress.totalSize );
                if ( 0 == progress.currentSize || 0 == progress.totalSize )
                {
                    return;
                }

                // 進捗 100.0% が 2 度続けてログ出力されないようにする
                if ( progress.currentSize != progress.totalSize )
                {
                    char progressStr[ 48 ];
                    const auto writtenSizeStr = GetDelimitedNumberString( progress.currentSize );
                    const auto totalSizeStr = GetDelimitedNumberString( progress.totalSize );

                    const float progressRate = static_cast< float >( progress.currentSize ) / static_cast< float >( progress.totalSize );
                    const float progressRatePercent = 100 * progressRate;
                    nn::util::SNPrintf( progressStr, sizeof( progressStr ), "%s / %s Byte(%.1f%)", writtenSizeStr.c_str(), totalSizeStr.c_str(), progressRatePercent );
                    m_pWriteView->SetProgressLabel( progressStr );
                    m_pWriteView->SetProgressBarValue( progressRate );
                    DEVMENU_LOG( "%s\n", progressStr );
                }
                m_StartTick = nn::os::GetSystemTick();
            }
        }
        break;
    case WriteState_Committed:
        {
            // 進捗 100% の表示
            char progressStr[48];
            const auto totalSizeStr = GetDelimitedNumberString( m_pWriteView->GetLastWrittenNspTotalSize() );
            nn::util::SNPrintf( progressStr, sizeof(progressStr), "%s / %s  Byte(100.0%)", totalSizeStr.c_str(), totalSizeStr.c_str() );
            m_pWriteView->SetProgressCaption( "[Write Completed]" );
            m_pWriteView->SetProgressLabel( progressStr );
            m_pWriteView->SetProgressBarValue(1.0f);
            DEVMENU_LOG( "%s\n", progressStr );

            // 次の state を決定する
            m_WriteState = WriteState_Completed;
        }
        break;
    case WriteState_Completed:
        {
            m_pWriteView->SetCloseButton();
            m_WriteState = WriteState_NotStarted;
        }
        break;
    case WriteState_Failed:
        {
            // TORIAEZU: SelectAll 時は失敗したらボタンを押すまでGameCard書き込み処理を止める
            m_pWriteView->SetErrorMessageView();
            m_WriteState = WriteState_WaitUserAction;
        }
        break;
    case WriteState_WaitUserAction:
        // ボタン押下待ち. 押下されると WriteState_NotStarted にステート遷移する
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

/**
* @brief nspリスト画面に切り替えます。
*
* @param[in]   contentMetaType      nspコンテンツタイプ
*/
void WriteScene::ListupCallBack( nn::ncm::ContentMetaType contentMetaType ) NN_NOEXCEPT
{
    m_pFileListSelectorScene->SetContentType( contentMetaType );
    m_pFileListSelectorScene->MakeNspList();
    m_pParentPage->SwitchScene( this, m_pFileListSelectorScene, false );
}

/**
* @brief nspリスト画面で選択されたnspファイルをコンテンツタイプ別に設定します。
*
* @param[in]   pNspFile          nspファイルパス文字列
*/
void WriteScene::SetNspFile( const char* pNspFile ) NN_NOEXCEPT
{
    switch ( m_pFileListSelectorScene->GetContentType() )
    {
    case nn::ncm::ContentMetaType::Application:
        m_ButtonWrite.UpdateFocusAndColor( true, true );
        nn::util::Strlcpy( m_ApplicationFileName, pNspFile, sizeof(m_ApplicationFileName) );
        m_pLabelApplication->setValue( ClipMessage( pNspFile, MaxVisibleFileNameLength ) );
        break;
    case nn::ncm::ContentMetaType::Patch:
        nn::util::Strlcpy( m_PatchFileName, pNspFile, sizeof(m_PatchFileName) );
        m_pLabelPatch->setValue( ClipMessage( pNspFile, MaxVisibleFileNameLength ) );
        break;
    default:
        break;
    }
}

/**
* @brief 選択されたnspテキストを初期状態にクリアします。
*
*/
void WriteScene::ClearScreenText() NN_NOEXCEPT
{
    m_ButtonWrite.UpdateFocusAndColor( false, false );
    m_pLabelApplication->setValue( NspFileNothing );
    m_pLabelPatch->setValue( NspFileNothing );
    nn::util::Strlcpy( m_ApplicationFileName, NspFileNothing.c_str(), sizeof(m_ApplicationFileName) );
    nn::util::Strlcpy( m_PatchFileName, NspFileNothing.c_str(), sizeof(m_PatchFileName) );
}

/**
* @brief       表示する文字数に上限を設けて超えた場合は省略した形を生成します。
*
* @param[in]   string          文字列
* @param[in]   maxLength       表示する文字の上限数( 9 以上 MaxDirectoryNameLength - 1 以内 )
*/
const std::string WriteScene::ClipMessage( const char* string, size_t maxLength ) NN_NOEXCEPT
{
    NN_ASSERT( 9 <= maxLength && maxLength < MaxDirectoryNameLength );

    const size_t stringLength = strlen( string );

    // 文字数が多い場合は、最初と最後から数文字だけを表示
    if ( stringLength > maxLength )
    {
        // " ... " と終端文字の長さを引く
        const size_t includeMessageLength = maxLength - 6;

        // 表示する文字を前半:後半 = 1:2 とする
        const size_t firstSize = includeMessageLength / 3;
        const size_t endSize = std::ceilf( includeMessageLength * 2 / 3 );

        const std::string firstMessage( string, string + firstSize - 1 );
        const std::string endMessage( string + stringLength - endSize, string + stringLength );

        return firstMessage + std::string( " ... " ) + endMessage;
    }
    return std::string( string );
}

}}} // ~namespace devmenu::application::write, ~namespace devmenu::application, ~namespace devmenu
