﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Accounts/DevMenu_AccountsSceneAsyncTask.h"
#include "Accounts/DevMenu_AccountsSceneDetail.h"
#include "Accounts/DevMenu_AccountsSceneFwdbg.h"
#include "Accounts/DevMenu_AccountsSceneList.h"
#include "Accounts/DevMenu_AccountsScenePlayData.h"
#include "Accounts/DevMenu_AccountsSceneSelector.h"
#include "Accounts/DevMenu_AccountsSceneSyncTask.h"
#include "Accounts/DevMenu_AccountsSdkHelper.h"
#include "Accounts/DevMenu_AccountsUiPolicy.h"
#include "DevMenu_Config.h"
#include "DevMenu_RootSurface.h"

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ResultForAdministrators.h>
#include <nn/account/account_ResultPrivate.h>

#include <glv_CustomVerticalListView.h>
#include <glv_texture.h>
#include <memory>
#include <vector>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_FormatString.h>

// "Accounts" is a page to manage user accounts (AccountManagementPage)

namespace devmenu { namespace accounts {

namespace
{
    std::vector<SelectorScene<NicknameSelection>::SelectItem> GetDebugNicknameItems() NN_NOEXCEPT
    {
        std::vector<SelectorScene<NicknameSelection>::SelectItem> nicknameItems;

        auto keyboardInputName = GetKeyboradInputName();
        glv::WideString label;
        glv::CopyWithConvertWideString(label, keyboardInputName, 0);
        NicknameSelection nSelection = { NicknameSelection::SelectionKind::Keyboard, { "" } };
        nn::util::Strlcpy(nSelection.data, keyboardInputName, sizeof(nSelection.data));
        nicknameItems.push_back(SelectorScene<NicknameSelection>::SelectItem(label, nSelection));

        int debugNicknameCount = 0;

        auto extraFontNicknames = GetNicknames(NicknameType::ExtraFont);
        for( const auto& nickname : extraFontNicknames )
        {
            const int LabelStrSize = 64;
            char labelStr[LabelStrSize];

            nn::util::SNPrintf(labelStr, LabelStrSize, "Debug name %c (extra font)", 'A' + debugNicknameCount);
            glv::WideString mainLabel;
            glv::CopyWithConvertWideString(mainLabel, labelStr, 0);

            glv::WideString subLabel;
            nn::util::SNPrintf(labelStr, LabelStrSize, "extra font : %s", nickname.name);
            glv::CopyWithConvertWideString(subLabel, labelStr, 0);

            NicknameSelection selection = { NicknameSelection::SelectionKind::String, { "" } };
            nn::util::Strlcpy(selection.data, nickname.name, sizeof(selection.data));
            nicknameItems.push_back(SelectorScene<NicknameSelection>::SelectItem(mainLabel, subLabel, selection));

            debugNicknameCount++;
        }

        auto noFontNicknames = GetNicknames(NicknameType::NoFont);
        for( const auto& nickname : noFontNicknames )
        {
            const int LabelStrSize = 64;
            char labelStr[LabelStrSize];

            nn::util::SNPrintf(labelStr, LabelStrSize, "Debug name %c (no font)", 'A' + debugNicknameCount);
            glv::WideString mainLabel;
            glv::CopyWithConvertWideString(mainLabel, labelStr, 0);

            glv::WideString subLabel;
            nn::util::SNPrintf(labelStr, LabelStrSize, "no font : %s", nickname.name);
            glv::CopyWithConvertWideString(subLabel, labelStr, 0);

            NicknameSelection selection = { NicknameSelection::SelectionKind::String, { "" } };
            nn::util::Strlcpy(selection.data, nickname.name, sizeof(selection.data));
            nicknameItems.push_back(SelectorScene<NicknameSelection>::SelectItem(mainLabel, subLabel, selection));

            debugNicknameCount++;
        }

        return nicknameItems;
    }

    struct ProfileImage
    {
        char name[32];
        char label[32];
    };

    std::vector<SelectorScene<ProfileImage>::SelectItem> GetDebugProfileImageItems() NN_NOEXCEPT
    {
        std::vector<SelectorScene<ProfileImage>::SelectItem> profileImageItems;

        ProfileImage profileImages[] =
        {
            {"Icon_Indigo.jpg", "Indigo"},
            {"Icon_Blue.jpg", "Blue"},
            {"Icon_LightCyan.jpg", "LightCyan"},
            {"Icon_Green.jpg", "Green"},
            {"Icon_GreenYellow.jpg", "GreenYellow"},
            {"Icon_Yellow.jpg", "Yellow"},
            {"Icon_Orange.jpg", "Orange"},
            {"Icon_Red.jpg", "Red"},
            {"Icon_LightPink.jpg", "LightPink"},
            {"Icon_Magenta_128KiB.jpg", "Magenta (128KiB)"},
            {"Icon_CORRUPTED.jpg", "<Corrupted JPEG>"}
        };

        for ( const auto& profileImage : profileImages )
        {
            const int LabelStrSize = 64;
            char labelStr[LabelStrSize];

            nn::util::SNPrintf(labelStr, LabelStrSize, "%s", profileImage.label);
            glv::WideString mainLabel;
            glv::CopyWithConvertWideString(mainLabel, labelStr, 0);

            profileImageItems.push_back(SelectorScene<ProfileImage>::SelectItem(mainLabel, profileImage));
        }

        return profileImageItems;
    }

    std::vector<SelectorScene<nn::Result>::SelectItem> GetDebugUserBadStatusItems() NN_NOEXCEPT
    {
        std::vector<SelectorScene<nn::Result>::SelectItem> userBadStatusItems;

#define DEVMENU_ACCOUNTS_BAD_STATE_DEF(name1, name2) \
    { "" # name1 ": \"" # name2 "\"", nn::account::Result ## name1 ## name2 () }

        const struct
        {
            const char* label;
            nn::Result result;
        } Definitions[] = {
#if defined(NN_DEVMENUSYSTEM)
            DEVMENU_ACCOUNTS_BAD_STATE_DEF(NetworkServiceAccount, CredentialBroken),
            DEVMENU_ACCOUNTS_BAD_STATE_DEF(NetworkServiceAccount, Unmanaged),
            DEVMENU_ACCOUNTS_BAD_STATE_DEF(NetworkServiceAccount, Banned),
            DEVMENU_ACCOUNTS_BAD_STATE_DEF(NintendoAccount, LinkageBroken),
#endif
            DEVMENU_ACCOUNTS_BAD_STATE_DEF(NintendoAccount, StateOtherButInteractionRequired),
            DEVMENU_ACCOUNTS_BAD_STATE_DEF(NintendoAccount, StateDeleted),
            DEVMENU_ACCOUNTS_BAD_STATE_DEF(NintendoAccount, StateBanned),
            DEVMENU_ACCOUNTS_BAD_STATE_DEF(NintendoAccount, StateSuspended),
            DEVMENU_ACCOUNTS_BAD_STATE_DEF(NintendoAccount, StateWithdrawn),
            DEVMENU_ACCOUNTS_BAD_STATE_DEF(NintendoAccount, StateTermsAgreementRequired),
            DEVMENU_ACCOUNTS_BAD_STATE_DEF(NintendoAccount, StateReauthorizationRequired),
        };
#undef DEVMENU_ACCOUNTS_BAD_STATE_DEF

        for (const auto& d : Definitions )
        {
            const int LabelStrSize = 64;
            char labelStr[LabelStrSize];

            nn::util::SNPrintf(labelStr, LabelStrSize, "%s", d.label);
            glv::WideString mainLabel;
            glv::CopyWithConvertWideString(mainLabel, labelStr, 0);

            userBadStatusItems.push_back(SelectorScene<nn::Result>::SelectItem(mainLabel, d.result));
        }
        return userBadStatusItems;
    }
}

class AccountManagementPage
    : public Page
{
private:
    ListScene m_AccountList;
    DetailScene m_AccountDetail;
    AsyncTaskScene m_AccountAsyncTask;
    SyncTaskScene m_AccountSyncTask;
    FwdbgScene m_AccountFwdbg;
    SelectorSceneWithKeyboardInput<NicknameSelection> m_AccountNicknameSelector;
    SelectorScene<ProfileImage> m_AccountProfileImageSelector;
    CancellableSelectorScene<nn::Result> m_AccountUserBadStatusSelector;
    PlayDataInfoScene m_AccountPlayDataInfo;

    glv::View* m_pFocusableChildView;

    // TODO Scene の push と pop でやりくりする
    struct Operators
        : public AbstractOperators
    {
        AccountManagementPage& parent;

        explicit Operators(AccountManagementPage& p) NN_NOEXCEPT
            : parent(p)
        {
        }
        virtual void RefreshAll() const NN_NOEXCEPT final NN_OVERRIDE
        {
        }

        virtual void AddUser() const NN_NOEXCEPT final NN_OVERRIDE
        {
            CreateUserAccount();
            parent.m_AccountList.RequireRefresh();
        }
        virtual void DeleteUser(const nn::account::Uid& user) const NN_NOEXCEPT final NN_OVERRIDE
        {
            parent.OnDeleteUser(user);
        }
        virtual void OpenUserInfo(const nn::account::Uid& user) const NN_NOEXCEPT final NN_OVERRIDE
        {
            parent.m_AccountDetail.SetAccount(user);
            parent.m_AccountDetail.RequireRefresh();

            // TODO enter leave で置き換える
            parent.m_AccountList.disable(glv::Property::Visible);
            parent.m_AccountDetail.enable(glv::Property::Visible);
        }
        virtual void CloseUserInfo() const NN_NOEXCEPT final NN_OVERRIDE
        {
            parent.m_AccountDetail.disable(glv::Property::Visible);
            parent.m_AccountList.enable(glv::Property::Visible);

            parent.m_AccountList.RequireRefresh();
            parent.m_AccountDetail.SetAccount(nn::account::InvalidUid);
            parent.m_AccountDetail.RequireRefresh();
        }
        virtual void BeginAsyncTask(nn::account::AsyncContext&& ctx, devmenu::Scene* prev) const NN_NOEXCEPT final NN_OVERRIDE
        {
            parent.m_AccountAsyncTask.SetContext(std::move(ctx), prev);

            prev->disable(glv::Property::Visible);
            parent.m_AccountAsyncTask.enable(glv::Property::Visible);

            parent.m_AccountAsyncTask.RequireRefresh();
            prev->RequireRefresh();
        }
        virtual void BeginSyncTask(std::function<nn::Result(void)> function, devmenu::Scene* prev) const NN_NOEXCEPT final NN_OVERRIDE
        {
            parent.m_AccountSyncTask.SetContext(prev, function);

            prev->disable(glv::Property::Visible);
            parent.m_AccountSyncTask.enable(glv::Property::Visible);

            parent.m_AccountSyncTask.RequireRefresh();
            prev->RequireRefresh();
        }
        virtual void OpenFwdbg() const NN_NOEXCEPT final NN_OVERRIDE
        {
            parent.m_AccountFwdbg.RequireRefresh();
            parent.m_AccountList.disable(glv::Property::Visible);
            parent.m_AccountFwdbg.enable(glv::Property::Visible);
        }
        virtual void PopScene(glv::View* current, glv::View* prev) const NN_NOEXCEPT final NN_OVERRIDE
        {
            current->disable(glv::Property::Visible);
            prev->enable(glv::Property::Visible);

            parent.m_AccountList.RequireRefresh();
            parent.m_AccountDetail.RequireRefresh();
        }
        virtual void OpenNicknameSelector(const nn::account::Uid& user) const NN_NOEXCEPT final NN_OVERRIDE
        {
            parent.m_AccountNicknameSelector.RequireRefresh();
            parent.m_AccountDetail.disable(glv::Property::Visible);
            parent.m_AccountNicknameSelector.enable(glv::Property::Visible);
            NN_UNUSED(user);
        }
        virtual void OpenProfileImageSelector(const nn::account::Uid& user) const NN_NOEXCEPT final NN_OVERRIDE
        {
            parent.m_AccountProfileImageSelector.RequireRefresh();
            parent.m_AccountDetail.disable(glv::Property::Visible);
            parent.m_AccountProfileImageSelector.enable(glv::Property::Visible);
            NN_UNUSED(user);
        }
        virtual void OpenBadStatusSelector(const nn::account::Uid& user) const NN_NOEXCEPT final NN_OVERRIDE
        {
            parent.m_AccountUserBadStatusSelector.RequireRefresh();
            parent.m_AccountDetail.disable(glv::Property::Visible);
            parent.m_AccountUserBadStatusSelector.enable(glv::Property::Visible);
            NN_UNUSED(user);
        }
        virtual void SetFocusableChild(glv::View* pView) const NN_NOEXCEPT final NN_OVERRIDE
        {
            parent.SetFocusableChild(pView);
        }
        virtual void OpenPlayDataInfo(const nn::account::Uid& user) const NN_NOEXCEPT final NN_OVERRIDE
        {
            parent.m_AccountPlayDataInfo.SetAccount(user);
            parent.m_AccountPlayDataInfo.RequireRefresh();
            parent.m_AccountDetail.disable(glv::Property::Visible);
            parent.m_AccountPlayDataInfo.enable(glv::Property::Visible);
        }
        virtual void ClosePlayDataInfo() const NN_NOEXCEPT final NN_OVERRIDE
        {
            parent.m_AccountPlayDataInfo.disable(glv::Property::Visible);
            parent.m_AccountDetail.enable(glv::Property::Visible);
        }
    } m_Op;

public:
    AccountManagementPage(int pageId, const glv::WideCharacterType* pageCaption, glv::Rect rect) NN_NOEXCEPT
        : Page(pageId, pageCaption, rect)
        , m_AccountList(m_Op, rect)
        , m_AccountDetail(m_Op, rect)
        , m_AccountAsyncTask(m_Op, rect)
        , m_AccountSyncTask(m_Op, rect)
        , m_AccountFwdbg(m_Op, rect, &m_AccountList)
        , m_AccountNicknameSelector(
            m_Op, rect, &m_AccountDetail, GLV_TEXT_API_WIDE_STRING("Select Nickname"),
            GetDebugNicknameItems(), [&](const NicknameSelection& selection)-> nn::Result { return m_AccountDetail.OnNicknameSelected(selection); })
        , m_AccountProfileImageSelector(
            m_Op, rect, &m_AccountDetail, GLV_TEXT_API_WIDE_STRING("Select ProfileImage"),
            GetDebugProfileImageItems(), [&](const ProfileImage& profileImage) { m_AccountDetail.OnProfileImageSelected(profileImage.name); })
        , m_AccountUserBadStatusSelector(
            m_Op, rect, this,
            &m_AccountDetail, GLV_TEXT_API_WIDE_STRING("Select resulted status"),
            GetDebugUserBadStatusItems(),
            { "Change Status", "Are you sure you want to change user state ?" },
            [&](const nn::Result& result) { m_AccountDetail.OnDebugUserBadStatusSelected(result); })
        , m_AccountPlayDataInfo(
            m_Op, rect)
        , m_pFocusableChildView( &m_AccountFwdbg )
        , m_Op(*this)
    {
        m_AccountList.enable(glv::Property::Visible);
        m_AccountDetail.disable(glv::Property::Visible);
        m_AccountAsyncTask.disable(glv::Property::Visible);
        m_AccountSyncTask.disable(glv::Property::Visible);
        m_AccountFwdbg.disable(glv::Property::Visible);
        m_AccountNicknameSelector.disable(glv::Property::Visible);
        m_AccountProfileImageSelector.disable(glv::Property::Visible);
        m_AccountUserBadStatusSelector.disable(glv::Property::Visible);

        // Set B Button Callbacks
        m_AccountList.SetBackButtonCallback( [&] { this->GetRootSurfaceContext()->MoveFocusToMenuTabs(); } );
        m_AccountDetail.SetBackButtonCallback( [&] { m_Op.CloseUserInfo(); } );
        m_AccountFwdbg.SetBackButtonCallback( [&] { m_AccountFwdbg.CloseScene(); } );
        m_AccountNicknameSelector.SetBackButtonCallback( [&] { m_AccountNicknameSelector.CloseScene(); } );
        m_AccountProfileImageSelector.SetBackButtonCallback( [&] { m_AccountProfileImageSelector.CloseScene(); } );
        m_AccountUserBadStatusSelector.SetBackButtonCallback( [&] { m_AccountUserBadStatusSelector.CloseScene(); } );
        m_AccountPlayDataInfo.SetBackButtonCallback([&] { m_Op.ClosePlayDataInfo(); });
        // Don't return from the SceneAsyncTask and SceneSyncTask scenes on B button. Instead, maintain current behavior to move focus to menu tabs.
        m_AccountAsyncTask.SetBackButtonCallback( [&] { this->GetRootSurfaceContext()->MoveFocusToMenuTabs(); } );
        m_AccountSyncTask.SetBackButtonCallback( [&] { this->GetRootSurfaceContext()->MoveFocusToMenuTabs(); } );

        *this
            << m_AccountList
            << m_AccountDetail
            << m_AccountAsyncTask
            << m_AccountSyncTask
            << m_AccountFwdbg
            << m_AccountNicknameSelector
            << m_AccountProfileImageSelector
            << m_AccountUserBadStatusSelector
            << m_AccountPlayDataInfo;
    }

    virtual bool onEvent(glv::Event::t e, glv::GLV& g) NN_NOEXCEPT NN_OVERRIDE
    {
        return Page::onEvent( e, g );
    }
    virtual void OnAttachedPage() NN_NOEXCEPT NN_OVERRIDE
    {
        if (m_AccountList.visible())
        {
            // 初回ページ表示時のみ容赦なくメインメニューからAddボタンへフォーカスが移動する現象回避用。
            // RequireRefresh()の遅延更新機構に基づいて g.setFocus( GetPrimaryView() )が初回時のみ予約された状態になってしまっていました。
            // OnAttachedPage() は GLVイベントコンテキストスコープ外のため、ここでのRefreshなどの構築処理は安全です。
            m_AccountList.Refresh();
        }
    }
    virtual void OnDetachedPage() NN_NOEXCEPT NN_OVERRIDE
    {
        m_AccountList.Clear();
    }
    virtual void OnActivatePage() NN_NOEXCEPT NN_OVERRIDE
    {
        RefreshAccountListView();
    }
    virtual void OnDeactivatePage() NN_NOEXCEPT NN_OVERRIDE
    {
    }

    virtual void OnLoopBeforeSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(context);
        NN_UNUSED(events);
    }
    virtual void OnLoopAfterSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(context);
        NN_UNUSED(events);

        // 内容の更新
        if (m_AccountAsyncTask.visible())
        {
            m_AccountAsyncTask.Update();
        }
        if (m_AccountSyncTask.visible())
        {
            m_AccountSyncTask.Update();
        }

        // 画面の再構成
        if (m_AccountList.IsRefreshRequired())
        {
            m_AccountList.Refresh();
        }
        if (m_AccountDetail.IsRefreshRequired() && m_AccountDetail.enabled( glv::Property::Visible ) )
        {
            // Only refresh account detail if it is visible in order to prevent it from stealing focus
            m_AccountDetail.Refresh();
        }
        if (m_AccountAsyncTask.IsRefreshRequired())
        {
            m_AccountAsyncTask.Refresh();
        }
        if (m_AccountFwdbg.IsRefreshRequired())
        {
            m_AccountFwdbg.Refresh();
        }
        if (m_AccountSyncTask.IsRefreshRequired())
        {
            m_AccountSyncTask.Refresh();
        }
        if (m_AccountNicknameSelector.IsRefreshRequired())
        {
            m_AccountNicknameSelector.Refresh();
        }
        if (m_AccountProfileImageSelector.IsRefreshRequired())
        {
            m_AccountProfileImageSelector.Refresh();
        }
        if (m_AccountUserBadStatusSelector.IsRefreshRequired())
        {
            m_AccountUserBadStatusSelector.Refresh();
        }
        if (m_AccountPlayDataInfo.IsRefreshRequired())
        {
            m_AccountPlayDataInfo.Refresh();
        }
    }

    virtual View* GetFocusableChild() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_ASSERT( m_pFocusableChildView );
        return m_pFocusableChildView;
    }

    void SetFocusableChild(glv::View* pView) NN_NOEXCEPT
    {
        m_pFocusableChildView = pView;
    }

    /* OnActivatedPage, OnChangeIntoForeground 時の表示更新対応
       AccountList をリフレッシュするとフォーカスが移動するのでひとまず無理矢理戻す*/
    void RefreshAccountListView() NN_NOEXCEPT
    {
        auto pRootSurface = GetRootSurfaceContext();
        NN_ASSERT(pRootSurface);
        auto pCurrentView = pRootSurface->focusedView();
        m_AccountList.Refresh();
        pRootSurface->setFocus(pCurrentView); // フォーカスを元に戻す
    }

    /**
    * @brief This sets up the Modal dialog to ask the user for confirmation before deleting the Account。
    */
    void OnDeleteUser(const nn::account::Uid& user) NN_NOEXCEPT
    {
        auto pView = new MessageView();

        // Set up the Title and Message
        pView->AddMessage("Delete User");
        pView->AddMessage("Are you sure you want to delete this User Account?");

        pView->AddButton( "Cancel" );

        // Add the button with the Lambda that will delete the account
        pView->AddButton(
            "Delete",
            [this, user](void*, nn::TimeSpan&)
            {
                this->m_Op.CloseUserInfo();

                m_Op.BeginSyncTask(
                    [this, user]() -> nn::Result
                    {
                        // 包括的な削除 API, 重い
                        NN_RESULT_DO(DeleteUserAccount(user));

                        this->m_AccountDetail.SetAccount(nn::account::InvalidUid);
                        this->m_AccountList.RequireRefresh();
                        NN_LOG("User account deleted\n");
                        NN_RESULT_SUCCESS;
                    },
                    &this->m_AccountList);
            },
            this,
            MessageView::ButtonTextColor::Red
        );
        GetRootSurfaceContext()->StartModal( pView, true );
    }
};

/**
 * @brief ページ生成 ( 専用クリエイター )
 */
template< size_t ID >
class AccountManagementPageCreator
    : PageCreatorBase
{
public:
    explicit AccountManagementPageCreator( const char* pageName ) NN_NOEXCEPT
        : PageCreatorBase( ID, pageName ) {}

protected:
    virtual glv::PageBase* newInstance() NN_NOEXCEPT NN_OVERRIDE
    {
        int resolution[ 2 ];
        const auto& d = glv::ApplicationFrameworkGetRuntimeContext().GetDisplay();
        d.GetResolution( resolution[ 0 ], resolution[ 1 ] );
        const auto width = static_cast< glv::space_t >( resolution[ 0 ] );
        const auto height = static_cast< glv::space_t >( resolution[ 1 ] );
        //const glv::Rect pageBounds( width - ( ( 12.0f ) * 2.0f ), height - 118.0f );    // 横は 8 + 4 マージン
        const glv::Rect pageBounds( width - 218.f, height - 118.0f );
        return new AccountManagementPage( ID, GLV_TEXT_API_WIDE_STRING( "Accounts" ), pageBounds );
    }
};

#define LOCAL_PAGE_CREATOR( _id, _name ) AccountManagementPageCreator< _id > g_AccountManagementPageCreator##_id( _name );
LOCAL_PAGE_CREATOR( DevMenuPageId_Accounts, "Accounts" );

}} // ~namespace devmenu::accounts, ~namespace devmenu
