﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "DevMenu_AccountsConfig.h"
#include "DevMenu_AccountsSdkHelper.h"
#include "DevMenu_AccountsUiComponents.h"
#include "../DevMenu_Config.h"
#include <nn/account/account_Api.h>
#include <nn/account/account_ApiDebug.h>

#include <algorithm>
#include <mutex>

#include <nn/image/image_JpegDecoder.h>

namespace devmenu { namespace accounts {

/* UserIcon -----------------------------------------------------------------------------------
*/

UserIcon::UserIcon(glv::GLsizei width, glv::GLsizei height, glv::space_t padX, glv::space_t padY) NN_NOEXCEPT
    : View(glv::Rect(width + padX * 2, height + padY * 2), glv::Place::t::TL)
    , m_Lock(false)
    , m_pTexture(new glv::Texture2(width, height, GL_RGB, GL_UNSIGNED_BYTE))
    , m_PadX(padX)
    , m_PadY(padY)
{
    auto scaleDenomX = 256 / width;
    auto scaleDenomY = 256 / height;
    NN_SDK_REQUIRES(scaleDenomX == scaleDenomY && (scaleDenomX == 1 || scaleDenomX == 2 || scaleDenomX == 4 || scaleDenomX == 8));
    NN_UNUSED(scaleDenomX);
    NN_UNUSED(scaleDenomY);

    m_pTexture->magFilter( GL_NEAREST );
    ClearTexture();

    disable(glv::Property::Controllable | glv::Property::DrawBack | glv::Property::DrawBorder | glv::Property::HitTest);
    enable(glv::Property::AlwaysBubble);
}

UserIcon::~UserIcon() NN_NOEXCEPT
{
    m_pTexture->destroy();
    delete m_pTexture;
}

void UserIcon::ActivateTexture() NN_NOEXCEPT
{
    glv::Texture2* pTexture;
    if ( nullptr != ( pTexture = m_pTexture ) )
    {
        if ( 0 != pTexture->id() )
        {
            pTexture->bind();
            pTexture->send();
        }
        else
        {
            pTexture->recreate();
        }
    }
}

void UserIcon::SetTexture(const Buffer& jpegData) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);

    nn::image::JpegDecoder decoder;
    decoder.SetImageData(jpegData.GetPointer(), jpegData.GetSize());
    decoder.SetPixelFormat(nn::image::PixelFormat_Rgb24);
    decoder.SetResolutionDenominator(256 / m_pTexture->width());
    bool decodeOk = false;
    auto r = decoder.Analyze();
    if (r == nn::image::JpegStatus_Ok)
    {
        auto dim = decoder.GetAnalyzedDimension();
        NN_ABORT_UNLESS(dim.width == m_pTexture->width());
        NN_ABORT_UNLESS(dim.height == m_pTexture->height());

        Buffer wb(decoder.GetAnalyzedWorkBufferSize());
        r = decoder.Decode(
            m_pTexture->buffer<unsigned char>(), m_pTexture->width() * m_pTexture->height() * 3, 1,
            wb.GetPointer(), wb.GetSize());
        decodeOk = (r == nn::image::JpegStatus_Ok);
    }

    if (!decodeOk)
    {
        NN_LOG("[DevMenu] Given jpegData is corrupted. Filling with dummy.\n");
        auto* pxs = m_pTexture->buffer<unsigned char>();
        for (auto i = 0; i < m_pTexture->height(); ++ i)
        {
            for (auto j = 0; j < m_pTexture->width(); ++ j)
            {
                auto isWhite = (((i / 8) % 2) == ((j / 8) % 2));
                auto c = (isWhite ? 255 : 128);
                auto pxIndex = i * m_pTexture->width() + j;
                pxs[pxIndex * 3 + 0] = c; // R
                pxs[pxIndex * 3 + 1] = c; // G
                pxs[pxIndex * 3 + 2] = c; // B
            }
        }
    }
    ActivateTexture();
}
void UserIcon::ClearTexture() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);

    auto* pxs = m_pTexture->buffer<unsigned char>();
    for (auto i = 0; i < m_pTexture->height(); ++ i)
    {
        for (auto j = 0; j < m_pTexture->width(); ++ j)
        {
            auto pxIndex = i * m_pTexture->width() + j;
            pxs[pxIndex * 3 + 0] = 128; // R
            pxs[pxIndex * 3 + 1] = 128; // G
            pxs[pxIndex * 3 + 2] = 128; // B
        }
    }
    ActivateTexture();
}

void UserIcon::onDraw(glv::GLV& g) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);

    glv::draw::enable(glv::draw::Texture2D);
    m_pTexture->begin();
    glv::draw::color(1, 1, 1, 1);
    m_pTexture->draw(m_PadX, m_pTexture->height() + m_PadY, m_pTexture->width() + m_PadX, m_PadY);
    m_pTexture->end();
    glv::draw::disable(glv::draw::Texture2D);
}

UserButton::Validity::Validity(bool validity) NN_NOEXCEPT
    : glv::View(glv::Rect(52, 36))
    , m_Label(
        validity? "Ready" : "N/A",
        glv::Label::Spec(glv::Place::CC, 0, 0, Default::FontSizeSmall))
{
    disable(glv::Property::Controllable | glv::Property::DrawBorder | glv::Property::DrawBack | glv::Property::HitTest);
    enable(glv::Property::AlwaysBubble);

    m_Style.color.set(glv::StyleColor::BlackOnWhite);
    if (validity)
    {
        m_Style.color.text.set(0.1, 0.8, 0.2);
    }
    else
    {
        m_Style.color.text.set(0.8, 0.1, 0.2);
    }
    m_Label.style(&m_Style);
    *this << m_Label;
}

UserButton::UserButton(const nn::account::Uid& uid, std::function<void()> callback, const glv::Rect& r) NN_NOEXCEPT
    : glv::View(r)
    , m_Callback(callback)
    , m_Placer(*this, glv::Direction::E, glv::Place::TL, 8.0f, 2.0f, 8.0f)
    , m_pValidity(new Validity(true))
    , m_Icon(32, 32, 0.0, 0.0)
{
    enable(glv::Property::Controllable | glv::Property::DrawBack | glv::Property::DrawBorder | glv::Property::HitTest);

    auto icon = LoadUserIcon(uid);
    icon ? m_Icon.SetTexture(icon) : m_Icon.ClearTexture();

    glv::CharTypeU16 nickname[64];
    GetNickname(nickname, sizeof(nickname) / sizeof(nickname[0]), uid);
    m_pNickname = new glv::Label(nickname, glv::Label::Spec(glv::Place::TL, 0, 0, CommonValue::InitialFontSize));

    m_Placer << m_pValidity;
    m_Placer << m_Icon;
    m_Placer << m_pNickname;
    changePadClickDetectableButtons( glv::BasicPadEventType::Button::Ok::Mask );
    changePadClickDetectableButtons( glv::DebugPadEventType::Button::Ok::Mask );
    attach( []( const glv::Notification& n )->void { n.receiver<UserButton>()->m_Callback(); }, glv::Update::Clicked, this );
}

UserButton::~UserButton() NN_NOEXCEPT
{
    if (m_pValidity != nullptr)
    {
        delete m_pValidity;
    }
    if (m_pNickname != nullptr)
    {
        delete m_pNickname;
    }
}

void UserButton::SetVailidity(bool valid) NN_NOEXCEPT
{
    if (valid)
    {
        enable(glv::Property::Controllable | glv::Property::DrawBorder | glv::Property::DrawBack | glv::Property::HitTest);
        disable(glv::Property::AlwaysBubble);
    }
    else
    {
        disable(glv::Property::Controllable | glv::Property::DrawBorder | glv::Property::DrawBack | glv::Property::HitTest);
        enable(glv::Property::AlwaysBubble);
    }
    delete m_pValidity;
    m_pValidity = new Validity(valid);
    *this << m_pValidity;
}

}} // ~namespace devmenu::accounts, ~namespace devmenu
