﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <glv.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/account/account_Result.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/ns/ns_UserResourceManagementApi.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

#include "DevMenu_AccountsSdkHelper.h"

namespace devmenu { namespace accounts {

namespace {

// SWKBD を起動するボタンネームにのみ使用する。ユーザーのニックネームに設定されることはない
const char* KeyboardInputName = "<Use software keyboard>";

const glv::CharTypeU16 NormalNicknames[][32] = {
    GLV_TEXT_API_STRING_UTF16("User"),
};

const glv::CharTypeU16 ExtraFontNicknames[][32] = {
    GLV_TEXT_API_STRING_UTF16("\u22EE\u24B5\u2603\u322A\u3388\u501A\u5369\u7554\u8F26\uFE39"),
    GLV_TEXT_API_STRING_UTF16("\u2474\u2593\u50C9\u671E\u21C4\u2584\uFF67\uFF50\uFE3B\u2FF0"),
    GLV_TEXT_API_STRING_UTF16("\u3120\u9CF3\u3400\u4B10\u5700\u6220\u7B30\u326E\u263A\u33D0"),
    GLV_TEXT_API_STRING_UTF16("\uB890\uC630\uD4C0\uF900\u32A4\u3136\u2776\u2552\u2299\uFE4C"),
};

const glv::CharTypeU16 NoFontNicknames[][32] = {
    GLV_TEXT_API_STRING_UTF16("\u09D0\u25F0\u324F\u4DBF\u9FCF\uA4CF\uD7AF\uFE00\uE450\uF8F0"),
    GLV_TEXT_API_STRING_UTF16("\u2EFA\uF81E\u10FD\u09D1\u070F\u0558\u0492\u027F\uE399\uFFA1"),
};

void GetNickname(nn::account::Nickname* pOutName, const uint16_t* nicknameU16, int length) NN_NOEXCEPT
{
    int dstLength;
    nn::util::GetLengthOfConvertedStringUtf16NativeToUtf8(&dstLength, nicknameU16, length);
    NN_ABORT_UNLESS(dstLength < sizeof(pOutName->name));
    auto r = nn::util::ConvertStringUtf16NativeToUtf8(pOutName->name, sizeof(pOutName->name), nicknameU16);
    NN_ABORT_UNLESS(r == nn::util::CharacterEncodingResult_Success);
    pOutName->name[dstLength] = '\0';
}

void GetNickname(nn::account::Nickname* pOutName, NicknameType nicknameType, int index) NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER_EQUAL(index, 0);
    switch( nicknameType )
    {
    case NicknameType::Normal:
        {
            NN_SDK_ASSERT_LESS(index, static_cast<int>(sizeof(NormalNicknames) / sizeof(NormalNicknames[0])));
            auto name = reinterpret_cast<const uint16_t*>(NormalNicknames[index]);
            auto length = nn::util::Strnlen(name, sizeof(NormalNicknames[index]) / sizeof(NormalNicknames[index][0]));
            GetNickname(pOutName, name, length);
        }
        return;

    case NicknameType::ExtraFont:
        {
            NN_SDK_ASSERT_LESS(index, static_cast<int>(sizeof(ExtraFontNicknames) / sizeof(ExtraFontNicknames[0])));
            auto name = reinterpret_cast<const uint16_t*>(ExtraFontNicknames[index]);
            auto length = nn::util::Strnlen(name, sizeof(ExtraFontNicknames[index]) / sizeof(ExtraFontNicknames[index][0]));
            GetNickname(pOutName, name, length);
        }
        return;

    case NicknameType::NoFont:
        {
            NN_SDK_ASSERT_LESS(index, static_cast<int>(sizeof(NoFontNicknames) / sizeof(NoFontNicknames[0])));
            auto name = reinterpret_cast<const uint16_t*>(NoFontNicknames[index]);
            auto length = nn::util::Strnlen(name, sizeof(NoFontNicknames[index]) / sizeof(NoFontNicknames[index][0]));
            GetNickname(pOutName, name, length);
        }
        return;

    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

} // ~namespace devmeenu::accounts::<anonymous>

// ユーザーアカウントの作成
nn::account::Uid CreateUserAccount() NN_NOEXCEPT
{
    nn::account::Uid uid;
    nn::account::ProfileEditor editor;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::BeginUserRegistration(&uid));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetProfileEditor(&editor, uid));
    SetDefaultProfile(editor);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::CompleteUserRegistrationForcibly(uid));
    return uid;
}

nn::Result DeleteUserAccount(const nn::account::Uid& user) NN_NOEXCEPT
{
    // 全体数を検索
    nn::ns::UserSaveDataStatistics stat;
    NN_RESULT_DO(nn::ns::CalculateUserSaveDataStatistics(&stat, user));

    // 削除開始
    NN_LOG("Deleting all user save data:\n");
    nn::ns::ProgressMonitorForDeleteUserSaveDataAll monitor;
    NN_RESULT_DO(nn::ns::DeleteUserSaveDataAll(&monitor, user));

    nn::os::SystemEvent e;
    monitor.GetSystemEvent(&e);
    nn::os::TimerEvent timer(nn::os::EventClearMode_AutoClear);
    timer.StartPeriodic(nn::TimeSpan::FromMilliSeconds(500), nn::TimeSpan::FromMilliSeconds(500));
    while (NN_STATIC_CONDITION(true))
    {
        nn::os::WaitAny(e.GetBase(), timer.GetBase());
        if (e.TryWait())
        {
            break;
        }
        NN_ABORT_UNLESS(timer.TryWait());
        auto progress = monitor.GetStatistics();
        NN_UNUSED(progress);
        NN_LOG(" - %d of %d (%zu of %zu bytes)\n", progress.count, stat.count, progress.sizeInBytes, stat.sizeInBytes);
    }
    NN_SDK_ASSERT(e.TryWait());
    NN_RESULT_DO(monitor.GetResult());

    // UA 削除
    return nn::account::DeleteUser(user);
}

void SetDefaultProfile(nn::account::ProfileEditor& editor) NN_NOEXCEPT
{
    static const char DefaultUserIconFilePath[] = "Contents:/accounts/defaultIcon.jpg";
    static const uint8_t DefaultUseData[nn::account::UserDataBytesMax] = {
#if defined(NN_BUILD_CONFIG_ENDIAN_LITTLE)
        0x01, 0x00, 0x00, 0x00, // version, mainImageType, padding(2),
        0x01, 0x00, 0x00, 0x00, // charId[LSB], charId>>8, charId>>16, charId[MSB]
        0x01, 0x00, 0x00, 0x01, // bgId[LSB], bgId>>8, bgId>>16, bgId[MSB]
#elif defined(NN_BUILD_CONFIG_ENDIAN_BIG)
        0x01, 0x00, 0x00, 0x00, // version, mainImageType, padding(2),
        0x00, 0x00, 0x00, 0x01, // charId[MSB], charId>>16, charId>>8, charId[LSB]
        0x01, 0x00, 0x00, 0x01, // bgId[MSB], bgId>>16, bgId>>8, bgId[LSB]
#else
    #error "Specified endian type is not supported."
#endif
    };

#if !defined(NN_SDK_BUILD_RELEASE)
    struct
    {
        uint8_t version;
        uint8_t mainImageType;
        uint8_t padding[2];
        uint32_t charactorId;
        uint32_t bgId;
        uint8_t reserve[4];
        char mii[16];
    } d;
    std::memcpy(&d, DefaultUseData, sizeof(d));
    NN_ASSERT(d.version == 0x01);
    NN_ASSERT(d.mainImageType == 0x00);
    NN_ASSERT(d.padding[0] == 0x00 && d.padding[1] == 0x00);
    NN_ASSERT(d.charactorId == 0x00000001);
    NN_ASSERT(d.bgId == 0x01000001);
#endif

    nn::account::Nickname name;
    GetNickname(&name, NicknameType::Normal, 0);
    editor.SetNickname(name);
    editor.SetUserData(reinterpret_cast<const char*>(DefaultUseData), sizeof(DefaultUseData));

    devmenu::Buffer b(128 * 1024);
    nn::fs::FileHandle file;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&file, DefaultUserIconFilePath, nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(file);
    };
    int64_t size;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&size, file));
    NN_SDK_ASSERT(static_cast<size_t>(size) <= b.GetSize());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(file, 0, b.GetPointer(), size));
    NN_ABORT_UNLESS_RESULT_SUCCESS(editor.FlushWithImage(b.GetPointer(), size));
}

bool IsNetworkServiceAccountRegistered(const nn::account::Uid& user) NN_NOEXCEPT
{
    nn::account::NetworkServiceAccountAdministrator admin;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountAdministrator(&admin, user));
    bool hasNsa;
    NN_ABORT_UNLESS_RESULT_SUCCESS(admin.IsNetworkServiceAccountRegistered(&hasNsa));
    return hasNsa;
}
NsaId GetNetworkServiceAccountId(const nn::account::Uid& user) NN_NOEXCEPT
{
    nn::account::NetworkServiceAccountManager manager;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountManager(&manager, user));
    bool hasNsa;
    NN_ABORT_UNLESS_RESULT_SUCCESS(manager.IsNetworkServiceAccountAvailable(&hasNsa));
    if (!hasNsa)
    {
        return NsaId();
    }
    nn::account::NetworkServiceAccountId nsaId;
    NN_ABORT_UNLESS_RESULT_SUCCESS(manager.GetNetworkServiceAccountId(&nsaId));
    return NsaId(nsaId);
}
bool GetNintendoAccountScreenName(char* out, size_t outSize, const nn::account::Uid& user) NN_NOEXCEPT
{
    nn::account::NetworkServiceAccountManager manager;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountManager(&manager, user));
    bool hasNsa;
    NN_ABORT_UNLESS_RESULT_SUCCESS(manager.IsNetworkServiceAccountAvailable(&hasNsa));
    if (!hasNsa)
    {
        return false;
    }

    static char g_NaCacheBuffer[nn::account::RequiredBufferSizeForCachedNintendoAccountInfo];
    nn::account::CachedNintendoAccountInfoForSystemService naInfo;
    auto r = manager.LoadCachedNintendoAccountInfo(&naInfo, g_NaCacheBuffer, sizeof(g_NaCacheBuffer));
    if (!r.IsSuccess())
    {
        return false;
    }

    size_t len;
    std::strncpy(out, naInfo.GetScreenName(&len), outSize);
    NN_SDK_ASSERT(len <= outSize);
    return true;
}

void GetNickname(glv::CharTypeU16* dst, int dstLength, const nn::account::Uid& uid) NN_NOEXCEPT
{
    nn::account::Nickname name;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNickname(&name, uid));

    char inBuffer[64];
    auto l = strnlen(name.name, sizeof(name.name));
    inBuffer[0] = '"';
    std::memcpy(inBuffer + 1, name.name, l);
    inBuffer[l + 1] = '"';
    inBuffer[l + 2] = '\0';

    int length;
    auto error = nn::util::GetLengthOfConvertedStringUtf8ToUtf16Native(&length, inBuffer, l + 2);
    NN_ABORT_UNLESS(error == nn::util::CharacterEncodingResult_Success);
    NN_ABORT_UNLESS(length < dstLength);
    NN_ABORT_UNLESS(nn::util::ConvertStringUtf8ToUtf16Native(reinterpret_cast<uint16_t*>(dst), dstLength, inBuffer) == nn::util::CharacterEncodingResult_Success);
}

devmenu::Buffer LoadUserIcon(const nn::account::Uid& uid) NN_NOEXCEPT
{
    size_t sizeActual;
    if (!nn::account::LoadProfileImage(&sizeActual, nullptr, 0, uid).IsSuccess())
    {
        return devmenu::Buffer();
    }
    devmenu::Buffer buffer(sizeActual);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::LoadProfileImage(&sizeActual, buffer.GetPointer(), buffer.GetSize(), uid));
    return buffer;
}

int UidToString(char* dst, size_t dstSize, const nn::account::Uid& uid) NN_NOEXCEPT
{
    return nn::util::SNPrintf(
        dst, dstSize, "%08x_%08x_%08x_%08x",
        static_cast<uint32_t>(uid._data[0] >> 32),
        static_cast<uint32_t>(uid._data[0] & 0xFFFFFFFFull),
        static_cast<uint32_t>(uid._data[1] >> 32),
        static_cast<uint32_t>(uid._data[1] & 0xFFFFFFFFull));
}

int NsaIdToString(char* dst, size_t dstSize, nn::account::NetworkServiceAccountId nsaId) NN_NOEXCEPT
{
    return nn::util::SNPrintf(
        dst, dstSize, "%08x_%08x",
        static_cast<uint32_t>(nsaId.id >> 32),
        static_cast<uint32_t>(nsaId.id & 0xFFFFFFFFull));
}

const char* GetKeyboradInputName() NN_NOEXCEPT
{
    return KeyboardInputName;
}

std::vector<nn::account::Nickname> GetNicknames(NicknameType nicknameType) NN_NOEXCEPT
{
    auto nicknames = std::vector<nn::account::Nickname>();
    unsigned int nicknameNum = 0;
    switch( nicknameType )
    {
    case NicknameType::Normal:
        nicknameNum = sizeof(NormalNicknames) / sizeof(NormalNicknames[0]);
        break;

    case NicknameType::ExtraFont:
        nicknameNum = sizeof(ExtraFontNicknames) / sizeof(ExtraFontNicknames[0]);
        break;

    case NicknameType::NoFont:
        nicknameNum = sizeof(NoFontNicknames) / sizeof(NoFontNicknames[0]);
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }

    for( unsigned int i = 0; i < nicknameNum; i++ )
    {
        nn::account::Nickname nickname;
        GetNickname(&nickname, nicknameType, i);
        nicknames.push_back(nickname);
    }
    return nicknames;
}

}} // ~namespace devmenu::accounts, ~namespace devmenu
