﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <string>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/hid/system/hid_UniquePad.h>

#include "ControllerSerialFlashTool_SerialFlashMap.h"

//!< TRow * TLine の　std::string を格納する構造体です
template <int TRow, int TLine>
struct StringTable
{
    std::string cell[TLine][TRow];
    static const int Row  = TRow;
    static const int LineMax = TLine;
    int lineCount;
};

inline size_t Split(std::string* pOutValue, const std::string& input, size_t offset, char delimiter) NN_NOEXCEPT
{
    auto delimiterIndex = input.find(delimiter, offset);
    if (delimiterIndex == std::string::npos)
    {
        *pOutValue = input.substr(offset);
    }

    *pOutValue = input.substr(offset, delimiterIndex - offset);

    return delimiterIndex;
};

template <int TRow, int TLine>
bool ParseCsv(StringTable<TRow, TLine>* pOutValue, const std::string& input) NN_NOEXCEPT
{
    size_t charIndex = 0;
    pOutValue->lineCount = 0;
    while (NN_STATIC_CONDITION(true))
    {
        NN_LOG("Parsing Line %d start from %d\n", pOutValue->lineCount + 1, charIndex);
        std::string line;
        charIndex = Split(&line, input, charIndex, '\n');
        NN_LOG("    Line -> %s\n", line.c_str());

        size_t charIndexInLine = 0;
        int rowCount = 0;
        while (NN_STATIC_CONDITION(true))
        {
            charIndexInLine = Split(&pOutValue->cell[pOutValue->lineCount][rowCount], line, charIndexInLine, ',');
            ++rowCount;
            if (charIndexInLine == std::string::npos)
            {
                // End of line
                if (rowCount < pOutValue->Row && line.length() > 1)
                {
                    // 列の数が期待より少ない
                    NN_LOG("Wrong Csv Format. %d cell[s] per line is expected. Only %d cell found at line:%d\n", pOutValue->Row, rowCount, pOutValue->lineCount + 1);
                }
                break;
            }
            if (rowCount >= pOutValue->Row)
            {
                // 列の数が期待より多い
                NN_LOG("Wrong Csv Format. %d cell[s] per line is expected. Failed at line %d\n", pOutValue->Row, pOutValue->lineCount + 1);
                return false;
            }
            ++charIndexInLine;
        }
        ++pOutValue->lineCount;
        if (charIndex == std::string::npos)
        {
            // End Of File
            break;
        }
        if (pOutValue->lineCount >= pOutValue->LineMax)
        {
            // 列の数が期待より多い
            NN_LOG("Wrong Csv Format. More than %d lines in the file\n", pOutValue->LineMax);
        }
        ++charIndex;
    }
    return true;
}

template <int TRow, int TLine>
std::string GetStringFromCsv(const StringTable<TRow, TLine>& input) NN_NOEXCEPT
{
    std::string output;
    for (int lineIndex = 0; lineIndex < input.lineCount; ++lineIndex)
    {
        for (int rowIndex = 0; rowIndex < input.Row; ++rowIndex)
        {
            output.append(input.cell[lineIndex][rowIndex]);
            if (rowIndex < input.Row - 1)
            {
                output.push_back(',');
            }
        }
        output.push_back('\n');
    }
    return output;
}
