﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>

#include "SimpleGfx_Types.h"

namespace nns { namespace sgx {

/**
 * @name    初期化 API
 * @{
 */

/**
 * @brief   ライブラリをデフォルトのパラメータで初期化
 */
void Initialize() NN_NOEXCEPT;

/**
 * @brief   パラメータを指定してライブラリを初期化
 */
void Initialize(const InitializationParameters& parameters) NN_NOEXCEPT;

/**
 * @brief   ライブラリを終了
 */
void Finalize() NN_NOEXCEPT;

/**
 * @brief   描画領域のサイズを取得
 */
void GetCanvasSize(Size* pOutSize) NN_NOEXCEPT;

/**
 * @}
 *
 * @name    描画準備 API
 * @{
 */

/**
 * @brief   描画を開始
 *
 * @details
 *  画面描画を開始します。
 *  実行すると、 @ref SetClearColor で指定した色で画面が消去されます。
 *  描画を終了する際は @ref EndRender を呼ぶか、または @ref EndRenderNoSwap と @ref SwapBuffer を呼び出す必要があります。
 */
void BeginRender() NN_NOEXCEPT;

/**
 * @brief   描画を終了し、フレームバッファを切り替える
 */
void EndRender() NN_NOEXCEPT;

/**
 * @brief   描画を終了。フレームバッファは切り替えない
 */
void EndRenderNoSwap() NN_NOEXCEPT;

/**
 * @brief   フレームバッファの切り替え
 */
void SwapBuffer() NN_NOEXCEPT;

/**
 * @brief   画面消去時の色を設定
 *
 * @details
 *  @ref BeginRender で画面を消去する際の背景色を設定します。
 *  未設定時の色は @ref Colors::Black です。
 */
void SetClearColor(const Color& color) NN_NOEXCEPT;

/**
 * @}
 *
 * @name    描画設定 API
 * @{
 */

/**
 * @brief   描画時のブレンド方法を設定
 */
void SetBlendMode(BlendMode mode) NN_NOEXCEPT;

/**
 * @brief   描画対象とする画像を指定
 *
 * @detail
 *  各種描画 API による描画対象を、指定した画像に変更します。
 *  描画対象をスクリーンに戻す場合は @ref ResetRenderTarget を呼び出してください。
 */
void SetRenderTarget(ImageData* pTargetImage) NN_NOEXCEPT;

/**
 * @brief   描画対象をスクリーンに戻す
 */
void ResetRenderTarget() NN_NOEXCEPT;

/**
 * @brief   描画範囲を指定した矩形内に制限する
 */
void ApplyRenderArea(float x, float y) NN_NOEXCEPT;
void ApplyRenderArea(float x, float y, float width, float height) NN_NOEXCEPT;
void ApplyRenderArea(const Rectangle& rect) NN_NOEXCEPT;

/**
 * @brief   制限した描画範囲を元に戻す
 */
void RestoreRenderArea() NN_NOEXCEPT;

/**
 * @brief   ステンシルバッファを有効化するかどうか設定する
 */
void SetStencilEnabled(bool isEnabled) NN_NOEXCEPT;

/**
 * @brief   ステンシルテストの方法を設定する
 *
 * @detail
 *  ステンシルテストの方法は、 @ref BeginRenderToStencil と @ref EndRenderToStencil の呼び出しによって初期化されます。
 *  使用時はその都度再設定する必要があります。
 */
void SetStencilFunc(StencilFunc func, int reference, uint32_t mask) NN_NOEXCEPT;

/**
 * @brief   ステンシルバッファへの描画を開始
 */
void BeginRenderToStencil() NN_NOEXCEPT;

/**
 * @brief   ステンシルバッファへの描画を終了
 */
void EndRenderToStencil() NN_NOEXCEPT;

/**
 * @brief   ステンシルバッファをクリア
 */
void ClearStencil() NN_NOEXCEPT;

/**
 * @}
 *
 * @name    点・線描画 API
 * @{
 */

/**
 * @brief   指定した位置に点を描画
 */
void DrawPoint(float x, float y, const Color& color, float size) NN_NOEXCEPT;

/**
 * @brief   始点と終点を結ぶ直線を描画
 */
void DrawLine(float x1, float y1, float x2, float y2, const Color& color, float lineWidth) NN_NOEXCEPT;
void DrawLine(const Point2D& startPoint, const Point2D& endPoint, const Color& color, float lineWidth) NN_NOEXCEPT;

/**
 * @brief   始点と終点を結ぶグラデーション直線を描画
 */
void DrawGradientLine(float x1, float y1, float x2, float y2, const Color& startColor, const Color& endColor, float lineWidth) NN_NOEXCEPT;
void DrawGradientLine(const Point2D& startPoint, const Point2D& endPoint, const Color& startColor, const Color& endColor, float lineWidth) NN_NOEXCEPT;

/**
 * @brief   指定した位置・サイズで矩形を描画
 */
void DrawRectangle(float x, float y, float width, float height, const Color& color, float lineWidth) NN_NOEXCEPT;
void DrawRectangle(const Rectangle& rectangle, const Color& color, float lineWidth) NN_NOEXCEPT;

/**
 * @brief   指定した頂点リストを結ぶ多角形を描画
 */
void DrawPolygon(int pointCount, const Point2D* points, const Color& color, float lineWidth) NN_NOEXCEPT;

/**
 * @brief   指定した中心・半径の円を描画
 */
void DrawCircle(float centerX, float centerY, float radius, const Color& color, float lineWidth) NN_NOEXCEPT;
void DrawCircle(const Point2D& center, float radius, const Color& color, float lineWidth) NN_NOEXCEPT;

/**
 * @brief   指定した中心・半径、開始角～終了角までの円弧を描画
 */
void DrawArc(float centerX, float centerY, float radius, float beginAngle, float endAngle, const Color& color, float lineWidth) NN_NOEXCEPT;
void DrawArc(const Point2D& center, float radius, float beginAngle, float endAngle, const Color& color, float lineWidth) NN_NOEXCEPT;

/**
 * @brief   指定した中心・半径、開始角～終了角までの扇形を描画
 */
void DrawPie(float centerX, float centerY, float radius, float beginAngle, float endAngle, const Color& color, float lineWidth) NN_NOEXCEPT;
void DrawPie(const Point2D& center, float radius, float beginAngle, float endAngle, const Color& color, float lineWidth) NN_NOEXCEPT;

/**
 * @}
 *
 * @name    塗り潰し API
 * @{
 */

/**
 * @brief   指定した位置・サイズの矩形を塗り潰す
 */
void FillRectangle(float x, float y, float width, float height, const Color& color) NN_NOEXCEPT;
void FillRectangle(const Rectangle& rectangle, const Color& color) NN_NOEXCEPT;

/**
 * @brief   指定した頂点リストを結ぶ多角形を塗り潰す
 */
void FillPolygon(int pointCount, const Point2D* points, const Color& color) NN_NOEXCEPT;

/**
 * @brief   指定した頂点リストと頂点カラーで多角形を塗り潰す
 */
void FillGradientPolygon(int pointCount, const Point2D* points, const Color* colors) NN_NOEXCEPT;

/**
 * @brief   指定した中心・半径の円を塗り潰す
 */
void FillCircle(float centerX, float centerY, float radius, const Color& color) NN_NOEXCEPT;
void FillCircle(const Point2D& center, float radius, const Color& color) NN_NOEXCEPT;

/**
 * @brief   指定した中心・半径、開始角～終了角までの扇形を塗り潰す
 */
void FillPie(float centerX, float centerY, float radius, float beginAngle, float endAngle, const Color& color) NN_NOEXCEPT;
void FillPie(const Point2D& center, float radius, float beginAngle, float endAngle, const Color& color) NN_NOEXCEPT;

/**
 * @}
 *
 * @name    画像描画 API
 * @{
 */

/**
 * @brief   画像を読み込む
 *
 * @details
 *  指定した画像ファイルを読み込みます。
 *  対応形式は 24bit / 32bit Bitmap と JPEG です。
 *  画像が不要になった際は @ref DestroyImage を呼び出して破棄してください。
 *
 * @return  処理結果
 * @retval  ResultCode::Ok              成功
 * @retval  ResultCode::ReadFailed      ファイルの読み込みに失敗
 * @retval  ResultCode::DecodeFailed    非対応の形式
 */
ResultCode LoadImage(ImageData* pOutImage, const char* imageFile) NN_NOEXCEPT;

/**
 * @brief   空の画像データを作成
 *
 * @details
 *  描画ターゲットとして指定可能な、空の画像データを作成します。
 *  画像が不要になった際は @ref DestroyImage を呼び出して破棄してください。
 */
void CreateImage(ImageData* pOutImage, PixelFormat pixelFormat, int width, int height) NN_NOEXCEPT;
void CreateImage(ImageData* pOutImage, PixelFormat pixelFormat, const Size& size) NN_NOEXCEPT;

/**
 * @brief   メモリ上のイメージから画像データを作成
 *
 * @details
 *  メモリ上のファイルイメージから画像データを作成します。
 *  対応形式は 24bit / 32bit Bitmap と JPEG です。
 *  画像が不要になった際は @ref DestroyImage を呼び出して破棄してください。
 *
 * @return  処理結果
 * @retval  ResultCode::Ok              成功
 * @retval  ResultCode::DecodeFailed    非対応の形式
 */
ResultCode CreateImageByMemory(ImageData* pOutImage, const char* pImageData, size_t imageSize, ImageFormat imageFormat) NN_NOEXCEPT;
ResultCode CreateImageByMemory(ImageData* pOutImage, const char* pImageData, size_t imageSize, ImageFormat imageFormat, PixelFormat pixelFormat) NN_NOEXCEPT;

/**
 * @brief   画像データの中身を差し替える
 *
 * @details
 *  メモリ上の画像ファイルイメージから作成した画像データを、既存の画像データに上書きします。
 *  対応形式は 24bit / 32bit Bitmap, JPEG, Raw です。
 *  差し替え後の画像は、差し替え前の画像と同一のサイズ、ピクセルフォーマットである必要があります。
 *
 * @return  処理結果
 * @retval  ResultCode::Ok              成功
 * @retval  ResultCode::DecodeFailed    非対応の形式、または画像のサイズやフォーマットが一致しない
 */
ResultCode UpdateImageByMemory(ImageData* pImage, const char* pImageFileData, size_t imageFileSize, ImageFormat format) NN_NOEXCEPT;

/**
 * @brief   フレームバッファを画像データとして取得
 */
void CaptureToImage(ImageData* pOutImage) NN_NOEXCEPT;

/**
 * @brief   画像の破棄
 */
void DestroyImage(ImageData* pImage) NN_NOEXCEPT;

NN_FORCEINLINE
void UnloadImage(ImageData* pImage) NN_NOEXCEPT
{
    DestroyImage(pImage);
}

/**
 * @brief   画像を指定した位置に描画
 */
void DrawImage(const ImageData& image, float x, float y) NN_NOEXCEPT;
void DrawImage(const ImageData& image, const Point2D& point) NN_NOEXCEPT;

/**
 * @brief   画像を指定した位置・サイズで描画
 */
void DrawImage(const ImageData& image, float x, float y, float width, float height) NN_NOEXCEPT;
void DrawImage(const ImageData& image, const Point2D& point, const Size& size) NN_NOEXCEPT;

/**
 * @brief   画像を指定した四角形に変形して描画
 */
void DrawImage(const ImageData& image, const Quadrangle& quad) NN_NOEXCEPT;

/**
 * @brief   画像を指定した矩形で切り抜いて、指定した位置に描画
 */
void DrawImage(const ImageData& image, const Point2D& destPoint, const Rectangle& srcRect) NN_NOEXCEPT;

/**
 * @brief   画像を指定した矩形で切り抜いて、指定した位置・サイズに描画
 */
void DrawImage(const ImageData& image, const Rectangle& destRect, const Rectangle& srcRect) NN_NOEXCEPT;

/**
 * @brief   画像描画時の不透明度を取得
 */
float GetImageOpacity() NN_NOEXCEPT;

/**
 * @brief   画像描画時の不透明度を指定
 *
 * @details
 *  不透明度は 0.0f ～ 1.0f の間で指定してください。
 *  範囲外の値を指定した場合は、自動的に範囲内にクランプされます。
 */
void SetImageOpacity(float opacity) NN_NOEXCEPT;

/**
 * @brief   画像描画時の回転角度を指定
 *
 * @details
 *  角度はラジアンで指定してください。
 */
void SetImageRotation(float angle) NN_NOEXCEPT;

/**
 * @}
 *
 * @name    文字列描画 API
 * @{
 */

/**
 * @brief   フォントコンテキストを保存して取得
 */
void SaveFontContext(FontContext* pOutContext) NN_NOEXCEPT;

/**
 * @brief   フォントコンテキストの復元
 */
void RestoreFontContext(const FontContext& context) NN_NOEXCEPT;

/**
 * @brief   文字列の描画色を取得
 */
void GetTextColor(Color* pOutColor) NN_NOEXCEPT;

/**
 * @brief   文字列の描画色を設定
 */
void SetTextColor(const Color& color) NN_NOEXCEPT;

/**
 * @brief   文字列の描画倍率を取得
 */
void GetTextScale(float* pOutScaleX, float* pOutScaleY) NN_NOEXCEPT;

/**
 * @brief   文字列の描画倍率を設定
 */
void SetTextScale(float scaleX, float scaleY) NN_NOEXCEPT;

/**
 * @brief   文字列の描画倍率を設定
 */
NN_FORCEINLINE
void SetTextScale(float scale) NN_NOEXCEPT
{
    SetTextScale(scale, scale);
}

/**
 * @brief   UTF-8 文字列を描画
 *
 * @details
 *  指定した位置に、UTF-8 文字列を描画します。
 *  任天堂外字に対応しています。使用する場合は \uXXXX の形式で文字コードを指定してください。
 *  printf と同等の書式指定に対応しています。
 */
void DrawText(float x, float y, const char* format, ...) NN_NOEXCEPT;

/**
 * @brief   UTF-8 文字列を指定矩形内に描画
 *
 * @details
 *  指定した矩形に、UTF-8 文字列を描画します。
 *  printf と同等の書式指定に対応しています。
 *  横方向の位置は @a alignment で指定し、縦方向は中央寄せになります。
 */
void DrawText(float x, float y, float width, float height, TextAlignment alignment, const char* format, ...) NN_NOEXCEPT;

/**
 * @brief   UTF-32 文字列を描画
 *
 * @details
 *  指定した位置に、UTF-32 文字列を描画します。
 *  任天堂外字に対応しています。
 *  対応している外字の定義は @ref ExtensionCharacter を参照してください。
 */
void DrawText(float x, float y, const uint32_t* text) NN_NOEXCEPT;

/**
 * @brief   UTF-32 文字列を描画
 *
 * @details
 *  指定した位置に、UTF-32 文字列を描画します。
 *  横方向の位置は @a alignment で指定し、縦方向は中央寄せになります。
 */
void DrawText(float x, float y, float width, float height, TextAlignment alignment, const uint32_t* text) NN_NOEXCEPT;

/**
 * @brief   UTF-8 文字列の描画サイズを取得
 *
 * @details
 *  指定した UTF-8 文字列を描画する際に必要な描画領域のサイズを取得します。
 */
void GetTextDrawSize(Size* pOutSize, const char* format, ...) NN_NOEXCEPT;

/**
 * @brief   UTF-32 文字列の描画サイズを取得
 *
 * @details
 *  指定した UTF-32 文字列を描画する際に必要な描画領域のサイズを取得します。
 */
void GetTextDrawSize(Size* pOutSize, const uint32_t* text) NN_NOEXCEPT;

/**
 * @}
 */

}}  // nns::sgx
