﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <mutex>
#include <memory>

#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/hid/system/hid_UniquePad.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/hid/debug/hid_FirmwareUpdate.h>
#include <nn/result/result_HandlingUtility.h>

#include "ControllerFirmwareUpdater_PadManager.h"
#include "ControllerFirmwareUpdater_Util.h"

namespace
{

// UniquePad のコントローラー番号と NpadId の対応
const nn::hid::NpadIdType NpadIdList[] =
{
    nn::hid::NpadId::No1,
    nn::hid::NpadId::No2,
    nn::hid::NpadId::No3,
    nn::hid::NpadId::No4,
    nn::hid::NpadId::No5,
    nn::hid::NpadId::No6,
    nn::hid::NpadId::No7,
    nn::hid::NpadId::No8,
    nn::hid::NpadId::Handheld
};

const int NpadIdListCount = NN_ARRAY_SIZE(NpadIdList);

}  // anonymous

nn::Result PadManager::GetNpadId(nn::hid::NpadIdType* pOutId, int padIndex) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutId);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (padIndex >= GetPadCount())
    {
        NN_RESULT_THROW(nn::hid::system::ResultUniquePadDisconnected());
    }

    auto padId = m_PadInfos[padIndex].id;
    if (!IsSupportedDevice(padId))
    {
        // 非サポートデバイス
        NN_RESULT_THROW(nn::hid::system::ResultUniquePadDisconnected());
    }

    int controllerNumber;
    auto result = nn::hid::system::GetUniquePadControllerNumber(
        &controllerNumber,
        padId);
    if (nn::hid::system::ResultUniquePadNotAvailable::Includes(result) ||
        controllerNumber == 0)
    {
        // 割り当て無しは Handheld と見なす
        *pOutId = nn::hid::NpadId::Handheld;
    }
    else if (result.IsFailure())
    {
        NN_RESULT_THROW(result);
    }
    else
    {
        // コントローラー番号に対応する NpadId
        NN_ABORT_UNLESS_LESS(controllerNumber, NpadIdListCount);
        *pOutId = NpadIdList[controllerNumber - 1];
    }

    NN_RESULT_SUCCESS;
}

nn::Result PadManager::GetUniquePadId(nn::hid::system::UniquePadId* pOutId, int padIndex) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutId);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (padIndex >= GetPadCount())
    {
        NN_RESULT_THROW(nn::hid::system::ResultUniquePadDisconnected());
    }

    auto padId = m_PadInfos[padIndex].id;
    if (!IsSupportedDevice(padId))
    {
        // 非サポートデバイス
        NN_RESULT_THROW(nn::hid::system::ResultUniquePadDisconnected());
    }

    *pOutId = padId;

    NN_RESULT_SUCCESS;
}

bool PadManager::GetFirmwareVersion(nn::hid::system::FirmwareVersion* pOutVersion, int padIndex) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutVersion);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (padIndex >= GetPadCount())
    {
        return false;
    }

    return m_PadInfos[padIndex].versionReader.Get(pOutVersion);
}

bool PadManager::IsFirmwareUpdating(nn::hid::system::UniquePadId id) NN_NOEXCEPT
{
    for (const auto& pad : m_PadInfos)
    {
        if (pad.id._storage == id._storage &&
            pad.isUpdating)
        {
            return true;
        }
    }

    return false;
}

void PadManager::RequestInvalidateFirmwareVersion() NN_NOEXCEPT
{
    m_IsInvalidateVersionRequested = true;
}

void PadManager::UpdateFirmwareUpdatingDevice() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    for (int i = 0; i < m_PadCount; i++)
    {
        auto& pad = m_PadInfos[i];
        bool isUpdating;
        if (nn::hid::debug::IsFirmwareUpdatingDevice(&isUpdating, pad.id).IsSuccess())
        {
            pad.isUpdating = isUpdating;
        }
        else
        {
            pad.isUpdating = false;
        }
    }
}

void PadManager::Refresh() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    nn::hid::system::UniquePadId ids[NN_ARRAY_SIZE(m_PadInfos)];
    m_PadCount = nn::hid::system::ListUniquePads(ids, NN_ARRAY_SIZE(m_PadInfos));
    for (int i = 0; i < NN_ARRAY_SIZE(m_PadInfos); i++)
    {
        auto& pad = m_PadInfos[i];
        pad.Clear();
        if (i < m_PadCount)
        {
            pad.id = ids[i];
        }
    }

    UpdateFirmwareUpdatingDevice();

#if 0
    NN_LOG("[PadManager:%s] PadCount: %d\n", NN_CURRENT_FUNCTION_NAME, m_PadCount);
    for (int i = 0; i < m_PadCount; i++)
    {
        auto& pad = m_PadInfos[i];

        nn::hid::system::UniquePadInterface padInterface;
        nn::hid::system::GetUniquePadInterface(&padInterface, pad.id);
        NN_LOG("  Pad%d: ID=%d, Type=%d, Interface=%d\n",
            i,
            pad.id,
            nn::hid::system::GetUniquePadType(pad.id),
            padInterface);
    }
#endif
}

void PadManager::Update() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    for (int i = 0; i < GetPadCountMax(); i++)
    {
        auto& pad    = m_PadInfos[i];
        auto& reader = pad.versionReader;
        if (i < m_PadCount && IsSupportedDevice(pad.id))
        {
            if (!reader.IsValid())
            {
                // サポートしているデバイスを有効化
                reader.SetUniquePadId(pad.id);
            }
        }
        else
        {
            // サポートしていないデバイスはバージョンを読まない
            reader.Invalidate();
        }

        reader.Update();
    }
}

void PadManager::InvalidateFirmwareVersionIfNeeded() NN_NOEXCEPT
{
    if (m_IsInvalidateVersionRequested)
    {
        m_IsInvalidateVersionRequested = false;

        for (auto& pad : m_PadInfos)
        {
            pad.versionReader.Invalidate();
        }
    }
}
