﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>

#include "ControllerConnectionAnalyzer_Definitions.h"
#include "ControllerConnectionAnalyzer_NpadProperty.h"

namespace {

const char* BtmSlotModeStr[] = {
    "2",
    "4",
};

const char* BtmWlanModeStr[] = {
    "Local 4P",
    "Local 8P",
    "None",
};

}

template <typename T>
void Selectable<T>::Next() NN_NOEXCEPT
{
    if (m_IsWaiting)
    {
        return;
    }

    m_IsSelecting = true;
    m_SelectingItem = static_cast<T>((m_SelectingItem + 1) % m_MaxNum);
}

template <typename T>
void Selectable<T>::Previous() NN_NOEXCEPT
{
    if (m_IsWaiting)
    {
        return;
    }

    m_IsSelecting = true;
    m_SelectingItem = static_cast<T>((m_MaxNum + m_SelectingItem - 1) % m_MaxNum);
}

template <typename T>
bool Selectable<T>::Submit() NN_NOEXCEPT
{
    if (!m_IsWaiting && m_IsSelecting)
    {
        m_IsSelecting = false;
        if (m_SelectingItem != m_SubmittedItem)
        {
            m_SubmittedItem = m_SelectingItem;
            return true;
        }
    }
    return false;
}

template <typename T>
bool Selectable<T>::IsSelecting() NN_NOEXCEPT
{
    return m_IsSelecting;
}

template <typename T>
void Selectable<T>::SetSubmittedItem(T item) NN_NOEXCEPT
{
    m_SubmittedItem = item;
}

template <typename T>
T Selectable<T>::GetSubmittedItem() NN_NOEXCEPT
{
    return m_SubmittedItem;
}

template <typename T>
T Selectable<T>::GetSelectingItem() NN_NOEXCEPT
{
    return m_SelectingItem;
}

template <typename T>
void Selectable<T>::SetWaitFlag(bool isWait) NN_NOEXCEPT
{
    m_IsWaiting = isWait;
}

template <typename T>
bool Selectable<T>::IsWaiting() NN_NOEXCEPT
{
    return m_IsWaiting;
}

Selectable<BtmWlanMode>* NpadProperty::s_pWlanMode;

void NpadProperty::Initialize() NN_NOEXCEPT
{
    m_StyleNo = NpadStyleNo_Other;

    for (int i = 0; i < SixAxisSensorNpadStyleMax; ++i)
    {
        m_DeviceCount[i] = nn::hid::GetSixAxisSensorHandles(m_Handle[i], DeviceCountMax, *m_pId, NpadStyles[i]);

        for (int j = 0; j < m_DeviceCount[i]; ++j)
        {
            nn::hid::StartSixAxisSensor(m_Handle[i][j]);
        }
    }

    for (int i = 0; i < DeviceCountMax; ++i)
    {
        m_FramerateFirstSample[i] = 0;
        m_FramerateComputation[i] = 0;
        m_PacketDropPercentage[i] = 0;
        m_IsResetSample[i] = false;

        m_pSlotMode[i] = new Selectable<BtmSlotMode>(BtmSlotMode_Max, BtmSlotMode_2);
    }

    if (s_pWlanMode == nullptr)
    {
        s_pWlanMode = new Selectable<BtmWlanMode>(BtmWlanMode_Max, BtmWlanMode_None);
    }

    m_IrHandle = nn::irsensor::GetIrCameraHandle(*m_pId);
    m_IrIsInitialized = false;

    CreateLocalCommunicationConfig();
}

void NpadProperty::Update() NN_NOEXCEPT
{
    m_PreviousButtonSet = m_CurrentButtonSet;
    m_StyleNo = UpdateNpadState(&m_CurrentButtonSet, m_pId);

    if (m_StyleNo == NpadStyleNo_Other)
    {
        return;
    }

    if (IsTrigger(m_CurrentButtonSet, m_PreviousButtonSet, nn::hid::NpadButton::R::Mask))
    {
        int deviceIndex = (m_StyleNo == NpadStyleNo_JoyDual) ? 1 : 0;
        m_pSlotMode[deviceIndex]->Next();
    }

    if (IsTrigger(m_CurrentButtonSet, m_PreviousButtonSet, nn::hid::NpadButton::ZR::Mask))
    {
        s_pWlanMode->Next();
    }

    if (IsTrigger(m_CurrentButtonSet, m_PreviousButtonSet, nn::hid::NpadButton::A::Mask))
    {
        int deviceIndex = (m_StyleNo == NpadStyleNo_JoyDual) ? 1 : 0;
        bool changed = m_pSlotMode[deviceIndex]->Submit();
        if (changed)
        {
            switch (m_pSlotMode[deviceIndex]->GetSelectingItem())
            {
            case BtmSlotMode_2:
                SetIrsensorOnOff(deviceIndex, false);
                break;
            case BtmSlotMode_4:
                SetIrsensorOnOff(deviceIndex, true);
                break;
            default:
                break;
            }
        }

        changed = s_pWlanMode->Submit();
        if (changed)
        {
            s_pWlanMode->SetWaitFlag(true);
            switch (s_pWlanMode->GetSelectingItem())
            {
            case BtmWlanMode_None:
                SetLocalCommunicationMode(BtmWlanMode_None);
                break;
            case BtmWlanMode_Local4:
                SetLocalCommunicationMode(BtmWlanMode_Local4);
                break;
            case BtmWlanMode_Local8:
                SetLocalCommunicationMode(BtmWlanMode_Local8);
                break;
            default:
                break;
            }
        }
    }

    for (int i = 0; i < m_DeviceCount[m_StyleNo]; ++i)
    {
        if (m_StyleNo == NpadStyleNo_JoyDual && !IsConnected((JoyDualControllerId)i))
        {
            continue;
        }

        nn::hid::GetSixAxisSensorState(&m_State[i], m_Handle[m_StyleNo][i]);

        // ResetIntervalsInFrame フレームごとに Packet Loss 向けデータを初期化します
        if (m_IsResetSample[i])
        {
            m_IsResetSample[i] = false;
            m_FramerateFirstTick[i] = nn::os::GetSystemTick();
            m_FramerateFirstSample[i] = m_State[i].samplingNumber;
        }
        nn::os::Tick currentTick = nn::os::GetSystemTick() - m_FramerateFirstTick[i];
        int64_t currentSample = m_State[i].samplingNumber - m_FramerateFirstSample[i];

        // UpdateIntervals 周期ごとに、Packet Loss を更新します
        if (currentTick.ToTimeSpan().GetMilliSeconds() >= UpdateIntervals)
        {
            m_FramerateComputation[i] = currentSample / (float(currentTick.GetInt64Value()) / nn::os::GetSystemTickFrequency());

            const float ExpectedFrameRate = 1000.0f / m_State[i].deltaTime.GetMilliSeconds();
            // NN_LOG("%d ExpectedFrameRate= %f (%f)\n", static_cast<uint32_t>(*m_pId) + i, ExpectedFrameRate, m_FramerateComputation[i]);
            m_PacketDropPercentage[i] = 1.0f - std::min( std::max( m_FramerateComputation[i] / ExpectedFrameRate, 0.0f ), 1.0f);

            m_IsResetSample[i] = true;
        }
    }
}

float NpadProperty::Draw(nn::gfx::util::DebugFontTextWriter* pTextWriter,  float regionOriginX, float regionOriginY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    if (m_StyleNo == NpadStyleNo_Other)
    {
        return regionOriginY;
    }

    pTextWriter->SetTextColor(Colors::White);
    pTextWriter->SetScale(1.4f, 1.4f);

    const float TextHeight = pTextWriter->GetFontHeight();
    const float PaddingY = 4.0f;

    float posY = regionOriginY;

    for (int i = 0; i < m_DeviceCount[m_StyleNo]; ++i)
    {
        if (m_StyleNo == NpadStyleNo_JoyDual && !IsConnected((JoyDualControllerId)i))
        {
            continue;
        }

        posY += (TextHeight + PaddingY);
        pTextWriter->SetCursor(regionOriginX, posY);
        if (m_StyleNo == NpadStyleNo_FullKey)
        {
            pTextWriter->Print("| %s", GetNpadStyleLabel(m_StyleNo));
        }
        else
        {
            pTextWriter->Print("| %s %s", GetNpadStyleLabel(m_StyleNo), GetJoyconLabel((JoyDualControllerId)i));
        }

        pTextWriter->SetCursor(regionOriginX + LedOriginLeft, posY);
        pTextWriter->Print("|");
        Color on  = Colors::GreenYellow;
        Color off = Colors::Gray;
        const nn::Bit8 pattern = GetLedPattern();
        float ledLeft = pTextWriter->GetCursorX();
        ledLeft = DrawIndicator(pTextWriter, ledLeft + 4, posY, ( 0 != ( pattern & 0x01 ) ) ? on : off);
        ledLeft = DrawIndicator(pTextWriter, ledLeft + 4, posY, ( 0 != ( pattern & 0x02 ) ) ? on : off);
        ledLeft = DrawIndicator(pTextWriter, ledLeft + 4, posY, ( 0 != ( pattern & 0x04 ) ) ? on : off);
        ledLeft = DrawIndicator(pTextWriter, ledLeft + 4, posY, ( 0 != ( pattern & 0x08 ) ) ? on : off);

        pTextWriter->SetTextColor(Colors::White);
        pTextWriter->SetScale(1.4f, 1.4f);
        pTextWriter->SetCursor(regionOriginX + TimeSlotOriginLeft, posY);
        pTextWriter->Print("| ");

        if (m_pSlotMode[i]->IsSelecting())
        {
            pTextWriter->SetTextColor(Colors::Yellow);
        }
        else
        {
            pTextWriter->SetTextColor(Colors::White);
        }

        pTextWriter->Print("%s", BtmSlotModeStr[m_pSlotMode[i]->GetSelectingItem()]);

        pTextWriter->SetTextColor(Colors::White);
        pTextWriter->SetScale(1.4f, 1.4f);
        pTextWriter->SetCursor(regionOriginX + WlanCoexOriginLeft, posY);
        pTextWriter->Print("| ");

        if (s_pWlanMode->IsSelecting())
        {
            pTextWriter->SetTextColor(Colors::Yellow);
            pTextWriter->Print("%s", BtmWlanModeStr[s_pWlanMode->GetSelectingItem()]);
        }
        else
        {
            if (!s_pWlanMode->IsWaiting())
            {
                pTextWriter->SetTextColor(Colors::White);
                pTextWriter->Print("%s", BtmWlanModeStr[s_pWlanMode->GetSubmittedItem()]);
            }
        }

        pTextWriter->SetTextColor(Colors::White);
        pTextWriter->SetScale(1.4f, 1.4f);
        pTextWriter->SetCursor(regionOriginX + PlrOriginLeft, posY);
        pTextWriter->Print("| %3.2f %%", m_PacketDropPercentage[i] < 0.0f || m_PacketDropPercentage[i] > 1.0f ? 0.0f : m_PacketDropPercentage[i] * 100.0f);
    }

    return posY;
}

const float NpadProperty::DrawIndicator(nn::gfx::util::DebugFontTextWriter* pTextWriter, const float left, const float top, const Color& color) NN_NOEXCEPT
{
    pTextWriter->SetTextColor(color);
    pTextWriter->SetScale(1.0f, 1.0f);

    pTextWriter->SetCursor(left, top + 4.0f);
    pTextWriter->Print("■");

    return pTextWriter->GetCursorX();
}

bool NpadProperty::IsConnected() const NN_NOEXCEPT
{
    auto style = nn::hid::GetNpadStyleSet(*m_pId);
    return style.Test<nn::hid::NpadStyleFullKey>() || style.Test<nn::hid::NpadStyleJoyDual>();
}

bool NpadProperty::IsConnected(const JoyDualControllerId id) const NN_NOEXCEPT
{
    auto style = nn::hid::GetNpadStyleSet(*m_pId);
    if (style.Test<nn::hid::NpadStyleJoyDual>())
    {
        nn::hid::NpadJoyDualState state;
        nn::hid::GetNpadState(&state, *m_pId);
        if (id == JoyDualControllerId_Sakyo && state.attributes.Test<nn::hid::NpadJoyAttribute::IsLeftConnected>())
        {
            return true;
        }
        else if (id == JoyDualControllerId_Ukyo && state.attributes.Test<nn::hid::NpadJoyAttribute::IsRightConnected>())
        {
            return true;
        }
    }

    return false;
}

NpadStyleNo NpadProperty::UpdateNpadState(nn::hid::NpadButtonSet* pOut, const nn::hid::NpadIdType* pId) const NN_NOEXCEPT
{
    auto style = nn::hid::GetNpadStyleSet(*pId);
    NpadStyleNo styleNo;

    if (style.Test<nn::hid::NpadStyleFullKey>() == true)
    {
        styleNo = NpadStyleNo_FullKey;

        nn::hid::NpadFullKeyState state;
        nn::hid::GetNpadState(&state, *pId);

        *pOut = state.buttons;
    }
    else if (style.Test<nn::hid::NpadStyleJoyDual>() == true)
    {
        styleNo = NpadStyleNo_JoyDual;

        nn::hid::NpadJoyDualState state;
        nn::hid::GetNpadState(&state, *pId);

        *pOut = state.buttons;
    }
    else
    {
        styleNo = NpadStyleNo_Other;
    }

    return styleNo;
}

const char* const NpadProperty::GetNpadStyleLabel(const NpadStyleNo style) NN_NOEXCEPT
{
    if (NpadStyleNo_FullKey <= style && style <= NpadStyleNo_JoyDual)
    {
        static const char* const s_StyleNameLabels[] =
        {
            "FullKeyCon", "Joy Dual", "Joy Right", "Joy Left",
        };
        return s_StyleNameLabels[style];
    }
    return "Unknown";
}

const char* const NpadProperty::GetJoyconLabel(const JoyDualControllerId id) NN_NOEXCEPT
{
    static const char* const s_JoyconLabels[] =
    {
        "(L)", "(R)",
    };
    if (JoyDualControllerId_Sakyo <= id && id <= JoyDualControllerId_Ukyo)
    {
        return s_JoyconLabels[id];
    }
    return "(-)";
}

const nn::Bit8 NpadProperty::GetLedPattern() const NN_NOEXCEPT
{
    static const nn::Bit8 s_PatternBits[] =
    {
        0x01, 0x03, 0x07, 0x0f, 0x09, 0x05, 0x0d, 0x06
    };
    int i = static_cast<uint32_t>(*m_pId);
    return s_PatternBits[i];
}

void NpadProperty::SetIrsensorOnOff(int deviceIndex, bool powerOn) NN_NOEXCEPT
{
    if (powerOn)
    {
        if (!m_IrIsInitialized)
        {
            m_IrIsInitialized = true;
            nn::irsensor::Initialize(m_IrHandle);
            nn::irsensor::MomentProcessorConfig config;
            nn::irsensor::GetMomentProcessorDefaultConfig(&config);
            nn::irsensor::RunMomentProcessor(m_IrHandle, config);
        }
    }
    else
    {
        if (m_IrIsInitialized)
        {
            m_IrIsInitialized = false;
            nn::irsensor::StopImageProcessor(m_IrHandle);
            nn::irsensor::Finalize(m_IrHandle);
        }
    }
}

void NpadProperty::CreateLocalCommunicationConfig() NN_NOEXCEPT
{
    std::memset(&m_Network, 0, sizeof(nn::ldn::NetworkConfig));
    m_Network.intentId = BasicSampleIntentId;
    m_Network.channel = nn::ldn::AutoChannel;
    m_Network.nodeCountMax = nn::ldn::NodeCountMax;
    m_Network.localCommunicationVersion = 0;

    static const uint8_t passphrase[16] = {
        0x70, 0x54, 0x32, 0x69, 0x63, 0x70, 0x6a, 0x37,
        0x59, 0x52, 0x65, 0x39, 0x4f, 0x61, 0x52, 0x62
    };
    std::memset(&m_Security, 0, sizeof(nn::ldn::SecurityConfig));
    m_Security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Debug);
    std::memcpy(m_Security.passphrase, passphrase, sizeof(passphrase));
    m_Security.passphraseSize = static_cast<uint16_t>(sizeof(passphrase));

    std::memset(&m_User, 0, sizeof(nn::ldn::UserConfig));
    std::strncpy(m_User.userName, "ControllerConnectionAnalyzer", nn::ldn::UserNameBytesMax);
}

void NpadProperty::StartLocalCommunication() NN_NOEXCEPT
{
    nn::ldn::State state = nn::ldn::GetState();

    if (state == nn::ldn::State_AccessPoint)
    {
        if (nn::ldn::CreateNetwork(m_Network, m_Security, m_User).IsFailure())
        {
            s_pWlanMode->SetSubmittedItem(BtmWlanMode_None);
        }
        return;
    }
    else if (state == nn::ldn::State_AccessPointCreated)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
    }
    else // state != nn::ldn::State_AccessPoint
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ldn::OpenAccessPoint());
    }

    // State_AccessPoint まで進める
    StartLocalCommunication();
}

void NpadProperty::SetLocalCommunicationMode(BtmWlanMode mode) NN_NOEXCEPT
{
    nn::ldn::State state = nn::ldn::GetState();

    switch (mode)
    {
    case BtmWlanMode_None:
        {
            if (state == nn::ldn::State_AccessPointCreated)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ldn::DestroyNetwork());
            }

            if (state != nn::ldn::State_Initialized)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ldn::CloseAccessPoint());
            }
        }
        break;
    case BtmWlanMode_Local4:
        {
            m_Network.nodeCountMax = 4;
            StartLocalCommunication();
        }
        break;
    case BtmWlanMode_Local8:
        {
            m_Network.nodeCountMax = 8;
            StartLocalCommunication();
        }
        break;
    default:
        break;
    }

    s_pWlanMode->SetWaitFlag(false);
}

bool NpadProperty::IsTrigger(const nn::hid::NpadButtonSet current,
                             const nn::hid::NpadButtonSet previous,
                             const nn::hid::NpadButtonSet buttonMask) NN_NOEXCEPT
{
    return (((current ^ previous) & current) & buttonMask).IsAnyOn();
}
