﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>
#include <cmath>
#include <sstream>
#include <iomanip>

#include "BluetoothSettingTool_Types.h"
#include "BluetoothSettingTool_Pad.h"
#include "BluetoothSettingTool_UIControl.h"

namespace BluetoothSettingTool {

/*!--------------------------------------------------------------------------*
  @brief        選択器

 *---------------------------------------------------------------------------*/
template<typename T>
class Selector : public UIControl{
public:
    Selector()
    {
        m_CurrentItem = 0;
        m_CursorPosition = 0;
        SetCursor('>');
        m_CursorSmoothness = 1.1;
        m_LineHeight = DISPLAY_LINE_HEIGHT;
        m_FontSize = DEFAULT_FONT_WIDTH;
        m_Offset = 10;
        m_NextKey = Button::DOWN;
        m_BackKey = Button::UP;
    };

    virtual ~Selector(){}

    virtual void Register(string name, T* item)
    {
        m_Names.push_back(name);
        m_Items.push_back(item);
    }

    virtual T* GetSelectingItem()
    {
        if( m_Items.size() == 0 )
        {
            return nullptr;
        }
        return m_Items[m_CurrentItem];
    }

    virtual void SetCursor(char cursor)
    {
        this->m_Cursor[0] = cursor;
        this->m_Cursor[1] = '\0';
    }

    virtual void ShowImpl(Display& display)
    {
        int available = Height / m_LineHeight;
        static int start = 0;

        if( (start + available - 1) < m_CurrentItem)
        {
            start += m_CurrentItem - (start + available - 1);
        }
        if( start > m_CurrentItem )
        {
            start -= start - m_CurrentItem;
        }

        if(start > 0)
        {
            display.DrawText(GetX(), GetY() - m_FontSize / 2, "^");
        }
        if(m_Items.size() > start + available)
        {
            display.DrawText(GetX(), GetY() + ((available - 1) * m_LineHeight) + m_FontSize / 2, "v");
        }

        for(int i=start; i<m_Items.size() && i<start + available; i++)
        {
            int x = GetX();
            int y = GetY() - start * m_LineHeight;

            if(i == m_CurrentItem)
            {
                display.DrawText(x + m_Offset, y + (i * m_LineHeight), "> " + GetItemName(i) );
            }
            else
            {
                display.DrawText(x + m_Offset, y + (i * m_LineHeight), "  " + GetItemName(i) );
            }
        }
    }

    virtual void InputPad(Pad& key)
    {
        if( key.IsTrigger(m_BackKey) )
        {
            if(m_CurrentItem == 0)
            {
                m_CurrentItem = m_Items.size() - 1;
            }
            else
            {
                m_CurrentItem--;
            }
        }
        else if( key.IsHold(m_BackKey) )
        {
            if(m_CurrentItem == 0)
            {
            }
            else
            {
                m_CurrentItem--;
            }
        }
        else if( key.IsTrigger(m_NextKey) )
        {
            if(m_CurrentItem == m_Items.size() - 1)
            {
                m_CurrentItem = 0;
            }
            else
            {
                m_CurrentItem++;
            }
        }
        else if( key.IsHold(m_NextKey) )
        {
            if(m_CurrentItem == m_Items.size() - 1)
            {
            }
            else
            {
                m_CurrentItem++;
            }
        }
    }

    virtual string GetItemName()
    {
        return GetItemName(m_CurrentItem);
    }

    virtual string GetItemName(int index)
    {
        return m_Names[index];
    }

    virtual void Clear()
    {
        m_Items.clear();
        m_Names.clear();
        m_CurrentItem = 0;
        m_CursorPosition = 0;
    }

protected:
    vector<T*>      m_Items;
    vector<string>  m_Names;
    int m_CurrentItem;
    int m_CursorPosition;
    char m_Cursor[2];

    int m_Offset;
    int m_LineHeight;
    int m_FontSize;

    double m_CursorSmoothness;
    Button m_NextKey;
    Button m_BackKey;
};

}    // namespace BluetoothSettingTool
