﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <glv.h>
#include <nn/util/util_FormatString.h>

#include "Base/BcatTestApp_Glv.h"
#include "Base/BcatTestApp_Sequence.h"
#include "Base/BcatTestApp_ConsoleCore.h"
#include "Base/BcatTestApp_Hid.h"
#include "Base/BcatTestApp_Utility.h"
#include "Common/BcatTestApp_Dialog.h"
#include "Common/BcatTestApp_Console.h"
#include "Common/BcatTestApp_BaseDisplay.h"

namespace app
{

void ExecDialog_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecDialog           ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecDialog_Finalize  ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void DrawDialog( void* arg ) NN_NOEXCEPT;

ExecCallbackGroup ExecDialogGroup = {
    ExecDialog_Initialize,
    ExecDialog,
    ExecDialog_Finalize,
    nullptr,

    DrawDialog,
    nullptr,
    DrawPriority_Dialog,
    0
};

namespace
{
    app::DialogParam* g_CurrentParam = nullptr;

    // 横幅
    int g_DialogWidth = 0;
    int g_DialogHeight = 0;

    char g_ResultStr[32];

    // ダイアログ位置
    const int DialogPositionX = Position_Dialog.l;
    const int DialogPositionY = Position_Dialog.t;

    app::DialogReturnParam g_DialogReturnParam;
    app::DialogReturnParam* g_pReturnParam;

    int g_DialogCursor = 0;

    glv::Color g_DialogColor;
    glv::Color g_DialogFrameColor;

    const int DialogBackColor[] =
        {
            DrawColorSet_DialogCautionBack,
            DrawColorSet_DialogYesNoBack,
            DrawColorSet_DialogNoticeBack,
        };

    const int DialogFrameColor[] =
        {
            DrawColorSet_DialogCautionFrame,
            DrawColorSet_DialogYesNoFrame,
            DrawColorSet_DialogNoticeFrame,
        };

    int g_DelayFrame;
    bool g_IsDrawDialog;

    const int DialogFontSize = 22;
    const int DialogFontWidth = 22;
    const int DialogFontHeight = 22;

} //namespace

namespace
{
//----------------------------------------------------------------
// コンソールプリンタ
//
void DrawDialogConsoleString16( int x, int y, char16_t* string, char* attr, int stringLen, void* arg ) NN_NOEXCEPT
{
    app::DrawProportionalConsoleGeneric16( DialogPositionX + 80, DialogPositionY + 40,
                                           DialogFontSize,
                                           DialogFontWidth, DialogFontHeight,
                                           x, y, string, attr, stringLen, arg );
}
} //namespace

//----------------------------------------------------------------
// パラメータ領域の確保
//
DialogParam* AllocDialogParam( DialogParam::InfoType infoType ) NN_NOEXCEPT
{
    DialogParam* p = new DialogParam;
    p->infoType = infoType;
    p->pConsole16 = new app::FixedProportionalConsole<char16_t>;

    int32_t bufferSize = APP_CONSOLE_BUFFER_SIZE( p->consoleSizeX, p->consoleSizeY );
    p->pConsoleBuffer = new char[ bufferSize ];
    p->pConsole16->SetBuffer( p->pConsoleBuffer, bufferSize, p->consoleSizeX, p->consoleSizeY );
    p->pConsole16->SetPrinter( DrawDialogConsoleString16, nullptr );

    return p;
}

//----------------------------------------------------------------
// パラメータ領域の解放
//
void FreeDialogParam() NN_NOEXCEPT
{
    delete g_CurrentParam->pConsoleBuffer;
    delete g_CurrentParam->pConsole16;

    delete g_CurrentParam;
}

//----------------------------------------------------------------
// 文字列幅などを計算しておく
void CalculateDialogSize() NN_NOEXCEPT
{
    if ( g_CurrentParam->dialogMessage[0] != '\0' )
    {
        glv::Font f;
        f.size(22);
        f.lineSpacing(1.0f);
        f.tabSpaces(4);

        float w;
        float h;
        f.getBounds( w, h, g_CurrentParam->dialogMessage );
        g_DialogWidth = w;
        g_DialogHeight = h;
    }
    else if ( g_CurrentParam->dialogMessage16[0] != u'\0' )
    {
        glv::Font f;
        f.size(22);
        f.lineSpacing(1.0f);
        f.tabSpaces(4);

        float w;
        float h;
        f.getBounds( w, h, g_CurrentParam->dialogMessage16 );
        g_DialogWidth = w;
        g_DialogHeight = h;
    }
}

//----------------------------------------------------------------
// 描画コールバック
//
void DrawDialog( void* arg ) NN_NOEXCEPT
{
    DialogParam::InfoType infoType = g_CurrentParam->infoType;
    NN_ASSERT( 0 <= infoType && infoType <= DialogParam::InfoType_Notice );

    if ( ! g_IsDrawDialog )
    {
        return;
    }

    // フレーム
    app::DrawFrameRectangle( Position_Dialog, DialogBackColor[ infoType ], DialogFrameColor[ infoType ], 4 );

    // セットメッセージの描画
    if ( g_CurrentParam->dialogMessage[0] != '\0' )
    {
        app::SetGlvColor( app::ColorSet_White );
        glv::draw::text( g_CurrentParam->dialogMessage,
                         DialogPositionX + 300 - g_DialogWidth / 2,
                         DialogPositionY + 40,
                         22, 1.0F );
    }
    else if ( g_CurrentParam->dialogMessage16[0] != u'\0' )
    {
        app::SetGlvColor( app::ColorSet_White );
        glv::draw::text( g_CurrentParam->dialogMessage16,
                         DialogPositionX + 300 - g_DialogWidth / 2,
                         DialogPositionY + 40,
                         22, 1.0F );
    }

    // コンソールの描画
    g_CurrentParam->pConsole16->Display();

    switch( infoType )
    {
        case DialogParam::InfoType_Caution:
        case DialogParam::InfoType_Notice:
            {
                if ( g_CurrentParam->isResultSet )
                {
                    app::SetGlvColor( app::ColorSet_White );
                    glv::draw::text( g_ResultStr, DialogPositionX + 100, DialogPositionY + 110, 22, 1.0F );
                }
            }
            break;
        case DialogParam::InfoType_YesNo:
            {
                if ( g_DialogCursor == 0 )
                {
                    app::SetGlvColor( DrawColorSet_DialogSelected );
                    glv::draw::text( IsEnglish()? u"CONFIRM": u"はい",
                                     DialogPositionX + 180, DialogPositionY + 110, 22 );
                    glv::draw::text( ">", DialogPositionX + 160, DialogPositionY + 110, 22 );
                    app::SetGlvColor( DrawColorSet_DialogNotSelected );
                    glv::draw::text( IsEnglish()? u"CANCEL": u"いいえ",
                                     DialogPositionX + 330, DialogPositionY + 110, 22 );
                }
                else
                {
                    app::SetGlvColor( DrawColorSet_DialogNotSelected );
                    glv::draw::text( IsEnglish()? u"CONFIRM": u"はい",
                                     DialogPositionX + 180, DialogPositionY + 110, 22 );
                    app::SetGlvColor( DrawColorSet_DialogSelected );
                    glv::draw::text( IsEnglish()? u"CANCEL": u"いいえ",
                                     DialogPositionX + 330, DialogPositionY + 110, 22 );
                    glv::draw::text( ">", DialogPositionX + 310, DialogPositionY + 110, 22 );
                }
            }
            break;
        default:
            break;
    }

    uint64_t count = app::sequence::GetDrawFrameCount();
    switch( infoType )
    {
        case DialogParam::InfoType_Caution:
            {
                // 大きい「！」マーク
                app::SetGlvColor( ( count & 0x20 )?
                                  DrawColorSet_DialogExclamationMark1 :
                                  DrawColorSet_DialogExclamationMark2 );
                glv::draw::text( "!", DialogPositionX + 20, DialogPositionY + 20, 48 );
            }
            break;
        case DialogParam::InfoType_YesNo:
            {
                // 大きい「?」マーク
                app::SetGlvColor( DrawColorSet_DialogQuestionMark );
                glv::draw::text( "?", DialogPositionX + 20, DialogPositionY + 20, 48 );
            }
            break;
        case DialogParam::InfoType_Notice:
            {
                // 大きい「i」マーク
                app::SetGlvColor( DrawColorSet_DialogInformationMark );
                glv::draw::text( "i", DialogPositionX + 20, DialogPositionY + 20, 48 );
            }
            break;
        default:
            break;
    }
}

//----------------------------------------------------------------
// ダイアログの実行コールバック (開始処理)
//
void ExecDialog_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    // 引数受け取り
    g_CurrentParam = reinterpret_cast<DialogParam*>(arg);

    // Result もあるなら文字列作成
    if ( g_CurrentParam->isResultSet )
    {
        nn::util::SNPrintf( g_ResultStr, sizeof(g_ResultStr), "ErrorCode: 0x%08x, %03d-%04d",
                            g_CurrentParam->result.GetInnerValueForDebug(),
                            g_CurrentParam->result.GetModule(),
                            g_CurrentParam->result.GetDescription() );
    }

    // エラー文字列幅の事前計算
    CalculateDialogSize();

    g_DialogReturnParam.selectedItem = DialogReturnParam::SelectedItem_No;
    g_DialogCursor = ( g_CurrentParam->startPosition == DialogParam::StartPosition_No )? 1: 0;

    // ディレイフレーム
    g_DelayFrame = 0;
    g_IsDrawDialog = true;
}

//----------------------------------------------------------------
// ダイアログの実行コールバック (終了処理)
//
void ExecDialog_Finalize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    FreeDialogParam();
}

//----------------------------------------------------------------
// タイミングをずらして戻る間の実行コールバック
//
void ExecDialog_Delaying( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    if ( ++ g_DelayFrame >= g_CurrentParam->delayFrame )
    {
        app::sequence::Return( g_pReturnParam );
    }
}

//----------------------------------------------------------------
// ダイアログから戻る指示 (ディレイ考慮)
//
void ExecDialog_Return() NN_NOEXCEPT
{
    if ( g_CurrentParam->delayFrame == 0 )
    {
        app::sequence::Return( g_pReturnParam );
    }
    else
    {
        g_IsDrawDialog = false;
        app::sequence::SlideTo( ExecDialog_Delaying );
    }
}

//----------------------------------------------------------------
// ダイアログの実行コールバック
//
void ExecDialog( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    DialogParam::InfoType infoType = g_CurrentParam->infoType;
    NN_ASSERT( 0 <= infoType && infoType <= DialogParam::InfoType_Notice );

    app::Pad pad( events );

    // A ボタン
    if ( pad.IsButtonDownA() )
    {
        if ( g_CurrentParam->infoType == DialogParam::InfoType_YesNo )
        {
            g_DialogReturnParam.selectedItem = (g_DialogCursor == 0)? DialogReturnParam::SelectedItem_Yes: DialogReturnParam::SelectedItem_No;
            g_pReturnParam = &g_DialogReturnParam;
        }
        else
        {
            g_pReturnParam = nullptr;
        }

        ExecDialog_Return();
        return;
    }
    // B ボタン
    if ( pad.IsButtonDownB() )
    {
        if ( g_CurrentParam->infoType == DialogParam::InfoType_YesNo )
        {
            g_DialogReturnParam.selectedItem = DialogReturnParam::SelectedItem_No;
            g_pReturnParam = &g_DialogReturnParam;
        }
        else
        {
            g_pReturnParam = nullptr;
        }

        ExecDialog_Return();
        return;
    }
    // 左 ボタン
    if ( pad.IsButtonDownLeft() )
    {
        g_DialogCursor = 0;
    }
    // 右 ボタン
    if ( pad.IsButtonDownRight() )
    {
        g_DialogCursor = 1;
    }
}

} //namespace app
