﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "BcatTestApp_Common.h"
#include "BcatTestApp_TopMenu.h"

#include <nn/nifm.h>
#include <nn/nn_TimeSpan.h>
#include <nn/time/time_StandardUserSystemClock.h>

namespace app
{
void ExecNetworkConnection_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecNetworkConnection           ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecNetworkConnection_Finalize  ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void DrawNetworkConnection( void* arg ) NN_NOEXCEPT;

ExecCallbackGroup ExecNetworkConnectionGroup = {
    ExecNetworkConnection_Initialize,
    ExecNetworkConnection,
    ExecNetworkConnection_Finalize,
    nullptr,

    DrawNetworkConnection,
    nullptr,
    DrawPriority_NetworkConnection,
    0
};
namespace
{
    // ウィンドウ位置
    const int PositionX = Position_NetworkConnection.l;
    const int PositionY = Position_NetworkConnection.t;

    // コンソール関連
    const int ConsoleWidth = 30;
    const int ConsoleHeight = 4;
    const int ConsoleBufferSize = APP_CONSOLE_BUFFER_SIZE_CHAR16( ConsoleWidth, ConsoleHeight );
    char* g_pConsoleBuffer;
    app::FixedProportionalConsole<char16_t>* g_pConsole;

    // フォント情報
    const int FontSize = 22;
    const int FontWidth = 22;
    const int FontHeight = 22;

    // 処理スレッド
    nn::os::ThreadType g_Thread;
    static const size_t ThreadStackSize = 32 * 1024;
    char* g_AllocatedStack;
    void* g_pStack;

    // 時刻
    nn::time::PosixTime g_StartTime;

    // 接続か、切断か
    enum ActionType
    {
        ActionType_Connect = 0,
        ActionType_Disconnect,
    };
    ActionType g_ActionType;

    // 状態変数
    bool g_IsThreadStarted = false;
    bool g_IsFinished = false;

    // 表示するメッセージ
    enum MessageType
    {
        MessageType_None = 0,
        MessageType_Connecting,
        MessageType_Connected,
        MessageType_ConnectionError,
        MessageType_Disconnecting,
        MessageType_Disconnected,
    };
    MessageType g_MessageType;
    MessageType g_ResultMessageType;
    nn::Result g_ConnectionResult;

} //namespace

//----------------------------------------------------------------
// ネットワーク接続切り替えのコンソールプリンタ
//
void DrawNetworkConnectionConsoleString( int x, int y, char16_t* string, char* attr, int stringLen, void* arg ) NN_NOEXCEPT
{
    app::DrawProportionalConsoleGeneric16( PositionX + 20, PositionY + 20,
                                           FontSize,
                                           FontWidth, FontHeight,
                                           x, y, string, attr, stringLen, arg );
}
//----------------------------------------------------------------
// ネットワーク接続切り替えの描画コールバック
//
void DrawNetworkConnection( void* arg ) NN_NOEXCEPT
{
    // 枠
    app::DrawFrameRectangle( Position_NetworkConnection, DrawColorSet_NetworkConnectionBack, DrawColorSet_NetworkConnectionFrame, DrawFrameWidth );

    g_pConsole->Clear();
    g_pConsole->SetAttribute( app::ConsoleColor_White );
    switch( g_MessageType )
    {
        case MessageType_Connecting:
            {
            g_pConsole->PrintfEx( 1, 1, u"ネットワークに接続しています" );
            g_pConsole->PrintfEx( 1, 2, u"B ボタンで中断できます" );

            // 時間がかかるかもしれないので簡単なアニメーション
            int n = (app::sequence::GetDrawFrameCount() / 0x10) % 3;
            g_pConsole->PrintfEx( 10 + n, 3, u"*" );
            }
            break;
        case MessageType_Connected:
            g_pConsole->PrintfEx( 1, 1, u"接続しました" );
            g_pConsole->PrintfEx( 6, 4, u"A ボタンを押してください" );
            break;
        case MessageType_Disconnecting:
            g_pConsole->PrintfEx( 1, 1, u"切断しています" );
            break;
        case MessageType_Disconnected:
            g_pConsole->PrintfEx( 1, 1, u"切断しました" );
            g_pConsole->PrintfEx( 6, 4, u"A ボタンを押してください" );
            break;
        default:
            break;
    }

    // コンソール描画
    g_pConsole->Display();
}

//----------------------------------------------------------------
// ネットワーク接続切り替えの実処理スレッドエントリ関数
//
void NetworkConnectionThreadFunc( void* arg ) NN_NOEXCEPT
{
    if ( g_ActionType == ActionType_Connect )
    {
        while(1)
        {
            app::GetSystemConsole().Printf("nn::nifm::SubmitNetworkRequestAndWait()\n");
            nn::nifm::SubmitNetworkRequestAndWait();

            if ( IsConnected() )
            {
                SetNetworkConnectionStatus( true );
                app::GetSystemConsole().Printf("nn::nifm::IsNetworkAvailable: true\n" );
                g_ResultMessageType = MessageType_Connected;
                break;
            }
            else
            {
                SetNetworkConnectionStatus( false );
                nn::Result result = nn::nifm::HandleNetworkRequestResult();

                if ( nn::nifm::ResultErrorHandlingCompleted::Includes( result ) )
                {
                    nn::nifm::CancelNetworkRequest();
                }
                else
                {
                    app::GetSystemConsole().Printf("nn::nifm::IsNetworkAvailable: false\n" );
                    app::PrintErrorCode( result );
                    g_ResultMessageType = MessageType_ConnectionError;
                    g_ConnectionResult = result;
                    break;
                }
            }
        }
    }
    else
    {
        app::GetSystemConsole().Printf("nn::nifm::CancelNetworkRequest()\n");
        nn::nifm::CancelNetworkRequest();
        SetNetworkConnectionStatus( false );
        g_ResultMessageType = MessageType_Disconnected;
    }

    g_IsFinished = true;
}

//----------------------------------------------------------------
// ネットワーク接続切り替えのキー説明
void PrintNetworkConnectionHelp() NN_NOEXCEPT
{
    app::GetHelpConsole().Clear();
    app::GetHelpConsole().PrintfEx( 2, 0,
                                    u"@1操作説明:@7 @4[B]@7...戻る" );
}
//----------------------------------------------------------------
// ネットワーク接続切り替え(開始処理)
//
void ExecNetworkConnection_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    g_ActionType = app::IsConnected()? ActionType_Disconnect: ActionType_Connect;
    g_IsFinished = false;
    g_MessageType = (g_ActionType == ActionType_Connect)? MessageType_Connecting: MessageType_Disconnecting;

    // コンソール作成
    g_pConsole = new app::FixedProportionalConsole<char16_t>;
    g_pConsoleBuffer = new char[ ConsoleBufferSize ];
    g_pConsole->SetBuffer( g_pConsoleBuffer, ConsoleBufferSize, ConsoleWidth, ConsoleHeight );
    g_pConsole->SetPrinter( DrawNetworkConnectionConsoleString, nullptr );

    // キー説明
    PrintNetworkConnectionHelp();

    // 開始タイミング
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetCurrentTime(&g_StartTime) );

    // 処理スレッド用スタック
    g_AllocatedStack = new char[ ThreadStackSize + nn::os::ThreadStackAlignment ];
    g_pStack = reinterpret_cast<void*>( (reinterpret_cast<uintptr_t>(g_AllocatedStack) + (nn::os::ThreadStackAlignment - 1)) & ~(nn::os::ThreadStackAlignment - 1) );

    // 処理スレッドを開始
    g_IsThreadStarted = true;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_Thread, NetworkConnectionThreadFunc, nullptr, g_pStack, ThreadStackSize, nn::os::DefaultThreadPriority ) );
    nn::os::SetThreadNamePointer( &g_Thread, "BcatTest_NetworkConnection" );
    nn::os::StartThread( &g_Thread );
}
//----------------------------------------------------------------
// ネットワーク接続切り替え(終了処理)
//
void ExecNetworkConnection_Finalize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    if ( g_IsThreadStarted )
    {
        nn::os::WaitThread( &g_Thread );
        nn::os::DestroyThread( &g_Thread );
    }

    // 各領域を破棄
    delete[] g_AllocatedStack;
    delete[] g_pConsoleBuffer;
    delete g_pConsole;
}

namespace {
//----------------------------------------------------------------
// ダイアログからの戻り
void AfterDialog( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    app::sequence::Return();
}
}

//----------------------------------------------------------------
// ネットワーク接続切り替え(処理の終了、ボタンまち)
//
void ExecNetworkConnection_AfterFinish( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    app::Pad pad( events );

    // A か B で戻る
    if ( pad.IsButtonDownA() || pad.IsButtonDownB() )
    {
        app::sequence::Return();
    }
}

//----------------------------------------------------------------
// ネットワーク接続切り替え
//
void ExecNetworkConnection( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    app::Pad pad( events );

    nn::time::PosixTime currentTime;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetCurrentTime(&currentTime) );
    nn::TimeSpan span = currentTime - g_StartTime;

    // 最初 1秒はボタン効かない
    if ( span < nn::TimeSpan::FromMilliSeconds(1000) )
    {
        return;
    }

    // 終了時
    if ( g_IsFinished )
    {
        g_MessageType = g_ResultMessageType;

        if ( g_MessageType == MessageType_ConnectionError )
        {
            app::SetErrorDialog( u"ネットワークへの接続に失敗しました", g_ConnectionResult );
            app::sequence::SetFromCall( AfterDialog, nullptr );
        }
        else
        {
            app::sequence::SlideTo( ExecNetworkConnection_AfterFinish );
        }
        return;
    }

    // B で戻る
    if ( pad.IsButtonDownB() )
    {
        if ( g_ActionType == ActionType_Connect )
        {
            app::GetSystemConsole().Printf("Cancel connection.\n");
            app::GetSystemConsole().Printf("nn::nifm::CancelNetworkRequest()\n");
            nn::nifm::CancelNetworkRequest();
        }
        app::sequence::Return();
        return;
    }
}

} // namespace app
