﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "BcatTestApp_Common.h"
#include "BcatTestApp_TopMenu.h"

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/account/account_Selector.h>

namespace app
{
void ExecInitializeSaveDataMenu_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecInitializeSaveDataMenu           ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecInitializeSaveDataMenu_Finalize  ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void DrawInitializeSaveDataMenu( void* arg ) NN_NOEXCEPT;

ExecCallbackGroup ExecInitializeSaveDataMenuGroup = {
    ExecInitializeSaveDataMenu_Initialize,
    ExecInitializeSaveDataMenu,
    ExecInitializeSaveDataMenu_Finalize,
    nullptr,

    DrawInitializeSaveDataMenu,
    nullptr,
    DrawPriority_InitializeSaveDataMenu,
    0
};

//----------------------------------------------------------------
namespace
{
    app::Menu g_Menu;
    enum
    {
        MenuIndex_InitSave = 0,
        MenuIndex_Back,
    };
    int g_InitialMenuIndex = MenuIndex_Back;
    const int MenuMax = MenuIndex_Back + 1;

} // namespace

//----------------------------------------------------------------
// セーブデータ初期化メニュー用描画コールバック
//
void DrawInitializeSaveDataMenu( void* arg ) NN_NOEXCEPT
{
    uint64_t count = app::sequence::GetDrawFrameCount();
    app::FixedProportionalConsole<char16_t>* p = g_Menu.GetConsole16();

    p->PrintfEx( 2, 2,
                 (count & 0x20)? app::ConsoleColor_Red: app::ConsoleColor_Yellow,
                 u"初期化してよろしいですか？" );
}

//----------------------------------------------------------------
// セーブデータ初期化メニューのメニュー文字列描画
//
void DrawInitializeSaveDataMenuItems() NN_NOEXCEPT
{
    app::FixedProportionalConsole<char16_t>* p = g_Menu.GetConsole16();
    if ( p )
    {
        app::SetGlvColor( app::ColorSet_White );
        p->PrintfEx( 1, 0,
                     u"-- セーブデータ初期化メニュー --" );
        p->PrintfEx( 2, 4,
                     app::ConsoleColor_White,
                     u"セーブデータを初期化する" );
        p->PrintfEx( 2, 5,
                     app::ConsoleColor_White,
                     u"戻る" );
    }
}

//----------------------------------------------------------------
// セーブデータ初期化メニューのキー説明
void PrintInitializeSaveDataMenuHelp() NN_NOEXCEPT
{
    app::GetHelpConsole().Clear();
    app::GetHelpConsole().PrintfEx( 2, 0,
                                    u"@1操作説明:@7 @6[上][下]@7...カーソル  @4[A]@7...決定  @2[B]@7...戻る" );
}

namespace
{
//----------------------------------------------------------------
// ダイアログからの戻り
void FromDialog( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    app::sequence::JumpTo( app::ExecTopMenuGroup );
}

// ユーザ選択しなかったというダイアログ
void SetDialogUserNotSelected() NN_NOEXCEPT
{
    app::DialogParam* p = app::AllocDialogParam( app::DialogParam::InfoType_Notice );
    p->GetConsole()->PrintfEx( 0, 0, u"セーブデータを初期化を中断しました。" );
    app::sequence::Call( app::ExecDialogGroup, p );
    app::sequence::SetFromCall( FromDialog, nullptr );
}

// 初期化したというダイアログ
void SetDialogInitialized() NN_NOEXCEPT
{
    app::DialogParam* p = app::AllocDialogParam( app::DialogParam::InfoType_Notice );
    p->GetConsole()->PrintfEx( 0, 0, u"セーブデータを初期化しました。" );
    app::sequence::Call( app::ExecDialogGroup, p );
    app::sequence::SetFromCall( FromDialog, nullptr );
}
} // namespace

//----------------------------------------------------------------
// セーブデータの初期化
//
void InitializeSaveData() NN_NOEXCEPT
{
    nn::Result result;

    // ユーザの選択
    if ( ! IsUserValid() )
    {
        SelectUser();
        if ( ! IsUserValid() )
        {
            SetDialogUserNotSelected();
            return;
        }
    }

    // セーブデータが存在するか？
    bool isExist = nn::fs::IsSaveDataExisting( app::GetUser() );
    app::GetSystemConsole().Printf("nn::fs::IsSaveDataExisting() = %d\n", isExist );

    if ( ! isExist )
    {
        // セーブデータ作成
        result = nn::fs::EnsureSaveData( app::GetUser() );
        app::GetSystemConsole().Printf("nn::fs::EnsureSaveData() :" );
        app::PrintErrorCode( result );

        // 容量が足りなかった場合
        // if ( nn::fs::ResultUsableSpaceNotEnough::Includes(result) ) で判定
    }

    isExist = nn::fs::IsSaveDataExisting( app::GetUser() );
    app::GetSystemConsole().Printf("nn::fs::IsSaveDataExisting() = %d\n", isExist );
    if ( isExist )
    {
        result = nn::fs::MountSaveData( app::MountSaveDataName, app::GetUser() );
        app::GetSystemConsole().Printf("nn::fs::MountSaveData(name,user) : ");
        app::PrintErrorCode( result );
        // 失敗時は必ずアボートすること、とAPIリファレンスに書かれている

        // セーブデータ初期化処理
        app::GetSystemConsole().Printf("nn::fs::CleanDirectoryRecursively()\n");
        char path[ app::PathLengthMax ];
        nn::util::SNPrintf( path, sizeof(path), "%s:/", app::MountSaveDataName );
        nn::fs::CleanDirectoryRecursively( path );

        app::GetSystemConsole().Printf("nn::fs::CommitSaveData()\n");
        nn::fs::CommitSaveData( app::MountSaveDataName );

        app::GetSystemConsole().Printf("nn::fs::Unmount()\n");
        nn::fs::Unmount( app::MountSaveDataName );
    }

    // 初期化しましたダイアログ
    SetDialogInitialized();
}

//----------------------------------------------------------------
// セーブデータ初期化メニュー(開始処理)
//
void ExecInitializeSaveDataMenu_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    // メニュー作成
    g_Menu.CreateConsole( app::ConsoleSize_Char16_t, app::Position_InitializeSaveDataMenu, 100, 7, 1, 4, 24, 4, 300, 0x230 );
    g_Menu.SetBackColor( app::ColorSet_DarkGreen, app::ColorSet_Green, app::DrawFrameWidth );
    g_Menu.SetItemParameter( MenuMax, g_InitialMenuIndex );
    DrawInitializeSaveDataMenuItems();

    // キー説明
    PrintInitializeSaveDataMenuHelp();
}
//----------------------------------------------------------------
// セーブデータ初期化メニュー(終了処理)
//
void ExecInitializeSaveDataMenu_Finalize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    g_Menu.DestroyConsole();
}

//----------------------------------------------------------------
// セーブデータ初期化メニュー
//
void ExecInitializeSaveDataMenu( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    g_Menu.SetHidEvent( &events );
    g_Menu.Update();

    switch( g_Menu.CheckButtonOk() )
    {
        case MenuIndex_InitSave:
            {
                // セーブデータの初期化
                InitializeSaveData();
                return;
            }
            break;
        case MenuIndex_Back:
            {
                app::sequence::JumpTo( ExecTopMenuGroup );
                return;
            }
            break;
        default:
            break;
    }


    // B ボタンで戻る
    if ( g_Menu.CheckButtonCancel() )
    {
        app::sequence::JumpTo( ExecTopMenuGroup );
    }

    if ( g_Menu.IsUpdated() )
    {
        DrawInitializeSaveDataMenuItems();
        g_Menu.ClearUpdated();
    }
}
} //namespace app


