﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "BcatTestApp_Common.h"
#include "BcatTestApp_StorageMenu.h"
#include "BcatTestApp_FileMenu.h"

#include <vector>
#include <nn/bcat/bcat_DeliveryCacheDirectory.h>

namespace app
{
void ExecDirectoryMenu_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecDirectoryMenu           ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecDirectoryMenu_Finalize  ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void DrawDirectoryMenu( void* arg ) NN_NOEXCEPT;

ExecCallbackGroup ExecDirectoryMenuGroup = {
    ExecDirectoryMenu_Initialize,
    ExecDirectoryMenu,
    ExecDirectoryMenu_Finalize,
    nullptr,

    DrawDirectoryMenu,
    nullptr,
    DrawPriority_DirectoryMenu,
    0
};

void ExecDirectoryActionMenu_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecDirectoryActionMenu           ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecDirectoryActionMenu_Finalize  ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void DrawDirectoryActionMenu( void* arg ) NN_NOEXCEPT;

ExecCallbackGroup ExecDirectoryActionMenuGroup = {
    ExecDirectoryActionMenu_Initialize,
    ExecDirectoryActionMenu,
    ExecDirectoryActionMenu_Finalize,
    nullptr,

    DrawDirectoryActionMenu,
    nullptr,
    DrawPriority_DirectoryActionMenu,
    0
};

//----------------------------------------------------------------
namespace
{
    //---------------- ディレクトリメニュー用
    enum {
        Sequence_Init = 0,
        Sequence_FromErrorDialog
    };

    // コンソール情報
    const int FixConsoleWidth = 78;
    const int FixConsoleHeight = 33;
    const int FixConsoleBufferSize = APP_CONSOLE_BUFFER_SIZE( FixConsoleWidth, FixConsoleHeight );
    const int PropConsoleWidth = 100;
    const int PropConsoleHeight = 33;
    const int PropConsoleBufferSize = APP_CONSOLE_BUFFER_SIZE( PropConsoleWidth, PropConsoleHeight );
    char* g_FixConsoleBuffer;
    char* g_PropConsoleBuffer;
    bool g_IsConsoleCreated = false;
    app::FixedConsole<char16_t> g_FixConsole;
    app::FixedProportionalConsole<char16_t> g_PropConsole;

    const float FontSize = 24.0F;
    const float GridSizeX = 18.0F;
    const float GridSizeY = 28.0F;

    // ディレクトリ情報
    nn::bcat::DirectoryName g_DirectoryName[ nn::bcat::DeliveryCacheDirectoryCountMax ];
    std::vector<nn::bcat::DirectoryName*> g_DirectoryNameList( nn::bcat::DeliveryCacheDirectoryCountMax );

    // ファイル情報
    nn::bcat::DeliveryCacheDirectoryEntry g_FileEntry[ nn::bcat::DeliveryCacheFileCountMaxPerDirectory ];
    int g_FileEntryNum = 0;

    const int DirListViewMax = 12;
    int g_DirListCursor = 0;
    int g_DirListViewTop = 0;
    int g_DirListNum = 0;
    bool g_IsDirListUpdate = false;
    int g_CurDirectory = -1;

    // 表示用
    int ConsoleOffsetX = 10;
    int ConsoleOffsetY = 10;
    bool g_IsDrawDirectoryMenuCallback = true;

    // ファイルメニュー呼び出し用
    app::FileMenuParameter g_FileMenuParameter;

    // ディレクトリ操作用
    nn::bcat::DeliveryCacheDirectory g_BcatDirectory;

    //---------------- ディレクトリアクションメニュー用
    app::Menu g_DirectoryActionMenu;

    enum
    {
        MenuIndex_OpenDirectory = 0,
        MenuIndex_EnumerateEntry,
        MenuIndex_FileMenu,
        MenuIndex_CloseDirectory
    };
    int g_InitialMenuIndex = MenuIndex_OpenDirectory;
    const int MenuMax = MenuIndex_CloseDirectory + 1;

    bool g_IsDirectoryOpened = false;
    int g_OpenedDirectory;
    bool g_IsEnumeratedDirectory = false;

} // namespace

namespace
{
//----------------------------------------------------------------
// 固定コンソールプリンタ
//
void DrawFixConsoleString( int x, int y, char16_t* string, char* attr, int stringLen, void* arg ) NN_NOEXCEPT
{
    DrawFixedConsoleGeneric16( Position_DirectoryMenu.l + ConsoleOffsetX,
                               Position_DirectoryMenu.t + ConsoleOffsetY,
                               FontSize, GridSizeX, GridSizeY,
                               x, y, string, attr, stringLen, arg );
}
//----------------------------------------------------------------
// プロポーショナルコンソールプリンタ
//
void DrawPropConsoleString( int x, int y, char16_t* string, char* attr, int stringLen, void* arg ) NN_NOEXCEPT
{
    DrawProportionalConsoleGeneric16( Position_DirectoryMenu.l + ConsoleOffsetX,
                                      Position_DirectoryMenu.t + ConsoleOffsetY,
                                      FontSize, GridSizeX, GridSizeY,
                                      x, y, string, attr, stringLen, arg );
}
}

//----------------------------------------------------------------
// ディレクトリメニュー用描画コールバック
//
void DrawDirectoryMenu( void* arg ) NN_NOEXCEPT
{
    // タイトル枠
    app::DrawFrameRectangle( Position_DirectoryMenu, DrawColorSet_DirectoryMenuBack, DrawColorSet_DirectoryMenuFrame, 3 );

    if ( ! g_IsDrawDirectoryMenuCallback )
    {
        return;
    }

    // カーソル
    if ( g_DirListNum > 0 )
    {
        int y = Position_DirectoryMenu.t + (g_DirListCursor * GridSizeY) + (4 * GridSizeY) + ConsoleOffsetY;
        app::DrawRectangle( Position_DirectoryMenu.l + GridSizeX * 3, y,
                            Position_DirectoryMenu.r - GridSizeX * 2, y + GridSizeY,
                            app::DrawColorSet_ItemCursorBack );

        // スクロールバー
        app::DrawScrollBar( g_DirListNum, DirListViewMax, g_DirListViewTop, ScrollBarDefaultViewWidth, Position_DirectoryListScrollBar );
    }

    // コンソール描画
    if ( g_IsConsoleCreated )
    {
        g_FixConsole.Display();
        g_PropConsole.Display();
    }
}

//----------------------------------------------------------------
void PrintDirectoryCursor() NN_NOEXCEPT
{
    // ディレクトリがあるならカーソル表示
    if ( g_DirListNum > 0 )
    {
        g_FixConsole.PrintfEx( 1, g_DirListCursor + 4, app::ConsoleColor_Yellow, u">" );
    }
}

//----------------------------------------------------------------
void PrintDirectoryList() NN_NOEXCEPT
{
    g_FixConsole.Clear();
    g_PropConsole.Clear();

    // ディレクトリ数
    g_PropConsole.PrintfEx( 0, 0, app::ConvertToChar16_t("Directory List :  %d dir(s)", g_DirListNum) );

    // ディレクトリリスト
    if ( g_DirListNum == 0 )
    {
        g_PropConsole.PrintfEx( 2, 3, app::ConsoleColor_Yellow, u"<ディレクトリがありません>" );
        return;
    }

    g_PropConsole.PrintfEx( 2, 2, u"No." );
    g_PropConsole.PrintfEx( 11, 2, u"Directory name" );
    g_FixConsole.PrintfEx( 2, 3, u"---  -------------------------------" );

    // ディレクトリ名
    for( int i=0; i<DirListViewMax; i++ )
    {
        int num = g_DirListViewTop + i;
        if ( num >= g_DirListNum )
        {
            break;
        }

        bool isSelected = ( g_IsDirectoryOpened && g_OpenedDirectory == num );
        g_FixConsole.PrintfEx( 2, i + 4,
                               isSelected? app::ConsoleColor_Yellow: app::ConsoleColor_White,
                               app::ConvertToChar16_t( "%3d: %s", num, g_DirectoryNameList[num]->value ) );
        if ( isSelected )
        {
            g_FixConsole.PrintfEx( 6, i + 4, app::ConsoleColor_Yellow, u"*" );
        }
    }
}

//----------------------------------------------------------------
// ディレクトリメニューのキー説明
void PrintDirectoryMenuHelp() NN_NOEXCEPT
{
    app::GetHelpConsole().Clear();
    app::GetHelpConsole().PrintfEx( 2, 0,
                                    u"@1操作説明:@7 @6[上][下]@7...カーソル  @4[A]@7...決定  @2[B]@7...戻る" );
}

//----------------------------------------------------------------
void FromCallErrorDialog( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    app::sequence::JumpTo( ExecStorageMenuGroup );
}

//----------------------------------------------------------------
nn::Result EnumerateDirectory() NN_NOEXCEPT
{
    // ディレクトリ情報取得
    app::GetSystemConsole().Printf( "nn::bcat::EnumerateDeliveryCacheDirectory() :" );
    int outCount;
    nn::Result result = nn::bcat::EnumerateDeliveryCacheDirectory( &outCount, g_DirectoryName, nn::bcat::DeliveryCacheDirectoryCountMax );
    app::PrintErrorCode( result );

    if ( ! result.IsSuccess() )
    {
        app::DialogParam* p = app::AllocDialogParam( DialogParam::InfoType_Caution );
        p->SetMessage16( u"ディレクトリ一覧の取得に失敗しました。" );
        p->SetResult( result );
        app::sequence::Call( app::ExecDialogGroup, p );
        app::sequence::SetFromCall( FromCallErrorDialog, nullptr );
        return result;
    }

    app::GetSystemConsole().Printf( "directory num = %d\n", outCount );

    g_DirListNum = outCount;
    g_DirListViewTop = 0;
    g_DirListCursor = 0;

// たくさん読めたことにする
#if 0
    g_DirListNum = 20;
    for( int i=outCount; i<g_DirListNum; i++ )
    {
        sprintf( g_DirectoryName[i].value, "DummyDirectoryName%d", i );
    }
#endif

    for( int i = 0; i<g_DirListNum; i++ )
    {
        g_DirectoryNameList[i] = &g_DirectoryName[i];
    }

    // ソート
    //xxxx();

    NN_RESULT_SUCCESS;
}

//----------------------------------------------------------------
// ディレクトリメニュー(開始処理)
//
void ExecDirectoryMenu_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    nn::Result result = EnumerateDirectory();
    if ( result.IsFailure() )
    {
        return;
    }

    // 固定コンソール
    g_FixConsoleBuffer = new char[ FixConsoleBufferSize ];
    g_FixConsole.SetBuffer( g_FixConsoleBuffer, FixConsoleBufferSize, FixConsoleWidth, FixConsoleHeight );
    g_FixConsole.SetPrinter( DrawFixConsoleString, nullptr );
    // プロポーショナルコンソール
    g_PropConsoleBuffer = new char[ PropConsoleBufferSize ];
    g_PropConsole.SetBuffer( g_PropConsoleBuffer, PropConsoleBufferSize, PropConsoleWidth, PropConsoleHeight );
    g_PropConsole.SetPrinter( DrawPropConsoleString, nullptr );

    g_IsConsoleCreated = true;
    g_IsDrawDirectoryMenuCallback = true;

    // ディレクトリオープン状況
    g_IsDirectoryOpened = false;
    g_IsEnumeratedDirectory = false;
    g_OpenedDirectory = -1;

    PrintDirectoryList();
    PrintDirectoryCursor();

    // キー説明
    PrintDirectoryMenuHelp();
}
//----------------------------------------------------------------
// ディレクトリメニュー(終了処理)
//
void ExecDirectoryMenu_Finalize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    if ( g_IsConsoleCreated )
    {
        delete g_FixConsoleBuffer;
        delete g_PropConsoleBuffer;
        g_FixConsoleBuffer = nullptr;
        g_PropConsoleBuffer = nullptr;
    }
    g_IsConsoleCreated = false;

    // ファイルがオープンされていたらクローズする
    if ( g_IsDirectoryOpened )
    {
        g_BcatDirectory.Close();
        app::GetSystemConsole().Printf("nn::bcat::DeliveryCacheDirectory::Close() : Success\n");

        g_IsDirectoryOpened = false;
        g_IsEnumeratedDirectory = false;
        g_OpenedDirectory = -1;
    }
}

namespace {
void FromCallFunc( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    app::g_IsDirListUpdate = true;
}
}

//----------------------------------------------------------------
// ディレクトリメニュー
//
void ExecDirectoryMenu( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    if ( app::sequence::GetSequence() == Sequence_FromErrorDialog )
    {
        app::sequence::JumpTo( ExecStorageMenuGroup );
        return;
    }

    app::Pad pad( events );

    // A ボタンでディレクトリアクションメニュー開く
    if ( pad.IsButtonDownA() && g_DirListNum > 0 )
    {
        g_CurDirectory = g_DirListViewTop + g_DirListCursor; // 対象のディレクトリ

        // ファイルがオープンされていたらクローズする
        if ( g_IsDirectoryOpened && g_OpenedDirectory != g_CurDirectory )
        {
            g_BcatDirectory.Close();
            g_IsDirectoryOpened = false;
            g_IsEnumeratedDirectory = false;
            g_OpenedDirectory = -1;
        }

        app::sequence::Call( ExecDirectoryActionMenuGroup );
        app::sequence::SetFromCall( FromCallFunc, nullptr );
        return;
    }

    // B ボタンで戻る
    if ( pad.IsButtonDownB() )
    {
        app::sequence::JumpTo( ExecStorageMenuGroup );
    }

    // カーソル上
    if ( pad.IsButtonDownUp() || pad.IsButtonRepeatUp() )
    {
        RotateWithinViewRange( g_DirListCursor, -1, g_DirListViewTop, DirListViewMax, g_DirListNum );
        g_IsDirListUpdate = true;
    }
    // カーソル下
    if ( pad.IsButtonDownDown() || pad.IsButtonRepeatDown() )
    {
        RotateWithinViewRange( g_DirListCursor, 1, g_DirListViewTop, DirListViewMax, g_DirListNum );
        g_IsDirListUpdate = true;
    }
    // カーソル左
    if ( pad.IsButtonDownLeft() )
    {
        g_DirListCursor = 0;
        g_DirListViewTop = 0;
        g_IsDirListUpdate = true;
    }
    // カーソル右
    if ( pad.IsButtonDownRight() )
    {
        g_DirListCursor = (g_DirListNum < DirListViewMax)? (g_DirListNum - 1): (DirListViewMax - 1);
        g_DirListViewTop = (g_DirListNum < DirListViewMax)? 0: (g_DirListNum - DirListViewMax);
        g_IsDirListUpdate = true;
    }

    // リストの更新
    if ( g_IsDirListUpdate )
    {
        g_IsDirListUpdate = false;
        PrintDirectoryList();
        PrintDirectoryCursor();
    }
}

//================================================================
// ディレクトリアクションメニュー

//----------------------------------------------------------------
// ディレクトリアクションメニュー用描画コールバック
//
void DrawDirectoryActionMenu( void* arg ) NN_NOEXCEPT
{
}

//----------------------------------------------------------------
// ディレクトリアクションメニューのメニュー文字列描画
//
void DrawDirectoryActionMenuItems() NN_NOEXCEPT
{
    app::FixedProportionalConsole<char16_t>* p = g_DirectoryActionMenu.GetConsole16();
    if ( p )
    {
        app::SetGlvColor( app::ColorSet_White );
        p->PrintfEx( 1, 0,
                     u"-- ディレクトリ操作メニュー --" );
        p->PrintfEx( 2, 2,
                     g_IsDirectoryOpened? app::ConsoleColor_BrightRed: app::ConsoleColor_White,
                     u"ディレクトリオープン : nn::bcat::DeliveryCacheDirectory::Open()" );
        p->PrintfEx( 2, 3,
                     g_IsDirectoryOpened? app::ConsoleColor_White: app::ConsoleColor_DarkWhite,
                     u"ファイルエントリ取得 : nn::bcat::DeliveryCacheDirectory::Read()" );
        p->PrintfEx( 2, 4,
                     g_IsEnumeratedDirectory? app::ConsoleColor_White: app::ConsoleColor_DarkWhite,
                     u"ファイル列挙..." );
        p->PrintfEx( 2, 5,
                     u"ディレクトリクローズ : nn::bcat::DeliveryCacheDirectory::Close()" );
    }

    // 選択したディレクトリ名とオープン状態
    if ( g_CurDirectory >= 0 )
    {
        p->PrintfEx( 16, 0, app::ConvertToChar16_t("[ %s ]", g_DirectoryName[ g_CurDirectory ].value ) );
    }

    // オープン状態
    if ( g_IsDirectoryOpened )
    {
        p->PrintfEx( 32, 0, app::ConsoleColor_Yellow, u"オープン中" );
    }
    else
    {
        p->ClearPartial( 32, 0 );
    }
}

//----------------------------------------------------------------
// ディレクトリアクションメニューのキー説明
void PrintDirectoryActionMenuHelp() NN_NOEXCEPT
{
    app::GetHelpConsole().Clear();
    app::GetHelpConsole().PrintfEx( 2, 0,
                                    u"@1操作説明:@7 @6[上][下]@7...カーソル  @4[A]@7...決定  @2[B]@7...戻る" );
}

//----------------------------------------------------------------
// ディレクトリアクションメニュー(開始処理)
//
void ExecDirectoryActionMenu_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    // メニュー作成
    g_DirectoryActionMenu.CreateConsole( app::ConsoleSize_Char16_t, app::Position_StorageMenu,
                                         100, 5, 1, 2, 24, 4,
                                         app::DrawPriority_DirectoryActionMenu + 1, 0x230 );
    g_DirectoryActionMenu.SetBackColor( app::ColorSet_DarkGreen, app::ColorSet_Green, app::DrawFrameWidth );
    g_DirectoryActionMenu.SetItemParameter( MenuMax, g_InitialMenuIndex );

    // メニュー項目のコンソールプリント
    DrawDirectoryActionMenuItems();

    // キー説明
    PrintDirectoryActionMenuHelp();
}
//----------------------------------------------------------------
// ディレクトリアクションメニュー(終了処理)
//
void ExecDirectoryActionMenu_Finalize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    g_DirectoryActionMenu.DestroyConsole();
}

// 「ディレクトリオープン」項目
bool DirectoryActionMenu_OpenDirectory() NN_NOEXCEPT
{
    app::GetSystemConsole().Printf("nn::bcat::DeliveryCacheDirectory::Open() :");
    nn::Result result = g_BcatDirectory.Open( g_DirectoryName[ g_CurDirectory ] );
    app::PrintErrorCode( result );
    if ( result.IsFailure() )
    {
        app::SetErrorDialog( u"ディレクトリのオープンに失敗しました", result );
        return false;
    }
    else
    {
        g_IsDirectoryOpened = true;
        g_OpenedDirectory = g_CurDirectory;
    }
    return true;
}

// 「ディレクトリクローズ」項目
void DirectoryActionMenu_CloseDirectory() NN_NOEXCEPT
{
    g_BcatDirectory.Close(); // 必ず成功
    app::GetSystemConsole().Printf("nn::bcat::DeliveryCacheDirectory::Close() : Success\n");

    g_IsDirectoryOpened = false;
    g_OpenedDirectory = -1;
    g_IsEnumeratedDirectory = false;
}

// 「ディレクトリのエントリ取得」項目
bool DirectoryActionMenu_EnumerateEntry() NN_NOEXCEPT
{
    if ( ! g_IsDirectoryOpened )
    {
        return true;
    }

    app::GetSystemConsole().Printf("nn::bcat::DeliveryCacheDirectory::Read() :");
    nn::Result result = g_BcatDirectory.Read( &g_FileEntryNum, g_FileEntry, nn::bcat::DeliveryCacheFileCountMaxPerDirectory );
    app::PrintErrorCode( result );

    if ( (g_IsEnumeratedDirectory = result.IsSuccess()) == false )
    {
        app::SetErrorDialog( u"ディレクトリのリードに失敗しました", result );
    }
    return g_IsEnumeratedDirectory;
}

void ExecFromFileMenuGroup( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    g_IsDrawDirectoryMenuCallback = true;
    g_DirectoryActionMenu.SetDrawInactive( true );
}

// 「ファイル列挙メニューへ」項目
bool DirectoryActionMenu_CallFileMenu() NN_NOEXCEPT
{
    if ( ! g_IsDirectoryOpened || ! g_IsEnumeratedDirectory )
    {
        return true;
    }

    g_IsDrawDirectoryMenuCallback = false;
    g_DirectoryActionMenu.SetDrawInactive( false );

    // 呼び出しパラメータ
    g_FileMenuParameter.pDirectoryName = &g_DirectoryName[ g_CurDirectory ];
    g_FileMenuParameter.fileEntryNum = g_FileEntryNum;
    g_FileMenuParameter.pFileEntryArray = g_FileEntry;

    app::sequence::Call( ExecFileMenuGroup, reinterpret_cast<void*>(&g_FileMenuParameter) );
    app::sequence::SetFromCall( ExecFromFileMenuGroup, nullptr );
    return false;
}

//----------------------------------------------------------------
// ディレクトリアクションメニュー
//
void ExecDirectoryActionMenu( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    g_DirectoryActionMenu.SetHidEvent( &events );
    g_DirectoryActionMenu.Update();

    // A ボタン
    bool isContinued = true;
    switch( g_DirectoryActionMenu.CheckButtonOk() )
    {
        case MenuIndex_OpenDirectory:
            isContinued = DirectoryActionMenu_OpenDirectory();
            break;
        case MenuIndex_EnumerateEntry:
            isContinued = DirectoryActionMenu_EnumerateEntry();
            break;
        case MenuIndex_FileMenu:
            isContinued = DirectoryActionMenu_CallFileMenu();
            break;
        case MenuIndex_CloseDirectory:
            DirectoryActionMenu_CloseDirectory();
            break;
        default:
            break;
    }
    if ( ! isContinued )
    {
        return;
    }

    // B ボタンで戻る
    if ( g_DirectoryActionMenu.CheckButtonCancel() )
    {
        app::sequence::Return();
    }

    if ( g_DirectoryActionMenu.IsUpdated() )
    {
        DrawDirectoryActionMenuItems();
        g_DirectoryActionMenu.ClearUpdated();
    }
}


//================================================================
// ソート


} // namespace app

