﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <memory>
#include <string>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/fs/fs_MemoryManagement.h>
#include <nn/fs/fs_AccessLogPrivate.h>
#include <nn/init/init_Malloc.h>
#include <nn/os/os_Argument.h>
#include <nn/os/os_Base.h>
#include <nn/os/os_MemoryHeap.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/oe.h>
#include <nn/nn_Log.h>
#include "BackupSaveData.h"

namespace {

//!< メモリヒープのサイズ
const size_t AllHeapSize =    12 * 1024 * 1024;
#if !defined(USE_MALLOC)
const size_t MemoryHeapSize = 8 * 1024 * 1024;
#endif

void Usage()
{
    // MEMO: 仕様公開しているのは exe の方なので nsp のパラメータは表示しないようにする
    // NN_LOG("BackupSaveData [-import|-export] [--replace-account] [--exist-account-only]\n");
}

enum ExecMode
{
    EXEC_IMPORT,
    EXEC_EXPORT,
    EXEC_OTHER,
};

struct ParsedArguments
{
    ExecMode    mode;
    bool        isPartial;
    bool        isReplaceAccount;
    bool        isExistAccountOnly;
    std::string titlesString;

    ParsedArguments()
    {
        mode = EXEC_OTHER;
        isPartial = false;
        isReplaceAccount = false;
        isExistAccountOnly = false;
        titlesString = "";
    }
};

bool ParseArguments(ParsedArguments* pArg)
{
    int argc = nn::os::GetHostArgc();
    char** argv = nn::os::GetHostArgv();

    if(argc < 2)
    {
        return false;
    }

    pArg->mode = EXEC_OTHER;

    for(int i=1; i<argc; i++)
    {
        if(!std::strcmp(argv[i], "-import"))
        {
            pArg->mode = EXEC_IMPORT;
        }
        else if(!std::strcmp(argv[i], "-export"))
        {
            pArg->mode = EXEC_EXPORT;
        }
        else if(!std::strcmp(argv[i], "-title"))
        {
            pArg->isPartial = true;
            i++;
            if(i>=argc)
            {
                NN_LOG("[ERROR] -title requires titleId\n");
                return false;
            }
            pArg->titlesString = argv[i];
            if(i>=argc || pArg->titlesString.substr(0,1) == "-")
            {
                NN_LOG("[ERROR] -title requires titleId\n");
                return false;
            }
        }
        else if(!std::strcmp(argv[i], "--replace-account"))
        {
            pArg->isReplaceAccount = true;
        }
        else if(!std::strcmp(argv[i], "--exist-account-only"))
        {
            pArg->isExistAccountOnly = true;
        }

    }

    if(pArg->mode == EXEC_OTHER)
    {
        return false;
    }

    return true;
}

int DoExport(bool isPartial, std::string titlesString, bool isExistAccountOnly)
{
    nn::Result result;
    if(!isPartial)
    {
        result = ExportSaveDataAll(isExistAccountOnly);
    }
    else
    {
        result = ExportSaveDataPartial(titlesString, isExistAccountOnly);
    }
    if(result.IsSuccess())
    {
        NN_LOG("Save data export succeeded.\n");
        return 0;
    }
    NN_LOG("Save data export failed.\n");
    return 1;
}

int DoImport(bool isReplaceAccount)
{
    nn::Result result = ImportSaveDataAll(isReplaceAccount);
    if(result.IsSuccess())
    {
        NN_LOG("Save data import succeeded.\n");
        return 0;
    }
    NN_LOG("Save data import failed.\n");
    return 1;
}

} // namespace

extern "C" void nninitStartup()
{
    // メモリヒープのサイズを設定
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::os::SetMemoryHeapSize(AllHeapSize));

    auto address = uintptr_t();

#if !defined(USE_MALLOC)
    // メモリヒープを確保
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::os::AllocateMemoryBlock(&address, MemoryHeapSize));

    // malloc 用のメモリヒープを設定
    ::nn::init::InitializeAllocator(reinterpret_cast<void*>(address),
                                    MemoryHeapSize);
#else
    // メモリヒープを確保
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::os::AllocateMemoryBlock(&address, AllHeapSize));

    // malloc 用のメモリヒープを設定
    ::nn::init::InitializeAllocator(reinterpret_cast<void*>(address),
                                    AllHeapSize);
#endif
}

extern "C" int nnMain()
{
    NN_LOG("BackupSaveData start\n");

    // ファイルシステム用のアロケータを設定
    // ::nn::fs::SetAllocator(AllocateForFileSystem, DeallocateForFileSystem);
    // FSログを無効化 (実はしなくてもそうなっている)
    nn::fs::SetLocalAccessLog(false);

    nn::Result result;
    int retVal = 1;
    ParsedArguments parsedArgument;

    if(!ParseArguments(&parsedArgument))
    {
        Usage();
        return 1;
    }

    nn::account::InitializeForSystemService();

    result = InitializeDirectories();
    if(!result.IsSuccess())
    {
        return 1;
    }
    result = InitilizeHeap();
    if(!result.IsSuccess())
    {
        return 1;
    }

    switch(parsedArgument.mode)
    {
    case EXEC_EXPORT:
        // セーブデータをエクスポートする
        retVal = DoExport(parsedArgument.isPartial, parsedArgument.titlesString, parsedArgument.isExistAccountOnly);
        break;
    case EXEC_IMPORT:
        // セーブデータをインポートする
        retVal = DoImport(parsedArgument.isReplaceAccount);
        break;
    default:
        break;
    }

    FinalizeHeap();

    NN_LOG("BackupSaveData finished\n");

    return retVal;
}
