﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "AlbumSynchronizer_ProgramOption.h"

#include <cstring>
#include <nn/util/util_StringUtil.h>
#include <nn/nn_Log.h>

#include "AlbumSynchronizer_Config.h"

#define ALBUMSYNC_TEST_STRING(expected,value)    \
    (TestStringEquality((expected), (value), static_cast<int>(std::strlen(expected) + 1)))

#define ALBUMSYNC_LOG_PARSE_ERROR(...) NN_LOG(__VA_ARGS__)

namespace detail{
    char ConvertToLowerChar(char c) NN_NOEXCEPT
    {
        if(c >= 'A' && c <= 'Z')
        {
            return c - 'A' + 'a';
        }
        return c;
    }

    bool TestStringEquality(const char* expected, const char* value, size_t size) NN_NOEXCEPT
    {
        for(size_t i = 0; i < size; i++)
        {
            char a = expected[i];
            char b = value[i];
            if(a == '\0' && b == '\0')
            {
                return true;
            }
            else if(a == '\0' || b == '\0')
            {
                break;
            }

            a = ConvertToLowerChar(a);
            b = ConvertToLowerChar(b);

            if(a != b)
            {
                break;
            }
        }
        return false;
    }

    enum class ParseResult
    {
        Success,
        NotMatched,
        Error,
    };

    // <StorageValue> ::= "NAND" | "SD"
    ParseResult TryParseStorageValue(char*** pNext, nn::fs::ImageDirectoryId* pOutValue, char** pCurrent, char** pEnd) NN_NOEXCEPT
    {
        if(pCurrent >= pEnd)
        {
            return ParseResult::NotMatched;
        }

        char** p = pCurrent;
        nn::fs::ImageDirectoryId value = nn::fs::ImageDirectoryId::Nand;
        if(ALBUMSYNC_TEST_STRING("NAND", *p))
        {
            p++;
            value = nn::fs::ImageDirectoryId::Nand;
        }
        else if(ALBUMSYNC_TEST_STRING("BuiltIn", *p))
        {
            p++;
            value = nn::fs::ImageDirectoryId::Nand;
        }
        else if(ALBUMSYNC_TEST_STRING("SD", *p))
        {
            p++;
            value = nn::fs::ImageDirectoryId::SdCard;
        }
        else if(ALBUMSYNC_TEST_STRING("SdCard", *p))
        {
            p++;
            value = nn::fs::ImageDirectoryId::SdCard;
        }
        else
        {
            return ParseResult::NotMatched;
        }
        *pOutValue = value;
        *pNext = p;
        return ParseResult::Success;
    }

    // <Storage> ::= "--storage" <StorageValue>
    ParseResult TryParseStorage(char*** pNext, nn::fs::ImageDirectoryId* pOutValue, char** pCurrent, char** pEnd) NN_NOEXCEPT
    {
        if(pCurrent >= pEnd)
        {
            return ParseResult::NotMatched;
        }

        char** p = pCurrent;
        nn::fs::ImageDirectoryId value = nn::fs::ImageDirectoryId::Nand;
        if(!ALBUMSYNC_TEST_STRING("--storage", *p))
        {
            return ParseResult::NotMatched;
        }
        p++;

        if(TryParseStorageValue(&p, &value, p, pEnd) != ParseResult::Success)
        {
            ALBUMSYNC_LOG_PARSE_ERROR("storage name required after '--storage'.\n");
            return ParseResult::Error;
        }

        *pOutValue = value;
        *pNext = p;
        return ParseResult::Success;
    }

    // <HostDirectoryValue> ::= string
    ParseResult TryParseHostDirectoryValue(char*** pNext, char* buffer, size_t bufferSize, char** pCurrent, char** pEnd) NN_NOEXCEPT
    {
        if(pCurrent >= pEnd)
        {
            return ParseResult::NotMatched;
        }

        char** p = pCurrent;
        if(nn::util::Strnlen(*p, static_cast<int>(bufferSize)) >= bufferSize)
        {
            ALBUMSYNC_LOG_PARSE_ERROR("host directory path is too long\n");
            return ParseResult::Error;
        }

        nn::util::Strlcpy(buffer, *p, static_cast<int>(bufferSize));
        p++;

        *pNext = p;
        return ParseResult::Success;
    }

    // <HostDirectory> ::= "--directory" <HostDirectoryValue>
    ParseResult TryParseHostDirectory(char*** pNext, char* buffer, size_t bufferSize, char** pCurrent, char** pEnd) NN_NOEXCEPT
    {
        if(pCurrent >= pEnd)
        {
            return ParseResult::NotMatched;
        }

        char** p = pCurrent;
        if(!ALBUMSYNC_TEST_STRING("--directory", *p))
        {
            return ParseResult::NotMatched;
        }
        p++;

        if(TryParseHostDirectoryValue(&p, buffer, bufferSize, p, pEnd) != ParseResult::Success)
        {
            ALBUMSYNC_LOG_PARSE_ERROR("host directory path is required after '--directory'.\n");
            return ParseResult::Error;
        }

        *pNext = p;
        return ParseResult::Success;
    }

    // <Force> ::= "--force"
    ParseResult TryParseForce(char*** pNext, bool* pOutValue, char** pCurrent, char** pEnd) NN_NOEXCEPT
    {
        if(pCurrent >= pEnd)
        {
            return ParseResult::NotMatched;
        }

        char** p = pCurrent;
        if(!ALBUMSYNC_TEST_STRING("--force", *p))
        {
            return ParseResult::NotMatched;
        }
        p++;

        *pOutValue = true;
        *pNext = p;
        return ParseResult::Success;
    }

    // <RawMode> ::= "--raw"
    ParseResult TryParseRawMode(char*** pNext, bool* pOutValue, char** pCurrent, char** pEnd) NN_NOEXCEPT
    {
        if(pCurrent >= pEnd)
        {
            return ParseResult::NotMatched;
        }

        char** p = pCurrent;
        if(!ALBUMSYNC_TEST_STRING("--raw", *p))
        {
            return ParseResult::NotMatched;
        }
        p++;

        *pOutValue = true;
        *pNext = p;
        return ParseResult::Success;
    }

    // <Action> ::= "--help" | "--upload" | "--download" | "--clean"
    ParseResult TryParseAction(char*** pNext, Action* pOutValue, char** pCurrent, char** pEnd) NN_NOEXCEPT
    {
        if(pCurrent >= pEnd)
        {
            return ParseResult::NotMatched;
        }

        char** p = pCurrent;
        Action action = Action_PrintHelp;
        if(NN_STATIC_CONDITION(false))
        {
        }
#if ALBUMSYNC_SUPPORT_ACTION_HELP
        else if(ALBUMSYNC_TEST_STRING("--help", *p))
        {
            p++;
            action = Action_PrintHelp;
        }
#endif
#if ALBUMSYNC_SUPPORT_ACTION_UPLOAD
        else if(ALBUMSYNC_TEST_STRING("--upload", *p))
        {
            p++;
            action = Action_Upload;
        }
#endif
#if ALBUMSYNC_SUPPORT_ACTION_DOWNLOAD
        else if(ALBUMSYNC_TEST_STRING("--download", *p))
        {
            p++;
            action = Action_Download;
        }
#endif
#if ALBUMSYNC_SUPPORT_ACTION_CLEAN
        else if(ALBUMSYNC_TEST_STRING("--clean", *p))
        {
            p++;
            action = Action_Clean;
        }
#endif
        else
        {
            return ParseResult::NotMatched;
        }
        *pOutValue = action;
        *pNext = p;
        return ParseResult::Success;
    }

    bool Validate(const ProgramOption& opts) NN_NOEXCEPT
    {
        auto action = opts.GetAction();
        if(action == Action_Upload || action == Action_Download || action == Action_Clean)
        {
            if(!opts.GetStorage())
            {
                ALBUMSYNC_LOG_PARSE_ERROR("'--storage' is required.\n");
                return false;
            }
        }
        if(action == Action_Upload || action == Action_Download)
        {
            if(!opts.GetHostDirectory())
            {
                ALBUMSYNC_LOG_PARSE_ERROR("'--directory' is required.\n");
                return false;
            }
        }

        // 安全のため directory にルート直下を指定させない
        if(const char* path = opts.GetHostDirectory())
        {
            // TORIAEZU:
            // 雑に判定
            int length = nn::util::Strnlen(path, ProgramOption::PathSizeMax);
            int delimiterCount = 0;
            bool endWithDelimiter = false;
            for(int i = 0; i < length; i++)
            {
                char c = path[i];
                if(c == '/' || c == '\\')
                {
                    delimiterCount++;
                    endWithDelimiter = true;
                }
                else
                {
                    endWithDelimiter = false;
                }
            }
            if(delimiterCount == 0 || (delimiterCount == 1 && endWithDelimiter))
            {
                ALBUMSYNC_LOG_PARSE_ERROR("directory path must not be a root path.\n");
                ALBUMSYNC_LOG_PARSE_ERROR("  path: %s\n", path);
                return false;
            }
        }

        return true;
    }

}


// <ProgramOption> ::= <Action> (<Storage> | <HostDirectory> | <Force>)*
// ただし <Storage>, <HostDirectory>, <Force> の指定は最大 1 回ずつまで。
bool ProgramOption::TryParseArgument(ProgramOption* pOutValue, int argc, char** argv) NN_NOEXCEPT
{
    ProgramOption option;

    char** p = argv;
    char** pEnd = argv + argc;
    if(detail::TryParseAction(&p, &option.m_Action, p, pEnd) != detail::ParseResult::Success)
    {
        ALBUMSYNC_LOG_PARSE_ERROR("action is required\n");
        return false;
    }

    while(p < pEnd)
    {
#if ALBUMSYNC_SUPPORT_OPTION_STORAGE
        {
            nn::fs::ImageDirectoryId storage;
            auto result = detail::TryParseStorage(&p, &storage, p, pEnd);
            if(result == detail::ParseResult::Error)
            {
                return false;
            }
            else if(result == detail::ParseResult::Success)
            {
                if(option.m_Storage)
                {
                    ALBUMSYNC_LOG_PARSE_ERROR("--storage option can't be specified twice.\n");
                    return false;
                }
                option.m_Storage = storage;
                continue;
            }
        }
#endif
#if ALBUMSYNC_SUPPORT_OPTION_DIRECTORY
        {
            auto result = detail::TryParseHostDirectory(&p, option.m_HostDirectory, sizeof(option.m_HostDirectory), p, pEnd);
            if(result == detail::ParseResult::Error)
            {
                return false;
            }
            else if(result == detail::ParseResult::Success)
            {
                if(option.m_IsHostDirectorySpecified)
                {
                    ALBUMSYNC_LOG_PARSE_ERROR("--directory option can't be specified twice.\n");
                    return false;
                }
                option.m_IsHostDirectorySpecified = true;
                continue;
            }
        }
#endif
#if ALBUMSYNC_SUPPORT_OPTION_FORCE
        {
            bool value = false;
            auto result = detail::TryParseForce(&p, &value, p, pEnd);
            if(result == detail::ParseResult::Success)
            {
                if(option.m_IsForceEnabled)
                {
                    ALBUMSYNC_LOG_PARSE_ERROR("--force option can't be specified twice.\n");
                    return false;
                }
                option.m_IsForceEnabled = value;
                continue;
            }
        }
#endif
#if ALBUMSYNC_SUPPORT_OPTION_RAW
        {
            bool value = false;
            auto result = detail::TryParseRawMode(&p, &value, p, pEnd);
            if(result == detail::ParseResult::Success)
            {
                if(option.m_IsRawModeEnabled)
                {
                    ALBUMSYNC_LOG_PARSE_ERROR("--raw option can't be specified twice.\n");
                    return false;
                }
                option.m_IsRawModeEnabled = value;
                continue;
            }
        }
#endif
        {
            ALBUMSYNC_LOG_PARSE_ERROR("Unknown option '%s'.\n", *p);
            return false;
        }
    }

    if(!detail::Validate(option))
    {
        return false;
    }

    *pOutValue = option;
    return true;
}
