﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <curl/curl.h>

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/friends/friends_Api.h>
#include <nn/fs.h>
#include <nn/init.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/migration/migration_UserMigrationServiceResource.h>
#include <nn/migration/detail/migration_Diagnosis.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/npns/npns_ApiSystem.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/os/os_Thread.h>
#include <nn/pdm/pdm_NotifyEventApi.h>
#include <nn/sf/sf_HipcSimpleAllInOneServerManager.h>
#include <nn/spl/spl_Api.h>
#include <nn/socket/socket_Api.h>
#include <nn/socket/socket_SystemConfig.h>
#include <nn/time/time_ApiForSystem.h>

// 例外関連コードのリンク抑制のためのオーバーライド
// 参考：http://spdlybra.nintendo.co.jp/jira/browse/SIGLO-34162
void* operator new(std::size_t size) { return malloc(size); }
void* operator new(std::size_t size, const std::nothrow_t&) NN_NOEXCEPT { return malloc(size); }
void  operator delete(void* ptr) NN_NOEXCEPT { free(ptr); }

void* operator new[](std::size_t size) { return malloc(size); }
void* operator new[](std::size_t size, const std::nothrow_t&) NN_NOEXCEPT{ return malloc(size); }
void  operator delete[](void* ptr) NN_NOEXCEPT{ free(ptr); }

namespace
{
#define HEAP_INITIALIZER_INITIALIZER {false}
struct HeapInitializer
{
    bool _initialized;
    nn::lmem::HeapHandle _handle;
    char _buffer[32 * 1024];

    void Initialize() NN_NOEXCEPT
    {
        _handle = nn::lmem::CreateExpHeap(_buffer, sizeof(_buffer), nn::lmem::CreationOption_NoOption);
        _initialized = true;
    }
    nn::lmem::HeapHandle GetHandle() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(_initialized);
        return _handle;
    }
} g_HeapInitializer = HEAP_INITIALIZER_INITIALIZER;

void* Allocate(size_t size) NN_NOEXCEPT
{
    return nn::lmem::AllocateFromExpHeap(g_HeapInitializer.GetHandle(), size);
}
void Deallocate(void* ptr, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    nn::lmem::FreeToExpHeap(g_HeapInitializer.GetHandle(), ptr);
}

// IPC サーバー
static const int SessionCountMax = 4;
static const int PortCountMax = 1;
struct ServerOption
{
    static const bool CanDeferInvokeRequest = false;
    static const int ObjectInSubDomainCountMax = 4 * 8;
    static const size_t PointerTransferBufferSize = 1024;
    static const int SubDomainCountMax = 4;
};
class Server
    : public nn::sf::HipcSimpleAllInOneServerManager<SessionCountMax, PortCountMax, ServerOption>
{
};
void IpcThreadFunction(void* p) NN_NOEXCEPT
{
    reinterpret_cast<Server*>(p)->LoopAuto();
}

struct StaticResource
{
    // socket
    nn::migration::DefaultImplConfiguration::SocketBuffer socketBuffer;
    // ipc
    NN_ALIGNAS(nn::os::MemoryPageSize) char ipcThreadStack[32 * 1024];
    // 非同期処理
    NN_ALIGNAS(nn::os::MemoryPageSize) char asyncThreadStack[nn::migration::DefaultUserMigrationServiceResource::ThreadStackSize][2];
    // 外部 NSA ログイン用のメモリ
    NN_ALIGNAS(nn::os::MemoryPageSize) char loginBuffer[nn::account::RequiredBufferSizeForExternalNetworkServiceAccountRegistrar];
    // libcurl 等, malloc 用
    NN_ALIGNAS(nn::os::MemoryPageSize) char commonAllocatorBuffer[1 * 1024 * 1024];
    // サービスオブジェクト自身のメモリ
    typedef std::aligned_storage<sizeof(nn::migration::DefaultUserMigrationServiceResource), std::alignment_of<nn::migration::DefaultUserMigrationServiceResource>::value>::type ResourceStorage;
    ResourceStorage resourceStorage;
    // ipc サーバーのメモリ
    typedef std::aligned_storage<sizeof(Server), std::alignment_of<Server>::value>::type ServerStorage;
    ServerStorage serverStorage;
} g_Resource;

} // ~namespace <anonymous>

extern "C" void nninitStartup()
{
    // malloc, free の初期化
    nn::init::InitializeAllocator(g_Resource.commonAllocatorBuffer, sizeof(g_Resource.commonAllocatorBuffer));
}
extern "C" void nnMain()
{
    nn::os::SetThreadNamePointer(nn::os::GetCurrentThread(), NN_SYSTEM_THREAD_NAME(migration, Main));

    // 環境の初期化 ---------------------------------------------------------------------------------
    // ヒープ
    g_HeapInitializer.Initialize();
    nn::fs::SetAllocator(Allocate, Deallocate);

    // サブシステム
    nn::account::InitializeForAdministrator();
    nn::friends::Initialize();
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::InitializeSystem());
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(nn::npns::InitializeForSystem());
    nn::ns::Initialize();
    nn::pdm::InitializeForNotification();
    nn::spl::InitializeForCrypto();
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(nn::time::InitializeForSystem());
    NN_ABORT_UNLESS(curl_global_init(CURL_GLOBAL_ALL) == CURLE_OK);

    // オブジェクト生成 ----------------------------------------------------------------------------
    nn::migration::UserMigrationRuntimeConfiguration runtimeConfig = {
        NN_SYSTEM_THREAD_NAME(migration, Executor),
        NN_SYSTEM_THREAD_PRIORITY(migration, Executor),
        g_Resource.asyncThreadStack,
        sizeof(g_Resource.asyncThreadStack),
        g_Resource.loginBuffer,
        sizeof(g_Resource.loginBuffer),
    };
    nn::migration::DefaultImplConfiguration::AttachSocketBuffer(g_Resource.socketBuffer);
    auto* pResource = new(&g_Resource.resourceStorage)nn::migration::DefaultUserMigrationServiceResource(runtimeConfig);
    NN_UTIL_SCOPE_EXIT
    {
        pResource->~UserMigrationServiceResource();
    };

    // サービスの準備 ------------------------------------------------------------------------------
    auto* pServer = new(&g_Resource.serverStorage)Server;
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(pServer->RegisterObjectForPort(
        pResource->GetServiceImpl(), SessionCountMax, nn::migration::UserMigrationServiceName));

    nn::os::ThreadType ipcThread;
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &ipcThread,
        IpcThreadFunction, pServer,
        g_Resource.ipcThreadStack, sizeof(g_Resource.ipcThreadStack),
        NN_SYSTEM_THREAD_PRIORITY(migration, IpcServer)));
    nn::os::SetThreadNamePointer(
        &ipcThread,
        NN_SYSTEM_THREAD_NAME(migration, IpcServer));

    NN_MIGRATION_DETAIL_INFO("Starting service: %zu\n", sizeof(g_Resource));

    // サービスの開始 ------------------------------------------------------------------------------
    pServer->Start();
    nn::os::StartThread(&ipcThread);

    nn::os::Event e(nn::os::EventClearMode_ManualClear);
    e.Wait();

    // 終了処理 -----------------------------------------------------------------------------------
#if 0
#endif
}
