﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <memory>
#include <new>
#include <nn/nn_Abort.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/fs/fs_MemoryManagement.h>
#include <nn/hid/hid_IHidDebugServer.sfdl.h>
#include <nn/hid/hid_IHidServer.sfdl.h>
#include <nn/hid/hid_IHidTemporaryServer.sfdl.h>
#include <nn/hid/hid_ServiceName.h>
#include <nn/irsensor/irsensor_IIrSensorServer.h>
#include <nn/irsensor/irsensor_IIrSensorSystemServer.h>
#include <nn/irsensor/irsensor_ServiceName.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Thread.h>
#include <nn/psc.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/util/util_ScopeExit.h>

#ifdef ENABLE_CPU_PROFILER
#include <nn/profiler.h>
#endif

#include "ahid_Hdr.h"
#include "cdhidUsb_Main.h"
#include "hid_HidDebugServer.h"
#include "hid_HidServer.h"
#include "hid_HidSystemServer.h"
#include "hid_HidTemporaryServer.h"
#include "hid_LockableMutexType.h"
#include "hid_ResourceManager-os.horizon.h"
#include "hid_Settings.h"
#include "hid_StaticObject.h"
#include "irsensor_ResourceManager.h"
#include "irsensor_IrSensorServer.h"
#include "irsensor_IrSensorSystemServer.h"
#include "hid_HidbusServer.h"

namespace {

//!< セッション数の最大値
const int SessionCountMax = 100;

//!< ポート数の最大値
const int PortCountMax = 6;

//!< サーバマネージャのオプション宣言です。
struct ServerManagerOption
{
    static const size_t PointerTransferBufferSize = 512;
};

//!< サーバマネージャを扱うためのクラスです。
class ServerManager final
    : public ::nn::sf::HipcSimpleAllInOneServerManager<SessionCountMax,
                                                       PortCountMax,
                                                       ServerManagerOption>
{
};

//!< 拡張ヒープのサイズ
const size_t HeapMemorySize = 4 * 1024;

//!< リソースデータを扱うファイルシステムのマウント名
const char* const MountName = "rom";

//!< 拡張ヒープにメモリ領域をアロケートします。
void* Allocate(size_t size) NN_NOEXCEPT;

//!< 拡張ヒープのメモリ領域をデアロケートします。
void Deallocate(void* p, size_t size) NN_NOEXCEPT;

//!< リソースデータを扱うファイルシステムをマウントを扱うクラスです。
class RomMounter final
{
    NN_DISALLOW_COPY(RomMounter);
    NN_DISALLOW_MOVE(RomMounter);
private:
    size_t m_Size;
    void* m_Buffer;
public:
    RomMounter() NN_NOEXCEPT;
    ~RomMounter() NN_NOEXCEPT;
};

//!< リソースマネージャをアクティブ化します。
void ActivateResourceManager() NN_NOEXCEPT;

//!< リソースマネージャを非アクティブ化します。
void DeactivateResourceManager() NN_NOEXCEPT;

//!< ServerManager に HidServer を登録します。
void RegisterHidServerWithServerManager(ServerManager* pManager) NN_NOEXCEPT;

//!< ServerManager に HidSystemServer を登録します。
void RegisterHidSystemServerWithServerManager(
    ServerManager* pManager) NN_NOEXCEPT;

//!< ServerManager に HidDebugServer を登録します。
void RegisterHidDebugServerWithServerManager(
    ServerManager* pManager) NN_NOEXCEPT;

//!< ServerManager に HidTemporaryServer を登録します。
void RegisterHidTemporaryServerWithServerManager(
    ServerManager* pManager) NN_NOEXCEPT;

//!< ServerManager に IrSensorServer を登録します。
void RegisterIrSensorServerWithServerManager(
    ServerManager* pManager) NN_NOEXCEPT;

//!< ServerManager に IrSensorSystemServer を登録します。
void RegisterIrSensorSystemServerWithServerManager(
    ServerManager* pManager) NN_NOEXCEPT;

//!< PmModule に依存モジュールを登録します。
void RegisterDependenciesWithPmModule(::nn::psc::PmModule* pModule) NN_NOEXCEPT;

//!< PmModule の要求を処理します。
void DispatchPmModuleRequest(::nn::psc::PmModule* pModule) NN_NOEXCEPT;

#ifdef ENABLE_CPU_PROFILER
//!< Cpu プロファイラ用のバッファ
static uint8_t g_ProfilerBuf[nn::profiler::MinimumBufferSize];
#endif

} // namespace

void* operator new(size_t count)
{
    return Allocate(count);
}

void* operator new(size_t count, const ::std::nothrow_t& tag) NN_NOEXCEPT
{
    NN_UNUSED(tag);
    return Allocate(count);
}

void operator delete(void* ptr) NN_NOEXCEPT
{
    Deallocate(ptr, 0);
}

void* operator new[](size_t count)
{
    return Allocate(count);
}

void* operator new[](size_t count, const ::std::nothrow_t& tag) NN_NOEXCEPT
{
    NN_UNUSED(tag);
    return Allocate(count);
}

void operator delete[](void* ptr) NN_NOEXCEPT
{
    Deallocate(ptr, 0);
}

extern "C" void nninitStartup()
{
    // 何もしない
}

extern "C" void nndiagStartup()
{
    // 何もしない
}

extern "C" void nnMain()
{
#ifdef ENABLE_CPU_PROFILER
    // Cpu プロファイラの初期化
    nn::profiler::Initialize(g_ProfilerBuf, sizeof(g_ProfilerBuf));
#endif

    ::nn::os::ThreadType* const pCurrentThread = ::nn::os::GetCurrentThread();
    ::nn::os::ChangeThreadPriority(
        pCurrentThread, NN_SYSTEM_THREAD_PRIORITY(hid, Main));
    ::nn::os::SetThreadNamePointer(
        pCurrentThread, NN_SYSTEM_THREAD_NAME(hid, Main));

    ServerManager& serverManager =
        ::nn::hid::detail::StaticObject<ServerManager>::Get();
    RegisterHidServerWithServerManager(&serverManager);
    RegisterHidSystemServerWithServerManager(&serverManager);
    RegisterHidDebugServerWithServerManager(&serverManager);
    RegisterHidTemporaryServerWithServerManager(&serverManager);
    RegisterIrSensorServerWithServerManager(&serverManager);
    RegisterIrSensorSystemServerWithServerManager(&serverManager);

    ::nn::hid::detail::FirmwareDebugSettings::Initialize();

    ::nn::fs::SetAllocator(Allocate, Deallocate);
    RomMounter mounter;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ahid::hdr::HdrInitialize());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::cdhid::usb::UsbHidInitialize());

    ActivateResourceManager();

    nn::hid::StartHidbusServer();

    ::nn::psc::PmModule& pmModule =
        ::nn::hid::detail::StaticObject<::nn::psc::PmModule>::Get();
    RegisterDependenciesWithPmModule(&pmModule);

    ::nn::os::MultiWaitHolderType pmModuleEventHolder = {};
    ::nn::os::InitializeMultiWaitHolder(
        &pmModuleEventHolder, pmModule.GetEventPointer()->GetBase());
    ::nn::os::SetMultiWaitHolderUserData(
        &pmModuleEventHolder, ::nn::psc::PmModuleId_Hid);
    serverManager.AddUserWaitHolder(&pmModuleEventHolder);

    ::nn::os::ChangeThreadPriority(
        pCurrentThread, NN_SYSTEM_THREAD_PRIORITY(hid, IpcServer));
    ::nn::os::SetThreadNamePointer(
        pCurrentThread, NN_SYSTEM_THREAD_NAME(hid, IpcServer));

    serverManager.Start();

    while (NN_STATIC_CONDITION(true))
    {
        ::nn::os::MultiWaitHolderType* waitId = serverManager.Wait();

        switch (::nn::os::GetMultiWaitHolderUserData(waitId))
        {
        case ServerManager::HipcSimpleAllInOneServerManager::AcceptTag:
        case ServerManager::HipcSimpleAllInOneServerManager::InvokeTag:
            serverManager.ProcessAuto(waitId);
            break;

        case ::nn::psc::PmModuleId_Hid:
            pmModule.GetEventPointer()->Clear();
            serverManager.AddUserWaitHolder(&pmModuleEventHolder);
            DispatchPmModuleRequest(&pmModule);
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    ::nn::os::ChangeThreadPriority(
        pCurrentThread, NN_SYSTEM_THREAD_PRIORITY(hid, Main));
    ::nn::os::SetThreadNamePointer(
        pCurrentThread, NN_SYSTEM_THREAD_NAME(hid, Main));

    NN_ABORT_UNLESS_RESULT_SUCCESS(pmModule.Finalize());

    DeactivateResourceManager();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::cdhid::usb::UsbHidFinalize());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ahid::hdr::HdrFinalize());
}

namespace {

//!< 拡張ヒープのハンドルを返します。
::nn::lmem::HeapHandle& GetHeapHandle() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(::std::atomic<bool>, s_IsInitialised, (false));

    static ::nn::hid::detail::LockableMutexType s_Mutex =
    {
        NN_OS_MUTEX_INITIALIZER(false)
    };

    static ::nn::lmem::HeapHandle s_HeapHandle;

    static char s_HeapMemory[HeapMemorySize];

    if (!s_IsInitialised)
    {
        ::std::lock_guard<decltype(s_Mutex)> locker(s_Mutex);

        if (!s_IsInitialised)
        {
            s_HeapHandle = ::nn::lmem::CreateExpHeap(
                s_HeapMemory,
                sizeof(s_HeapMemory),
                ::nn::lmem::CreationOption_NoOption);

            s_IsInitialised = true;
        }
    }

    return s_HeapHandle;
}

//!< DebugPad をアクティブ化します。
::nn::Result ActivateDebugPad() NN_NOEXCEPT
{
    if (::nn::hid::detail::FirmwareDebugSettings::ManagesDevices())
    {
        NN_RESULT_DO(
            ::nn::hid::detail::GetResourceManager()
                .GetDebugPadTask().ActivateDebugPad());
    }

    NN_RESULT_SUCCESS;
}

//!< DebugPad を非アクティブ化します。
::nn::Result DeactivateDebugPad() NN_NOEXCEPT
{
    if (::nn::hid::detail::FirmwareDebugSettings::ManagesDevices())
    {
        NN_RESULT_DO(
            ::nn::hid::detail::GetResourceManager()
                .GetDebugPadTask().DeactivateDebugPad());
    }

    NN_RESULT_SUCCESS;
}

//!< Keyboard をアクティブ化します。
::nn::Result ActivateKeyboard() NN_NOEXCEPT
{
    if (::nn::hid::detail::FirmwareDebugSettings::ManagesDevices())
    {
        NN_RESULT_DO(
            ::nn::hid::detail::GetResourceManager()
                .GetKeyboardTask().ActivateKeyboard());
    }

    NN_RESULT_SUCCESS;
}

//!< Keyboard を非アクティブ化します。
::nn::Result DeactivateKeyboard() NN_NOEXCEPT
{
    if (::nn::hid::detail::FirmwareDebugSettings::ManagesDevices())
    {
        NN_RESULT_DO(
            ::nn::hid::detail::GetResourceManager()
                .GetKeyboardTask().DeactivateKeyboard());
    }

    NN_RESULT_SUCCESS;
}

//!< Mouse をアクティブ化します。
::nn::Result ActivateMouse() NN_NOEXCEPT
{
    if (::nn::hid::detail::FirmwareDebugSettings::ManagesDevices())
    {
        NN_RESULT_DO(
            ::nn::hid::detail::GetResourceManager()
                .GetMouseTask().ActivateMouse());
    }

    NN_RESULT_SUCCESS;
}

//!< Mouse を非アクティブ化します。
::nn::Result DeactivateMouse() NN_NOEXCEPT
{
    if (::nn::hid::detail::FirmwareDebugSettings::ManagesDevices())
    {
        NN_RESULT_DO(
            ::nn::hid::detail::GetResourceManager()
                .GetMouseTask().DeactivateMouse());
    }

    NN_RESULT_SUCCESS;
}

//!< TouchScreen をアクティブ化します。
::nn::Result ActivateTouchScreen() NN_NOEXCEPT
{
    if (::nn::hid::detail::FirmwareDebugSettings::ManagesDevices())
    {
        if (::nn::hid::detail::FirmwareDebugSettings::ManagesTouchIcI2c())
        {
            NN_RESULT_DO(
                ::nn::hid::detail::GetResourceManager()
                .GetTouchScreenTask().ActivateTouchScreen());
        }
        else
        {
            NN_RESULT_DO(
                ::nn::hid::detail::GetResourceManager()
                .DeassertTouchIcReset());
        }
    }

    NN_RESULT_SUCCESS;
}

//!< TouchScreen を非アクティブ化します。
::nn::Result DeactivateTouchScreen() NN_NOEXCEPT
{
    if (::nn::hid::detail::FirmwareDebugSettings::ManagesDevices())
    {
        if (::nn::hid::detail::FirmwareDebugSettings::ManagesTouchIcI2c())
        {
            NN_RESULT_DO(
                ::nn::hid::detail::GetResourceManager()
                .GetTouchScreenTask().DeactivateTouchScreen());
        }
        else
        {
            /*
            NN_RESULT_DO(
                ::nn::hid::detail::GetResourceManager()
                .DeassertTouchIcReset());
*/
        }
    }

    NN_RESULT_SUCCESS;
}

//!< TouchScreen を起床状態に遷移させます。
::nn::Result WakeTouchScreenUp() NN_NOEXCEPT
{
    if (::nn::hid::detail::FirmwareDebugSettings::ManagesDevices())
    {
        NN_RESULT_DO(
            ::nn::hid::detail::GetResourceManager()
                .GetTouchScreenTask().WakeTouchScreenUp());
    }

    NN_RESULT_SUCCESS;
}

//!< TouchScreen をスリープ状態に遷移させます。
::nn::Result PutTouchScreenToSleep() NN_NOEXCEPT
{
    if (::nn::hid::detail::FirmwareDebugSettings::ManagesDevices())
    {
        NN_RESULT_DO(
            ::nn::hid::detail::GetResourceManager()
                .GetTouchScreenTask().PutTouchScreenToSleep());
    }

    NN_RESULT_SUCCESS;
}

//!< Gesture をアクティブ化します。
::nn::Result ActivateGesture() NN_NOEXCEPT
{
    if (::nn::hid::detail::FirmwareDebugSettings::ManagesDevices())
    {
        NN_RESULT_DO(
            ::nn::hid::detail::GetResourceManager()
                .GetGestureTask().ActivateGesture());
    }

    NN_RESULT_SUCCESS;
}

//!< Gesture を非アクティブ化します。
::nn::Result DeactivateGesture() NN_NOEXCEPT
{
    if (::nn::hid::detail::FirmwareDebugSettings::ManagesDevices())
    {
        NN_RESULT_DO(
            ::nn::hid::detail::GetResourceManager()
                .GetGestureTask().DeactivateGesture());
    }

    NN_RESULT_SUCCESS;
}

//!< ConsoleSixAxisSensor をアクティブ化します。
::nn::Result ActivateConsoleSixAxisSensor() NN_NOEXCEPT
{
    if (::nn::hid::detail::FirmwareDebugSettings::ManagesDevices())
    {
        NN_RESULT_DO(
            ::nn::hid::detail::GetResourceManager()
                .GetConsoleSixAxisSensorTask().ActivateConsoleSixAxisSensor());
    }

    NN_RESULT_SUCCESS;
}

//!< ConsoleSixAxisSensor を非アクティブ化します。
::nn::Result DeactivateConsoleSixAxisSensor() NN_NOEXCEPT
{
    if (::nn::hid::detail::FirmwareDebugSettings::ManagesDevices())
    {
        NN_RESULT_DO(
            ::nn::hid::detail::GetResourceManager()
                .GetConsoleSixAxisSensorTask().DeactivateConsoleSixAxisSensor());
    }

    NN_RESULT_SUCCESS;
}

//!< ConsoleSixAxisSensor を起床状態に遷移させます。
::nn::Result WakeConsoleSixAxisSensorUp() NN_NOEXCEPT
{
    if (::nn::hid::detail::FirmwareDebugSettings::ManagesDevices())
    {
        NN_RESULT_DO(
            ::nn::hid::detail::GetResourceManager()
                .GetConsoleSixAxisSensorTask().WakeConsoleSixAxisSensorUp());
    }

    NN_RESULT_SUCCESS;
}

//!< ConsoleSixAxisSensor をスリープ状態に遷移させます。
::nn::Result PutConsoleSixAxisSensorToSleep() NN_NOEXCEPT
{
    if (::nn::hid::detail::FirmwareDebugSettings::ManagesDevices())
    {
        NN_RESULT_DO(
            ::nn::hid::detail::GetResourceManager()
                .GetConsoleSixAxisSensorTask().PutConsoleSixAxisSensorToSleep());
    }

    NN_RESULT_SUCCESS;
}

//!< Xpad をアクティブ化します。
::nn::Result ActivateXpad() NN_NOEXCEPT
{
    NN_RESULT_DO(
        ::nn::hid::detail::GetResourceManager()
            .GetXpadTask().ActivateXpad());

    NN_RESULT_SUCCESS;
}

//!< Npad を非アクティブ化します。
::nn::Result DeactivateXpad() NN_NOEXCEPT
{
    NN_RESULT_DO(
        ::nn::hid::detail::GetResourceManager()
            .GetXpadTask().DeactivateXpad());

    NN_RESULT_SUCCESS;
}

//!< コントロールマネージャー をアクティブ化します。
::nn::Result ActivateControlManager() NN_NOEXCEPT
{
    NN_RESULT_DO(
        ::nn::hid::detail::GetResourceManager()
            .GetAbstractedPadXcdTask().ActivateAbstractedPadXcdManager());

    NN_RESULT_SUCCESS;
}

//!< コントロールマネージャー を非アクティブ化します。
::nn::Result DeactivateControlManager() NN_NOEXCEPT
{
    NN_RESULT_DO(
        ::nn::hid::detail::GetResourceManager()
            .GetAbstractedPadXcdTask().DeactivateAbstractedPadXcdManager());

    NN_RESULT_SUCCESS;
}

//!< Npad / UniquePad 共通のタイマーをアクティブ化します
::nn::Result ActivatePadCommonTimer() NN_NOEXCEPT
{
    NN_RESULT_DO(
        ::nn::hid::detail::GetResourceManager()
            .ActivatePadCommonTimer());

    NN_RESULT_SUCCESS;
}

//!< Npad / UniquePad 共通のタイマーを非アクティブ化します
::nn::Result DeactivatePadCommonTimer() NN_NOEXCEPT
{
    NN_RESULT_DO(
        ::nn::hid::detail::GetResourceManager()
            .DeactivatePadCommonTimer());

    NN_RESULT_SUCCESS;
}

//!< Npad をアクティブ化します。
::nn::Result ActivateNpad() NN_NOEXCEPT
{
    NN_RESULT_DO(
        ::nn::hid::detail::GetResourceManager()
            .GetNpadResourceManager().Activate());

    NN_RESULT_SUCCESS;
}

//!< Npad を非アクティブ化します。
::nn::Result DeactivateNpad() NN_NOEXCEPT
{
    NN_RESULT_DO(
        ::nn::hid::detail::GetResourceManager()
            .GetNpadResourceManager().Deactivate());

    NN_RESULT_SUCCESS;
}

//!< Npad を起床状態に遷移させます。
::nn::Result WakeNpadUp() NN_NOEXCEPT
{
    ::nn::hid::detail::GetResourceManager()
        .GetNpadResourceManager().ResumeNpad();
    NN_RESULT_SUCCESS;
}

//!< 振動のFW不具合対応のため、接続されているデバイスを再起動させます。
::nn::Result ResetDevices() NN_NOEXCEPT
{
    NN_RESULT_DO(
        ::nn::hid::detail::GetResourceManager()
        .GetAbstractedPadXcdTask().ResetDevices());

    NN_RESULT_SUCCESS;
}
//!< UniquePad をアクティブ化します。
::nn::Result ActivateUniquePad() NN_NOEXCEPT
{
    NN_RESULT_DO(
        ::nn::hid::detail::GetResourceManager()
            .GetUniquePadResourceManager().Activate());

    NN_RESULT_SUCCESS;
}

//!< UniquePad を非アクティブ化します。
::nn::Result DeactivateUniquePad() NN_NOEXCEPT
{
    NN_RESULT_DO(
        ::nn::hid::detail::GetResourceManager()
            .GetUniquePadResourceManager().Deactivate());

    NN_RESULT_SUCCESS;
}

//!< Palma をアクティブ化します。
::nn::Result ActivatePalma() NN_NOEXCEPT
{
    NN_RESULT_DO(
        ::nn::hid::detail::GetResourceManager()
            .GetPalmaResourceManager().Activate());

    NN_RESULT_SUCCESS;
}

//!< Palma を非アクティブ化します。
::nn::Result DeactivatePalma() NN_NOEXCEPT
{
    NN_RESULT_DO(
        ::nn::hid::detail::GetResourceManager()
            .GetPalmaResourceManager().Deactivate());

    NN_RESULT_SUCCESS;
}

//!< Ahid をアクティブ化します。
::nn::Result ActivateAhid() NN_NOEXCEPT
{
    for (auto& task : ::nn::hid::detail::GetResourceManager().GetAhidTasks())
    {
        NN_RESULT_DO(task.ActivateAhid());
    }

    NN_RESULT_SUCCESS;
}

//!< Ahid を非アクティブ化します。
::nn::Result DeactivateAhid() NN_NOEXCEPT
{
    for (auto& task : ::nn::hid::detail::GetResourceManager().GetAhidTasks())
    {
        NN_RESULT_DO(task.DeactivateAhid());
    }

    NN_RESULT_SUCCESS;
}

//!< SixAxisSensor をアクティブ化します。
::nn::Result ActivateSixAxisSensor() NN_NOEXCEPT
{
    NN_RESULT_DO(
        ::nn::hid::detail::GetResourceManager()
            .GetXpadTask().ActivateSixAxisSensor());

    NN_RESULT_SUCCESS;
}

//!< SixAxisSensor を非アクティブ化します。
::nn::Result DeactivateSixAxisSensor() NN_NOEXCEPT
{
    NN_RESULT_DO(
        ::nn::hid::detail::GetResourceManager()
            .GetXpadTask().DeactivateSixAxisSensor());

    NN_RESULT_SUCCESS;
}

//!< JoySixAxisSensor をアクティブ化します。
::nn::Result ActivateJoySixAxisSensor() NN_NOEXCEPT
{
    NN_RESULT_DO(
        ::nn::hid::detail::GetResourceManager()
            .GetXpadTask().ActivateJoySixAxisSensor());

    NN_RESULT_SUCCESS;
}

//!< JoySixAxisSensor を非アクティブ化します。
::nn::Result DeactivateJoySixAxisSensor() NN_NOEXCEPT
{
    NN_RESULT_DO(
        ::nn::hid::detail::GetResourceManager()
            .GetXpadTask().DeactivateJoySixAxisSensor());

    NN_RESULT_SUCCESS;
}

//!< Xcd ドライバをアクティブ化します。
::nn::Result ActivateXcdDriver() NN_NOEXCEPT
{
    NN_RESULT_DO(
        ::nn::hid::detail::GetResourceManager()
            .GetXcdTask().ActivateXcdDriver());

    NN_RESULT_SUCCESS;
}

//!< Xcd ドライバを非アクティブ化します。
::nn::Result DeactivateXcdDriver() NN_NOEXCEPT
{
    NN_RESULT_DO(
        ::nn::hid::detail::GetResourceManager()
            .GetXcdTask().DeactivateXcdDriver());

    NN_RESULT_SUCCESS;
}

//!< Xcd ドライバ を MinimumAwake 状態に遷移させます。
::nn::Result PutXcdDriverToMinimumAwake() NN_NOEXCEPT
{
    NN_RESULT_DO(
        ::nn::hid::detail::GetResourceManager()
            .GetXcdTask().Resume());

    NN_RESULT_SUCCESS;
}

//!< Xcd ドライバ を SleepReady 状態に遷移させます。
::nn::Result PutXcdDriverToSleepReady() NN_NOEXCEPT
{
/* SIGLO-49256 MinimumAwake 中も Nwcp 通信が有効にするための W/A
    NN_RESULT_DO(
        ::nn::hid::detail::GetResourceManager()
            .GetXcdTask().DisableWiredDeviceRegistration());
*/
    NN_RESULT_DO(
        ::nn::hid::detail::GetResourceManager()
            .GetXcdTask().Suspend());

    NN_RESULT_SUCCESS;
}

//!< Xcd ドライバ を FullAwake 状態に遷移させます。
::nn::Result PutXcdDriverToFullAwake() NN_NOEXCEPT
{
/* SIGLO-49256 MinimumAwake 中の Nwcp 通信が有効にするための W/A
    NN_RESULT_DO(
        ::nn::hid::detail::GetResourceManager()
            .GetXcdTask().EnableWiredDeviceRegistration());
*/
    NN_RESULT_SUCCESS;
}

//!< オーバーレイ通知サービスをアクティブ化します。
::nn::Result ActivateSenderForOverlay() NN_NOEXCEPT
{
    ::nn::hid::detail::GetResourceManager()
        .ActivateSenderForOverlay();
    NN_RESULT_SUCCESS;
}

//!< オーバーレイ通知サービスを非アクティブ化します。
::nn::Result DeactivateSenderForOverlay() NN_NOEXCEPT
{
    ::nn::hid::detail::GetResourceManager()
        .DeactivateSenderForOverlay();

    NN_RESULT_SUCCESS;
}

//!< プレイレポートサービスをアクティブ化します。
::nn::Result ActivatePlayReport() NN_NOEXCEPT
{
    ::nn::hid::detail::GetResourceManager()
        .GetPlayReportTask()
        .ActivatePlayReport();
    NN_RESULT_SUCCESS;
}

//!< プレイレポートサービスを非アクティブ化します。
::nn::Result DeactivatePlayReport() NN_NOEXCEPT
{
    ::nn::hid::detail::GetResourceManager()
        .GetPlayReportTask()
        .DeactivatePlayReport();

    NN_RESULT_SUCCESS;
}

//!< 登録デバイス管理をアクティブ化します。
::nn::Result ActivateRegisteredDeviceManagement() NN_NOEXCEPT
{
    ::nn::hid::detail::GetResourceManager()
        .GetRegisteredDeviceTask()
        .ActivateRegisteredDeviceManagement();
    NN_RESULT_SUCCESS;
}

//!< 登録デバイス管理サービスを非アクティブ化します。
::nn::Result DeactivateRegisteredDeviceManagement() NN_NOEXCEPT
{
    ::nn::hid::detail::GetResourceManager()
        .GetRegisteredDeviceTask()
        .DeactivateRegisteredDeviceManagement();

    NN_RESULT_SUCCESS;
}

::nn::Result ActivateAbstractedPadManager() NN_NOEXCEPT
{
    ::nn::hid::detail::GetResourceManager()
        .GetAbstractedPadManager()
        .Activate();
    NN_RESULT_SUCCESS;
}

::nn::Result DeactivateAbstractedPadManager() NN_NOEXCEPT
{
    ::nn::hid::detail::GetResourceManager()
        .GetAbstractedPadManager()
        .Deactivate();
    NN_RESULT_SUCCESS;
}

void* Allocate(size_t size) NN_NOEXCEPT
{
    return ::nn::lmem::AllocateFromExpHeap(GetHeapHandle(), size);
}

void Deallocate(void* p, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    ::nn::lmem::FreeToExpHeap(GetHeapHandle(), p);
}

RomMounter::RomMounter() NN_NOEXCEPT
    : m_Size(0)
    , m_Buffer(nullptr)
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(::nn::fs::QueryMountRomCacheSize(&m_Size));

    m_Buffer = Allocate(m_Size);

    NN_ABORT_UNLESS_NOT_NULL(m_Buffer);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::fs::MountRom(MountName, m_Buffer, m_Size));
}

RomMounter::~RomMounter() NN_NOEXCEPT
{
    ::nn::fs::Unmount(MountName);

    Deallocate(m_Buffer, m_Size);
}

void ActivateResourceManager() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(ActivateDebugPad());
    NN_ABORT_UNLESS_RESULT_SUCCESS(ActivateKeyboard());
    NN_ABORT_UNLESS_RESULT_SUCCESS(ActivateMouse());
    NN_ABORT_UNLESS_RESULT_SUCCESS(ActivateTouchScreen());
    NN_ABORT_UNLESS_RESULT_SUCCESS(ActivateGesture());
    NN_ABORT_UNLESS_RESULT_SUCCESS(ActivateXcdDriver());
    NN_ABORT_UNLESS_RESULT_SUCCESS(ActivateXpad());
    NN_ABORT_UNLESS_RESULT_SUCCESS(ActivateSenderForOverlay());
    NN_ABORT_UNLESS_RESULT_SUCCESS(ActivatePlayReport());
    NN_ABORT_UNLESS_RESULT_SUCCESS(ActivateRegisteredDeviceManagement());
    NN_ABORT_UNLESS_RESULT_SUCCESS(ActivateNpad());
    NN_ABORT_UNLESS_RESULT_SUCCESS(ActivateUniquePad());
    NN_ABORT_UNLESS_RESULT_SUCCESS(ActivatePalma());
    NN_ABORT_UNLESS_RESULT_SUCCESS(ActivatePadCommonTimer());
    NN_ABORT_UNLESS_RESULT_SUCCESS(ActivateAhid());
    NN_ABORT_UNLESS_RESULT_SUCCESS(ActivateControlManager());
    NN_ABORT_UNLESS_RESULT_SUCCESS(ActivateSixAxisSensor());
    NN_ABORT_UNLESS_RESULT_SUCCESS(ActivateJoySixAxisSensor());
    NN_ABORT_UNLESS_RESULT_SUCCESS(ActivateConsoleSixAxisSensor());
    NN_ABORT_UNLESS_RESULT_SUCCESS(ActivateAbstractedPadManager());
}

void DeactivateResourceManager() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(DeactivateDebugPad());
    NN_ABORT_UNLESS_RESULT_SUCCESS(DeactivateKeyboard());
    NN_ABORT_UNLESS_RESULT_SUCCESS(DeactivateMouse());
    NN_ABORT_UNLESS_RESULT_SUCCESS(DeactivateTouchScreen());
    NN_ABORT_UNLESS_RESULT_SUCCESS(DeactivateGesture());
    NN_ABORT_UNLESS_RESULT_SUCCESS(DeactivateAhid());
    NN_ABORT_UNLESS_RESULT_SUCCESS(DeactivateControlManager());
    NN_ABORT_UNLESS_RESULT_SUCCESS(DeactivateSenderForOverlay());
    NN_ABORT_UNLESS_RESULT_SUCCESS(DeactivatePlayReport());
    NN_ABORT_UNLESS_RESULT_SUCCESS(DeactivateRegisteredDeviceManagement());
    NN_ABORT_UNLESS_RESULT_SUCCESS(DeactivateNpad());
    NN_ABORT_UNLESS_RESULT_SUCCESS(DeactivateUniquePad());
    NN_ABORT_UNLESS_RESULT_SUCCESS(DeactivatePalma());
    NN_ABORT_UNLESS_RESULT_SUCCESS(DeactivatePadCommonTimer());
    NN_ABORT_UNLESS_RESULT_SUCCESS(DeactivateXpad());
    NN_ABORT_UNLESS_RESULT_SUCCESS(DeactivateSixAxisSensor());
    NN_ABORT_UNLESS_RESULT_SUCCESS(DeactivateJoySixAxisSensor());
    NN_ABORT_UNLESS_RESULT_SUCCESS(DeactivateXcdDriver());
    NN_ABORT_UNLESS_RESULT_SUCCESS(DeactivateConsoleSixAxisSensor());
    NN_ABORT_UNLESS_RESULT_SUCCESS(DeactivateAbstractedPadManager());
}

void RegisterHidServerWithServerManager(ServerManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    ::nn::sf::SharedPointer<::nn::hid::IHidServer> pHidServer = {};

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::hid::detail::CreateHidServerProxy(&pHidServer));

    const char* const serviceName = ::nn::hid::HidServiceName;

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        pManager->RegisterObjectForPort(
            pHidServer, SessionCountMax, serviceName));
}

void RegisterHidSystemServerWithServerManager(
    ServerManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    ::nn::sf::SharedPointer<::nn::hid::IHidSystemServer> pHidSystemServer = {};

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::hid::detail::CreateHidSystemServerProxy(&pHidSystemServer));

    const char* const serviceName = ::nn::hid::HidSystemServiceName;

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        pManager->RegisterObjectForPort(
            pHidSystemServer, SessionCountMax, serviceName));
}

void RegisterHidDebugServerWithServerManager(
    ServerManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    ::nn::sf::SharedPointer<::nn::hid::IHidDebugServer> pHidDebugServer = {};

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::hid::detail::CreateHidDebugServerProxy(&pHidDebugServer));

    const char* const serviceName = ::nn::hid::HidDebugServiceName;

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        pManager->RegisterObjectForPort(
            pHidDebugServer, SessionCountMax, serviceName));
}

void RegisterHidTemporaryServerWithServerManager(
    ServerManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    ::nn::sf::SharedPointer<::nn::hid::IHidTemporaryServer> pHidTemporaryServer = {};

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::hid::detail::CreateHidTemporaryServerProxy(&pHidTemporaryServer));

    const char* const serviceName = ::nn::hid::HidTemporaryServiceName;

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        pManager->RegisterObjectForPort(
            pHidTemporaryServer, SessionCountMax, serviceName));
}

void RegisterIrSensorServerWithServerManager(
    ServerManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    ::nn::sf::SharedPointer<
        ::nn::irsensor::IIrSensorServer> pIrSensorServer = {};

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::irsensor::detail::CreateIrSensorServerProxy(&pIrSensorServer));

    const char* const serviceName = ::nn::irsensor::IrSensorServiceName;

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        pManager->RegisterObjectForPort(
            pIrSensorServer, SessionCountMax, serviceName));
}

void RegisterIrSensorSystemServerWithServerManager(
    ServerManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    ::nn::sf::SharedPointer<
        ::nn::irsensor::IIrSensorSystemServer> pIrSensorSystemServer = {};

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::irsensor::detail::CreateIrSensorSystemServerProxy(
            &pIrSensorSystemServer));

    const char* const serviceName = ::nn::irsensor::IrSensorSystemServiceName;

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        pManager->RegisterObjectForPort(
            pIrSensorSystemServer, SessionCountMax, serviceName));
}

void RegisterDependenciesWithPmModule(::nn::psc::PmModule* pModule) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pModule);

    const ::nn::psc::PmModuleId dependencies[] = {
        ::nn::psc::PmModuleId_Usb,
        ::nn::psc::PmModuleId_Gpio,
        ::nn::psc::PmModuleId_Uart,
        ::nn::psc::PmModuleId_I2c,
        ::nn::psc::PmModuleId_Spi,
        ::nn::psc::PmModuleId_Psm,
        ::nn::psc::PmModuleId_Bluetooth,
        ::nn::psc::PmModuleId_Btm,
        ::nn::psc::PmModuleId_Bpc,
        ::nn::psc::PmModuleId_Sasbus,
    };

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        pModule->Initialize(
            ::nn::psc::PmModuleId_Hid,
            dependencies, NN_ARRAY_SIZE(dependencies),
            ::nn::os::EventClearMode_ManualClear));
}

void DispatchPmModuleRequest(::nn::psc::PmModule* pModule) NN_NOEXCEPT
{
    auto pmState = ::nn::psc::PmState();
    auto pmFlags = ::nn::psc::PmFlagSet();

    static nn::psc::PmState currentState = nn::psc::PmState_FullAwake;

    NN_ABORT_UNLESS_RESULT_SUCCESS(pModule->GetRequest(&pmState, &pmFlags));

    ::nn::Result result = ::nn::ResultSuccess();

    auto folder = [](::nn::Result* pResult, ::nn::Result result) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pResult);

        if (pResult->IsSuccess())
        {
            *pResult = result;
        }
    };

    switch (pmState)
    {
    case ::nn::psc::PmState_FullAwake:
        folder(&result, PutXcdDriverToFullAwake());
        folder(&result, WakeTouchScreenUp());
        folder(&result, WakeNpadUp());
        folder(&result, WakeConsoleSixAxisSensorUp());
        break;

    case ::nn::psc::PmState_MinimumAwake:
        if (currentState == ::nn::psc::PmState_FullAwake)
        {
            folder(&result, ResetDevices());
        }
        folder(&result, PutXcdDriverToMinimumAwake());
        folder(&result, PutTouchScreenToSleep());
        folder(&result, PutConsoleSixAxisSensorToSleep());
        break;

    case ::nn::psc::PmState_SleepReady:
        folder(&result, PutXcdDriverToSleepReady());
        folder(&result, PutTouchScreenToSleep());
        folder(&result, PutConsoleSixAxisSensorToSleep());
        break;

    default:
        break;
    }

    currentState = pmState;

    NN_ABORT_UNLESS_RESULT_SUCCESS(pModule->Acknowledge(pmState, result));
}

} // namespace
