﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/htcs.h>
#include <cstring>
#include <mutex>

#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>

#include <nn/arp/arp_Api.h>
#include <nn/arp/arp_Result.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_Bis.h>
#include <nn/htc/tenv/htc_Tenv.h>
#include <nn/ldr/ldr_ShellApi.h>
#include <nn/lr/lr_LocationResolver.h>
#include <nn/lr/lr_Service.h>
#include <nn/ncm/ncm_ProgramId.h>
#include <nn/ns/ns_DevelopApi.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/os.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/settings/system/settings_FirmwareVersion.h>
#include <nn/tc/tc.h>
#include <nn/tc/tc_VirtualTemperature.h>
#include <nn/util/util_StringUtil.h>
#include <nn/capsrv/capsrv_ScreenShotControl.h>

#include "cs_CommandProcessor.h"
#include "cs_Result.public.h"

namespace nn { namespace cs {

namespace {

    const size_t DataSizeMax = 4 * 1024; //sizeof(ResponseHeader) + sizeof(uint32_t) + (sizeof(nn::ns::ApplicationTitle::name));
    Bit8 s_Data[DataSizeMax]; // Buffer for response

    Result GetTitleName(size_t* pOutSize, char* pOutName, size_t nameSize, os::ProcessId processId)
    {
        static ns::ApplicationControlProperty s_Property;
        memset(&s_Property, 0, sizeof(s_Property));

        Result result;
        for (int i = 0; i < 50; i++)
        {
            result = arp::GetApplicationControlProperty(&s_Property, processId);
            if (!arp::ResultNotRegistered::Includes(result))
            {
                break;
            }

            // We need to return as failure once the process whose title is being requested is terminated.
            // WA until SIGLO-84591 is completed.
            // Prerequisite: It is not necessary to return title for non-Application
            {
                os::ProcessId applicationPid;
                if (ns::GetRunningApplicationProcessIdForDevelop(&applicationPid).IsFailure() || applicationPid != processId)
                {
                    result = ns::ResultApplicationNotRunning();
                    break;
                }
            }
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        }
        if( result.IsFailure() )
        {
            return result;
        }

        const auto& title = s_Property.GetDefaultTitle();
        const auto length = nn::util::Strnlen( title.name, sizeof( title.name ) );
        NN_ABORT_UNLESS( length < sizeof( title.name ) );

        if ( length > 0 )
        {
            // NULL 終端するために +1 文字
            nn::util::Strlcpy( pOutName, reinterpret_cast<const char*>( title.name ), length + 1 );
            *pOutSize = length + 1;
        }
        else
        {
            const char DefaultName[] = "No Name";
            const auto DefaultNameLength = sizeof( DefaultName );
            nn::util::Strlcpy( pOutName, DefaultName, DefaultNameLength );
            *pOutSize = DefaultNameLength;
        }

        return ResultSuccess();
    }

    void SendNullData(int socket, size_t size)
    {
        std::lock_guard<os::SdkRecursiveMutexType> lock(*nn::scs::GetHtcsSendMutex());
        std::memset(s_Data, 0, DataSizeMax);
        while (size > 0)
        {
            size_t sendSize = std::min(size, DataSizeMax);
            nn::htcs::Send(socket, s_Data, sendSize, nn::htcs::HTCS_MSG_WAITALL);
            size -= sendSize;
        }
    }
}

    void CommandProcessor::SendFirmwareVersion(int socket, const CommandHeader& command)
    {
        ResponseFirmwareVersion rfv=
        {
            {
                command.id,
                Response_FirmwareVersion,
                sizeof(ResponseFirmwareVersion) - sizeof(ResponseHeader)
            },
            {0}
        };

        nn::settings::system::GetFirmwareVersion(&rfv.firmwareVersion);
        std::lock_guard<os::SdkRecursiveMutexType> lock(*nn::scs::GetHtcsSendMutex());
        htcs::Send(socket, &rfv, sizeof(rfv), htcs::HTCS_MSG_WAITALL);
    }

    void CommandProcessor::SendTitleName(const CommandHeader& header, const void* pBody, int socket)
    {
        os::ProcessId processId;
        processId.value = *(static_cast<const Bit64*>(pBody));

        const size_t bufferSize = sizeof(ns::ApplicationTitle::name) + 1;
        char buffer[bufferSize];
        size_t outSize;
        auto result = GetTitleName(&outSize, buffer, bufferSize, processId);
        if (result.IsFailure())
        {
            // 失敗をクライアントに通知
            SendErrorResult(socket, header, result);
            return;
        }

        uint32_t titleLength = outSize;
        ResponseHeader rh =
        {
            header.id,
            Response_TitleName,
            4 + titleLength
        };

        size_t dataSize = sizeof(rh) + sizeof(titleLength) + (sizeof(char) * outSize);
        Bit8* pData(new(&s_Data) Bit8[dataSize]);
        std::memcpy(pData, &rh, sizeof(rh));
        std::memcpy(pData + sizeof(rh), &titleLength, sizeof(titleLength));
        std::memcpy(pData + sizeof(titleLength) + sizeof(rh), buffer, sizeof(char) * titleLength);

        std::lock_guard<os::SdkRecursiveMutexType> lock(*nn::scs::GetHtcsSendMutex());
        htcs::Send(socket, pData, dataSize, htcs::HTCS_MSG_WAITALL);
    }

    void CommandProcessor::ProcessCommandControlVirtualTemperature(
        const CommandHeader& header,
        const void* pBody,
        int socket )
    {
        CommandControlVirtualTemperature ccvt;
        std::memcpy(&ccvt, pBody, sizeof(ccvt));

#if defined(NN_BUILD_CONFIG_SPEC_NX)
        nn::tc::Initialize();
        Result result;
        if( ccvt.isEnable )
        {
            result = nn::tc::SetVirtualTemperature(nn::tc::Location_ThermalSensorInternal, ccvt.temperature);
            if( result.IsFailure() )
            {
                SendErrorResult(socket, header, result);
                nn::tc::Finalize();
                return;
            }

            result = nn::tc::EnableVirtualTemperature(nn::tc::Location_ThermalSensorInternal);
            if( result.IsFailure() )
            {
                SendErrorResult(socket, header, result);
                nn::tc::Finalize();
                return;
            }
        }
        else
        {
            result = nn::tc::DisableVirtualTemperature(nn::tc::Location_ThermalSensorInternal);
            if( result.IsFailure() )
            {
                SendErrorResult(socket, header, result);
                nn::tc::Finalize();
                return;
            }

            result = nn::tc::SetVirtualTemperature(nn::tc::Location_ThermalSensorInternal, 48000);
            if( result.IsFailure() )
            {
                SendErrorResult(socket, header, result);
                nn::tc::Finalize();
                return;
            }
        }
        nn::tc::Finalize();

        // 成功をクライアントに通知
        SendSuccess(socket, header);
#else
        SendErrorResult(socket, header, ResultNotImplemented());
#endif
    }

    void CommandProcessor::ProcessCommandTakeScreenShot(const CommandHeader& header, int socket)
    {
        Result result = nn::capsrv::InitializeScreenShotControl();
        if (result.IsFailure())
        {
            SendErrorResult(socket, header, result);
            return;
        }

        size_t dataSize;
        int width;
        int height;
        result = nn::capsrv::OpenRawScreenShotReadStreamForDevelop(&dataSize, &width, &height,
            nn::vi::LayerStack::LayerStack_ApplicationForDebug, nn::TimeSpan::FromSeconds(10));
        if (result.IsFailure())
        {
            SendErrorResult(socket, header, result);
            nn::capsrv::FinalizeScreenShotControl();
            return;
        }

        ResponseHeader rh =
        {
            header.id,
            Response_ScreenShot,
        };
        rh.dataSize = sizeof(int32_t) + sizeof(int32_t) + sizeof(int32_t) + dataSize;

        std::memcpy(s_Data, &rh, sizeof(rh));
        int32_t dataSize32 = (int32_t)dataSize;
        std::memcpy(s_Data + sizeof(rh), &dataSize32, sizeof(dataSize32));
        int32_t width32 = (int32_t)width;
        std::memcpy(s_Data + sizeof(rh) + sizeof(dataSize32), &width32, sizeof(width32));
        int32_t height32 = (int32_t)height;
        std::memcpy(s_Data + sizeof(rh) + sizeof(dataSize32) + sizeof(width32), &height32, sizeof(height32));

        std::lock_guard<os::SdkRecursiveMutexType> lock(*nn::scs::GetHtcsSendMutex());
        htcs::Send(socket, s_Data, sizeof(rh) + sizeof(dataSize32) + sizeof(width32) + sizeof(height32), htcs::HTCS_MSG_WAITALL);

        size_t readSize;
        size_t totalReadSize = 0;
        while (totalReadSize < dataSize)
        {
            result = nn::capsrv::ReadRawScreenShotReadStreamForDevelop(&readSize, s_Data, DataSizeMax, totalReadSize);
            if (result.IsFailure())
            {
                SendNullData(socket, dataSize - totalReadSize);
                SendErrorResult(socket, header, result);
                nn::capsrv::CloseRawScreenShotReadStreamForDevelop();
                nn::capsrv::FinalizeScreenShotControl();
                return;
            }
            htcs::Send(socket, s_Data, readSize, htcs::HTCS_MSG_WAITALL);
            totalReadSize += readSize;
        }
        nn::capsrv::CloseRawScreenShotReadStreamForDevelop();
        nn::capsrv::FinalizeScreenShotControl();
        SendSuccess(socket, header);
    }

    bool CommandProcessor::ProcessCommand(const CommandHeader& header, const Bit8* pBody, int socket)
    {
        switch(header.command)
        {
        // ファームウェアバージョン取得
        case Command_GetFirmwareVersion:
            {
                SendFirmwareVersion(socket, header);
            }
            break;

        // タイトル文字列取得
        case Command_GetTitleName:
            {
                SendTitleName(header, pBody, socket);
            }
            break;

        // 仮想温度制御
        case Command_ControlVirtualTemperature:
            {
                ProcessCommandControlVirtualTemperature(header, pBody, socket);
            }
            break;

        // スクリーンショット
        case Command_TakeScreenShot:
            {
                ProcessCommandTakeScreenShot(header, socket);
            }
            break;

        // scs と共通のコマンド
        default:
            {
                nn::scs::CommandProcessor::ProcessCommand(header, pBody, socket);
            }
            break;
        }

        return true;
    }

}}  // namespace nn::cs
