﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//#define NN_CS_ENABLE_AUDIO_THREAD_LOG

#ifdef NN_CS_ENABLE_AUDIO_THREAD_LOG
#define NN_CS_AUDIO_LOG(...) NN_SDK_LOG(__VA_ARGS__)
#else
#define NN_CS_AUDIO_LOG(...)
#endif

#include <nn/nn_SdkLog.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/audio/audio_FinalOutputRecorder.h>
#include <nn/htcs.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/util/util_StringUtil.h>

#include <nn/util/util_TinyMt.h>

#include "cs_AudioServer.h"

namespace nn { namespace cs {

namespace {

const char PortName[] = "iywys@$audio";

const size_t RecorderThreadStackSize = 4 * nn::os::ThreadStackAlignment;

const int InvalidSocket = -1;

const int ChannelCount = 6;

const int SamplingRate = 48000;

const int IntervalMsec = 20;

const int BitDepth = 16;

const int ProcessingBufferCount = 4;

NN_ALIGNAS(nn::audio::FinalOutputRecorderBuffer::AddressAlignment)
    char g_Buffer[ProcessingBufferCount][nn::audio::FinalOutputRecorderBuffer::AddressAlignment *
                                         (((BitDepth / 8) * ChannelCount * SamplingRate * IntervalMsec / 1000 / nn::audio::FinalOutputRecorderBuffer::AddressAlignment) + 1)];
}

bool RecvMessage(int sock, char* buffer, size_t bufferSize, const char* const msg)
{
    while(true)
    {
        nn::htcs::ssize_t receivedBytes = nn::htcs::Recv(sock, buffer, bufferSize - 1, 0);


        if (receivedBytes < 0)
        {
            return false;
        }
        else if (receivedBytes == 0)
        {
            return false;
        }

        buffer[receivedBytes] = '\0';
        // BinaryWriter で Write した場合、最初の一バイト目に何かを示す数字がやってくるので、それを考慮して文字列比較する
        if (strstr(buffer, msg) != nullptr)
        {
            break;
        }
    }
    return true;
}

bool SendMessage(int sock, void* buffer, size_t bufferSize)
{
    nn::htcs::ssize_t sentBytes = nn::htcs::Send(sock, buffer, bufferSize, 0);
    if (sentBytes < 0)
    {
        return false;
    }
    return true;
}

bool SendWaveForm(int sock, nn::audio::FinalOutputRecorder* pRecorder, nn::os::SystemEvent* pEvent)
{
    pEvent->Wait();
    nn::audio::FinalOutputRecorderBuffer* pAudioInBuffer = nullptr;
    pAudioInBuffer = GetReleasedFinalOutputRecorderBuffer(pRecorder);

    while(pAudioInBuffer != nullptr)
    {
        // TODO: メッセージのやり取りを行うかを決め、行う場合はメッセージの内容を決める
        char buf[128];
        const char WaveRequireMessage[] = "msg[Waveform Require]";
        bool result = RecvMessage(sock, buf, sizeof(buf), WaveRequireMessage);
        if (!result)
        {
            return false;
        }

        void* inBuffer = nn::audio::GetFinalOutputRecorderBufferDataPointer(pAudioInBuffer);
        size_t inSize = (BitDepth / 8) * SamplingRate * ChannelCount * IntervalMsec / 1000;
        result = SendMessage(sock, inBuffer, inSize);
        if (!result)
        {
            return false;
        }

        int32_t adspLoad = nn::audio::GetAdspLoad();
        result = SendMessage(sock, &adspLoad, sizeof(adspLoad));
        if (!result)
        {
            return false;
        }

        nn::audio::AppendFinalOutputRecorderBuffer(pRecorder, pAudioInBuffer);
        pAudioInBuffer = GetReleasedFinalOutputRecorderBuffer(pRecorder);
    }
    return true;
}

void RecorderThreadFunc(void* arg) NN_NOEXCEPT
{
    // サービスの登録
    nn::htcs::SockAddrHtcs serviceAddr;
    serviceAddr.family = nn::htcs::HTCS_AF_HTCS;
    serviceAddr.peerName = nn::htcs::GetPeerNameAny();
    ::nn::util::Strlcpy(
        serviceAddr.portName.name,
        PortName,
        ::std::extent<decltype(serviceAddr.portName.name)>::value);

    while(true)
    {
        int serviceSocket = nn::htcs::Socket();
        if(serviceSocket == InvalidSocket)
        {
            // TODO: error handling
            NN_CS_AUDIO_LOG("%s%d, error: %d\n", __func__, __LINE__, nn::htcs::GetLastError());
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
            continue;
        }

        if(nn::htcs::Bind(serviceSocket, &serviceAddr) != 0)
        {
            // TODO: error handling
            NN_CS_AUDIO_LOG("%s%d, error: %d\n", __func__, __LINE__, nn::htcs::GetLastError());
            nn::htcs::Close(serviceSocket);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
            continue;
        }

        if(nn::htcs::Listen(serviceSocket, 1) != 0)
        {
            // TODO: error handling
            NN_CS_AUDIO_LOG("%s%d, error: %d\n", __func__, __LINE__, nn::htcs::GetLastError());
            nn::htcs::Close(serviceSocket);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
            continue;
        }

        // 接続待ち受け
        int socket = nn::htcs::Accept(serviceSocket, nullptr);
        if(socket < 0)
        {
            // TODO: error handling
            NN_CS_AUDIO_LOG("%s%d, error: %d\n", __func__, __LINE__, nn::htcs::GetLastError());
            nn::htcs::Close(serviceSocket);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
            continue;
        }

        // FinalOutputRecorder の初期化
        nn::audio::FinalOutputRecorder recorder;
        nn::os::SystemEvent recorderBufferEvent;
        nn::audio::FinalOutputRecorderParameter paramRecorder{ 0 };
        nn::audio::InitializeFinalOutputRecorderParameter(&paramRecorder);
        paramRecorder.sampleRate = SamplingRate;
        paramRecorder.channelCount = ChannelCount;

        while (nn::audio::OpenFinalOutputRecorder(&recorder, &recorderBufferEvent, paramRecorder).IsFailure())
        {
            // TODO: error handling
            NN_CS_AUDIO_LOG("%s%d\n", __func__, __LINE__);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
        }
        const int dataSize = (BitDepth / 8) * ChannelCount * SamplingRate * IntervalMsec / 1000;
        const int bufferSize = nn::util::align_up(dataSize, nn::audio::FinalOutputRecorderBuffer::SizeGranularity);
        nn::audio::FinalOutputRecorderBuffer audioInBuffer[ProcessingBufferCount];

        void* inBuffer[ProcessingBufferCount];

        for (int i = 0; i < ProcessingBufferCount; ++i)
        {
            inBuffer[i] = g_Buffer[i];
            nn::audio::SetFinalOutputRecorderBufferInfo(&audioInBuffer[i], inBuffer[i], bufferSize, dataSize);
            nn::audio::AppendFinalOutputRecorderBuffer(&recorder, &audioInBuffer[i]);
        }
        while(nn::audio::StartFinalOutputRecorder(&recorder).IsFailure())
        {
            // TODO: error handling
            NN_CS_AUDIO_LOG("%s%d\n", __func__, __LINE__);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
        }

        // 波形送信
        while (SendWaveForm(socket, &recorder, &recorderBufferEvent))
        {
        }

        nn::audio::StopFinalOutputRecorder(&recorder);
        nn::audio::CloseFinalOutputRecorder(&recorder);

        nn::os::DestroySystemEvent(recorderBufferEvent.GetBase());

        nn::htcs::Close(socket);
        nn::htcs::Close(serviceSocket);
    }
} // NOLINT(impl/function_size)

void InitializeAudioServer() NN_NOEXCEPT
{
    static nn::os::ThreadType s_RecorderThread = {};
    NN_OS_ALIGNAS_THREAD_STACK static char s_RecorderThreadStack[RecorderThreadStackSize];
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::CreateThread(
            &s_RecorderThread,
            &RecorderThreadFunc,
            nullptr,
            reinterpret_cast<void*>(s_RecorderThreadStack),
            sizeof(s_RecorderThreadStack),
            NN_SYSTEM_THREAD_PRIORITY(cs, AudioRecorder)));
    nn::os::SetThreadNamePointer(
        &s_RecorderThread, NN_SYSTEM_THREAD_NAME(cs, AudioRecorder));
    nn::os::StartThread(&s_RecorderThread);
}

}}
