﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/web/web_OfflineApi.h>

#include <mutex>
#include <string>

#include <nn/applet/applet_Types.h>
#include <nn/web/web_OfflineApiPrivate.h>
#include <nn/web/web_Result.h>
#include <nn/la/la_CommonArgumentsWriter.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os/os_Mutex.h>
#include <nn/web/common/web_CommonApi.h>
#include <nn/web/common/web_CommonArgData.h>
#include <nn/web/common/web_CommonTypes.h>
#include <nn/web/detail/web_OfflineApiDetail.h>
#include <nn/web/detail/web_OfflineTypesDetail.h>

namespace nn { namespace web {

struct StaticMutex
{
    nn::os::MutexType mutex;

    void lock() NN_NOEXCEPT
    {
        nn::os::LockMutex(&mutex);
    }

    void unlock() NN_NOEXCEPT
    {
        nn::os::UnlockMutex(&mutex);
    }
};

static nn::applet::LibraryAppletHandle g_Handle = nn::applet::InvalidLibraryAppletHandle;
static StaticMutex g_Mutex = { NN_OS_MUTEX_INITIALIZER(false) };

//------------------------------------------------------------------------
nn::Result ShowOfflineHtmlPage(OfflineHtmlPageReturnValue* pOutputOfflineHtmlPageReturnValue,
    const ShowOfflineHtmlPageArg& arg) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutputOfflineHtmlPageReturnValue);

    {
        std::lock_guard<StaticMutex> lock(g_Mutex);
        g_Handle = nn::web::common::CommonApi::PrepareLibraryApplet(applet::AppletId_LibraryAppletOfflineWeb);
    }

    // 共通引数をプッシュ
    nn::la::CommonArgumentsWriter writer(nn::web::common::ShimVersion, detail::OfflineShimVersion);
    writer.PushToInChannel(g_Handle);

    // アプリケーションの「遊び方」を開くための起動引数をプッシュ
    {
        auto writer = reinterpret_cast<const detail::ShowOfflineHtmlPageArgWriter*>(&arg);
        writer->PushToInChannel(g_Handle);
    }

    // アプリケーションの「遊び方」を開く、その後終了する迄待機します。
    auto result = nn::web::common::CommonApi::StartLibraryAppletAndWait(pOutputOfflineHtmlPageReturnValue, sizeof(*pOutputOfflineHtmlPageReturnValue), g_Handle);
    g_Handle = nn::applet::InvalidLibraryAppletHandle;

    if (result.IsSuccess())
    {
        return nn::ResultSuccess();
    }
    else
    {
        return nn::web::ResultTerminated();
    }
}

//------------------------------------------------------------------------
void RequestExitOfflineHtmlPage() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_Mutex);
    // 起動中なら終了要求を送ります。ハンドルが無効または終了済みなら何もしません。
    if (g_Handle != nn::applet::InvalidLibraryAppletHandle &&
        !nn::os::TryWaitSystemEvent(nn::applet::GetLibraryAppletExitEvent(g_Handle))) {
        nn::applet::RequestExitLibraryApplet(g_Handle);
    }
}

}} // namespace nn::web
