﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/os.h>
#include <nn/ns/ns_ApplicationDownloadApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_RetailInteractiveDisplayApi.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/rid/rid_Result.h>
#include <nn/rid/rid_MenuUpdateApi.h>
#include "detail/rid_NetworkConnection.h"
#include "detail/rid_Settings.h"

namespace nn { namespace rid {

    MenuUpdater::MenuUpdater() NN_NOEXCEPT : m_Mutex(false), m_IsCancelled(false)
    {
        m_State = MenuUpdateState::DoNothing;
    }

    Result MenuUpdater::Execute() NN_NOEXCEPT
    {
        Result result = ExecuteImpl();
        NN_RESULT_TRY(result)
            NN_RESULT_CATCH(ResultMenuUpdateCancelled)
            {
                UpdateState(MenuUpdateState::Cancelled);
            }
            NN_RESULT_CATCH_ALL
            {
                UpdateState(MenuUpdateState::Failed);
            }
        NN_RESULT_END_TRY

        return result;
    }

    void MenuUpdater::Cancel() NN_NOEXCEPT
    {
        m_IsCancelled = true;
    }

    MenuUpdateState MenuUpdater::GetState() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_Mutex);

        return m_State;
    }

    Result MenuUpdater::ExecuteImpl() NN_NOEXCEPT
    {
        m_IsCancelled = false;
        m_State = MenuUpdateState::DoNothing;

        NetworkConnection connection;
        if (!connection.IsAvailable())
        {
            NN_RESULT_THROW(ResultNetworkNotConnected());
        }

        ncm::ApplicationId menuApplicationId = GetMenuApplicationId();

        // 追加コンテンツのダウンロードタスクが残っていた場合のために、タスクをキャンセルする
        NN_RESULT_DO(ns::CancelApplicationDownload(menuApplicationId));

        NN_RESULT_TRY(ns::DownloadApplication(menuApplicationId, ncm::StorageId::BuiltInUser));
            NN_RESULT_CATCH(ns::ResultAlreadyUpToDate)
            {
                UpdateState(MenuUpdateState::NeedNoUpdate);
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH_ALL{}
        NN_RESULT_END_TRY

        ns::ApplicationDownloadTaskStatus status;
        NN_RESULT_DO(ns::GetApplicationDownloadTaskStatus(&status, menuApplicationId));

        int unavailableCount = 0;
        while (status == ns::ApplicationDownloadTaskStatus::Running)
        {
            if (m_IsCancelled)
            {
                NN_RESULT_DO(ns::CancelApplicationDownload(menuApplicationId));
                NN_RESULT_THROW(ResultMenuUpdateCancelled());
            }

            NN_RESULT_DO(ns::GetApplicationDownloadTaskStatus(&status, menuApplicationId));
            UpdateState(MenuUpdateState::Downloading);

            if (!connection.IsAvailable())
            {
                if (unavailableCount > 60)
                {
                    NN_RESULT_DO(ns::CancelApplicationDownload(menuApplicationId));
                    NN_RESULT_THROW(ResultNetworkNotConnected());
                }
                unavailableCount++;

                connection.SubmitRequestAndWait();
            }
            else
            {
                unavailableCount = 0;
            }

            os::SleepThread(TimeSpan::FromSeconds(1));
        }

        if (status == ns::ApplicationDownloadTaskStatus::Downloaded)
        {
            UpdateState(MenuUpdateState::Completed);
        }
        else
        {
            NN_RESULT_DO(ns::CancelApplicationDownload(menuApplicationId));
            NN_RESULT_THROW(ResultMenuUpdateFailed());
        }

        NN_RESULT_SUCCESS;
    }

    void MenuUpdater::UpdateState(MenuUpdateState state) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(m_Mutex);

        m_State = state;
    }
}}
