﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/init.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <glv.h>
#include <nn/repair/repair_LabelText.h>
#include <nn/util/util_FormatString.h>
#include <nn/settings/factory/settings_SerialNumber.h>
#include <nn/repair/repair_Api.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nifm/nifm_ApiWirelessCommunicationControl.h>
#include "detail/repair_FirmwareVersion.h"

namespace nn {
    namespace repair {
        void GetFirmwareVersionLabel(glv::Label* pOutLabel) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pOutLabel);

            detail::FirmwareVersion fw_version;

            char buffer[1024];
            nn::util::SNPrintf(buffer, 1024, "System version\t\t\t%s", fw_version.DisplayVersion());
            pOutLabel->setValue(std::string(buffer));
        }

        nn::Result GetDeviceIdLabel(glv::Label* pOutLabel) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pOutLabel);

            char deviceId[nn::repair::DeviceIdLength];
            nn::Result result = nn::repair::GetDeviceId(deviceId);

            char buffer[1024];

            if (result.IsSuccess())
            {
                nn::util::SNPrintf(buffer, 1024, "Device ID\t\t\t %s", deviceId);
            }
            else
            {
                nn::util::SNPrintf(buffer, 1024, "Failed to get Device ID", deviceId);
            }
            pOutLabel->setValue(std::string(buffer));

            return result;
        }

        nn::Result GetSerialNumberLabel(glv::Label* pOutLabel) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pOutLabel);

            nn::settings::factory::SerialNumber serialNumber;
            nn::Result result = nn::settings::factory::GetSerialNumber(&serialNumber);

            char buffer[1024];

            if (result.IsSuccess())
            {
                nn::util::SNPrintf(buffer, 1024, "Serial number\t\t\t %s", serialNumber.string);
            }
            else
            {
                nn::util::SNPrintf(buffer, 1024, "Failed to get Serial number");
            }

            pOutLabel->setValue(std::string(buffer));

            return result;
        }

        void GetToolInformationLabel(glv::Label* pOutLabel, const char* toolName, int majorVersion, int minorVersion) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pOutLabel);

            char buffer[1024];
            nn::util::SNPrintf(buffer, 1024, "%s\t\t\t\t\t\tVer.%d.%d", toolName, majorVersion, minorVersion);
            pOutLabel->setValue(std::string(buffer));
            pOutLabel->pos(10, 25);
            pOutLabel->size(32.0f);
        }

        void GetTransfererSerialNumberLabel(glv::Label* pOutLabel, const nn::settings::factory::SerialNumber* pSerialNumber) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pOutLabel);
            NN_SDK_REQUIRES_NOT_NULL(pSerialNumber);

            char buffer[1024];
            nn::util::SNPrintf(buffer, 1024, "Transferer Serial number\t\t %s", pSerialNumber->string);
            pOutLabel->setValue(std::string(buffer));
        }

        void GetAirplaneModeLabel(glv::Label* pOutLabel) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pOutLabel);

            bool enables = nn::nifm::IsWirelessCommunicationEnabled();

            char buffer[1024];
            nn::util::SNPrintf(buffer, 1024, "Airplane Mode\t\t\t %s", enables?"disabled":"enabled");
            pOutLabel->setValue(std::string(buffer));
        }
    }
}
