﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/fs.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/repair/repair_CryptUtility.h>
#include <nn/repair/repair_FileSystem.h>
#include <string>
#include "repair_BlackList.h"
#include "repair_CachedArchiveFile.h"
#include "repair_SaveData.h"


namespace nn { namespace repair { namespace detail {

//!< サムネイルデータエントリを扱うクラスです。
class ThumbnailDataEntry final
{
private:
    //!< ヘッダ用の構造体です。
    struct Header final
    {
        //!< ファイル数
        int64_t count;

        //!< セーブデータ情報
        nn::fs::SaveDataInfo info;
    };

    //!< ファイルメタ用の構造体です。
    struct Meta final
    {
        //!< ファイルサイズ
        int64_t size;
    };

    //!< マウント名
    static const std::string MountName;

    //!< ルートパス
    static const std::string RootPath;

    //!< スペース Id
    nn::fs::SaveDataSpaceId m_SpaceId;

public:
    ThumbnailDataEntry() NN_NOEXCEPT
        : m_SpaceId(nn::fs::SaveDataSpaceId::User)
    {
    }

    //!< エントリを数えます。
    nn::Result CountEntries(
            int64_t *pOutCount, const BlackList* pBlackList) const NN_NOEXCEPT;

    //!< アーカイブファイルに書き込みます。
    nn::Result WriteArchive(
            std::shared_ptr<CachedArchiveFile> pFile, const BlackList* pBlackList) const NN_NOEXCEPT;

    //!< エントリを書き込みます。
    nn::Result WriteEntries(
            int64_t* pOutSize,
            std::shared_ptr<CachedArchiveFile> pFile,
            int64_t entriesCount,
            int64_t offset) const NN_NOEXCEPT;

private:
    //!< ヘッダを書き込みます。
    nn::Result WriteHeader(std::shared_ptr<CachedArchiveFile> pFile, const SaveData& saveData) const NN_NOEXCEPT;

    //!< ボディを書き込みます。
    nn::Result WriteBody(std::shared_ptr<CachedArchiveFile> pFile, const SaveData& saveData) const NN_NOEXCEPT;

    //!< ファイルメタ&データを書き込みます。
    nn::Result WriteFileMetaAndData(std::shared_ptr<CachedArchiveFile> pFile, const SaveData& saveData) const NN_NOEXCEPT;

    //!< セーブデータ情報とファイル数を取得します。
    nn::Result GetInfoAndFileCount(
            nn::fs::SaveDataInfo* pOutInfo, int64_t* pOutCount, std::shared_ptr<CachedArchiveFile> pFile, int64_t offset) const NN_NOEXCEPT;

    //!< ファイルサイズを取得します。
    nn::Result GetFileSize(
            int64_t* pOutSize, std::shared_ptr<CachedArchiveFile> pFile, int64_t offset) const NN_NOEXCEPT;

    //!< すべてのファイルを書き込みます。
    nn::Result WriteFiles(
            int64_t* pOutSize, std::shared_ptr<CachedArchiveFile> pFile,
            const nn::fs::SaveDataInfo& info, int64_t fileCount, int64_t offset) const NN_NOEXCEPT;

    //!< ファイルをベリファイします。
    nn::Result VerifyFiles(const SaveData& saveData) const NN_NOEXCEPT;

    //!< ファイルが有効か確認します。
    bool IsValidFiles(const SaveData& saveData) const NN_NOEXCEPT;

    //!< 破損情報を記録します。
    nn::Result SendCorruptionLog(const char* pFormat, ...) const NN_NOEXCEPT;
};

}}} // namespace nn::repair::detail

