﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/fs/fs_BcatSaveData.h>
#include <nn/fs/fs_DeviceSaveData.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/repair/repair_Result.h>
#include <string>

#include "../repair_Utility.h"
#include "repair_BlackList.h"
#include "repair_MessageReporter.h"
#include "repair_SaveData.h"

namespace nn { namespace repair { namespace detail {

bool SaveData::IsNecessary(const BlackList* pList) const NN_NOEXCEPT
{
    bool isNeedsBackup;
    switch (m_Info.saveDataType)
    {
    case nn::fs::SaveDataType::System:
        {
            NN_SDK_ASSERT_NOT_NULL(pList);
            isNeedsBackup = pList->IsHit(m_Info.systemSaveDataId) ? false : true;
        }
        break;
    case nn::fs::SaveDataType::Temporary:
        {
            // Temporary は fs 起動時に削除されるので
            // バックアップツール起動時には存在しない
            // 何らかの理由で存在したとしても当然移行する必要もない
            isNeedsBackup = false;
        }
        break;
    case nn::fs::SaveDataType::Cache:
        {
            // Cache は移行しなくともユーザが手元で再取得可能
            // 容量も通常のセーブデータより圧倒的に大きいため移行しない
            isNeedsBackup = false;
        }
        break;
    default:
        {
            isNeedsBackup = this->IsPseudo() ? false : true;
        }
        break;
    }

    return isNeedsBackup;
}

bool SaveData::IsPseudo() const NN_NOEXCEPT
{
    static const nn::ncm::ApplicationId PseudoApplicationId = { 0x0000000000000000 };
    return (m_Info.applicationId.value == PseudoApplicationId.value) ? true: false;
}

bool SaveData::HasThumbnail() const NN_NOEXCEPT
{
    return (m_Info.saveDataType == nn::fs::SaveDataType::Account)
        && (m_Info.saveDataUserId != nn::fs::InvalidUserId);
}

nn::Result SaveData::Mount(const std::string& name) const NN_NOEXCEPT
{
    switch (m_Info.saveDataType)
    {
    case nn::fs::SaveDataType::Account:
        {
            NN_REPAIR_RESULT_DO(nn::fs::MountSaveData(name.c_str(), m_Info.applicationId, m_Info.saveDataUserId));
        }
        break;
    case nn::fs::SaveDataType::Bcat:
        {
            NN_REPAIR_RESULT_DO(nn::fs::MountBcatSaveData(name.c_str(), m_Info.applicationId));
        }
        break;
    case nn::fs::SaveDataType::Device:
        {
            NN_REPAIR_RESULT_DO(nn::fs::MountDeviceSaveData(name.c_str(), m_Info.applicationId));
        }
        break;
    case nn::fs::SaveDataType::System:
        {
            NN_REPAIR_RESULT_DO(nn::fs::MountSystemSaveData(
                        name.c_str(), nn::fs::SaveDataSpaceId::ProperSystem, m_Info.systemSaveDataId, m_Info.saveDataUserId));
        }
        break;
    default:
        {
            SendMessage(
                    "Unexpected SaveDataType %d\n",
                    static_cast<int>(m_Info.saveDataType));
        }
        return ResultUnexpectedSaveDataType();
    }

    NN_RESULT_SUCCESS;
}

void SaveData::Unmount(const std::string& name) const NN_NOEXCEPT
{
    nn::fs::Unmount(name.c_str());
}

nn::Result SaveData::Create(
        nn::Bit64 ownerId, int64_t availableSize, int64_t journalSize, uint32_t flags) const NN_NOEXCEPT
{
    auto result = nn::Result();
    switch (m_Info.saveDataType)
    {
    case nn::fs::SaveDataType::System:
        {
            result =
                nn::fs::CreateSystemSaveData(
                        nn::fs::SaveDataSpaceId::ProperSystem,
                        m_Info.systemSaveDataId, m_Info.saveDataUserId, ownerId, availableSize, journalSize, flags);
        }
        break;
    case nn::fs::SaveDataType::Account:
        {
            result =
                nn::fs::CreateSaveData(
                        m_Info.applicationId, m_Info.saveDataUserId, ownerId, availableSize, journalSize, flags);
        }
        break;
    case nn::fs::SaveDataType::Bcat:
        {
            result =
                nn::fs::CreateBcatSaveData(m_Info.applicationId, availableSize);
        }
        break;
    case nn::fs::SaveDataType::Device:
        {
            result =
                nn::fs::CreateDeviceSaveData(
                        m_Info.applicationId, ownerId, availableSize, journalSize, flags);
        }
        break;
    default:
        {
            SendMessage(
                    "Unexpected SaveDataType %d\n",
                    static_cast<int>(m_Info.saveDataType));
        }
        return ResultUnexpectedSaveDataType();
    }

    // 既に存在する場合は SUCCESS とする
    if (nn::fs::ResultPathAlreadyExists().Includes(result))
    {
        result = nn::ResultSuccess();
    }

    return result;
}

nn::Result SaveData::GetSaveDataId(nn::fs::SaveDataId* pOutId, const nn::fs::SaveDataSpaceId& spaceId, const nn::fs::SaveDataInfo& ref) NN_NOEXCEPT
{
    std::unique_ptr<nn::fs::SaveDataIterator> iter;
    NN_REPAIR_RESULT_DO(
        nn::fs::OpenSaveDataIterator(&iter, spaceId));

    for ( ; ; )
    {
        auto count = int64_t();
        auto info = nn::fs::SaveDataInfo();
        NN_REPAIR_RESULT_DO(iter->ReadSaveDataInfo(&count, &info, 1));

        if (count == 0)
        {
            break;
        }

        if (info.saveDataType == ref.saveDataType
            && info.systemSaveDataId == ref.systemSaveDataId
            && info.applicationId.value == ref.applicationId.value
            && info.saveDataUserId == ref.saveDataUserId)
        {
            *pOutId = info.saveDataId;
            NN_RESULT_SUCCESS;
        }
    }

    return nn::fs::ResultTargetNotFound();
}

nn::Result SaveData::Delete(const nn::fs::SaveDataSpaceId& spaceId) const NN_NOEXCEPT
{
    std::unique_ptr<nn::fs::SaveDataIterator> iter;
    NN_REPAIR_RESULT_DO(
            nn::fs::OpenSaveDataIterator(&iter, spaceId));

    for ( ; ; )
    {
        auto count = int64_t();
        auto info = nn::fs::SaveDataInfo();
        NN_REPAIR_RESULT_DO(iter->ReadSaveDataInfo(&count, &info, 1));

        if (count == 0)
        {
            break;
        }

        switch (m_Info.saveDataType)
        {
        case nn::fs::SaveDataType::System:
            {
                if (info.saveDataType == m_Info.saveDataType
                        && info.systemSaveDataId == m_Info.systemSaveDataId
                        && info.saveDataUserId == m_Info.saveDataUserId)
                {
                    NN_REPAIR_RESULT_DO(
                            nn::fs::DeleteSaveData(spaceId, info.saveDataId));
                    NN_RESULT_SUCCESS;
                }
            }
            break;
        case nn::fs::SaveDataType::Account:
            {
                if (info.saveDataType == m_Info.saveDataType
                        && info.applicationId.value == m_Info.applicationId.value
                        && info.saveDataUserId == m_Info.saveDataUserId)
                {
                    NN_REPAIR_RESULT_DO(
                            nn::fs::DeleteSaveData(spaceId, info.saveDataId));
                    NN_RESULT_SUCCESS;
                }
            }
            break;
        case nn::fs::SaveDataType::Bcat:
            {
                if (info.saveDataType == m_Info.saveDataType
                        && info.applicationId.value == m_Info.applicationId.value)
                {
                    NN_REPAIR_RESULT_DO(
                            nn::fs::DeleteSaveData(spaceId, info.saveDataId));
                    NN_RESULT_SUCCESS;
                }
            }
            break;
        case nn::fs::SaveDataType::Device:
            {
                if (info.saveDataType == m_Info.saveDataType
                        && info.applicationId.value == m_Info.applicationId.value)
                {
                    NN_REPAIR_RESULT_DO(
                            nn::fs::DeleteSaveData(spaceId, info.saveDataId));
                    NN_RESULT_SUCCESS;
                }
            }
            break;
        case nn::fs::SaveDataType::Temporary:
            {
                // Temporary は移行対象ではないので消す必要がない
                SendMessage("SaveDataType::Temporary is not deleted.");
            }
            break;
        case nn::fs::SaveDataType::Cache:
            {
                // Cache は移行対象ではないので消す必要がない
                SendMessage("SaveDataType::Cache is not deleted.");
            }
            break;
        default:
            {
                SendMessage("Unexpected SaveDataType %d\n",
                        static_cast<int>(m_Info.saveDataType));
            }
            NN_RESULT_THROW(ResultUnexpectedSaveDataType());
        }
    }
    NN_RESULT_SUCCESS;
}

nn::Result SaveData::Commit(const std::string& name) const NN_NOEXCEPT
{
    NN_REPAIR_RESULT_DO(nn::fs::CommitSaveData(name.c_str()));
    NN_RESULT_SUCCESS;
}

nn::fs::SaveDataInfo SaveData::GetInfo() const NN_NOEXCEPT
{
    return m_Info;
}

nn::Result SaveData::GetOwnerId(
        nn::Bit64* pOutOwnerId, const nn::fs::SaveDataSpaceId& spaceId) const NN_NOEXCEPT
{
    NN_REPAIR_RESULT_DO(nn::fs::GetSaveDataOwnerId(pOutOwnerId, spaceId, m_Info.saveDataId));
    NN_RESULT_SUCCESS;
}

nn::Result SaveData::GetAvailableSize(
        int64_t* pOutAvailableSize, const nn::fs::SaveDataSpaceId& spaceId) const NN_NOEXCEPT
{
    NN_REPAIR_RESULT_DO(nn::fs::GetSaveDataAvailableSize(pOutAvailableSize, spaceId, m_Info.saveDataId));
    NN_RESULT_SUCCESS;
}

nn::Result SaveData::GetJournalSize(
        int64_t* pOutJournalSize, const nn::fs::SaveDataSpaceId& spaceId) const NN_NOEXCEPT
{
    NN_REPAIR_RESULT_DO(nn::fs::GetSaveDataJournalSize(pOutJournalSize, spaceId, m_Info.saveDataId));
    NN_RESULT_SUCCESS;
}

nn::Result SaveData::GetFlags(
        uint32_t* pOutFlags, const nn::fs::SaveDataSpaceId& spaceId) const NN_NOEXCEPT
{
    NN_REPAIR_RESULT_DO(nn::fs::GetSaveDataFlags(pOutFlags, spaceId, m_Info.saveDataId));
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::repair::detail

