﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>

#include "../repair_Utility.h"
#include "repair_AlbumDataEntry.h"
#include "repair_BlackList.h"
#include "repair_Converter.h"
#include "repair_MessageReporter.h"
#include "repair_SaveData.h"
#include "repair_SaveDataEntry.h"
#include "repair_ThumbnailDataEntry.h"

namespace nn { namespace repair { namespace detail {

const uint32_t Converter::MAGIC_NUMBER = 0xCAFECAFE;
const uint32_t Converter::CURRENT_VERSION = 3;

nn::Result Converter::ToArchive(const BlackList* pList) const NN_NOEXCEPT
{
    NN_REPAIR_RESULT_DO(this->WriteHeader(pList));
    NN_REPAIR_RESULT_DO(this->WriteBody(pList));

    NN_RESULT_SUCCESS;
}

nn::Result Converter::ToArchive(
        nn::Bit64 id, const BlackList* pList) const NN_NOEXCEPT
{
    NN_REPAIR_RESULT_DO(this->WriteHeader(pList, 1));
    NN_REPAIR_RESULT_DO(this->WriteBody(pList, id));

    NN_RESULT_SUCCESS;
}

nn::Result Converter::WriteHeader(
        const BlackList* pList, int64_t count) const NN_NOEXCEPT
{
    if (count == 0)
    {
        int64_t entriesCount;
        NN_REPAIR_RESULT_DO(this->CountEntries(&entriesCount, pList));
        count = entriesCount;
    }

    Header header;
    header.type = m_Type;
    header.count = count;

    auto offset = m_pFile->GetEndOffset();
    NN_REPAIR_RESULT_DO(m_pFile->Write(offset, &header, sizeof(Header), true));

    NN_RESULT_SUCCESS;
}

nn::Result Converter::CountEntries(
        int64_t* pOutCount, const BlackList* pList) const NN_NOEXCEPT
{
    switch (m_Type)
    {
    case DataType::System:
        {
            SaveDataEntry entry(nn::fs::SaveDataSpaceId::ProperSystem, Converter::CURRENT_VERSION);
            NN_REPAIR_RESULT_DO(entry.CountEntries(pOutCount, pList));
        }
        break;
    case DataType::User:
        {
            SaveDataEntry entry(nn::fs::SaveDataSpaceId::User, Converter::CURRENT_VERSION);
            NN_REPAIR_RESULT_DO(entry.CountEntries(pOutCount));
        }
        break;
    case DataType::Thumbnail:
        {
            ThumbnailDataEntry entry;
            NN_REPAIR_RESULT_DO(entry.CountEntries(pOutCount, pList));
        }
        break;
    case DataType::Album:
        {
            AlbumDataEntry entry;
            NN_REPAIR_RESULT_DO(entry.CountEntries(pOutCount));
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
}

nn::Result Converter::WriteBody(
        const BlackList* pList, nn::Bit64 id) const NN_NOEXCEPT
{
    switch (m_Type)
    {
     case DataType::System:
        {
            SaveDataEntry entry(nn::fs::SaveDataSpaceId::ProperSystem, Converter::CURRENT_VERSION);
            NN_REPAIR_RESULT_DO(entry.WriteArchive(m_pFile, pList, id));
        }
        break;
    case DataType::User:
        {
            SaveDataEntry entry(nn::fs::SaveDataSpaceId::User, Converter::CURRENT_VERSION);
            NN_REPAIR_RESULT_DO(entry.WriteArchive(m_pFile));
        }
        break;
    case DataType::Thumbnail:
        {
            ThumbnailDataEntry entry;
            NN_REPAIR_RESULT_DO(entry.WriteArchive(m_pFile, pList));
        }
        break;
    case DataType::Album:
        {
            AlbumDataEntry entry;
            NN_REPAIR_RESULT_DO(entry.WriteArchive(m_pFile));
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
}

nn::Result Converter::ToData(int64_t* pOutSize, int64_t offset) const NN_NOEXCEPT
{
    // ファイルヘッダのチェック
    Header header;

    NN_REPAIR_RESULT_DO(m_pFile->Read(offset, &header, sizeof(Header)));

    // for debug
    // SendMessage("Coverter header check %llX, %08x, %d \n",offset, header.magic, header.version );

    bool check1 = (header.magic == MAGIC_NUMBER);
    bool check2 = (header.version <= CURRENT_VERSION); // 互換の為ここでのチェックはゆるめ。

    if(check1 && check2)
    {
        if (header.type != m_Type)
        {
            // ok but do nothing
            return nn::ResultSuccess();
        }
    }
    else
    {
        return nn::repair::ResultBadArchiveFile();
    }

    int64_t size;

    switch (m_Type)
    {
    case DataType::System:
        {
            SaveDataEntry entry(nn::fs::SaveDataSpaceId::ProperSystem, header.version);
            NN_REPAIR_RESULT_DO(
                entry.WriteEntries(
                    &size, m_pFile, header.count, offset + sizeof(Header)));
        }
        break;
    case DataType::User:
        {
            SaveDataEntry entry(nn::fs::SaveDataSpaceId::User, header.version);
            NN_REPAIR_RESULT_DO(
                entry.WriteEntries(
                    &size, m_pFile, header.count, offset + sizeof(Header)));
        }
        break;
    case DataType::Thumbnail:
        {
            ThumbnailDataEntry entry;
            NN_REPAIR_RESULT_DO(
                entry.WriteEntries(
                    &size, m_pFile, header.count, offset + sizeof(Header)));
        }
        break;
    case DataType::Album:
        {
            AlbumDataEntry entry;
            NN_REPAIR_RESULT_DO(
                entry.WriteEntries(
                    &size, m_pFile, header.count, offset + sizeof(Header)));
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    *pOutSize = sizeof(Header) + size;

    NN_RESULT_SUCCESS;
}


}}} // namespace nn::repair::detail

