﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d/res/g3d_ResTexPatternAnim.h>
#include <algorithm>
#include <nw/g3d/res/g3d_ResFile.h>

NW_G3D_PRAGMA_PUSH_WARNINGS
NW_G3D_DISABLE_WARNING_SHADOW

namespace nw { namespace g3d { namespace res {

void ResTexPatternMatAnim::Eval(
    u16* pResult, float frame, const s8* pSubBindIndex) const
{
    const PatAnimInfo* pInfo = ref().ofsPatAnimInfoArray.to_ptr<PatAnimInfo>();

    for (int idxPatAnim = 0, numPatAnim = GetPatAnimCount();
        idxPatAnim < numPatAnim; ++idxPatAnim, ++pInfo)
    {
        if (pSubBindIndex[idxPatAnim] == -1 ||
            pInfo->curveIndex == -1)
        {
            continue;
        }

        const ResAnimCurve* pCurve = GetCurve(pInfo->curveIndex);
        pResult[idxPatAnim] = static_cast<u16>(pCurve->EvalInt(frame));
    }
}

void ResTexPatternMatAnim::Eval(
    u16* pResult, float frame, const s8* pSubBindIndex, AnimFrameCache* pFrameCache) const
{
    const PatAnimInfo* pInfo = ref().ofsPatAnimInfoArray.to_ptr<PatAnimInfo>();

    for (int idxPatAnim = 0, numPatAnim = GetPatAnimCount();
        idxPatAnim < numPatAnim; ++idxPatAnim, ++pInfo)
    {
        if (pSubBindIndex[idxPatAnim] == -1 ||
            pInfo->curveIndex == -1)
        {
            continue;
        }

        const ResAnimCurve* pCurve = GetCurve(pInfo->curveIndex);
        pResult[idxPatAnim] = static_cast<u16>(
            pCurve->EvalInt(frame, &pFrameCache[pInfo->curveIndex]));
    }
}

BindResult ResTexPatternMatAnim::PreBind(const ResMaterial* pTarget)
{
    BindResult result;
    PatAnimInfo* pInfo = ref().ofsPatAnimInfoArray.to_ptr<PatAnimInfo>();
    for (int idxPatAnim = 0, numPatAnim = GetPatAnimCount();
        idxPatAnim < numPatAnim; ++idxPatAnim, ++pInfo)
    {
        const ResName* pName = pInfo->ofsName.GetResName();
        int idxTarget = pTarget->GetSamplerIndex(pName);
        pInfo->subbindIndex = static_cast<s8>(idxTarget);
        if (idxTarget >= 0)
        {
            result |= BindResult::Bound();
        }
        else
        {
            result |= BindResult::NotBound();
        }
    }
    return result;
}

BindResult ResTexPatternMatAnim::BindCheck(const ResMaterial* pTarget) const
{
    BindResult result;
    const PatAnimInfo* pInfo = ref().ofsPatAnimInfoArray.to_ptr<PatAnimInfo>();
    for (int idxPatAnim = 0, numPatAnim = GetPatAnimCount();
        idxPatAnim < numPatAnim; ++idxPatAnim, ++pInfo)
    {
        const ResName* pName = pInfo->ofsName.GetResName();
        int idxTarget = pTarget->GetSamplerIndex(pName);
        if (idxTarget >= 0)
        {
            result |= BindResult::Bound();
        }
        else
        {
            result |= BindResult::NotBound();
        }
    }
    return result;
}

void ResTexPatternMatAnim::Init(u16* pResult) const
{
    // バインドされていない部分も初期化してしまう。
    const u16* pBaseValueArray = ref().ofsBaseValueArray.to_ptr<u16>();
    for (int idxPatAnim = 0, numPatAnim = GetPatAnimCount(); idxPatAnim < numPatAnim; ++idxPatAnim)
    {
        pResult[idxPatAnim] = pBaseValueArray[idxPatAnim];
    }
}

//--------------------------------------------------------------------------------------------------

BindResult ResTexPatternAnim::PreBind(const ResModel* pModel)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pModel, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    ref().ofsBindModel.set_ptr(pModel->ptr());
    u16* pBindIndexArray = ref().ofsBindIndexArray.to_ptr<u16>();

    BindResult result;
    for (int idxMatAnim = 0, numMatAnim = GetMatAnimCount(); idxMatAnim < numMatAnim; ++idxMatAnim)
    {
        ResTexPatternMatAnim* pMatAnim = GetMatAnim(idxMatAnim);

        const ResName* pName = pMatAnim->ref().ofsName.GetResName();
        const ResMaterial* pMaterial = pModel->GetMaterial(pName);
        if (pMaterial != NULL)
        {
            pBindIndexArray[idxMatAnim] = static_cast<u16>(pMaterial->GetIndex());
            BindResult subBindResult = pMatAnim->PreBind(pMaterial);
            result |= subBindResult;
        }
        else
        {
            pBindIndexArray[idxMatAnim] = AnimFlag::NOT_BOUND;
            result |= BindResult::NotBound();
        }
    }
    return result;
}

BindResult ResTexPatternAnim::BindCheck(const ResModel* pModel) const
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pModel, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    BindResult result;
    for (int idxMatAnim = 0, numMatAnim = GetMatAnimCount(); idxMatAnim < numMatAnim; ++idxMatAnim)
    {
        const ResTexPatternMatAnim* pMatAnim = GetMatAnim(idxMatAnim);

        const ResName* pName = pMatAnim->ref().ofsName.GetResName();
        const ResMaterial* pMaterial = pModel->GetMaterial(pName);
        if (pMaterial != NULL)
        {
            BindResult subBindResult = pMatAnim->BindCheck(pMaterial);
            result |= subBindResult;
        }
        else
        {
            result |= BindResult::NotBound();
        }
    }
    return result;
}

BindResult ResTexPatternAnim::Bind(const ResFile* pFile)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pFile, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    BindResult result;

    for (int idxTex = 0, numTex = GetTextureRefCount(); idxTex < numTex; ++idxTex)
    {
        ResTextureRef* pTextureRef = GetTextureRef(idxTex);
        if (!pTextureRef->IsBound())
        {
            result |= pTextureRef->Bind(pFile);
        }
    }

    return result;
}

BindResult ResTexPatternAnim::Bind(const ResFile* pFile, BindCallback* pCallback)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pFile, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pCallback, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    BindResult result;

    for (int idxTex = 0, numTex = GetTextureRefCount(); idxTex < numTex; ++idxTex)
    {
        ResTextureRef* pTextureRef = GetTextureRef(idxTex);
        if (!pTextureRef->IsBound())
        {
            BindResult curResult = pTextureRef->Bind(pFile);
            result |= curResult.IsComplete() ? curResult : pCallback->AnimTexNotBound(pTextureRef);
        }
    }

    return result;
}

bool ResTexPatternAnim::ForceBind(const ResTexture* pTexture, const char* name)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pTexture, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    if (ResTextureRef* pTextureRef = GetTextureRef(name))
    {
        pTextureRef->ForceBind(pTexture);
        return true;
    }
    return false;
}

void ResTexPatternAnim::Release()
{
    for (int idxTex = 0, numTex = GetTextureRefCount(); idxTex < numTex; ++idxTex)
    {
        ResTextureRef* pTextureRef = GetTextureRef(idxTex);
        pTextureRef->Release();
    }
}

bool ResTexPatternAnim::BakeCurve(void* pBuffer, size_t bufferSize)
{
    if (bufferSize == 0)
    {
        return true;
    }
    if (pBuffer == NULL ||
        bufferSize < GetBakedSize())
    {
        return false;
    }
    for (int idxMatAnim = 0, numMatAnim = GetMatAnimCount(); idxMatAnim < numMatAnim; ++idxMatAnim)
    {
        ResTexPatternMatAnim* pMatAnim = GetMatAnim(idxMatAnim);
        for (int idxCurve = 0, numCurve = pMatAnim->GetCurveCount();
            idxCurve < numCurve; ++idxCurve)
        {
            ResAnimCurve* curve = pMatAnim->GetCurve(idxCurve);
            size_t size = curve->CalcBakedIntSize();
            curve->BakeInt(pBuffer, size);
            pBuffer = AddOffset(pBuffer, size);
        }
    }
    ref().flag |= CURVE_BAKED;

    return true;
}

void* ResTexPatternAnim::ResetCurve()
{
    void* pBuffer = NULL;
    if (IsCurveBaked())
    {
        bool foundCurve = false;
        for (int idxMatAnim = 0, numMatAnim = GetMatAnimCount(); idxMatAnim < numMatAnim; ++idxMatAnim)
        {
            ResTexPatternMatAnim* pMatAnim = GetMatAnim(idxMatAnim);
            for (int idxCurve = 0, numCurve = pMatAnim->GetCurveCount();
                idxCurve < numCurve; ++idxCurve)
            {
                ResAnimCurve* curve = pMatAnim->GetCurve(idxCurve);
                bit32 type = curve->ref().flag & ResAnimCurve::CURVE_MASK;
                if (!foundCurve && type == ResAnimCurve::CURVE_BAKED_INT)
                {
                    pBuffer = curve->ref().ofsKeyArray.to_ptr();
                    foundCurve = true;
                }
                curve->Reset();
            }
        }
        ref().flag ^= CURVE_BAKED;
    }

    return pBuffer;
}

void ResTexPatternAnim::Reset()
{
    ref().ofsBindModel.set_ptr(NULL);
    u16* pBindIndexArray = ref().ofsBindIndexArray.to_ptr<u16>();
    for (int idxMatAnim = 0, numMatAnim = GetMatAnimCount(); idxMatAnim < numMatAnim; ++idxMatAnim)
    {
        ResTexPatternMatAnim* pMatAnim = GetMatAnim(idxMatAnim);
        ResTexPatternMatAnim::PatAnimInfo* pInfo = pMatAnim->ref().ofsPatAnimInfoArray.to_ptr<ResTexPatternMatAnim::PatAnimInfo>();
        for (int idxPatAnim = 0, numPatAnim = pMatAnim->GetPatAnimCount();
            idxPatAnim < numPatAnim; ++idxPatAnim, ++pInfo)
        {
            pInfo->subbindIndex = -1;
        }
        pBindIndexArray[idxMatAnim] = AnimFlag::NOT_BOUND;
    }
    ResetCurve();
}

}}} // namespace nw::g3d::res

NW_G3D_PRAGMA_POP_WARNINGS
