﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d/res/g3d_ResShaderParamAnim.h>
#include <algorithm>

NW_G3D_PRAGMA_PUSH_WARNINGS
NW_G3D_DISABLE_WARNING_SHADOW

namespace nw { namespace g3d { namespace res {

template <bool useContext>
void ResShaderParamMatAnim::EvalImpl(
    void* pResult, float frame, const u16* pSubBindIndex, AnimFrameCache* pFrameCache) const
{
    int idxCurve = 0;
    const ParamAnimInfo* pInfo = ref().ofsParamAnimInfoArray.to_ptr<ParamAnimInfo>();
    const ResAnimCurveData* const pCurves = ref().ofsCurveArray.to_ptr<ResAnimCurveData>();

    for (int idxParamAnim = 0, numParamAnim = GetParamAnimCount();
        idxParamAnim < numParamAnim; ++idxParamAnim, ++pInfo)
    {
        int numInt = pInfo->numIntCurve;
        int numFloat = pInfo->numFloatCurve;
        if (pSubBindIndex[idxParamAnim] == AnimFlag::NOT_BOUND)
        {
            idxCurve += numInt + numFloat;
            continue;
        }

        // texsrt の mode をカーブすると、int カーブと float カーブが混ざる
        // 以下の処理では、int カーブが先に詰まっているとしている
        const ResAnimCurveData* pCurve = pCurves + idxCurve;
        int* pIntResult = AddOffset<int>(pResult, sizeof(int) * idxCurve);
        idxCurve += numInt;
        for (int idxInt = 0; idxInt < numInt; ++idxInt, ++pIntResult, ++pCurve, ++pFrameCache)
        {
            const ResAnimCurve* pIntCurve = ResAnimCurve::ResCast(pCurve);
            if (NW_G3D_STATIC_CONDITION(useContext))
            {
                *pIntResult = pIntCurve->EvalInt(frame, pFrameCache);
            }
            else
            {
                *pIntResult = pIntCurve->EvalInt(frame);
            }
        }

        float* pFloatResult = AddOffset<float>(pResult, sizeof(float) * idxCurve);
        idxCurve += numFloat;
        for (int idxFloat = 0; idxFloat < numFloat;
            ++idxFloat, ++pFloatResult, ++pCurve, ++pFrameCache)
        {
            const ResAnimCurve* pFloatCurve = ResAnimCurve::ResCast(pCurve);
            if (NW_G3D_STATIC_CONDITION(useContext))
            {
                *pFloatResult = pFloatCurve->EvalFloat(frame, pFrameCache);
            }
            else
            {
                *pFloatResult = pFloatCurve->EvalFloat(frame);
            }
        }
    }
}

template void ResShaderParamMatAnim::EvalImpl<true>(void*, float, const u16*, AnimFrameCache*) const;
template void ResShaderParamMatAnim::EvalImpl<false>(void*, float, const u16*, AnimFrameCache*) const;

BindResult ResShaderParamMatAnim::PreBind(const ResMaterial* target)
{
    BindResult result;
    ParamAnimInfo* pInfo = ref().ofsParamAnimInfoArray.to_ptr<ParamAnimInfo>();
    for (int idxParamAnim = 0, numParamAnim = GetParamAnimCount();
        idxParamAnim < numParamAnim; ++idxParamAnim, ++pInfo)
    {
        const ResName* pName = pInfo->ofsName.GetResName();
        int idxTarget = target->GetShaderParamIndex(pName);
        pInfo->subbindIndex = static_cast<u16>(idxTarget); // -1 -> 0xFFFF
        if (idxTarget >= 0)
        {
            result |= BindResult::Bound();
        }
        else
        {
            result |= BindResult::NotBound();
        }
    }
    return result;
}

BindResult ResShaderParamMatAnim::BindCheck(const ResMaterial* target) const
{
    BindResult result;
    const ParamAnimInfo* pInfo = ref().ofsParamAnimInfoArray.to_ptr<ParamAnimInfo>();
    for (int idxParamAnim = 0, numParamAnim = GetParamAnimCount();
        idxParamAnim < numParamAnim; ++idxParamAnim, ++pInfo)
    {
        const ResName* pName = pInfo->ofsName.GetResName();
        int idxTarget = target->GetShaderParamIndex(pName);
        if (idxTarget >= 0)
        {
            result |= BindResult::Bound();
        }
        else
        {
            result |= BindResult::NotBound();
        }
    }
    return result;
}

//--------------------------------------------------------------------------------------------------

BindResult ResShaderParamAnim::PreBind(const ResModel* pModel)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pModel, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    ref().ofsBindModel.set_ptr(pModel->ptr());
    u16* pBindIndexArray = ref().ofsBindIndexArray.to_ptr<u16>();

    BindResult result;
    for (int idxMatAnim = 0, numMatAnim = GetMatAnimCount(); idxMatAnim < numMatAnim; ++idxMatAnim)
    {
        ResShaderParamMatAnim* pMatAnim = GetMatAnim(idxMatAnim);

        const ResName* pName = pMatAnim->ref().ofsName.GetResName();
        const ResMaterial* pMaterial = pModel->GetMaterial(pName);
        if (pMaterial != NULL)
        {
            pBindIndexArray[idxMatAnim] = static_cast<u16>(pMaterial->GetIndex());
            BindResult subBindResult = pMatAnim->PreBind(pMaterial);
            result |= subBindResult;
        }
        else
        {
            pBindIndexArray[idxMatAnim] = AnimFlag::NOT_BOUND;
            result |= BindResult::NotBound();
        }
    }
    return result;
}

BindResult ResShaderParamAnim::BindCheck(const ResModel* pModel) const
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pModel, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    BindResult result;
    for (int idxMatAnim = 0, numMatAnim = GetMatAnimCount(); idxMatAnim < numMatAnim; ++idxMatAnim)
    {
        const ResShaderParamMatAnim* pMatAnim = GetMatAnim(idxMatAnim);

        const ResName* pName = pMatAnim->ref().ofsName.GetResName();
        const ResMaterial* pMaterial = pModel->GetMaterial(pName);
        if (pMaterial != NULL)
        {
            BindResult subBindResult = pMatAnim->BindCheck(pMaterial);
            result |= subBindResult;
        }
        else
        {
            result |= BindResult::NotBound();
        }
    }
    return result;
}

bool ResShaderParamAnim::BakeCurve(void* pBuffer, size_t bufferSize)
{
    if (bufferSize == 0)
    {
        return true;
    }
    if (pBuffer == NULL ||
        bufferSize < GetBakedSize())
    {
        return false;
    }
    for (int idxMatAnim = 0, numMatAnim = GetMatAnimCount(); idxMatAnim < numMatAnim; ++idxMatAnim)
    {
        ResShaderParamMatAnim* pMatAnim = GetMatAnim(idxMatAnim);
        for (int idxCurve = 0, numCurve = pMatAnim->GetCurveCount();
            idxCurve < numCurve; ++idxCurve)
        {
            ResAnimCurve* curve = pMatAnim->GetCurve(idxCurve);
            size_t size = curve->CalcBakedSize();
            curve->Bake(pBuffer, size);
            pBuffer = AddOffset(pBuffer, size);
        }
    }
    ref().flag |= CURVE_BAKED;

    return true;
}

void* ResShaderParamAnim::ResetCurve()
{
    void* pBuffer = NULL;
    if (IsCurveBaked())
    {
        bool foundCurve = false;
        for (int idxMatAnim = 0, numMatAnim = GetMatAnimCount(); idxMatAnim < numMatAnim; ++idxMatAnim)
        {
            ResShaderParamMatAnim* pMatAnim = GetMatAnim(idxMatAnim);
            for (int idxCurve = 0, numCurve = pMatAnim->GetCurveCount();
                idxCurve < numCurve; ++idxCurve)
            {
                ResAnimCurve* curve = pMatAnim->GetCurve(idxCurve);
                bit32 type = curve->ref().flag & ResAnimCurve::CURVE_MASK;
                if (!foundCurve && (type == ResAnimCurve::CURVE_BAKED_FLOAT ||
                    type == ResAnimCurve::CURVE_BAKED_INT || type == ResAnimCurve::CURVE_BAKED_BOOL))
                {
                    pBuffer = curve->ref().ofsKeyArray.to_ptr();
                    foundCurve = true;
                }
                curve->Reset();
            }
        }
        ref().flag ^= CURVE_BAKED;
    }

    return pBuffer;
}

void ResShaderParamAnim::Reset()
{
    ref().ofsBindModel.set_ptr(NULL);
    u16* pBindIndexArray = ref().ofsBindIndexArray.to_ptr<u16>();
    for (int idxMatAnim = 0, numMatAnim = GetMatAnimCount(); idxMatAnim < numMatAnim; ++idxMatAnim)
    {
        ResShaderParamMatAnim* pMatAnim = GetMatAnim(idxMatAnim);
        ResShaderParamMatAnim::ParamAnimInfo* pInfo = pMatAnim->ref().ofsParamAnimInfoArray.to_ptr<ResShaderParamMatAnim::ParamAnimInfo>();
        for (int idxParamAnim = 0, numParamAnim = pMatAnim->GetParamAnimCount();
            idxParamAnim < numParamAnim; ++idxParamAnim, ++pInfo)
        {
            pInfo->subbindIndex = AnimFlag::NOT_BOUND;
        }
        pBindIndexArray[idxMatAnim] = AnimFlag::NOT_BOUND;
    }
    ResetCurve();
}

}}} // namespace nw::g3d::res

NW_G3D_PRAGMA_POP_WARNINGS
