﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d/res/g3d_ResAnimCurve.h>
#include <nw/g3d/math/g3d_MathCommon.h>
#include <nw/g3d/ut/g3d_Flag.h>
#include <nw/g3d/ut/g3d_Perf.h>

#include <limits.h>

NW_G3D_PRAGMA_PUSH_WARNINGS
NW_G3D_DISABLE_WARNING_SHADOW
NW_G3D_DISABLE_WARNING_REINTERPRET_CAST

namespace nw { namespace g3d { namespace res {

// メンバ関数ポインタはサイズの解釈が一定でないためヘッダに出さない
class ResAnimCurve::Impl
{
public:
    static void (ResAnimCurve::* const s_pFuncFindFrame[])(AnimFrameCache*, float) const;
    static float (ResAnimCurve::* const s_pFuncEvalFloat[][NUM_KEY])(float, AnimFrameCache*) const;
    static int (ResAnimCurve::* const s_pFuncEvalInt[][NUM_KEY])(float, AnimFrameCache*) const;
};

void (ResAnimCurve::* const ResAnimCurve::Impl::s_pFuncFindFrame[])(AnimFrameCache*, float) const = {
    &ResAnimCurve::FindFrame<float>,
    &ResAnimCurve::FindFrame<s16>,
    &ResAnimCurve::FindFrame<u8>
};

float (ResAnimCurve::* const ResAnimCurve::Impl::s_pFuncEvalFloat[][NUM_KEY])(float, AnimFrameCache*) const = {
    {
        &ResAnimCurve::EvalCubic<float>,
        &ResAnimCurve::EvalCubic<s16>,
        &ResAnimCurve::EvalCubic<s8>
    },
    {
        &ResAnimCurve::EvalLinear<float>,
        &ResAnimCurve::EvalLinear<s16>,
        &ResAnimCurve::EvalLinear<s8>
    },
    {
        &ResAnimCurve::EvalBakedFloat<float>,
        &ResAnimCurve::EvalBakedFloat<s16>,
        &ResAnimCurve::EvalBakedFloat<s8>
    },
};

int (ResAnimCurve::* const ResAnimCurve::Impl::s_pFuncEvalInt[][NUM_KEY])(float, AnimFrameCache*) const = {
    {
        &ResAnimCurve::EvalStepInt<s32>,
        &ResAnimCurve::EvalStepInt<s16>,
        &ResAnimCurve::EvalStepInt<s8>
    },
    {
        &ResAnimCurve::EvalBakedInt<s32>,
        &ResAnimCurve::EvalBakedInt<s16>,
        &ResAnimCurve::EvalBakedInt<s8>
    },
    {
        &ResAnimCurve::EvalStepBool,
        &ResAnimCurve::EvalStepBool,
        &ResAnimCurve::EvalStepBool
    },
    {
        &ResAnimCurve::EvalBakedBool,
        &ResAnimCurve::EvalBakedBool,
        &ResAnimCurve::EvalBakedBool
    }
};

//--------------------------------------------------------------------------------------------------

float ResAnimCurve::EvalFloat(float frame, AnimFrameCache* pFrameCache) const
{
    NW_G3D_PERF_LEVEL3_FUNC();
    NW_G3D_ASSERT(IsFloatCurve());
    NW_G3D_ASSERT_NOT_NULL(pFrameCache);

    float offset;
    frame = WrapFrame(&offset, frame);
    int curveType = GetCurveType() >> CURVE_SHIFT;
    int keyType = GetKeyType() >> KEY_SHIFT;
    float result = (this->*Impl::s_pFuncEvalFloat[curveType][keyType])(frame, pFrameCache);
    return result * ref().fScale + offset;
}

int ResAnimCurve::EvalInt(float frame, AnimFrameCache* pFrameCache) const
{
    NW_G3D_PERF_LEVEL3_FUNC();
    NW_G3D_ASSERT(IsIntCurve());
    NW_G3D_ASSERT_NOT_NULL(pFrameCache);

    float offset;
    frame = WrapFrame(&offset, frame);
    int curveType = (GetCurveType() - CURVE_INT_BASE) >> CURVE_SHIFT;
    int keyType = GetKeyType() >> KEY_SHIFT;
    int result = (this->*Impl::s_pFuncEvalInt[curveType][keyType])(frame, pFrameCache);
    return result + ref().iOffset;
}

float ResAnimCurve::WrapFrame(float *pOutOffset, float frame) const
{
    // カーブ単位でのフレームの折り返しを処理する。
    // アニメーション単位でのフレームの変更はより上位層で行う。
    *pOutOffset = ref().fOffset;
    float startFrame = GetStartFrame();
    float endFrame = GetEndFrame();
    if (frame >= startFrame && frame <= endFrame )
    {
        return frame;
    }
    float frameDiff = startFrame - frame;
    WrapMode wrapMode;
    bool outWrap = false;
    if (frameDiff > 0)
    {
        wrapMode = GetPreWrapMode();
        if (WRAP_CLAMP == wrapMode)
        {
            return startFrame;
        }
    }
    else
    {
        wrapMode = GetPostWrapMode();
        if (WRAP_CLAMP == wrapMode)
        {
            return endFrame;
        }
        frameDiff = frame - endFrame;
        outWrap = true;
    }

    // クランプ以外の場合の処理。
    float duration = endFrame - startFrame;
    int count = static_cast<int>(frameDiff / duration);
    float frameMod = frameDiff - duration * count;
    bool repeat = true;
    if (wrapMode == WRAP_RELATIVE_REPEAT)
    {
        float relative = ( count + 1 ) * ref().fDelta;
        if (!outWrap)
        {
            relative *= -1.0f;
        }
        *pOutOffset += relative;
    }
    else if (wrapMode == WRAP_MIRROR && !(count & 0x01))
    {
        repeat = false;
    }
    if (repeat != outWrap) // outWrap の場合は inWarp と逆になるので裏返す。
    {
        frame = endFrame - frameMod;
    }
    else
    {
        frame = startFrame + frameMod;
    }

    NW_G3D_ASSERT(ref().startFrame <= frame && frame <= ref().endFrame);
    return frame;
}

void ResAnimCurve::UpdateFrameCache(AnimFrameCache* pFrameCache, float frame) const
{
    if (frame < pFrameCache->start || pFrameCache->end <= frame)
    {
        // FrameCache の区間に収まっていない場合は探索する。
        // 前回の keyIndex から探索した方が近い場合を効率的に判定できれば切り替えたい。
        float denom = Math::Rcp(ref().endFrame - ref().startFrame);
        pFrameCache->keyIndex = FastCast<u16>((frame - ref().startFrame) * denom * ref().numKey);

        if (pFrameCache->keyIndex > ref().numKey - 1)
        {
            pFrameCache->keyIndex = ref().numKey - 1;
        }

        int frameType = GetFrameType() >> FRAME_SHIFT;
        (this->*Impl::s_pFuncFindFrame[frameType])(pFrameCache, frame);
    }
}


template <typename T>
void ResAnimCurve::FindFrame(AnimFrameCache *pFrameCache, float frame) const
{
    NW_G3D_PERF_LEVEL3_FUNC();
    // 長さ0のキーは終端を除いて存在しない前提。
    typedef Frame<T> FrameType;
    const FrameType* pFrameArray = ref().ofsFrameArray.to_ptr<FrameType>();
    T quantized = FrameType::Quantize(frame);
    int endFrameIndex = ref().numKey - 1;

    if (quantized >= pFrameArray[endFrameIndex].frame)
    {
        // 終端フレームは長さ0で除算しないよう end を適当に設定しておく。
        pFrameCache->keyIndex = static_cast<u16>(endFrameIndex);
        pFrameCache->start = pFrameArray[pFrameCache->keyIndex].Get();
        pFrameCache->end = pFrameCache->start + 1.0f;
        return;
    }

    // 線形探索
    int keyIndex = pFrameCache->keyIndex;
    if (quantized < pFrameArray[keyIndex].frame)
    {
        do
        {
            NW_G3D_ASSERT(0 < keyIndex);
            --keyIndex;
        } while (quantized < pFrameArray[keyIndex].frame);
    }
    else
    {
        do
        {
            NW_G3D_ASSERT(keyIndex < endFrameIndex);
            ++keyIndex;
        } while (pFrameArray[keyIndex].frame <= quantized);

        // ここにきた時点で1つ通り過ぎている。
        NW_G3D_ASSERT(0 < keyIndex);
        --keyIndex;
    }

    pFrameCache->keyIndex = keyIndex;
    pFrameCache->start = pFrameArray[keyIndex].Get();
    pFrameCache->end = pFrameArray[keyIndex + 1].Get();
}

//--------------------------------------------------------------------------------------------------

template <typename T>
float ResAnimCurve::EvalCubic(float frame, AnimFrameCache* pFrameCache) const
{
    NW_G3D_PERF_LEVEL3_FUNC();

    UpdateFrameCache(pFrameCache, frame);

    // セグメント内での比率を求める。
    float start = pFrameCache->start;
    float end = pFrameCache->end;
    float ratio = (frame - start) * Math::Rcp(end - start);

    // インデックスと比率から求めた値を返す。
    typedef ResCubicKey<T> KeyType;
    const KeyType* pKeyArray = ref().ofsKeyArray.to_ptr<KeyType>();
    return pKeyArray[pFrameCache->keyIndex].Get(ratio);
}

template <typename T>
float ResAnimCurve::EvalLinear(float frame, AnimFrameCache* pFrameCache) const
{
    NW_G3D_PERF_LEVEL3_FUNC();

    UpdateFrameCache(pFrameCache, frame);

    // セグメント内での比率を求める。
    float start = pFrameCache->start;
    float end = pFrameCache->end;
    float ratio = (frame - start) * Math::Rcp(end - start);

    // インデックスと比率から求めた値を返す。
    typedef ResLinearKey<T> KeyType;
    const KeyType* pKeyArray = ref().ofsKeyArray.to_ptr<KeyType>();
    return pKeyArray[pFrameCache->keyIndex].Get(ratio);
}

template <typename T>
float ResAnimCurve::EvalBakedFloat(float frame, AnimFrameCache* /*pFrameCache*/) const
{
    NW_G3D_PERF_LEVEL3_FUNC();
    typedef ResFloatKey<T> KeyType;

    int start = static_cast<int>(GetStartFrame());
    int keyIndex = static_cast<int>(frame) - start;
    float ratio = frame - (start + keyIndex);

    // keyIndex + 1 のキーが必ず存在するようコンバートされている前提。
    NW_G3D_ASSERT(keyIndex + 1 < ref().numKey);

    // ratio : 1-ratio で補間。
    const KeyType* pKeyArray = ref().ofsKeyArray.to_ptr<KeyType>();
    return pKeyArray[keyIndex + 1].Get() * ratio + pKeyArray[keyIndex].Get() * (1.0f - ratio);
}

template <typename T>
int ResAnimCurve::EvalStepInt(float frame, AnimFrameCache* pFrameCache) const
{
    NW_G3D_PERF_LEVEL3_FUNC();

    UpdateFrameCache(pFrameCache, frame);

    // インデックスと比率から求めた値を返す。
    typedef ResIntKey<T> KeyType;
    const KeyType* pKeyArray = ref().ofsKeyArray.to_ptr<KeyType>();
    return pKeyArray[pFrameCache->keyIndex].Get();
}

template <typename T>
int ResAnimCurve::EvalBakedInt(float frame, AnimFrameCache* /*pFrameCache*/) const
{
    NW_G3D_PERF_LEVEL3_FUNC();
    typedef ResIntKey<T> KeyType;

    int start = static_cast<int>(GetStartFrame());
    int keyIndex = static_cast<int>(frame) - start;

    // float 以外は補間しない。
    const KeyType* pKeyArray = ref().ofsKeyArray.to_ptr<KeyType>();
    return pKeyArray[keyIndex].Get();
}

int ResAnimCurve::EvalStepBool(float frame, AnimFrameCache* pFrameCache) const
{
    NW_G3D_PERF_LEVEL3_FUNC();

    UpdateFrameCache(pFrameCache, frame);

    // インデックスと比率から求めた値を返す。
    typedef s32 KeyType;
    const KeyType* pKeyArray = ref().ofsKeyArray.to_ptr<KeyType>();
    return (pKeyArray[pFrameCache->keyIndex >> 5] >> (pFrameCache->keyIndex & 0x1F)) & 0x1;
}

int ResAnimCurve::EvalBakedBool(float frame, AnimFrameCache* /*pFrameCache*/) const
{
    NW_G3D_PERF_LEVEL3_FUNC();
    typedef s32 KeyType;

    int start = static_cast<int>(GetStartFrame());
    int keyIndex = static_cast<int>(frame) - start;

    // float 以外は補間しない。
    const KeyType* pKeyArray = ref().ofsKeyArray.to_ptr<KeyType>();
    return (pKeyArray[keyIndex >> 5] >> (keyIndex & 0x1F)) & 0x1;
}

//--------------------------------------------------------------------------------------------------

template <typename T>
void ResAnimCurve::BakeImpl(void* pBuffer, float start, int numKey)
{
    // キーフレームのキャッシュを使いながら評価する。
    AnimFrameCache frameCache;
    frameCache.start = std::numeric_limits<float>::infinity();
    int last = numKey - 1;
    T* pBaked = static_cast<T*>(pBuffer);

    pBaked[0] = static_cast<T>(EvalStepInt<T>(GetStartFrame(), &frameCache));
    for (int idxFrame = 1; idxFrame < last; ++idxFrame)
    {
        pBaked[idxFrame] = static_cast<T>(EvalStepInt<T>(start + idxFrame, &frameCache));
    }
    pBaked[last] = static_cast<T>(EvalStepInt<T>(GetEndFrame(), &frameCache));
}

template <>
void ResAnimCurve::BakeImpl<bool>(void* pBuffer, float start, int numKey)
{
    // キーフレームのキャッシュを使いながら評価する。
    AnimFrameCache frameCache;
    frameCache.start = std::numeric_limits<float>::infinity();
    int last = numKey - 1;
    bit32* pBaked = static_cast<bit32*>(pBuffer);

    SetBit(pBaked, 0, EvalStepBool(GetStartFrame(), &frameCache));
    for (int idxFrame = 1; idxFrame < numKey - 1; ++idxFrame)
    {
        SetBit(pBaked, idxFrame, EvalStepBool(start + idxFrame, &frameCache));
    }
    SetBit(pBaked, last, EvalStepBool(GetEndFrame(), &frameCache));
}

template <>
void ResAnimCurve::BakeImpl<float>(void* pBuffer, float start, int numKey)
{
    // キーフレームのキャッシュを使いながら評価する。
    AnimFrameCache frameCache;
    frameCache.start = std::numeric_limits<float>::infinity();
    int last = numKey - 1;
    float* pBaked = static_cast<float*>(pBuffer);

    pBaked[0] = EvalFloat(GetStartFrame(), &frameCache);
    for (int idxFrame = 1; idxFrame < last; ++idxFrame)
    {
        pBaked[idxFrame] = EvalFloat(start + idxFrame, &frameCache);
    }
    pBaked[last] = EvalFloat(GetEndFrame(), &frameCache);

    // 両端は外挿により整数フレームの値を計算する。
    float end = start + (numKey - 1);
    float diffFrame;
    diffFrame = GetStartFrame() - start;
    pBaked[0] = (pBaked[0] - diffFrame * pBaked[1]) / (1.0f - diffFrame);
    diffFrame = end - GetEndFrame();
    if (diffFrame < 1.0f) // 最終フレームが整数の場合、+1 フレームを計算する必要はない。
    {
        pBaked[last] = (pBaked[last] - diffFrame * pBaked[last - 1]) / (1.0f - diffFrame);
    }
}

void ResAnimCurve::BakeFloat(void* pBuffer, size_t bufferSize)
{
    NW_G3D_ASSERT_NOT_NULL(pBuffer);
    NW_G3D_ASSERT(bufferSize >= CalcBakedFloatSize());
    (void)bufferSize;

    float start = Math::Floor(GetStartFrame());
    float end = Math::Floor(GetEndFrame()) + 1.0f; // 補間するので EndFrame より大きな整数にする。
    int numKey = static_cast<int>(end) - static_cast<int>(start) + 1;

    if (GetCurveType() == CURVE_BAKED_FLOAT || numKey <= 2)
    {
        return; // 2フレーム以下の場合、ベイクしても意味はない、あるいは正しくベイクできない。
    }

    BakeImpl<float>(pBuffer, start, numKey);

    // pBuffer にバックアップをとる。
    static const size_t BAKED_BACKUP_SIZE =
        sizeof(ref().flag) + sizeof(ref().numKey) + sizeof(ref().fOffset) + sizeof(ref().fScale) + sizeof(ref().ofsKeyArray);
    void* backupBuffer = AddOffset(pBuffer, CalcBakedFloatSize() - BAKED_BACKUP_SIZE);
    *static_cast<bit16*>(backupBuffer) = ref().flag;
    backupBuffer = AddOffset(backupBuffer, sizeof(ref().flag));
    *static_cast<u16*>(backupBuffer) = ref().numKey;
    backupBuffer = AddOffset(backupBuffer, sizeof(ref().numKey));
    *static_cast<f32*>(backupBuffer) = ref().fScale;
    backupBuffer = AddOffset(backupBuffer, sizeof(ref().fScale));
    *static_cast<f32*>(backupBuffer) = ref().fOffset;
    backupBuffer = AddOffset(backupBuffer, sizeof(ref().fOffset));
    *static_cast<u32*>(backupBuffer) = reinterpret_cast<u32>(ref().ofsKeyArray.to_ptr());

    ref().flag = (ref().flag & ~(CURVE_MASK | KEY_MASK)) | (CURVE_BAKED_FLOAT | KEY32);
    ref().numKey = static_cast<u16>(numKey);
    ref().fScale = 1.0f;
    ref().fOffset = 0.0f;
    ref().ofsKeyArray.set_ptr(pBuffer);
}

void ResAnimCurve::BakeInt(void* pBuffer, size_t bufferSize)
{
    NW_G3D_ASSERT_NOT_NULL(pBuffer);
    (void)bufferSize;

    float start = Math::Floor(GetStartFrame());
    float end = Math::Floor(GetEndFrame()); // 切り捨てで評価されるので Floor でよい。
    int numKey = static_cast<int>(end) - static_cast<int>(start) + 1;
    bit16 flag = 0;

    if (GetCurveType() == CURVE_BAKED_FLOAT ||
        GetCurveType() == CURVE_BAKED_BOOL ||
        numKey <= 2)
    {
        return; // 2フレーム以下の場合、ベイクしても意味はない、あるいは正しくベイクできない。
    }

    if (GetCurveType() == CURVE_STEP_INT)
    {
        size_t size = numKey;
        if (GetKeyType() == KEY8)
        {
            NW_G3D_ASSERT(bufferSize >= size);

            BakeImpl<s8>(pBuffer, start, numKey);
        }
        else if (GetKeyType() == KEY16)
        {
            size <<= 1;
            NW_G3D_ASSERT(bufferSize >= size);

            BakeImpl<s16>(pBuffer, start, numKey);
        }
        else
        {
            size <<= 2;
            NW_G3D_ASSERT(bufferSize >= size);

            BakeImpl<s32>(pBuffer, start, numKey);
        }

        flag = (ref().flag & ~CURVE_MASK) | CURVE_BAKED_INT;
    }
    else if (GetCurveType() == CURVE_STEP_BOOL)
    {
        BakeImpl<bool>(pBuffer, start, numKey);

        flag = CURVE_BAKED_BOOL;
    }
    // pBuffer にバックアップをとる。
    static const size_t BAKED_BACKUP_SIZE =
        sizeof(ref().flag) + sizeof(ref().numKey) + sizeof(ref().ofsKeyArray);
    void* backupBuffer = AddOffset(pBuffer, CalcBakedIntSize() - BAKED_BACKUP_SIZE);
    *static_cast<bit16*>(backupBuffer) = ref().flag;
    backupBuffer = AddOffset(backupBuffer, sizeof(ref().flag));
    *static_cast<u16*>(backupBuffer) = ref().numKey;
    backupBuffer = AddOffset(backupBuffer, sizeof(ref().numKey));
    *static_cast<u32*>(backupBuffer) = reinterpret_cast<u32>(ref().ofsKeyArray.to_ptr());

    ref().flag = flag;
    ref().numKey = static_cast<u16>(numKey);
    ref().ofsKeyArray.set_ptr(pBuffer);
}

size_t ResAnimCurve::CalcBakedFloatSize() const
{
    float start = Math::Floor(GetStartFrame());
    float end = Math::Floor(GetEndFrame()) + 1.0f; // 補間するので EndFrame より大きな整数にする。
    int numKey = static_cast<int>(end) - static_cast<int>(start) + 1;

    if (numKey <= 2)
    {
        return 0; // 2フレーム以下の場合、ベイクしても意味はない、あるいは正しくベイクできない。
    }

    static const size_t BAKED_BACKUP_SIZE =
        sizeof(ref().flag) + sizeof(ref().numKey) + sizeof(ref().fOffset) + sizeof(ref().fScale) + sizeof(ref().ofsKeyArray);
    return sizeof(float) * numKey + BAKED_BACKUP_SIZE;
}

size_t ResAnimCurve::CalcBakedIntSize() const
{
    float start = Math::Floor(GetStartFrame());
    float end = Math::Floor(GetEndFrame()); // 切り捨てで評価されるので Floor でよい。
    int numKey = static_cast<int>(end) - static_cast<int>(start) + 1;

    if (numKey <= 2)
    {
        return 0; // 2フレーム以下の場合、ベイクしても意味はない、あるいは正しくベイクできない。
    }

    size_t size = 0;
    if (GetCurveType() == CURVE_STEP_INT ||
        GetCurveType() == CURVE_BAKED_INT)
    {
        if (GetKeyType() == KEY8)
        {
            size = numKey;
        }
        else if (GetKeyType() == KEY16)
        {
            size = (numKey << 1);
        }
        else
        {
            size = (numKey << 2);
        }
    }
    else if (GetCurveType() == CURVE_STEP_BOOL ||
             GetCurveType() == CURVE_BAKED_BOOL)
    {
        size = Align(numKey, sizeof(bit32) * CHAR_BIT) / CHAR_BIT;
    }

    static const size_t BAKED_BACKUP_SIZE =
        sizeof(ref().flag) + sizeof(ref().numKey) + sizeof(ref().ofsKeyArray);
    return size + BAKED_BACKUP_SIZE;
}

void ResAnimCurve::ResetFloat()
{
    if (GetCurveType() != CURVE_BAKED_FLOAT)
    {
        return;
    }
    void* pBuffer = ref().ofsKeyArray.to_ptr();

    // pBuffer にとったバックアップから復元する
    void* backupBuffer = AddOffset(pBuffer, CalcBakedFloatSize() - 16);
    ref().flag = *static_cast<u16*>(backupBuffer);
    backupBuffer = AddOffset(backupBuffer, 2);
    ref().numKey = *static_cast<u16*>(backupBuffer);
    backupBuffer = AddOffset(backupBuffer, 2);
    ref().fScale = *static_cast<f32*>(backupBuffer);
    backupBuffer = AddOffset(backupBuffer, 4);
    ref().fOffset = *static_cast<f32*>(backupBuffer);
    backupBuffer = AddOffset(backupBuffer, 4);
    ref().ofsKeyArray.set_ptr(reinterpret_cast<void*>(*static_cast<u32*>(backupBuffer)));
}

void ResAnimCurve::ResetInt()
{
    if (GetCurveType() != CURVE_BAKED_INT &&
        GetCurveType() != CURVE_BAKED_BOOL)
    {
        return;
    }
    void* pBuffer = ref().ofsKeyArray.to_ptr();

    // pBuffer にとったバックアップから復元する
    void* backupBuffer = AddOffset(pBuffer, CalcBakedIntSize() - 8);
    ref().flag = *static_cast<u16*>(backupBuffer);
    backupBuffer = AddOffset(backupBuffer, 2);
    ref().numKey = *static_cast<u16*>(backupBuffer);
    backupBuffer = AddOffset(backupBuffer, 2);
    ref().ofsKeyArray.set_ptr(reinterpret_cast<void*>(*static_cast<u32*>(backupBuffer)));
}

}}} // namespace nw::g3d::res

NW_G3D_PRAGMA_POP_WARNINGS
