﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d/g3d_SceneAnimObj.h>
#include <nw/g3d/ut/g3d_Perf.h>
#include <nw/g3d/math/g3d_Vector3.h>
#include <nw/g3d/math/g3d_Matrix34.h>
#include <nw/g3d/math/g3d_Matrix44.h>

namespace nw { namespace g3d {

void CameraAnimObj::Sizer::Calc(const InitArg& arg)
{
    NW_G3D_ASSERT(arg.IsValid());

    int numCurve = arg.GetMaxCurveCount();

    // サイズ計算
    int idx = 0;
    chunk[idx++].size = sizeof(CameraAnimResult);
    chunk[idx++].size = arg.IsContextEnabled() ? Align(sizeof(AnimFrameCache) * numCurve) : 0;
    NW_G3D_ASSERT(idx == NUM_CHUNK);

    CalcOffset(chunk, NUM_CHUNK);
}

size_t CameraAnimObj::CalcBufferSize(const InitArg& arg)
{
    Sizer& sizer = arg.GetSizer();
    sizer.Calc(arg);
    return sizer.GetTotalSize();
}

bool CameraAnimObj::Init(const InitArg& arg, void* pBuffer, size_t bufferSize)
{
    NW_G3D_ASSERT_NOT_NULL(pBuffer);
    NW_G3D_WARNING(IsAligned(pBuffer, BUFFER_ALIGNMENT), "pBuffer must be aligned.");

    Sizer& sizer = arg.GetSizer();
    if (!sizer.IsValid())
    {
        // キャッシュが残っていない場合は再計算する。
        sizer.Calc(arg);
    }
    if (sizer.GetTotalSize() > bufferSize)
    {
        // バッファが必要なサイズに満たない場合は失敗。
        return false;
    }

    int numCurve = arg.GetMaxCurveCount();

    // メンバの初期化。
    void* ptr = pBuffer;
    SetBufferPtr(pBuffer);
    // フレーム関係はリソース設定時に初期化
    m_pRes = NULL;
    GetContext().Init(sizer.GetPtr<AnimFrameCache>(ptr, Sizer::FRAMECACHE_ARRAY), numCurve);
    SetResultBuffer(sizer.GetPtr(ptr, Sizer::RESULT_BUFFER));

    return true;
}

void CameraAnimObj::SetResource(ResCameraAnim* pRes)
{
    NW_G3D_ASSERT_NOT_NULL(pRes);

    m_pRes = pRes;

    bool loop = (pRes->ref().flag & AnimFlag::PLAYPOLICY_LOOP) != 0;
    ResetFrameCtrl(pRes->GetFrameCount(), loop);
    GetContext().SetCurveCount(pRes->GetCurveCount());
    CameraAnimObj::ClearResult();
}

void CameraAnimObj::ClearResult()
{
    NW_G3D_ASSERT_NOT_NULL(m_pRes);

    m_pRes->Init(GetResult());
}

void CameraAnimObj::Calc()
{
    NW_G3D_PERF_LEVEL1_FUNC();
    NW_G3D_ASSERT_NOT_NULL(m_pRes);

    if (IsFrameChanged())
    {
        float frame = GetFrameCtrl().GetFrame();
        AnimContext& context = GetContext();
        if (context.IsFrameCacheValid())
        {
            m_pRes->Eval(GetResult(), frame, context.GetFrameCacheArray(0));
        }
        else
        {
            m_pRes->Eval(GetResult(), frame);
        }
        UpdateLastFrame();
    }
}

//--------------------------------------------------------------------------------------------------

void LightAnimObj::Sizer::Calc(const InitArg& arg)
{
    NW_G3D_ASSERT(arg.IsValid());

    int numCurve = arg.GetMaxCurveCount();

    // サイズ計算
    int idx = 0;
    chunk[idx++].size = sizeof(LightAnimResult);
    chunk[idx++].size = arg.IsContextEnabled() ? Align(sizeof(AnimFrameCache) * numCurve) : 0;
    NW_G3D_ASSERT(idx == NUM_CHUNK);

    CalcOffset(chunk, NUM_CHUNK);
}

size_t LightAnimObj::CalcBufferSize(const InitArg& arg)
{
    Sizer& sizer = arg.GetSizer();
    sizer.Calc(arg);
    return sizer.GetTotalSize();
}

bool LightAnimObj::Init(const InitArg& arg, void* pBuffer, size_t bufferSize)
{
    NW_G3D_ASSERT_NOT_NULL(pBuffer);
    NW_G3D_WARNING(IsAligned(pBuffer, BUFFER_ALIGNMENT), "pBuffer must be aligned.");

    Sizer& sizer = arg.GetSizer();
    if (!sizer.IsValid())
    {
        // キャッシュが残っていない場合は再計算する。
        sizer.Calc(arg);
    }
    if (sizer.GetTotalSize() > bufferSize)
    {
        // バッファが必要なサイズに満たない場合は失敗。
        return false;
    }

    int numCurve = arg.GetMaxCurveCount();

    // メンバの初期化。
    void* ptr = pBuffer;
    SetBufferPtr(pBuffer);
    // フレーム関係はリソース設定時に初期化
    m_pRes = NULL;
    GetContext().Init(sizer.GetPtr<AnimFrameCache>(ptr, Sizer::FRAMECACHE_ARRAY), numCurve);
    SetResultBuffer(sizer.GetPtr(ptr, Sizer::RESULT_BUFFER));

    return true;
}

void LightAnimObj::SetResource(ResLightAnim* pRes)
{
    NW_G3D_ASSERT_NOT_NULL(pRes);

    m_pRes = pRes;

    bool loop = (pRes->ref().flag & AnimFlag::PLAYPOLICY_LOOP) != 0;
    ResetFrameCtrl(pRes->GetFrameCount(), loop);
    GetContext().SetCurveCount(pRes->GetCurveCount());
    LightAnimObj::ClearResult();
}

void LightAnimObj::ClearResult()
{
    NW_G3D_ASSERT_NOT_NULL(m_pRes);

    m_pRes->Init(GetResult());
}

void LightAnimObj::Calc()
{
    NW_G3D_PERF_LEVEL1_FUNC();
    NW_G3D_ASSERT_NOT_NULL(m_pRes);

    if (IsFrameChanged())
    {
        float frame = GetFrameCtrl().GetFrame();
        AnimContext& context = GetContext();
        if (context.IsFrameCacheValid())
        {
            m_pRes->Eval(GetResult(), frame, context.GetFrameCacheArray(0));
        }
        else
        {
            m_pRes->Eval(GetResult(), frame);
        }
        UpdateLastFrame();
    }
}

//--------------------------------------------------------------------------------------------------

void FogAnimObj::Sizer::Calc(const InitArg& arg)
{
    NW_G3D_ASSERT(arg.IsValid());

    int numCurve = arg.GetMaxCurveCount();

    // サイズ計算
    int idx = 0;
    chunk[idx++].size = sizeof(FogAnimResult);
    chunk[idx++].size = arg.IsContextEnabled() ? Align(sizeof(AnimFrameCache) * numCurve) : 0;
    NW_G3D_ASSERT(idx == NUM_CHUNK);

    CalcOffset(chunk, NUM_CHUNK);
}

size_t FogAnimObj::CalcBufferSize(const InitArg& arg)
{
    Sizer& sizer = arg.GetSizer();
    sizer.Calc(arg);
    return sizer.GetTotalSize();
}

bool FogAnimObj::Init(const InitArg& arg, void* pBuffer, size_t bufferSize)
{
    NW_G3D_ASSERT_NOT_NULL(pBuffer);
    NW_G3D_WARNING(IsAligned(pBuffer, BUFFER_ALIGNMENT), "pBuffer must be aligned.");

    Sizer& sizer = arg.GetSizer();
    if (!sizer.IsValid())
    {
        // キャッシュが残っていない場合は再計算する。
        sizer.Calc(arg);
    }
    if (sizer.GetTotalSize() > bufferSize)
    {
        // バッファが必要なサイズに満たない場合は失敗。
        return false;
    }

    int numCurve = arg.GetMaxCurveCount();

    // メンバの初期化。
    void* ptr = pBuffer;
    SetBufferPtr(pBuffer);
    // フレーム関係はリソース設定時に初期化
    m_pRes = NULL;
    GetContext().Init(sizer.GetPtr<AnimFrameCache>(ptr, Sizer::FRAMECACHE_ARRAY), numCurve);
    SetResultBuffer(sizer.GetPtr(ptr, Sizer::RESULT_BUFFER));

    return true;
}

void FogAnimObj::SetResource(ResFogAnim* pRes)
{
    NW_G3D_ASSERT_NOT_NULL(pRes);

    m_pRes = pRes;

    bool loop = (pRes->ref().flag & AnimFlag::PLAYPOLICY_LOOP) != 0;
    ResetFrameCtrl(pRes->GetFrameCount(), loop);
    GetContext().SetCurveCount(pRes->GetCurveCount());
    FogAnimObj::ClearResult();
}

void FogAnimObj::ClearResult()
{
    NW_G3D_ASSERT_NOT_NULL(m_pRes);

    m_pRes->Init(GetResult());
}

void FogAnimObj::Calc()
{
    NW_G3D_PERF_LEVEL1_FUNC();
    NW_G3D_ASSERT_NOT_NULL(m_pRes);

    if (IsFrameChanged())
    {
        float frame = GetFrameCtrl().GetFrame();
        AnimContext& context = GetContext();
        if (context.IsFrameCacheValid())
        {
            m_pRes->Eval(GetResult(), frame, context.GetFrameCacheArray(0));
        }
        else
        {
            m_pRes->Eval(GetResult(), frame);
        }
        UpdateLastFrame();
    }
}

//--------------------------------------------------------------------------------------------------

void SceneAnimHelper::CalcAimCameraMtx(Mtx34* pMtx, const CameraAnimResult* pResult)
{
    NW_G3D_ASSERT_NOT_NULL(pMtx);
    NW_G3D_ASSERT_NOT_NULL(pResult);

    const Vec3& pos = *Vec3::Cast(pResult->view.pos);
    const Vec3& aim = *Vec3::Cast(pResult->view.aim);
    Vec3 back;
    back.Sub(pos, aim);

    if (back.x == 0.0f && back.z == 0.0f)
    {
        // カメラと注視点の XZ 座標が同じ場合ツイストは定義されない。
        pMtx->m00 = 1.0f;
        pMtx->m01 = 0.0f;
        pMtx->m02 = 0.0f;
        pMtx->m03 = -pos.x;

        pMtx->m10 = 0.0f;
        pMtx->m11 = 0.0f;

        pMtx->m20 = 0.0f;
        pMtx->m22 = 0.0f;

        if( back.y <= 0.0f )
        {
            // 真上を見る。
            pMtx->m12 = 1.0f;
            pMtx->m13 = -pos.z;

            pMtx->m21 = -1.0f;
            pMtx->m23 = pos.y;
        }
        else
        {
            // 真下を見る。
            pMtx->m12 = -1.0f;
            pMtx->m13 = pos.z;

            pMtx->m21 = 1.0f;
            pMtx->m23 = -pos.y;
        }
    }
    else
    {
        float st, ct;
        Vec3 right, up, rightTwist, upTwist;

        right.Set(back.z, 0.0f, -back.x); // right = cross(camUp(0, 1, 0), back)
        right.Normalize(right);
        back.Normalize(back);
        up.Cross(back, right);

        Math::SinCos(&st, &ct, pResult->view.twist);

        //
        // r軸, u軸をru平面上でcameraTwistだけ半時計回りに回転させて rightTwist を求める。
        // right.y == 0であることに注意。
        //
        rightTwist.x = st * up.x + ct * right.x;
        rightTwist.y = st * up.y;
        rightTwist.z = st * up.z + ct * right.z;

        upTwist.x    = ct * up.x - st * right.x;
        upTwist.y    = ct * up.y;
        upTwist.z    = ct * up.z - st * right.z;

        // rightTwist
        pMtx->m00 = rightTwist.x;
        pMtx->m01 = rightTwist.y;
        pMtx->m02 = rightTwist.z;
        pMtx->m03 = -Vec3::Dot(pos, rightTwist);

        // upTwist
        pMtx->m10 = upTwist.x;
        pMtx->m11 = upTwist.y;
        pMtx->m12 = upTwist.z;
        pMtx->m13 = -Vec3::Dot(pos, upTwist);

        // look
        pMtx->m20 = back.x;
        pMtx->m21 = back.y;
        pMtx->m22 = back.z;
        pMtx->m23 = -Vec3::Dot(pos, back);
    }
}

void SceneAnimHelper::CalcRotateCameraMtx(Mtx34* pMtx, const CameraAnimResult* pResult)
{
    NW_G3D_ASSERT_NOT_NULL(pMtx);
    NW_G3D_ASSERT_NOT_NULL(pResult);

    const Vec3& pos = *Vec3::Cast(pResult->view.pos);
    const Vec3& rotate = *Vec3::Cast(pResult->view.rotate);
    float sx, sy, sz, cx, cy, cz;
    Math::SinCos(&sx, &cx, rotate.x);
    Math::SinCos(&sy, &cy, rotate.y);
    Math::SinCos(&sz, &cz, rotate.z);

    // Z軸, X軸, Y軸の順番で回転させている。
    Vec3 right, up, back;

    right.x = sx * sy * sz + cy * cz;
    right.y = cx * sz;
    right.z = sx * cy * sz - sy * cz;

    up.x    = sx * sy * cz - cy * sz;
    up.y    = cx * cz;
    up.z    = sx * cy * cz + sy * sz;

    back.x  = cx * sy;
    back.y  = - sx;
    back.z  = cx * cy;

    pMtx->m00 = right.x;
    pMtx->m01 = right.y;
    pMtx->m02 = right.z;
    pMtx->m03 = -Vec3::Dot(pos, right);

    pMtx->m10 = up.x;
    pMtx->m11 = up.y;
    pMtx->m12 = up.z;
    pMtx->m13 = -Vec3::Dot(pos, up);

    pMtx->m20 = back.x;
    pMtx->m21 = back.y;
    pMtx->m22 = back.z;
    pMtx->m23 = -Vec3::Dot(pos, back);
}

void SceneAnimHelper::CalcOrthoProjMtx(Mtx44* pMtx, const CameraAnimResult* pResult)
{
    NW_G3D_ASSERT_NOT_NULL(pMtx);
    NW_G3D_ASSERT_NOT_NULL(pResult);

    float halfHeight = pResult->proj.height;
    float halfWidth = pResult->proj.aspect * halfHeight;
    float nearZ =  pResult->proj.nearZ;
    float farZ = pResult->proj.farZ;
    pMtx->Ortho(-halfWidth, halfWidth, -halfHeight, halfHeight, nearZ, farZ);
}

void SceneAnimHelper::CalcPerspProjMtx(Mtx44* pMtx, const CameraAnimResult* pResult)
{
    NW_G3D_ASSERT_NOT_NULL(pMtx);
    NW_G3D_ASSERT_NOT_NULL(pResult);

    float fovy = pResult->proj.fovy;
    float aspect = pResult->proj.aspect;
    float nearZ =  pResult->proj.nearZ;
    float farZ = pResult->proj.farZ;
    pMtx->Perspective(fovy, aspect, nearZ, farZ);
}

void SceneAnimHelper::CalcOrthoProjTexMtx(Mtx34* pMtx, const CameraAnimResult* pResult)
{
    NW_G3D_ASSERT_NOT_NULL(pMtx);
    NW_G3D_ASSERT_NOT_NULL(pResult);

    float halfHeight = pResult->proj.height;
    float halfWidth = pResult->proj.aspect * halfHeight;
    pMtx->TexProjOrtho(-halfWidth, halfWidth, -halfHeight, halfHeight);
}

void SceneAnimHelper::CalcPerspProjTexMtx(Mtx34* pMtx, const CameraAnimResult* pResult)
{
    NW_G3D_ASSERT_NOT_NULL(pMtx);
    NW_G3D_ASSERT_NOT_NULL(pResult);

    float fovy = pResult->proj.fovy;
    float aspect = pResult->proj.aspect;
    pMtx->TexProjPerspective(fovy, aspect);
}

}} // namespace nw::g3d
