﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d/g3d_MaterialObj.h>
#include <algorithm>

namespace nw { namespace g3d {

void MaterialObj::Sizer::Calc(const InitArg& arg)
{
    NW_G3D_ASSERTMSG(arg.GetBufferingCount() > 0, "%s\n", NW_G3D_RES_GET_NAME(arg.GetResource(), GetName()));
    ResMaterial* pRes = arg.GetResource();
    int numDirtyFlag = pRes->GetShaderParamCount();
    size_t dirtyFlagBufferSize = Align(numDirtyFlag, 32) >> 3; // 1 パラメータあたり 1 ビット。
    int numBuffering = arg.GetBufferingCount();

    // サイズ計算
    int idx = 0;
    chunk[idx++].size = dirtyFlagBufferSize;
    chunk[idx++].size = numBuffering > 1 ? dirtyFlagBufferSize * numBuffering : 0;
    chunk[idx++].size = Align(pRes->GetSrcParamSize());
    chunk[idx++].size = sizeof(ResTexture*) * pRes->GetTextureCount();
    NW_G3D_ASSERTMSG(idx == NUM_CHUNK, "%s\n", NW_G3D_RES_GET_NAME(arg.GetResource(), GetName()));

    CalcOffset(chunk, NUM_CHUNK);
}

//--------------------------------------------------------------------------------------------------

size_t MaterialObj::CalcBufferSize(const InitArg& arg)
{
    Sizer& sizer = arg.GetSizer();
    sizer.Calc(arg);
    return sizer.GetTotalSize();
}

bool MaterialObj::Init(const InitArg& arg, void* pBuffer, size_t bufferSize)
{
    NW_G3D_ASSERTMSG(bufferSize == 0 || pBuffer, "%s\n", NW_G3D_RES_GET_NAME(arg.GetResource(), GetName()));  // ShaderParam が無い場合は 0 になる。
    NW_G3D_WARNING(IsAligned(pBuffer, BUFFER_ALIGNMENT), "pBuffer must be aligned.");

    Sizer& sizer = arg.GetSizer();
    if (!sizer.IsValid())
    {
        // キャッシュが残っていない場合は再計算する。
        sizer.Calc(arg);
    }
    if (sizer.GetTotalSize() > bufferSize)
    {
        // バッファが必要なサイズに満たない場合は失敗。
        return false;
    }

    ResMaterial* pRes = arg.GetResource();
    int numTexture = pRes->GetTextureCount();
    int numDirtyFlag = pRes->GetShaderParamCount();
    int numFlag32 = (numDirtyFlag + 31) >> 5;

    // メンバの初期化。
    void* ptr = pBuffer;
    m_pRes = pRes;
    m_Flag = BLOCK_BUFFER_SWAP;
    m_DirtyFlags = 0;
    m_NumBuffering = static_cast<u8>(arg.GetBufferingCount());
    m_NumFlag32 = static_cast<u16>(numFlag32);
    m_pDirtyFlagArray = sizer.GetPtr<u32>(ptr, Sizer::DIRTY_FLAG_ARRAY);
    m_pBufferFlagArray = sizer.GetPtr<u32>(ptr, Sizer::BUFFER_FLAG_ARRAY);
    memset(&m_MatBlock, 0, sizeof(GfxBuffer));
    m_pSrcParam = sizer.GetPtr(ptr, Sizer::SRC_PARAM_ARRAY);
    memcpy(m_pSrcParam, pRes->ref().ofsSrcParam.to_ptr(), pRes->GetSrcParamSize());
    m_ppTextureArray = sizer.GetPtr<ResTexture*>(ptr, Sizer::TEXTURE_ARRAY);
    m_pUserPtr = NULL;
    m_pBufferPtr = pBuffer;
    m_pBlockBuffer = NULL;

    if (m_NumBuffering == 1)
    {
        // 単一バッファの場合はメインのダーティフラグで判定します。
        m_pBufferFlagArray = m_pDirtyFlagArray;
    }

    for (int idxTexture = 0; idxTexture < numTexture; ++idxTexture)
    {
        m_ppTextureArray[idxTexture] = pRes->GetTextureRef(idxTexture)->Get();
    }

    InitDependPointer();

    return true;
}

size_t MaterialObj::CalcBlockBufferSize() const
{
    return Align(m_pRes->GetRawParamSize(), BLOCK_BUFFER_ALIGNMENT) * m_NumBuffering;
}

bool MaterialObj::SetupBlockBuffer(void* pBuffer, size_t bufferSize)
{
    NW_G3D_ASSERTMSG(bufferSize == 0 || pBuffer, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    NW_G3D_ASSERT_ADDR_ALIGNMENT_DETAIL(pBuffer, BLOCK_BUFFER_ALIGNMENT, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    NW_G3D_ASSERTMSG((m_Flag & BLOCK_BUFFER_VALID) == 0, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

    size_t size = CalcBlockBufferSize();

    if (size > bufferSize)
    {
        // バッファが必要なサイズに満たない場合は失敗。
        return false;
    }

    m_pBlockBuffer = pBuffer;

    // アプリケーションがデフォルト値で UniformBlock を埋められるよう、
    // ここではシェーダパラメータを変換せず、最初の CalcMatBlock で変換する。
    ResetDirtyFlags();
    // シェーダがパラメータを持たないケースでは、m_pDirtyFlagArray=NULLのためチェックする。
    if (m_NumBuffering > 1 && m_pDirtyFlagArray)
    {
        // 単一バッファの場合はメインのダーティフラグで判定します。
        std::fill_n(m_pDirtyFlagArray + m_NumFlag32, m_NumFlag32 * m_NumBuffering, 0);
    }

    // size は複数バッファ分のサイズなので 1 枚分に変換して SetData します。
    m_MatBlock.SetData(pBuffer, size / m_NumBuffering, m_NumBuffering);
    m_MatBlock.Setup();
    m_Flag |= BLOCK_BUFFER_VALID;

    return true;
}

void MaterialObj::CleanupBlockBuffer()
{
    NW_G3D_ASSERTMSG(m_Flag & BLOCK_BUFFER_VALID, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

    m_MatBlock.Cleanup();
    m_Flag &= ~BLOCK_BUFFER_VALID;
}

void MaterialObj::CalcMatBlock(int bufferIndex /*= 0*/)
{
    NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(bufferIndex, m_NumBuffering, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

    if (m_Flag & PARAM_DIRTY)
    {
        if (m_NumBuffering > 1) // 単一バッファの場合はメインのダーティフラグで判定します。
        {
            // バッファ毎のダーティフラグへ伝搬します。
            for (int idxFlag32 = 0; idxFlag32 < m_NumFlag32; ++idxFlag32)
            {
                bit32 flag32 = m_pDirtyFlagArray[idxFlag32];
                for (int idxBuffer = 0; idxBuffer < m_NumBuffering; ++idxBuffer)
                {
                    m_pBufferFlagArray[idxBuffer * m_NumFlag32 + idxFlag32] |= flag32;
                }
                m_pDirtyFlagArray[idxFlag32] = 0;
            }
        }

        // 全バッファのフラグを立てます。
        m_DirtyFlags = bit8(~0);
        m_Flag ^= PARAM_DIRTY;
    }

    int bufferBit = 0x1 << bufferIndex;
    if (!(m_DirtyFlags & bufferBit) && !m_pRes->HasVolatile())
    {
        return;
    }

    // bufferIndex のバッファとダーティフラグで更新します。
    void* pBuffer = m_MatBlock.GetData(bufferIndex);
    if (IsBlockSwapEnabled())
    {
        ConvertDirtyParams<true>(pBuffer, &m_pBufferFlagArray[bufferIndex * m_NumFlag32]);
    }
    else
    {
        ConvertDirtyParams<false>(pBuffer, &m_pBufferFlagArray[bufferIndex * m_NumFlag32]);
    }
    m_DirtyFlags &= ~bufferBit;

    m_MatBlock.DCFlush(bufferIndex);
}

template <bool swap>
void MaterialObj::ConvertParams(void* pBuffer)
{
    if (m_pRes->GetRawParamSize() <= 0)
    {
        return;
    }

    int numParam = m_pRes->GetShaderParamCount();
    for (int idxParam = 0; idxParam < numParam; ++idxParam)
    {
        ResShaderParam* pParam = m_pRes->GetShaderParam(idxParam);
        s32 offset = pParam->GetOffset();
        if (offset >= 0)
        {
            void* pDst = AddOffset(pBuffer, offset);
            const void* pSrc = AddOffset(m_pSrcParam, pParam->GetSrcOffset());
            pParam->Convert<swap>(pDst, pSrc, m_pUserPtr);
        }
    }
}

template <bool swap>
void MaterialObj::ConvertDirtyParams(void* pBuffer, bit32* pDirtyFlagArray)
{
    int numDirtyFlag = m_pRes->GetShaderParamCount();
    const bit32* pVolatileFlagArray = m_pRes->ref().ofsVolatileFlag.to_ptr<const bit32>();

    for (int idxFlag32 = 0, numFlag32 = (numDirtyFlag + 31) >> 5;
        idxFlag32 < numFlag32; ++idxFlag32)
    {
        bit32 flag32 = pDirtyFlagArray[idxFlag32];
        bit32 flagVolatile = pVolatileFlagArray[idxFlag32];
        flag32 |= flagVolatile;
        while (flag32)
        {
            int bitIndex = 31 - CountLeadingZeros(flag32);
            int paramIndex = (idxFlag32 << 5) + bitIndex;
            ResShaderParam* pParam = GetResShaderParam(paramIndex);
            NW_G3D_ASSERTMSG(pParam->GetOffset() >= 0, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName())); // オフセットが負ならフラグは立たない。
            void* pDst = AddOffset(pBuffer, pParam->GetOffset());
            const void* pSrc = AddOffset(m_pSrcParam, pParam->GetSrcOffset());
            pParam->Convert<swap>(pDst, pSrc, m_pUserPtr);
            flag32 ^= 0x1 << bitIndex;
        }
        pDirtyFlagArray[idxFlag32] = flag32;
    }
}

template <bool swap>
void MaterialObj::ConvertVolatileParams(void* pBuffer)
{
    int numDirtyFlag = m_pRes->GetShaderParamCount();

    // 外部データを参照するパラメータはダーティフラグでは更新を判定できないので、
    // 常に計算しなおす。
    for (int idxParam = numDirtyFlag, end = m_pRes->GetShaderParamCount();
        idxParam < end; ++idxParam)
    {
        ResShaderParam* pParam = GetResShaderParam(idxParam);
        s32 offset = pParam->GetOffset();
        if (offset >= 0)
        {
            void* pDst = AddOffset(pBuffer, offset);
            const void* pSrc = AddOffset(m_pSrcParam, pParam->GetSrcOffset());
            pParam->Convert<swap>(pDst, pSrc, m_pUserPtr);
        }
    }
}

void MaterialObj::ResetDirtyFlags()
{
    int numDirtyFlag = m_pRes->GetShaderParamCount();
    memset(m_pDirtyFlagArray, 0, Align(numDirtyFlag, 32) >> 3); // 1 パラメータあたり 1 ビット。

    int numFlag32 = Align(numDirtyFlag, 32) >> 5;
    for (int idxFlag32 = 0; idxFlag32 < numFlag32; ++idxFlag32)
    {
        u32 flag32 = 0;
        int numBit = std::min(32, numDirtyFlag - idxFlag32 * 32);
        for (int bitShift = 0; bitShift < numBit; ++bitShift)
        {
            ResShaderParam* pParam = GetResShaderParam(idxFlag32 * 32 + bitShift);
            // UniformBlock 内のオフセットが解決されていないパラメータは変換できないので、
            // 解決されているパラメータのみダーティフラグを立てる。
            if (pParam->GetOffset() >= 0)
            {
                flag32 |= 0x1 << bitShift;
            }
        }
        m_pDirtyFlagArray[idxFlag32] = flag32;
    }
    m_Flag |= PARAM_DIRTY;
}

void MaterialObj::InitDependPointer()
{
    for (int idxShaderParam = 0, numShaderParam = m_pRes->GetShaderParamCount();
        idxShaderParam < numShaderParam; ++idxShaderParam)
    {
        ResShaderParam *pResShaderParam = m_pRes->GetShaderParam(idxShaderParam);
        int dependIndex = pResShaderParam->GetDependIndex();
        if (dependIndex != idxShaderParam)
        {
            size_t ofsToPtr = pResShaderParam->GetSrcSize() - sizeof(bit32);
            void** ptr = AddOffset<void*>(m_pSrcParam, pResShaderParam->GetSrcOffset() + ofsToPtr);
            *ptr = AddOffset(m_pSrcParam, m_pRes->GetShaderParam(dependIndex)->GetSrcOffset());
        }
    }
}

}} // namespace nw::g3d
