﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d/g3d_Billboard.h>

namespace nw { namespace g3d {

void Billboard::CalcWorld(Mtx34* pWorldView, const Mtx34& view, const Mtx34& world)
{
    // ワールドビュー行列の回転成分を以下の優先順位に従って変更する
    //   1. ワールド座標系の Z 軸がカメラの視線軸と平行になるようにする
    //   2. Y 軸の向きは通常のワールドビュー行列と同じまま維持する

/*
    vx, vy, vz 算出の未省略のコードは
        Mtx34 wv;
        wv.Mul(view, world);
        Vec3 vx, vy, vz;
        vy.Set(wv.m01,  wv.m11,  wv.m21);
        vz.Set(0.0f,    0.0f,    1.0f);

        vy.Normalize(vy);       // 外積計算後に正規化するため不要
        vz.Normalize(vz);       //

        vx.Cross(vy, vz);       // 計算簡略化可能 vx={vy.y, -vy.x, 0.0f}
                                // この処理はSetSR 内で行っている
        vx.Normalize(vx);       // y,zが正規化されているため不要

        vy.Cross(vz, vx);       // 計算簡略化可能 vy={-vx.y, vx.x, 0.0f}
                                //                  ={ vy.x, vy.y, 0.0f}
                                // よって vy.z を 0.0f とする事で不要
        vx.Normalize(vx);       // x,zが正規化されているため不要
    である。
*/

    Vec3 vy;
    vy.x = view.m00 * world.m01 + view.m01 * world.m11 + view.m02 * world.m21;
    vy.y = view.m10 * world.m01 + view.m11 * world.m11 + view.m12 * world.m21;
    vy.z = 0.0f;

    if (!( vy.Normalize(vy) > 0.0f ))
    {
        // 'world' ビルボードの Y 軸が視線軸と平行です
        NW_G3D_WARNING( false, "Y axis of a 'world' billboard is parallel to eye axis." );
        vy.Set(0.0f, 1.0f, 0.0f);
    }

    SetR(pWorldView, vy);
}

void Billboard::CalcWorldViewpoint(Mtx34* pWorldView, const Mtx34& view, const Mtx34& world)
{
    // ワールドビュー行列の回転成分を以下の優先順位に従って変更する
    //   1. ワールド座標系の Z 軸がカメラの方を向くようにする
    //   2. Y 軸の向きは通常のワールドビュー行列と同じまま維持する

/*
    vx, vy, vz 算出の未省略のコードは
        Mtx34 wv;
        wv.Mul(view, world);
        Vec3 vx, vy, vz;
        vy.Set( wv.m01,  wv.m11,  wv.m21);
        vz.Set(-wv.m03, -wv.m13, -wv.m23);

        vy.Normalize(vy);       // 外積計算後に正規化するため不要
        vz.Normalize(vz);       //

        vx.Cross(vy, vz);       //
        vx.Normalize(vx);       //

        vy.Cross(vz, vx);       //
        vx.Normalize(vx);       // x,zが正規化されているため不要
    である。
*/

    Vec3 vx, vy, vz;
    vy.x = view.m00 * world.m01 + view.m01 * world.m11 + view.m02 * world.m21;
    vy.y = view.m10 * world.m01 + view.m11 * world.m11 + view.m12 * world.m21;
    vy.z = view.m20 * world.m01 + view.m21 * world.m11 + view.m22 * world.m21;
    vz.x = -pWorldView->m03;
    vz.y = -pWorldView->m13;
    vz.z = -pWorldView->m23;

    if (!( vz.Normalize(vz) > 0.0f ))
    {
        // カメラがモデルと同じ座標に位置しています
        NW_G3D_WARNING( false, "The camera is locating at a 'world_viewpoint' billboard." );
        vz.Set(0.0f, 0.0f, 1.0f);
    }

    vx.Cross(vy, vz);
    if (vx.Normalize(vx) > 0.0f)
    {
        vy.Cross(vz, vx);
    }
    else
    {
        // カメラが 'world_viewpoint' ビルボードの Y 軸上に位置しています
        NW_G3D_WARNING( false, "The camera is locating on Y axis of a 'world_viewpoint' billboard." );
        vx.Set(vz.z, 0.0f, -vz.x); // (0, 1, 0) と vz の外積
        if (vx.Normalize(vx) > 0.0f)
        {
            vy.Cross(vz, vx);
        }
        else
        {
            // z 軸が (0, 1, 0) と平行
            vx.Set(1.0f, 0.0f, 0.0f);
            vy.Set(0.0f, 0.0f, -vz.y); // vz と vx の外積
        }
    }

    SetR(pWorldView, vx, vy, vz);
}

void Billboard::CalcYAxis(Mtx34* pWorldView, const Mtx34& view, const Mtx34& world)
{
    // ワールドビュー行列の回転成分を以下の優先順位に従って変更する
    //   1. Y 軸の向きは通常のワールドビュー行列と同じまま維持する
    //   2. ワールド座標系の Z 軸がカメラの視線軸と平行になるようにする
/*
    vx, vy, vz 算出の未省略のコードは
        Mtx34 wv;
        wv.Mul(view, world);
        Vec3 vx, vy, vz;
        vy.Set(wv.m01,  wv.m11,  wv.m21);
        vz.Set(0.0f,    0.0f,    1.0f);

        vy.Normalize(vy);       // 外積計算後に正規化するため不要
        vz.Normalize(vz);       //

        vx.Cross(vy, vz);       // 計算簡略化可能 vx={vy.y, -vy.x, 0.0f}
                                // よって vx をあらかじめ指定することで不要
        vx.Normalize(vx);       //

        vz.Cross(vx, vy);       //
        vz.Normalize(vz);       // x,yが正規化されているため不要
    である。
*/

    Vec3 vx, vy, vz;
    vy.x = view.m00 * world.m01 + view.m01 * world.m11 + view.m02 * world.m21;
    vy.y = view.m10 * world.m01 + view.m11 * world.m11 + view.m12 * world.m21;
    vy.z = view.m20 * world.m01 + view.m21 * world.m11 + view.m22 * world.m21;
    vx.Set(vy.y, -vy.x, 0.0f); // vy と vz の外積

    if (!( vy.Normalize(vy) > 0.0f ))
    {
        // 行列が縮退しています
        NW_G3D_WARNING( false, "Y axis of a 'yaxis' billboard is degenerated." );
        vy.Set(0.0f, 1.0f, 0.0f);
        vx.Set(1.0f, 0.0f, 0.0f); // vy と vz の外積
    }

    if (vx.Normalize(vx) > 0.0f)
    {
        vz.Cross(vx, vy);
    }
    else
    {
        // 'yaxis' ビルボードの Y 軸が視線軸と平行です
        NW_G3D_WARNING( false, "Y axis of a 'yaxis' billboard is parallel to eye axis." );
        vx.Set(1.0f, 0.0f, 0.0f);
        vz.Set(0.0f, -vy.z, 0.0f); // vx と vy の外積
    }

    SetR(pWorldView, vx, vy, vz);
}

void Billboard::CalcYAxisViewpoint(Mtx34* pWorldView, const Mtx34& view, const Mtx34& world)
{
    // ワールドビュー行列の回転成分を以下の優先順位に従って変更する
    //   1. Y 軸の向きは通常のワールドビュー行列と同じまま維持する
    //   2. モデル座標系の Z 軸がカメラの方を向くようにする

/*
    vx, vy, vz 算出の未省略のコードは
        Mtx34 wv;
        wv.Mul(view, world);
        Vec3 vx, vy, vz;
        vy.Set( wv.m01,  wv.m11,  wv.m21);
        vz.Set(-wv.m03, -wv.m13, -wv.m23);

        vy.Normalize(vy);       //
        vz.Normalize(vz);       // 外積計算後に正規化するため不要

        vx.Cross(vy, vz);       //
        vx.Normalize(vx);       //

        vz.Cross(vx, vy);       //
        vz.Normalize(vz);       // x,yが正規化されているため不要
    である。
*/

    Vec3 vx, vy, vz;
    vy.x = view.m00 * world.m01 + view.m01 * world.m11 + view.m02 * world.m21;
    vy.y = view.m10 * world.m01 + view.m11 * world.m11 + view.m12 * world.m21;
    vy.z = view.m20 * world.m01 + view.m21 * world.m11 + view.m22 * world.m21;
    vz.x = -pWorldView->m03;
    vz.y = -pWorldView->m13;
    vz.z = -pWorldView->m23;

    if (!( vy.Normalize(vy) > 0.0f ))
    {
        // 行列が縮退しています
        NW_G3D_WARNING( false, "Y axis of a 'yaxis_viewpoint' billboard is degenerated." );
        vy.Set(0.0f, 1.0f, 0.0f);
    }

    vx.Cross(vy, vz);
    if (vx.Normalize(vx) > 0.0f)
    {
        vz.Cross(vx, vy);
    }
    else
    {
        // カメラが 'yaxis_viewpoint' ビルボードの Y 軸上に位置しています
        NW_G3D_WARNING( false, "The camera is locating on Y axis of a 'yaxis_viewpoint' billboard." );
        vx.Set(vy.y, -vy.x, 0.0f); // vy と (0, 0, 1) の外積
        if (vx.Normalize(vx) > 0.0f)
        {
            vz.Cross(vx, vy);
        }
        else
        {
            // y 軸が (0, 0, 1) と平行
            vx.Set(1.0f, 0.0f, 0.0f);
            vz.Set(0.0f, -vy.z, 0.0f); // vx と vy の外積
        }
    }

    SetR(pWorldView, vx, vy, vz);
}

void Billboard::CalcScreen(Mtx34* pWorldView, const Mtx34& local)
{
    // ワールドビュー行列の回転成分を以下の優先順位に従って変更する
    //   1. ワールド座標系の Z 軸がカメラの視線軸と平行になるようにする
    //   2. ワールド座標系の Y 軸がビュー座標系のY軸と平行になるようにする

/*
    vx, vy, vz 算出の未省略のコードは
        Mtx34 wv;
        wv.Mul(view, world);
        Vec3 vx, vy, vz;
        vy.Set(local.m01, local.m11, local.m21);
        vz.Set(     0.0f,      0.0f,      1.0f);

        vy.Normalize(vy);       //
        vz.Normalize(vz);       // 既に正規化されているので不要

        vx.Cross(vy, vz);       // 計算簡略化可能 vx={vy.y, -vy.x, 0.0f}
                                // この処理はSetSR 内で行っている
        vx.Normalize(vx);       // y,zが正規化されているため不要

        vy.Cross(vz, vx);       // 計算簡略化可能 vy={-vx.y, vx.x, 0.0f}
                                //                  ={ vy.x, vy.y, 0.0f}
                                // よって vy.z を 0.0f とする事で不要
        vx.Normalize(vx);       // x,zが正規化されているため不要
    である。
*/

    Vec3 vy;
    vy.x = local.m01;
    vy.y = local.m11;
    vy.z = 0.0f;

    if (!( vy.Normalize(vy) > 0.0f ))
    {
        // 'screen' ビルボードの Y 軸が視線軸と平行です
        NW_G3D_WARNING( false, "Y axis of a 'screen' billboard is parallel to eye axis." );
        vy.Set(0.0f, 1.0f, 0.0f);
    }

    SetR(pWorldView, vy);
}

void Billboard::CalcScreenViewpoint(Mtx34* pWorldView, const Mtx34& local)
{
    // ワールドビュー行列の回転成分を以下の優先順位に従って変更する
    //   1. ワールド座標系の Z 軸がカメラの方を向くようにする
    //   2. ワールド座標系の Y 軸がビュー座標系のY軸と平行になるようにする

/*
    vx, vy, vz 算出の未省略のコードは
        Vec3 vx, vy, vz;
        vy.Set(local.m01, local.m11, local.m21);
        vz.Set(  -wv.m03,   -wv.m13,   -wv.m23);

        vy.Normalize(vy);       // 外積計算後に正規化するため不要
        vz.Normalize(vz);       //

        vx.Cross(vy, vz);       //
        vx.Normalize(vx);       //

        vy.Cross(vz, vx);       //
        vx.Normalize(vx);       // x,zが正規化されているため不要
    である。
*/

    Vec3 vx, vy, vz;
    vy.x = local.m01;
    vy.y = local.m11;
    vy.z = 0.0f;
    vz.x = -pWorldView->m03;
    vz.y = -pWorldView->m13;
    vz.z = -pWorldView->m23;

    if (!( vz.Normalize(vz) > 0.0f ))
    {
        // カメラがモデルと同じ座標に位置しています
        NW_G3D_WARNING( false, "The camera is locating at a 'screen_viewpoint' billboard." );
        vz.Set(0.0f, 0.0f, 1.0f);
    }

    vx.Cross(vy, vz);
    if (vx.Normalize(vx) > 0.0f)
    {
        vy.Cross(vz, vx);
    }
    else
    {
        // カメラが 'screen_viewpoint' ビルボードの Y 軸上に位置しています
        NW_G3D_WARNING( false, "The camera is locating on Y axis of a 'screen_viewpoint' billboard." );
        vx.Set(vz.z, 0.0f, -vz.x); // (0, 1, 0) と vz の外積
        if (vx.Normalize(vx) > 0.0f)
        {
            vy.Cross(vz, vx);
        }
        else
        {
            // z 軸が (0, 1, 0) と平行
            vx.Set(1.0f, 0.0f, 0.0f);
            vy.Set(0.0f, 0.0f, -vz.y); // vz と vx の外積
        }
    }

    SetR(pWorldView, vx, vy, vz);
}

void Billboard::Calc(bit32 billboardMode, Mtx34* pWorldView,
    const Mtx34& view, const Mtx34& world, const Mtx34& local)
{
    NW_G3D_ASSERT(ResBone::BILLBOARD_WORLD_VIEWVECTOR <= billboardMode &&
        billboardMode <= ResBone::BILLBOARD_MAX);
    // Billboard クラスの関数群はスケールを維持しないので退避する。
    Vec3 s;
    s.x = Mtx34::ExtractBaseScale(world, 0);
    s.y = Mtx34::ExtractBaseScale(world, 1);
    s.z = Mtx34::ExtractBaseScale(world, 2);

    static void (* const pCalcBillboard[])(Mtx34*, const Mtx34&, const Mtx34&, const Mtx34&) = {
        &Billboard::CalcWorld,
        &Billboard::CalcWorldViewpoint,
        &Billboard::CalcScreen,
        &Billboard::CalcScreenViewpoint,
        &Billboard::CalcYAxis,
        &Billboard::CalcYAxisViewpoint
    };

    int funcIndex = (billboardMode - ResBone::BILLBOARD_WORLD_VIEWVECTOR) >> ResBone::BILLBOARD_SHIFT;
    pCalcBillboard[funcIndex](pWorldView, view, world, local);

    // 退避しておいたスケールを復元する。
    pWorldView->ScaleBases(*pWorldView, s);
}

void Billboard::SetR(Mtx34* pWorldView, const Vec3& vy)
{
    pWorldView->m00 = vy.y;     // = vx.x
    pWorldView->m01 = vy.x;
    pWorldView->m02 = 0.0f;

    pWorldView->m10 = -vy.x;    // = vx.y
    pWorldView->m11 =  vy.y;
    pWorldView->m12 = 0.0f;

    pWorldView->m20 = 0.0f;
    pWorldView->m21 = 0.0f;
    pWorldView->m22 = 1.0f;
}

void Billboard::SetR(Mtx34* pWorldView, const Vec3& vx, const Vec3& vy, const Vec3& vz)
{
    pWorldView->m00 = vx.x;
    pWorldView->m01 = vy.x;
    pWorldView->m02 = vz.x;

    pWorldView->m10 = vx.y;
    pWorldView->m11 = vy.y;
    pWorldView->m12 = vz.y;

    pWorldView->m20 = vx.z;
    pWorldView->m21 = vy.z;
    pWorldView->m22 = vz.z;
}

}} // namespace nw::g3d
