﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d/fnd/g3d_GX2Utility.h>
#include <cafe/gx2/gx2Enum.h>
#include <nw/g3d/ut/g3d_Inlines.h>
#include <nw/g3d/fnd/g3d_GLUtility.h>

#ifndef NW_STRIP_GL // GL をはがす際には TexConv も切り離す。
#ifdef _WIN32
#ifndef WIN32
#define WIN32
#endif
#endif
#pragma warning(push)
#pragma warning(disable:4100)
#pragma warning(disable:4063)
#if NW_G3D_IS_HOST_WIN
#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <Windows.h>
#include <texUtils.h>
#endif // NW_G3D_IS_HOST_WIN
#pragma warning(pop)
#include <sdk_ver.h>
#if CAFE_OS_SDK_VERSION < 20903
using namespace TexUtils;
#endif
#endif // NW_STRIP_GL

#include <functional>
#include <algorithm>

namespace nw { namespace g3d { namespace fnd {

#ifndef NW_STRIP_GL

GLPrimitiveType s_GLPrimType[] =
{
    { 0xFFFFFFFF,                   0xFFFFFFFF      },
    { GL_POINTS,                    GL_POINTS       },
    { GL_LINES,                     GL_LINES        },
    { GL_LINE_STRIP,                GL_LINES        },
    { GL_TRIANGLES,                 GL_TRIANGLES    },
    { GL_TRIANGLE_FAN,              GL_TRIANGLES    },
    { GL_TRIANGLE_STRIP,            GL_TRIANGLES    },
    { 0xFFFFFFFF,                   0xFFFFFFFF      },
    { 0xFFFFFFFF,                   0xFFFFFFFF      },
    { 0xFFFFFFFF,                   0xFFFFFFFF      },
#if !defined(NW_G3D_IS_GL_ES)
    { GL_LINES_ADJACENCY,           GL_LINES        },
    { GL_LINE_STRIP_ADJACENCY,      GL_LINES        },
    { GL_TRIANGLES_ADJACENCY,       GL_TRIANGLES    },
    { GL_TRIANGLE_STRIP_ADJACENCY,  GL_TRIANGLES    },
#else
    { 0xFFFFFFFF,                   0xFFFFFFFF      },
    { 0xFFFFFFFF,                   0xFFFFFFFF      },
    { 0xFFFFFFFF,                   0xFFFFFFFF      },
    { 0xFFFFFFFF,                   0xFFFFFFFF      },
#endif
    { 0xFFFFFFFF,                   0xFFFFFFFF      },
    { 0xFFFFFFFF,                   0xFFFFFFFF      },
    { 0xFFFFFFFF,                   0xFFFFFFFF      },
    { GL_LINES,                     GL_LINES        }, // 対応するものが無いので数が一致するラインにしておく。
    { GL_LINE_LOOP,                 GL_LINES        },
#if !defined(NW_G3D_IS_GL_ES)
    { GL_QUADS,                     GL_TRIANGLES    },
    { GL_QUAD_STRIP,                GL_TRIANGLES    },
#else
    { 0xFFFFFFFF,                   0xFFFFFFFF      },
    { 0xFFFFFFFF,                   0xFFFFFFFF      },
#endif
    // tessellation は未対応
    { 0xFFFFFFFF,                   0xFFFFFFFF      },
    { 0xFFFFFFFF,                   0xFFFFFFFF      },
    { 0xFFFFFFFF,                   0xFFFFFFFF      },
    { 0xFFFFFFFF,                   0xFFFFFFFF      },
    { 0xFFFFFFFF,                   0xFFFFFFFF      },
    { 0xFFFFFFFF,                   0xFFFFFFFF      }
};

GLSurfaceFormat s_GLSurfaceFormat[] =
{
    { GX2_SURFACE_FORMAT_INVALID,                  GL_INVALID_ENUM,      GL_INVALID_ENUM,          GL_INVALID_ENUM                 },
    { GX2_SURFACE_FORMAT_TC_R8_UNORM,              GL_R8,                GL_RED,                   GL_UNSIGNED_BYTE                },
    { GX2_SURFACE_FORMAT_TC_R8_UINT,               GL_R8UI,              GL_RED_INTEGER,           GL_UNSIGNED_BYTE                },
    { GX2_SURFACE_FORMAT_TC_R8_SNORM,              GL_R8_SNORM,          GL_RED,                   GL_BYTE                         },
    { GX2_SURFACE_FORMAT_TC_R8_SINT,               GL_R8I,               GL_RED_INTEGER,           GL_BYTE                         },
#if !defined(NW_G3D_IS_GL_ES)
    { GX2_SURFACE_FORMAT_TCD_R16_UNORM,            GL_R16,               GL_RED,                   GL_UNSIGNED_SHORT               },
#else
    { GX2_SURFACE_FORMAT_TCD_R16_UNORM,            GL_INVALID_ENUM,      GL_RED,                   GL_UNSIGNED_SHORT               },
#endif
    { GX2_SURFACE_FORMAT_TC_R16_UINT,              GL_R16UI,             GL_RED_INTEGER,           GL_UNSIGNED_SHORT               },
#if !defined(NW_G3D_IS_GL_ES)
    { GX2_SURFACE_FORMAT_TC_R16_SNORM,             GL_R16_SNORM,         GL_RED,                   GL_SHORT                        },
#else
    { GX2_SURFACE_FORMAT_TC_R16_SNORM,             GL_INVALID_ENUM,      GL_RED,                   GL_SHORT                        },
#endif
    { GX2_SURFACE_FORMAT_TC_R16_SINT,              GL_R16I,              GL_RED_INTEGER,           GL_SHORT                        },
    { GX2_SURFACE_FORMAT_TC_R16_FLOAT,             GL_R16F,              GL_RED,                   GL_HALF_FLOAT                   },
    { GX2_SURFACE_FORMAT_TC_R8_G8_UNORM,           GL_RG8,               GL_RG,                    GL_UNSIGNED_BYTE                },
    { GX2_SURFACE_FORMAT_TC_R8_G8_UINT,            GL_RG8UI,             GL_RG_INTEGER,            GL_UNSIGNED_BYTE                },
    { GX2_SURFACE_FORMAT_TC_R8_G8_SNORM,           GL_RG8_SNORM,         GL_RG,                    GL_BYTE                         },
    { GX2_SURFACE_FORMAT_TC_R8_G8_SINT,            GL_RG8I,              GL_RG_INTEGER,            GL_BYTE                         },
#if !defined(NW_G3D_IS_GL_ES)
    { GX2_SURFACE_FORMAT_TCS_R5_G6_B5_UNORM,       GL_RGB,               GL_RGB,                   GL_UNSIGNED_SHORT_5_6_5_REV     },
    { GX2_SURFACE_FORMAT_TC_R5_G5_B5_A1_UNORM,     GL_RGB5_A1,           GL_RGBA,                  GL_UNSIGNED_SHORT_1_5_5_5_REV   },
    { GX2_SURFACE_FORMAT_TC_R4_G4_B4_A4_UNORM,     GL_RGBA4,             GL_RGBA,                  GL_UNSIGNED_SHORT_4_4_4_4_REV   },
    { GX2_SURFACE_FORMAT_TC_A1_B5_G5_R5_UNORM,     GL_RGBA8,             GL_ABGR_EXT,              GL_UNSIGNED_SHORT_5_5_5_1       },
#else
    { GX2_SURFACE_FORMAT_TCS_R5_G6_B5_UNORM,       GL_RGB,               GL_RGB,                   GL_INVALID_ENUM                 },
    { GX2_SURFACE_FORMAT_TC_R5_G5_B5_A1_UNORM,     GL_RGB5_A1,           GL_RGBA,                  GL_INVALID_ENUM                 },
    { GX2_SURFACE_FORMAT_TC_R4_G4_B4_A4_UNORM,     GL_RGBA4,             GL_RGBA,                  GL_INVALID_ENUM                 },
    { GX2_SURFACE_FORMAT_TC_A1_B5_G5_R5_UNORM,     GL_RGBA8,             GL_INVALID_ENUM,          GL_INVALID_ENUM                 },
#endif
    { GX2_SURFACE_FORMAT_TC_R32_UINT,              GL_R32UI,             GL_RED_INTEGER,           GL_UNSIGNED_INT                 },
    { GX2_SURFACE_FORMAT_TC_R32_SINT,              GL_R32I,              GL_RED_INTEGER,           GL_UNSIGNED_INT                 },
    { GX2_SURFACE_FORMAT_TCD_R32_FLOAT,            GL_R32F,              GL_RED,                   GL_FLOAT                        },
#if !defined(NW_G3D_IS_GL_ES)
    { GX2_SURFACE_FORMAT_TC_R16_G16_UNORM,         GL_RG16,              GL_RG,                    GL_UNSIGNED_SHORT               },
#else
    { GX2_SURFACE_FORMAT_TC_R16_G16_UNORM,         GL_INVALID_ENUM,      GL_RG,                    GL_UNSIGNED_SHORT               },
#endif
    { GX2_SURFACE_FORMAT_TC_R16_G16_UINT,          GL_RG16UI,            GL_RG_INTEGER,            GL_UNSIGNED_SHORT               },
#if !defined(NW_G3D_IS_GL_ES)
    { GX2_SURFACE_FORMAT_TC_R16_G16_SNORM,         GL_RG16_SNORM,        GL_RG,                    GL_SHORT                        },
#else
    { GX2_SURFACE_FORMAT_TC_R16_G16_SNORM,         GL_INVALID_ENUM,      GL_RG,                    GL_SHORT                        },
#endif
    { GX2_SURFACE_FORMAT_TC_R16_G16_SINT,          GL_RG16I,             GL_RG_INTEGER,            GL_SHORT                        },
    { GX2_SURFACE_FORMAT_TC_R16_G16_FLOAT,         GL_RG16F,             GL_RG,                    GL_HALF_FLOAT                   },
#if !defined(NW_G3D_IS_GL_ES)
    { GX2_SURFACE_FORMAT_TC_R11_G11_B10_FLOAT,     GL_RGB16F,            GL_RGB,                   GL_UNSIGNED_INT_10F_11F_11F_REV },
#else
    { GX2_SURFACE_FORMAT_TC_R11_G11_B10_FLOAT,     GL_R11F_G11F_B10F,    GL_RGB,                   GL_UNSIGNED_INT_10F_11F_11F_REV },
#endif
    { GX2_SURFACE_FORMAT_TCS_R10_G10_B10_A2_UNORM, GL_RGB10_A2,          GL_RGBA,                  GL_UNSIGNED_INT_2_10_10_10_REV  },
    { GX2_SURFACE_FORMAT_TC_R10_G10_B10_A2_UINT,   GL_RGB10_A2UI,        GL_RGBA_INTEGER,          GL_UNSIGNED_INT_2_10_10_10_REV  },
    { GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM,    GL_RGBA8,             GL_RGBA,                  GL_UNSIGNED_BYTE                },
    { GX2_SURFACE_FORMAT_TC_R8_G8_B8_A8_UINT,      GL_RGBA8UI,           GL_RGBA_INTEGER,          GL_UNSIGNED_BYTE                },
    { GX2_SURFACE_FORMAT_TC_R8_G8_B8_A8_SNORM,     GL_RGBA8_SNORM,       GL_RGBA,                  GL_BYTE                         },
    { GX2_SURFACE_FORMAT_TC_R8_G8_B8_A8_SINT,      GL_RGBA8I,            GL_RGBA_INTEGER,          GL_BYTE                         },
    { GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_SRGB,     GL_SRGB8_ALPHA8,      GL_RGBA,                  GL_UNSIGNED_BYTE                },
#if !defined(NW_G3D_IS_GL_ES)
    { GX2_SURFACE_FORMAT_TCS_A2_B10_G10_R10_UNORM, GL_RGBA16,            GL_ABGR_EXT,              GL_UNSIGNED_INT_10_10_10_2      },
    { GX2_SURFACE_FORMAT_TC_A2_B10_G10_R10_UINT,   GL_RGBA16UI,          GL_ABGR_EXT,              GL_UNSIGNED_INT_10_10_10_2      },
#else
    { GX2_SURFACE_FORMAT_TCS_A2_B10_G10_R10_UNORM, GL_INVALID_ENUM,      GL_INVALID_ENUM,          GL_INVALID_ENUM                 },
    { GX2_SURFACE_FORMAT_TC_A2_B10_G10_R10_UINT,   GL_RGBA16UI,          GL_INVALID_ENUM,          GL_INVALID_ENUM                 },
#endif
    { GX2_SURFACE_FORMAT_TC_R32_G32_UINT,          GL_RG32UI,            GL_RG_INTEGER,            GL_UNSIGNED_INT                 },
    { GX2_SURFACE_FORMAT_TC_R32_G32_SINT,          GL_RG32I,             GL_RG_INTEGER,            GL_INT                          },
    { GX2_SURFACE_FORMAT_TC_R32_G32_FLOAT,         GL_RG32F,             GL_RG,                    GL_FLOAT                        },
#if !defined(NW_G3D_IS_GL_ES)
    { GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_UNORM, GL_RGBA16,            GL_RGBA,                  GL_UNSIGNED_SHORT               },
#else
    { GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_UNORM, GL_INVALID_ENUM,      GL_RGBA,                  GL_UNSIGNED_SHORT               },
#endif
    { GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_UINT,  GL_RGBA16UI,          GL_RGBA_INTEGER,          GL_UNSIGNED_SHORT               },
#if !defined(NW_G3D_IS_GL_ES)
    { GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_SNORM, GL_RGBA16_SNORM,      GL_RGBA,                  GL_SHORT                        },
#else
    { GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_SNORM, GL_INVALID_ENUM,      GL_RGBA,                  GL_SHORT                        },
#endif
    { GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_SINT,  GL_RGBA16I,           GL_RGBA_INTEGER,          GL_SHORT                        },
    { GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_FLOAT, GL_RGBA16F,           GL_RGBA,                  GL_HALF_FLOAT                   },
    { GX2_SURFACE_FORMAT_TC_R32_G32_B32_A32_UINT,  GL_RGBA32UI,          GL_RGBA_INTEGER,          GL_UNSIGNED_INT                 },
    { GX2_SURFACE_FORMAT_TC_R32_G32_B32_A32_SINT,  GL_RGBA32I,           GL_RGBA_INTEGER,          GL_INT                          },
    { GX2_SURFACE_FORMAT_TC_R32_G32_B32_A32_FLOAT, GL_RGBA32F,           GL_RGBA,                  GL_FLOAT                        },
#if !defined(NW_G3D_IS_GL_ES)
    { GX2_SURFACE_FORMAT_T_BC1_UNORM,              GL_COMPRESSED_RGBA_S3TC_DXT1_EXT,         GL_INVALID_ENUM, GL_INVALID_ENUM      },
    { GX2_SURFACE_FORMAT_T_BC1_SRGB,               GL_COMPRESSED_SRGB_ALPHA_S3TC_DXT1_EXT,   GL_INVALID_ENUM, GL_INVALID_ENUM      },
    { GX2_SURFACE_FORMAT_T_BC2_UNORM,              GL_COMPRESSED_RGBA_S3TC_DXT3_EXT,         GL_INVALID_ENUM, GL_INVALID_ENUM      },
    { GX2_SURFACE_FORMAT_T_BC2_SRGB,               GL_COMPRESSED_SRGB_ALPHA_S3TC_DXT3_EXT,   GL_INVALID_ENUM, GL_INVALID_ENUM      },
    { GX2_SURFACE_FORMAT_T_BC3_UNORM,              GL_COMPRESSED_RGBA_S3TC_DXT5_EXT,         GL_INVALID_ENUM, GL_INVALID_ENUM      },
    { GX2_SURFACE_FORMAT_T_BC3_SRGB,               GL_COMPRESSED_SRGB_ALPHA_S3TC_DXT5_EXT,   GL_INVALID_ENUM, GL_INVALID_ENUM      },
    { GX2_SURFACE_FORMAT_T_BC4_UNORM,              GL_COMPRESSED_RED_RGTC1,                  GL_INVALID_ENUM, GL_INVALID_ENUM      },
    { GX2_SURFACE_FORMAT_T_BC4_SNORM,              GL_COMPRESSED_SIGNED_RED_RGTC1,           GL_INVALID_ENUM, GL_INVALID_ENUM      },
    { GX2_SURFACE_FORMAT_T_BC5_UNORM,              GL_COMPRESSED_RG_RGTC2,                   GL_INVALID_ENUM, GL_INVALID_ENUM      },
    { GX2_SURFACE_FORMAT_T_BC5_SNORM,              GL_COMPRESSED_SIGNED_RG_RGTC2,            GL_INVALID_ENUM, GL_INVALID_ENUM      },
#else
    { GX2_SURFACE_FORMAT_T_BC1_UNORM,              GL_INVALID_ENUM,                          GL_INVALID_ENUM, GL_INVALID_ENUM      },
    { GX2_SURFACE_FORMAT_T_BC1_SRGB,               GL_INVALID_ENUM,                          GL_INVALID_ENUM, GL_INVALID_ENUM      },
    { GX2_SURFACE_FORMAT_T_BC2_UNORM,              GL_INVALID_ENUM,                          GL_INVALID_ENUM, GL_INVALID_ENUM      },
    { GX2_SURFACE_FORMAT_T_BC2_SRGB,               GL_INVALID_ENUM,                          GL_INVALID_ENUM, GL_INVALID_ENUM      },
    { GX2_SURFACE_FORMAT_T_BC3_UNORM,              GL_INVALID_ENUM,                          GL_INVALID_ENUM, GL_INVALID_ENUM      },
    { GX2_SURFACE_FORMAT_T_BC3_SRGB,               GL_INVALID_ENUM,                          GL_INVALID_ENUM, GL_INVALID_ENUM      },
    { GX2_SURFACE_FORMAT_T_BC4_UNORM,              GL_INVALID_ENUM,                          GL_INVALID_ENUM, GL_INVALID_ENUM      },
    { GX2_SURFACE_FORMAT_T_BC4_SNORM,              GL_INVALID_ENUM,                          GL_INVALID_ENUM, GL_INVALID_ENUM      },
    { GX2_SURFACE_FORMAT_T_BC5_UNORM,              GL_INVALID_ENUM,                          GL_INVALID_ENUM, GL_INVALID_ENUM      },
    { GX2_SURFACE_FORMAT_T_BC5_SNORM,              GL_INVALID_ENUM,                          GL_INVALID_ENUM, GL_INVALID_ENUM      },
#endif
    { GX2_SURFACE_FORMAT_LAST,                     GL_INVALID_ENUM,                          GL_INVALID_ENUM, GL_INVALID_ENUM      },
};

GLSurfaceFormat s_GLDepthFormat[] =
{
    { GX2_SURFACE_FORMAT_INVALID,                  GL_INVALID_ENUM,      GL_INVALID_ENUM,    GL_INVALID_ENUM                       },
    { GX2_SURFACE_FORMAT_TCD_R16_UNORM,            GL_DEPTH_COMPONENT16, GL_DEPTH_COMPONENT, GL_UNSIGNED_SHORT                     },
#if !defined(NW_G3D_IS_GL_ES)
    { GX2_SURFACE_FORMAT_TCD_R32_FLOAT,            GL_DEPTH_COMPONENT32, GL_DEPTH_COMPONENT, GL_FLOAT                              },
#else
    { GX2_SURFACE_FORMAT_TCD_R32_FLOAT,            GL_DEPTH_COMPONENT32F,GL_DEPTH_COMPONENT, GL_FLOAT                              },
#endif
    { GX2_SURFACE_FORMAT_D_D24_S8_UNORM,           GL_DEPTH24_STENCIL8,  GL_DEPTH_STENCIL,   GL_UNSIGNED_INT_24_8                  },
    { GX2_SURFACE_FORMAT_D_D32_FLOAT_S8_UINT_X24,  GL_DEPTH32F_STENCIL8, GL_DEPTH_STENCIL,   GL_FLOAT_32_UNSIGNED_INT_24_8_REV     },
    { GX2_SURFACE_FORMAT_LAST,                     GL_INVALID_ENUM,      GL_INVALID_ENUM,    GL_INVALID_ENUM                       },
};

GLAttribFormat s_GLAttribFormat[] =
{
    { GX2_ATTRIB_FORMAT_8_UNORM,                   GL_UNSIGNED_BYTE,                1, GL_TRUE  },
    { GX2_ATTRIB_FORMAT_8_UINT,                    GL_UNSIGNED_BYTE,                1, GL_FALSE },
    { GX2_ATTRIB_FORMAT_8_SNORM,                   GL_BYTE,                         1, GL_TRUE  },
    { GX2_ATTRIB_FORMAT_8_SINT,                    GL_BYTE,                         1, GL_FALSE },
    { GX2_ATTRIB_FORMAT_8_UINT_TO_FLOAT,           GL_UNSIGNED_BYTE,                1, GL_TRUE  },
    { GX2_ATTRIB_FORMAT_8_SINT_TO_FLOAT,           GL_BYTE,                         1, GL_TRUE  },
    { GX2_ATTRIB_FORMAT_4_4_UNORM,                 GL_UNSIGNED_BYTE,                2, GL_TRUE  },
    { GX2_ATTRIB_FORMAT_16_UNORM,                  GL_UNSIGNED_SHORT,               1, GL_TRUE  },
    { GX2_ATTRIB_FORMAT_16_UINT,                   GL_UNSIGNED_SHORT,               1, GL_FALSE },
    { GX2_ATTRIB_FORMAT_16_SNORM,                  GL_SHORT,                        1, GL_TRUE  },
    { GX2_ATTRIB_FORMAT_16_SINT,                   GL_SHORT,                        1, GL_FALSE },
    { GX2_ATTRIB_FORMAT_16_FLOAT,                  GL_HALF_FLOAT,                   1, GL_TRUE  },
    { GX2_ATTRIB_FORMAT_16_UINT_TO_FLOAT,          GL_UNSIGNED_SHORT,               1, GL_TRUE  },
    { GX2_ATTRIB_FORMAT_16_SINT_TO_FLOAT,          GL_SHORT,                        1, GL_TRUE  },
    { GX2_ATTRIB_FORMAT_8_8_UNORM,                 GL_UNSIGNED_BYTE,                2, GL_TRUE  },
    { GX2_ATTRIB_FORMAT_8_8_UINT,                  GL_UNSIGNED_BYTE,                2, GL_FALSE },
    { GX2_ATTRIB_FORMAT_8_8_SNORM,                 GL_BYTE,                         2, GL_TRUE  },
    { GX2_ATTRIB_FORMAT_8_8_SINT,                  GL_BYTE,                         2, GL_FALSE },
    { GX2_ATTRIB_FORMAT_8_8_UINT_TO_FLOAT,         GL_UNSIGNED_BYTE,                2, GL_TRUE  },
    { GX2_ATTRIB_FORMAT_8_8_SINT_TO_FLOAT,         GL_BYTE,                         2, GL_TRUE  },
    { GX2_ATTRIB_FORMAT_32_UINT,                   GL_UNSIGNED_INT,                 1, GL_FALSE },
    { GX2_ATTRIB_FORMAT_32_SINT,                   GL_INT,                          1, GL_FALSE },
    { GX2_ATTRIB_FORMAT_32_FLOAT,                  GL_FLOAT,                        1, GL_FALSE },
    { GX2_ATTRIB_FORMAT_16_16_UNORM,               GL_UNSIGNED_SHORT,               2 ,GL_TRUE  },
    { GX2_ATTRIB_FORMAT_16_16_UINT,                GL_UNSIGNED_SHORT,               2, GL_FALSE },
    { GX2_ATTRIB_FORMAT_16_16_SNORM,               GL_SHORT,                        2, GL_TRUE  },
    { GX2_ATTRIB_FORMAT_16_16_SINT,                GL_SHORT,                        2, GL_FALSE },
    { GX2_ATTRIB_FORMAT_16_16_FLOAT,               GL_HALF_FLOAT,                   2, GL_FALSE },
    { GX2_ATTRIB_FORMAT_16_16_UINT_TO_FLOAT,       GL_UNSIGNED_SHORT,               2, GL_TRUE  },
    { GX2_ATTRIB_FORMAT_16_16_SINT_TO_FLOAT,       GL_SHORT,                        2, GL_TRUE  },
    { GX2_ATTRIB_FORMAT_10_11_11_FLOAT,            GL_UNSIGNED_INT_10F_11F_11F_REV, 3, GL_FALSE },
    { GX2_ATTRIB_FORMAT_8_8_8_8_UNORM,             GL_UNSIGNED_BYTE,                4, GL_TRUE  },
    { GX2_ATTRIB_FORMAT_8_8_8_8_UINT,              GL_UNSIGNED_BYTE,                4, GL_FALSE },
    { GX2_ATTRIB_FORMAT_8_8_8_8_SNORM,             GL_BYTE,                         4, GL_TRUE  },
    { GX2_ATTRIB_FORMAT_8_8_8_8_SINT,              GL_BYTE,                         4, GL_FALSE },
    { GX2_ATTRIB_FORMAT_8_8_8_8_UINT_TO_FLOAT,     GL_UNSIGNED_BYTE,                4, GL_TRUE  },
    { GX2_ATTRIB_FORMAT_8_8_8_8_SINT_TO_FLOAT,     GL_BYTE,                         4, GL_TRUE  },
    { GX2_ATTRIB_FORMAT_10_10_10_2_UNORM,          GL_UNSIGNED_INT_2_10_10_10_REV,  4, GL_TRUE  },
    { GX2_ATTRIB_FORMAT_10_10_10_2_UINT,           GL_UNSIGNED_INT_2_10_10_10_REV,  4, GL_FALSE },
    { GX2_ATTRIB_FORMAT_10_10_10_2_SNORM,          GL_INT_2_10_10_10_REV,           4, GL_TRUE  },
    { GX2_ATTRIB_FORMAT_10_10_10_2_SINT,           GL_INT_2_10_10_10_REV,           4, GL_FALSE },
    { GX2_ATTRIB_FORMAT_32_32_UINT,                GL_UNSIGNED_INT,                 2, GL_FALSE },
    { GX2_ATTRIB_FORMAT_32_32_SINT,                GL_INT,                          2, GL_FALSE },
    { GX2_ATTRIB_FORMAT_32_32_FLOAT,               GL_FLOAT,                        2, GL_FALSE },
    { GX2_ATTRIB_FORMAT_16_16_16_16_UNORM,         GL_UNSIGNED_SHORT,               4, GL_TRUE  },
    { GX2_ATTRIB_FORMAT_16_16_16_16_UINT,          GL_UNSIGNED_SHORT,               4, GL_FALSE },
    { GX2_ATTRIB_FORMAT_16_16_16_16_SNORM,         GL_SHORT,                        4, GL_TRUE  },
    { GX2_ATTRIB_FORMAT_16_16_16_16_SINT,          GL_SHORT,                        4, GL_FALSE },
    { GX2_ATTRIB_FORMAT_16_16_16_16_FLOAT,         GL_HALF_FLOAT,                   4, GL_FALSE },
    { GX2_ATTRIB_FORMAT_16_16_16_16_UINT_TO_FLOAT, GL_UNSIGNED_SHORT,               4, GL_TRUE  },
    { GX2_ATTRIB_FORMAT_16_16_16_16_SINT_TO_FLOAT, GL_SHORT,                        4, GL_TRUE  },
    { GX2_ATTRIB_FORMAT_32_32_32_UINT,             GL_UNSIGNED_INT,                 3, GL_FALSE },
    { GX2_ATTRIB_FORMAT_32_32_32_SINT,             GL_INT,                          3, GL_FALSE },
    { GX2_ATTRIB_FORMAT_32_32_32_FLOAT,            GL_FLOAT,                        3, GL_FALSE },
    { GX2_ATTRIB_FORMAT_32_32_32_32_UINT,          GL_UNSIGNED_INT,                 4, GL_FALSE },
    { GX2_ATTRIB_FORMAT_32_32_32_32_SINT,          GL_INT,                          4, GL_FALSE },
    { GX2_ATTRIB_FORMAT_32_32_32_32_FLOAT,         GL_FLOAT,                        4, GL_FALSE },
    { GX2_ATTRIB_FORMAT_LAST,                      GL_INVALID_ENUM,                 0, GL_FALSE },
};

#if NW_G3D_IS_HOST_WIN
class TexUtilsDll
{
    typedef BOOL (*TC2InitializeType)(TC2Config* pConfig);
    typedef void (*TC2DestroyType)();
    typedef BOOL (*TC2ConvertTilingType)(GX2Surface* pInSurface,
        GX2TileMode dstTileMode, u32 initialSwizzle, GX2Surface* pOutSurface);
    typedef BOOL (*TC2DestroyGX2SurfaceType)(GX2Surface* pGX2Surface);
    typedef BOOL (*TC2GetSourceSurfaceSizeType)(GX2Surface* pSurface);

public:
    TexUtilsDll()
        : m_hDll()
        , TC2Initialize()
        , TC2Destroy()
        , TC2ConvertTiling()
    {
    }

    ~TexUtilsDll()
    {
        Finalize();
    }

    void Initialize(const char* dllPath)
    {
        if ((m_hDll = LoadLibraryA(dllPath)) == NULL)
        {
            return;
        }
        TC2Initialize = GetRequiredFunction<TC2InitializeType>("TC2Initialize");
        TC2Destroy = GetRequiredFunction<TC2DestroyType>("TC2Destroy");
        TC2ConvertTiling = GetRequiredFunction<TC2ConvertTilingType>("TC2ConvertTiling");
        TC2DestroyGX2Surface = GetRequiredFunction<TC2DestroyGX2SurfaceType>("TC2DestroyGX2Surface");
        TC2GetSourceSurfaceSize = GetRequiredFunction<TC2GetSourceSurfaceSizeType>("TC2GetSourceSurfaceSize");
    }

    void Finalize()
    {
        if (m_hDll)
        {
            FreeLibrary(m_hDll);
            m_hDll = NULL;
        }
    }

    bool IsValid()
    {
        return m_hDll != NULL;
    }

    TC2InitializeType TC2Initialize;
    TC2DestroyType TC2Destroy;
    TC2ConvertTilingType TC2ConvertTiling;
    TC2DestroyGX2SurfaceType TC2DestroyGX2Surface;
    TC2GetSourceSurfaceSizeType TC2GetSourceSurfaceSize;

private:
    template< typename TFunc >
    TFunc GetRequiredFunction(const char* pFunctionName)
    {
        TFunc ret = reinterpret_cast<TFunc>(GetProcAddress(m_hDll, pFunctionName));
        NW_G3D_ASSERTMSG(ret, "failed to load function: %s", pFunctionName);
        return ret;
    }

    HMODULE m_hDll;
} g_TexUtilsDll;

const char* GetTexUtilsDllPath()
{
#if _DEBUG
    return "TexUtilsD.dll";
#else
    return "TexUtils.dll";
#endif
}
#endif

const GLPrimitiveType& FindGLPrimitiveType(GX2PrimitiveType type)
{
    return s_GLPrimType[type];
}

struct GLSurfaceCmp : public std::binary_function<GLSurfaceFormat, GX2SurfaceFormat, bool>
{
    bool operator ()(const GLSurfaceFormat& lhs, GX2SurfaceFormat rhs) const
    {
        return lhs.gx2Format == rhs;
    }
};

const GLSurfaceFormat& FindGLFormat(GX2SurfaceFormat format)
{
    return *std::find_if(
        s_GLSurfaceFormat,
        s_GLSurfaceFormat + (sizeof(s_GLSurfaceFormat) / sizeof(GLSurfaceFormat) - 1),
        std::bind2nd(GLSurfaceCmp(), format));
}

const GLSurfaceFormat& FindGLDepthFormat(GX2SurfaceFormat format)
{
    return *std::find_if(
        s_GLDepthFormat,
        s_GLDepthFormat + (sizeof(s_GLDepthFormat) / sizeof(GLSurfaceFormat) - 1),
        std::bind2nd(GLSurfaceCmp(), format));
}

struct AttribCmp : public std::binary_function<GLAttribFormat, GX2AttribFormat, bool>
{
    bool operator ()(const GLAttribFormat& lhs, GX2AttribFormat rhs) const
    {
        return lhs.gx2Format == rhs;
    }
};

const GLAttribFormat& FindGLFormat(GX2AttribFormat format)
{
    return *std::find_if(
        s_GLAttribFormat,
        s_GLAttribFormat + (sizeof(s_GLAttribFormat) / sizeof(GLAttribFormat) - 1),
        std::bind2nd(AttribCmp(), format));
}

void ConvertToGLSurface(GX2Surface& surface, int arrayLength)
{
    // NOTE: バグ回避のために GX2_SURFACE_DIM_2D_ARRAY に一旦変更してタイリング解除します。
#if NW_G3D_IS_HOST_WIN
    if (!g_TexUtilsDll.IsValid())
    {
        g_TexUtilsDll.Initialize(GetTexUtilsDllPath());
    }
    NW_G3D_ASSERTMSG(g_TexUtilsDll.IsValid(), "Failed to load GX2 texure library");

    bool changeDim = false;
    if (surface.dim == GX2_SURFACE_DIM_CUBE &&
        arrayLength > 1)
    {
        surface.dim = GX2_SURFACE_DIM_2D_ARRAY;
        changeDim = true;
    }
    GX2Surface linear = surface;
    TC2Config tc2_config = {};
    tc2_config.gbTilingConfig = 0;
    tc2_config.gpu = GPU_Cafe;
    g_TexUtilsDll.TC2Initialize( &tc2_config );
    bool isConvertSuccess = !! g_TexUtilsDll.TC2ConvertTiling(
        reinterpret_cast< ::GX2Surface* >(&surface),
        ::GX2_TILE_MODE_LINEAR_SPECIAL,
        0,
        &linear);
    NW_G3D_ASSERT(isConvertSuccess);
    (void)isConvertSuccess;

    if (changeDim)
    {
        linear.dim = surface.dim = GX2_SURFACE_DIM_CUBE;
    }

    const int heightFactor = surface.dim == GX2_SURFACE_DIM_1D ||
        surface.dim == GX2_SURFACE_DIM_1D_ARRAY ? 0 : 1;
    const int depthFactor = surface.dim == GX2_SURFACE_DIM_3D ? 1 : 0;

    for (u32 mipLevel = 0; mipLevel < surface.numMips; ++mipLevel)
    {
        void* dstImage = GetImagePtr(surface, mipLevel);
        const void* srcImage = GetImagePtr(linear, mipLevel);

        u32 width = std::max<u32>(1, surface.width >> mipLevel);
        u32 height = std::max<u32>(1, surface.height >> (mipLevel * heightFactor));
        u32 depth = std::max<u32>(1, surface.depth >> (mipLevel * depthFactor));
        u32 size = CalcImageSize(surface.format, width, height, depth);
        memcpy(dstImage, srcImage, size);
    }

    surface.tileMode = linear.tileMode;
    NW_G3D_ASSERT(surface.imageSize >= linear.imageSize);
    NW_G3D_ASSERT(surface.mipSize >= linear.mipSize);

    bool isDestroySuccess = !! g_TexUtilsDll.TC2DestroyGX2Surface(&linear);
    NW_G3D_ASSERT(isDestroySuccess);
    (void)isDestroySuccess;
#endif // NW_G3D_IS_HOST_WIN
}

void CalcSurfaceSizeAndAlignment(GX2Surface& surface)
{
#if NW_G3D_IS_HOST_WIN
    if (!g_TexUtilsDll.IsValid())
    {
        g_TexUtilsDll.Initialize(GetTexUtilsDllPath());
    }
    if (g_TexUtilsDll.IsValid())
    {
        TC2Config tc2_config = {};
        tc2_config.gbTilingConfig = 0;
        tc2_config.gpu = GPU_Cafe;
        g_TexUtilsDll.TC2Initialize( &tc2_config );

        bool success = !!g_TexUtilsDll.TC2GetSourceSurfaceSize(reinterpret_cast< GX2Surface* >(&surface));
        (void)success;
        NW_G3D_ASSERT(success);
    }
    else
#endif // NW_G3D_IS_HOST_WIN
    {
        surface.imageSize = CalcImageSize(surface.format, surface.width, surface.height, surface.depth);
    }
}

#endif // NW_STRIP_GL

u32 CalcImageSize(GX2SurfaceFormat format, u32 width, u32 height, u32 depth)
{
    u32 size = 0;
    u32 type = format & 0xFF;
    if (IsCompressed(format))
    {
        width  = ( width + 3 ) >> 2;
        height = ( height + 3 ) >> 2;
        size = (type == 0x31 || type == 0x34) ? 8 : 16; // (BC1 BC4) or (BC2 BC3 BC5)
    }
    else
    {
        size = type == 0 ? 0 :      // GX2_SURFACE_FORMAT_INVALID
            type <= 0x02 ? 1 :      // GX2_SURFACE_FORMAT_T_4_4_UNORM
            type <= 0x0C ? 2 :      // GX2_SURFACE_FORMAT_TC_5_5_5_1_UNORM
            type <= 0x1C ? 4 :      // GX2_SURFACE_FORMAT_T_X24_8_X32_UINT
            type <= 0x20 ? 8 :      // GX2_SURFACE_FORMAT_TC_16_16_16_16_FLOAT
            type <= 0x23 ? 16 :     // GX2_SURFACE_FORMAT_TC_32_32_32_32_FLOAT
            type >= 0x2F ? 12 :     // GX2_SURFACE_FORMAT_T_32_32_32_UINT
            type == 0x2B ? 4 : 0;   // GX2_SURFACE_FORMAT_T_5_9_9_9_SHAREDEXP
    }
    return width * height * depth * size;
}

bool IsCompressed(GX2SurfaceFormat format)
{
    return (format & 0xFF) >= (GX2_SURFACE_FORMAT_T_BC1_UNORM & 0xFF);
}

void* GetImagePtr(GX2Surface& surface, u32 mipLevel)
{
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
    return
        mipLevel == 0 ? surface.imagePtr :
        mipLevel == 1 ? surface.mipPtr :
        surface.mipPtr ? AddOffset(surface.mipPtr, surface.mipOffset[mipLevel - 1]) : NULL;
#else
    NW_G3D_UNUSED(surface);
    NW_G3D_UNUSED(mipLevel);
    return NULL;
#endif
}

const void* GetImagePtr(const GX2Surface& surface, u32 mipLevel)
{
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
    return
        mipLevel == 0 ? surface.imagePtr :
        mipLevel == 1 ? surface.mipPtr :
        surface.mipPtr ? AddOffset(surface.mipPtr, surface.mipOffset[mipLevel - 1]) : NULL;
#else
    NW_G3D_UNUSED(surface);
    NW_G3D_UNUSED(mipLevel);
    return NULL;
#endif
}

}}} // namespace nw::g3d::fnd
