﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/g3d/edit/detail/g3d_EditDetailDefs.h>

#if NW_G3D_CONFIG_USE_HOSTIO

#include <nw/g3d/edit/g3d_IAllocator.h>

namespace nw { namespace g3d { namespace edit { namespace ut {

namespace detail {

template<typename T>
class DynamicArray
{
public:
    explicit DynamicArray(nw::g3d::edit::IAllocator* allocator, size_t alignment)
        : m_pAllocator(allocator)
        , m_Alignment(alignment)
        , m_ArrayLength(0)
        , m_ArrayLengthMax(DEFAULT_ARRAY_SIZE)
        , m_Array(NULL)
    {
        SetAllocator(allocator);
    }

    explicit DynamicArray(size_t alignment)
        : m_pAllocator(NULL)
        , m_Alignment(alignment)
        , m_ArrayLength(0)
        , m_ArrayLengthMax(DEFAULT_ARRAY_SIZE)
        , m_Array(NULL)
    {
    }

    ~DynamicArray()
    {
        if (m_pAllocator == NULL)
        {
            return;
        }

        m_pAllocator->Free(m_Array);
        m_Array = NULL;
        m_ArrayLength = 0;
        m_pAllocator = NULL;
    }

    int Size() const
    {
        return m_ArrayLength;
    }

    void Clear()
    {
        m_ArrayLength = 0;
    }

    void SetAllocator(nw::g3d::edit::IAllocator* allocator)
    {
        NW_G3D_ASSERT_NOT_NULL(allocator);
        m_pAllocator = allocator;
        m_Array = static_cast<T*>(m_pAllocator->Alloc(sizeof(T) * m_ArrayLengthMax, m_Alignment));
        NW_G3D_ASSERT_NOT_NULL(m_Array);
    }

    bool PushBack(T elem)
    {
        if (m_ArrayLength < m_ArrayLengthMax)
        {
            m_Array[m_ArrayLength] = elem;
            ++m_ArrayLength;
            return true;
        }

        if (Resize(m_ArrayLengthMax + 1))
        {
            m_Array[m_ArrayLength] = elem;
            ++m_ArrayLength;
            return true;
        }

        return false;
    }

    int IndexOf(T target) const
    {
        int index = 0;
        for (const T* iter = m_Array, *end = m_Array + m_ArrayLength; iter != end; ++iter, ++index)
        {
            if (*iter == target)
            {
                return index;
            }
        }

        return -1;
    }

    void Erase(int pos)
    {
        NW_G3D_ASSERT(pos >= 0);
        NW_G3D_ASSERT((pos + 1) <= m_ArrayLength);

        int movePos = pos + 1;
        if (movePos < m_ArrayLength)
        {
            CopyOverlap(&m_Array[pos], &m_Array[movePos], (m_ArrayLength - movePos) * sizeof(T));
        }
        m_ArrayLength = m_ArrayLength - 1;
    }

    void Erase(T target)
    {
        int index = IndexOf(target);
        if (index == -1)
        {
            return;
        }

        Erase(index);
    }

    T& operator[] (int index)
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(index, m_ArrayLength);
        return m_Array[index];
    }

    const T& operator[] (int index) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(index, m_ArrayLength);
        return m_Array[index];
    }

private:
    bool Resize(size_t size)
    {
        NW_G3D_ASSERT_NOT_NULL(m_pAllocator);

        size_t currentSize = static_cast<size_t>(m_ArrayLengthMax);
        if (currentSize >= size)
        {
            return true;
        }

        int bufferCount = m_ArrayLengthMax << 2;

        T* newPtrArray = static_cast<T*>(m_pAllocator->Alloc(sizeof(T) * bufferCount, m_Alignment));
        if (newPtrArray == NULL)
        {
            return false;
        }

        m_ArrayLengthMax = bufferCount;

        for (int i = 0; i < m_ArrayLength; ++i)
        {
            newPtrArray[i] = m_Array[i];
        }

        m_pAllocator->Free(m_Array);
        m_Array = newPtrArray;
        return true;
    }

    void* CopyOverlap(void* dst, const void* src, u32 size)
    {
#if NW_G3D_IS_HOST_CAFE
        return OSBlockMove(dst, src, size, FALSE);
#else
        return std::memmove(dst, src, size);
#endif
    }

private:
    static const int DEFAULT_ARRAY_SIZE = 8;

    nw::g3d::edit::IAllocator* m_pAllocator;
    size_t m_Alignment;
    int m_ArrayLength;
    int m_ArrayLengthMax;
    T* m_Array;
};

} // namespace detail

}}}} // namespace nw::g3d::edit::ut

#endif // NW_G3D_CONFIG_USE_HOSTIO
