﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_EditShadingModel.h"

#if NW_G3D_CONFIG_USE_HOSTIO

#include <nw/g3d/res/g3d_ResShader.h>
#include <nw/g3d/edit/g3d_IAllocator.h>
#include "g3d_EditShaderProgram.h"

namespace nw { namespace g3d { namespace edit { namespace detail {

using namespace ut::detail;

EditShadingModel::EditShadingModel(
    IAllocator* allocator,
    u32 shaderArchiveKey,
    int shadingModelIndex,
    ResShadingModel* resShadingModel,
    EditShaderArchive* editShaderArchive)
    : m_pAllocator(allocator)
    , m_pTargetResShadingModel(resShadingModel)
    , m_ShaderArchiveKey(shaderArchiveKey)
    , m_ShadingModelIndex(shadingModelIndex)
    , m_pEditShaderArchive(editShaderArchive)
    , m_pUpdateResShadingModel(NULL)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(allocator, "%s\n", NW_G3D_RES_GET_NAME(resShadingModel, GetName()));
    NW_G3D_ASSERT_NOT_NULL(resShadingModel);
    resShadingModel->ref().pCallbackParam.set_ptr(this);

#if NW_G3D_IS_HOST_WIN
    m_pOriginalGLShaderInfo = resShadingModel->ref().ofsGLShaderInfo.to_ptr();
#endif
}

bool
EditShadingModel::Init()
{
    NW_G3D_ASSERT_NOT_NULL(m_pTargetResShadingModel);

    int shaderProgramCount = m_pTargetResShadingModel->GetShaderProgramCount();

    // 編集用のシェーダプログラムインスタンスを作成
    {
        size_t bufferSize = FixedSizeArray<EditShaderProgram>::CalcBufferSize(shaderProgramCount);
        void* buffer = m_pAllocator->Alloc(bufferSize, DEFAULT_ALIGNMENT);
        if (buffer == NULL) // バッファ確保失敗
        {
            return false;
        }

        m_ShaderProgramArray.SetArrayBuffer(buffer, shaderProgramCount);
        for(int i = 0, end = m_ShaderProgramArray.Size(); i < end; ++i)
        {
            nw::g3d::res::ResShaderProgram* shaderProgram = m_pTargetResShadingModel->GetShaderProgram(i);
            EditShaderProgram* editShaderProgram = m_ShaderProgramArray.UnsafeAt(i);
            new (editShaderProgram) EditShaderProgram(m_pAllocator, m_ShaderArchiveKey, m_ShadingModelIndex, i, shaderProgram);
        }
    }
    return true;
}

void
EditShadingModel::UpdateShadingModel(ResShaderArchive* resShaderArchive)
{
    NW_G3D_EDIT_ASSERT_NOT_NULL(resShaderArchive);
    NW_G3D_EDIT_ASSERT(resShaderArchive->GetShadingModelCount() > 0);

    // PC 版の場合は共有ソースを展開する
#if NW_G3D_IS_HOST_WIN
    ResShadingModel* resShadingModel = resShaderArchive->GetShadingModel(0);
    if(m_pUpdateResShadingModel)
    {
        m_pUpdateResShadingModel->Cleanup();
    }
    m_pUpdateResShadingModel = resShadingModel;
    m_pUpdateResShadingModel->Setup();
    m_pTargetResShadingModel->ref().ofsGLShaderInfo.set_ptr(resShadingModel->ref().ofsGLShaderInfo.to_ptr());
#else
    NW_G3D_UNUSED(resShaderArchive);
#endif
}

void
EditShadingModel::Cleanup()
{
#if NW_G3D_IS_HOST_WIN
    if(m_pUpdateResShadingModel)
    {
        m_pUpdateResShadingModel->Cleanup();
        m_pUpdateResShadingModel = NULL;
    }
    m_pTargetResShadingModel->ref().ofsGLShaderInfo.set_ptr(m_pOriginalGLShaderInfo);
#endif

    {
        int size = m_ShaderProgramArray.Size();
        for (int i = 0; i < size; ++i)
        {
            EditShaderProgram* editShaderProgram = m_ShaderProgramArray.UnsafeAt(i);
            editShaderProgram->Cleanup();
        }
    }
}

void
EditShadingModel::Destroy()
{
    {
        int size = m_ShaderProgramArray.Size();
        for (int i = 0; i < size; ++i)
        {
            EditShaderProgram* editShaderProgram = m_ShaderProgramArray.UnsafeAt(i);
            editShaderProgram->Destroy();
        }
        void* buffer = m_ShaderProgramArray.GetBufferPtr();
        if (buffer != NULL)
        {
            m_ShaderProgramArray.Clear();
            m_pAllocator->Free(buffer);
        }
    }
}

}}}} // namespace nw::g3d::edit::detail

#endif // NW_G3D_CONFIG_USE_HOSTIO
