﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_EditShaderArchive.h"

#if NW_G3D_CONFIG_USE_HOSTIO

#include <nw/g3d/edit/g3d_IAllocator.h>
#include "g3d_EditShadingModel.h"
#include "g3d_EditShaderProgram.h"

namespace nw { namespace g3d { namespace edit { namespace detail {

using namespace ut::detail;

void
UpdateProgram(ResShadingModel* pShader, int programIndex)
{
    EditShadingModel* editShadingModel = pShader->ref().pCallbackParam.to_ptr<EditShadingModel>();
    NW_G3D_EDIT_ASSERT_NOT_NULL_DETAIL(editShadingModel, "%s\n", NW_G3D_RES_GET_NAME(pShader, GetName()));

    EditShaderProgram* editShaderProgram = editShadingModel->GetEditShaderProgram(programIndex);
    NW_G3D_EDIT_ASSERT_NOT_NULL_DETAIL(editShaderProgram, "%s\n", NW_G3D_RES_GET_NAME(pShader, GetName()));

    const EditShaderArchive* editShaderArchive = editShadingModel->GetEditShaderArchive();
    NW_G3D_EDIT_ASSERT_NOT_NULL_DETAIL(editShaderArchive, "%s\n", NW_G3D_RES_GET_NAME(pShader, GetName()));

    if (!editShaderProgram->GetSendInfoCallbackFlag())
    {
        editShaderArchive->CallOriginalUpdateCallback(pShader, programIndex);
        return;
    }

    ResShaderProgram* program = pShader->GetShaderProgram(programIndex);
    NW_G3D_EDIT_ASSERT_NOT_NULL_DETAIL(program, "%s\n", NW_G3D_RES_GET_NAME(pShader, GetName()));

    // ハンドル未作成の場合は元のコールバックを呼んでおく
#if NW_G3D_IS_HOST_WIN
    if (program->ref().handle == 0)
    {
        IAllocator* pAllocator = editShaderProgram->m_pAllocator;
        u32 shaderArchiveKey = editShaderProgram->m_ShaderArchiveKey;
        int shadingModelIndex = editShaderProgram->m_ShadingModelIndex;
        int shaderProgramIndex = editShaderProgram->m_ShaderProgramIndex;

        void* pGLShaderInfo = editShadingModel->m_pTargetResShadingModel->ref().ofsGLShaderInfo.to_ptr();
        editShadingModel->m_pTargetResShadingModel->ref().ofsGLShaderInfo.set_ptr(
            editShadingModel->m_pOriginalGLShaderInfo);

        editShaderProgram->Cleanup();
        editShaderProgram->Destroy();
        editShaderArchive->CallOriginalUpdateCallback(pShader, programIndex);
        new (editShaderProgram) EditShaderProgram(
            pAllocator,
            shaderArchiveKey,
            shadingModelIndex,
            shaderProgramIndex,
            program);

        editShadingModel->m_pTargetResShadingModel->ref().ofsGLShaderInfo.set_ptr(pGLShaderInfo);
    }
#endif

    editShaderProgram->ClearOptionInfo();

    const bit32* staticKey = pShader->GetStaticKey(programIndex);
    int staticOptionCount = pShader->GetStaticOptionCount();

    for (int i = 0; i < staticOptionCount; ++i)
    {
        ResShaderOption* option = pShader->GetStaticOption(i);
        const char* optionId = option->GetName();
        int choiceIndex = option->ReadStaticKey(staticKey);
        const char* choice = option->GetChoiceName(choiceIndex);
        bool result = editShaderProgram->PushOptionInfo(optionId, choice);
        NW_G3D_ASSERTMSG(result, "%s\n", NW_G3D_RES_GET_NAME(pShader, GetName()));
    }

    const bit32* dynamicKey = pShader->GetDynamicKey(programIndex);
    int dynamicOptionCount = pShader->GetDynamicOptionCount() - 1; // 最後の1つはシステム予約

    for (int i = 0; i < dynamicOptionCount; ++i)
    {
        ResShaderOption* option = pShader->GetDynamicOption(i);
        const char* optionId = option->GetName();
        int choiceIndex = option->ReadDynamicKey(dynamicKey);
        const char* choice = option->GetChoiceName(choiceIndex);
        bool result = editShaderProgram->PushOptionInfo(optionId, choice);
        NW_G3D_ASSERTMSG(result, "%s\n", NW_G3D_RES_GET_NAME(pShader, GetName()));
    }

    {
        bool result = editShaderProgram->MakeModifiedShaderProgramPacket();
        NW_G3D_ASSERTMSG(result, "%s\n", NW_G3D_RES_GET_NAME(pShader, GetName()));
    }

    program->ref().flag &= ~ResShaderProgram::UPDATE_REQURIED;
    editShaderProgram->SetSendInfoCallbackFlag(false);
}

EditShaderArchive::EditShaderArchive(IAllocator* allocator, ResShaderArchive* resShaderArchive)
    : m_pAllocator(allocator)
    , m_pTargetResShaderArchive(resShaderArchive)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(allocator, "%s\n", NW_G3D_RES_GET_NAME(resShaderArchive, GetName()));
    NW_G3D_ASSERT_NOT_NULL(resShaderArchive);

    m_pOriginalUpdateProgramCallback = m_pTargetResShaderArchive->GetUpdateProgramCallback();
}

bool
EditShaderArchive::Attach()
{
    NW_G3D_ASSERT_NOT_NULL(m_pTargetResShaderArchive);

    int shadingModelCount = m_pTargetResShaderArchive->GetShadingModelCount();
    if (shadingModelCount > 0)
    {
        size_t bufferSize = FixedSizeArray<EditShadingModel>::CalcBufferSize(shadingModelCount);
        void* buffer = m_pAllocator->Alloc(bufferSize, DEFAULT_ALIGNMENT);
        if (buffer == NULL) // バッファ確保失敗
        {
            return false;
        }

        m_ShadingModelArray.SetArrayBuffer(buffer, shadingModelCount);
        bool isFailed = false;
        for(int i = 0, end = m_ShadingModelArray.Size(); i < end; ++i)
        {
            nw::g3d::res::ResShadingModel* shadingModel = m_pTargetResShaderArchive->GetShadingModel(i);
            EditShadingModel* editShadingModel = m_ShadingModelArray.UnsafeAt(i);
            new (editShadingModel) EditShadingModel(
                m_pAllocator, reinterpret_cast<u32>(m_pTargetResShaderArchive), i,shadingModel, this);
            isFailed |= !editShadingModel->Init();
        }

        // 初期化失敗のものが一つでもあった場合は、インスタンスを破棄して失敗にする。
        if (isFailed)
        {
            Detach();
            return false;
        }
    }
    return true;
}

void
EditShaderArchive::Detach()
{
    Cleanup();
    Destroy();

    // UpdateProgramCallback をオリジナルのものに戻す
    if (m_pOriginalUpdateProgramCallback != m_pTargetResShaderArchive->GetUpdateProgramCallback())
    {
        m_pTargetResShaderArchive->SetUpdateProgramCallback(m_pOriginalUpdateProgramCallback);
    }
}

void
EditShaderArchive::Cleanup()
{
    int shadingModelSize = m_pTargetResShaderArchive->GetShadingModelCount();
    for (int i = 0; i < shadingModelSize; ++i)
    {
        EditShadingModel* editShadingModel = m_ShadingModelArray.UnsafeAt(i);
        editShadingModel->Cleanup();
    }
}

void
EditShaderArchive::Destroy()
{
    DestroyShadingModels();
}

void
EditShaderArchive::UpdateShaderProgram(int shadingModelIndex, int shaderProgramIndex, ResShaderArchive* resShaderArchive)
{
    NW_G3D_ASSERTMSG(shadingModelIndex >= 0, "%s\n", NW_G3D_RES_GET_NAME(resShaderArchive, GetName()));
    if (shadingModelIndex >= m_ShadingModelArray.Size())
    {
        return;// 範囲外なので処理しない
    }

    EditShadingModel* editShadingModel = m_ShadingModelArray.UnsafeAt(shadingModelIndex);
    editShadingModel->UpdateShadingModel(resShaderArchive);
    EditShaderProgram* editShaderProgram = editShadingModel->GetEditShaderProgram(shaderProgramIndex);
    if (editShaderProgram == NULL)
    {
        return;
    }
    editShaderProgram->UpdateShaderProgram(resShaderArchive);
}

void
EditShaderArchive::UpdateShadingModels(int shadingModelIndices[], u32 indexSize)
{
    m_pTargetResShaderArchive->SetUpdateProgramCallback(UpdateProgram);
    for (size_t i = 0; i < indexSize; ++i)
    {
        ResShadingModel* resShadingModel = m_pTargetResShaderArchive->GetShadingModel(shadingModelIndices[i]);
        EditShadingModel* editShadingModel = m_ShadingModelArray.UnsafeAt(shadingModelIndices[i]);
        int shaderProgramSize = resShadingModel->GetShaderProgramCount();
        for (int j = 0; j < shaderProgramSize; ++j)
        {
            ResShaderProgram* resShaderProgram = resShadingModel->GetShaderProgram(j);
            resShaderProgram->ref().flag |= ResShaderProgram::UPDATE_REQURIED;
            editShadingModel->GetEditShaderProgram(j)->SetSendInfoCallbackFlag(true);
        }
    }
}

void
EditShaderArchive::DestroyShadingModels()
{
    for(int i = 0, end = m_ShadingModelArray.Size(); i < end; ++i)
    {
        EditShadingModel* editShadingModel = m_ShadingModelArray.UnsafeAt(i);
        editShadingModel->Destroy();
    }

    if (void* buffer = m_ShadingModelArray.GetArrayBufferPtr())
    {
        m_ShadingModelArray.Clear();
        m_pAllocator->Free(buffer);
    }
}

void
EditShaderArchive::ClearHandles()
{
    int shadingModelSize = m_pTargetResShaderArchive->GetShadingModelCount();
    for (int i = 0; i < shadingModelSize; ++i)
    {
        ResShadingModel* shadingModel = m_pTargetResShaderArchive->GetShadingModel(i);
        int shaderProgramSize = shadingModel->GetShaderProgramCount();
        for (int j = 0; j < shaderProgramSize; ++j)
        {
            ResShaderProgram* shaderProgram = shadingModel->GetShaderProgram(j);
            shaderProgram->ref().handle = 0;
        }
    }
}

void
EditShaderArchive::SendModifiedShaderPrograms(EditSocket* socket)
{
    int shadingModelCount = m_ShadingModelArray.Size();
    for (int i = 0; i < shadingModelCount; ++i)
    {
        EditShadingModel* shadingModel = m_ShadingModelArray.UnsafeAt(i);
        int shaderProgramCount = shadingModel->GetEditShaderProgramCount();
        for (int j = 0; j < shaderProgramCount; ++j)
        {
            EditShaderProgram* shaderProgram = shadingModel->GetEditShaderProgram(j);
            shaderProgram->SendModifiedShaderProgram(socket);
        }
    }
}

}}}} // namespace nw::g3d::edit::detail

#endif // NW_G3D_CONFIG_USE_HOSTIO
